import smtpd
import asyncore
import smtplib
import email
import logging
import random
import string
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
from email.mime.nonmultipart import MIMENonMultipart
import content_types
import os
import subprocess
from tempfile import NamedTemporaryFile
from enum import Enum
from email import encoders

logging.basicConfig(level=logging.DEBUG, format="%(asctime)s %(message)s")


class SMTPfilter(smtpd.SMTPServer):
    """
    Main filter class
    """

    def __init__(self, localaddr, outaddr, tmp_path, data_size_limit=smtpd.DATA_SIZE_DEFAULT, m=None,
                 enable_smtputf8=False, decode_data=None):
        logging.info("SMTP filtering started on " + str(localaddr) + ".")
        self.out_port = outaddr[1]
        self.out_addr = outaddr[0]
        self.tmp_path = tmp_path
        super().__init__(localaddr, None, data_size_limit, m, enable_smtputf8, decode_data)

    def process_message(self, peer, mailfrom, rcpttos, data, **kwargs):
        """
        Method called after receiving e-mail 
        
        :param peer: Tuple containing ip address and port of peer
        :param mailfrom: Address of sender
        :param rcpttos: List of recipients
        :param data: Data part of message
        :param kwargs: Ignored arguments
        :return: 
        """

        msg_id = "".join([random.choice(string.digits+string.ascii_letters) for _ in range(10)])
        logging.info("Received message from"+repr(peer)+", FROM address "+repr(mailfrom)+", TO addresses " +
                     repr(rcpttos)+". Let's call it "+msg_id+".")

        message = email.message_from_string(data)

        logging.debug("Starting conversion of " + msg_id + ".")
        msg_format, msg_new = self.convert_message(message, msg_id=msg_id)
        logging.debug("Conversion of " + msg_id + " ended.")
        logging.info(msg_id+" had format "+repr(msg_format)+".")
        # message.set_payload(msg_new)

        sender = smtplib.SMTP(self.out_addr, self.out_port)
        msgas = msg_new.as_string()
        sender.sendmail(mailfrom, rcpttos, msgas)
        sender.quit()

    def convert_message(self, message, msg_id="?"):
        """
        Method controlling conversion of messages
        
        :param message: `email.message.Message` object
        :param msg_id: Id of message for logging
        :return: (Representation of original message types, Converted message)
        """

        if message.is_multipart():
            subresults = list()
            subtypes = list()
            for submessage in message.get_payload():
                sub_type, sub_result = self.convert_message(submessage, msg_id=msg_id)
                subresults.append(sub_result)
                subtypes.append(sub_type)

            result = MIMEMultipart()
            for h in [x for x in message.keys() if x not in ['MIME-Version', 'Content-Type', 'Content-Disposition']]:
                result[h] = message.get(h)  # copying headers

            print(message.keys())
            result.set_payload(subresults)
            return subtypes, result
        else:
            converted_message = None
            msg_type = message.get_content_type()
            msg_err = "This format (" + msg_type + ") is not supported or error happened.\n" \
                                                   "filename: " + repr(message.get_filename()) + "\n" \
                                                   "message id: " + msg_id + "\n"
            if msg_type == "text/plain":  # skip plaintext conversion
                return msg_type, message
            if content_types.LibreOffice.match(msg_type) is not None \
                    or content_types.office.match(msg_type) is not None:
                cmd = ["libreoffice", "--headless", "--convert-to", "pdf", "--outdir", self.tmp_path,
                       Value.ORIGINAL_FILE_NAME]
                converted_message, msg_err_n = self.run_external_filter(message, msg_id, cmd, lambda x: x + ".pdf",
                                                                        "application", "pdf")
                msg_err += msg_err_n
            elif content_types.image.match(msg_type) is not None:
                cmd = ["convert", "-resize", "1024x1024", Value.ORIGINAL_FILE_NAME, Value.CONVERTED_FILE_NAME]
                converted_message, msg_err_n = self.run_external_filter(message, msg_id, cmd, lambda x: x + ".png",
                                                                        "image", "png")
                msg_err += msg_err_n
            elif content_types.PDF.match(msg_type) is not None:
                cmd = ["pdftops", Value.ORIGINAL_FILE_NAME, Value.CONVERTED_FILE_NAME]
                converted_message, msg_err_n = self.run_external_filter(message, msg_id, cmd, lambda x: x + ".ps",
                                                                        "application", "postscript")
                cmd = ["ps2pdf", Value.ORIGINAL_FILE_NAME, Value.CONVERTED_FILE_NAME]
                converted_message, msg_err_n = self.run_external_filter(converted_message, msg_id, cmd,
                                                                        lambda x: x + ".pdf",
                                                                        "application", "pdf")
                msg_err += msg_err_n

            # add custom rules here

            if converted_message is None:  # no conversion happened
                converted_message = MIMEText(msg_err.encode(encoding='UTF-8'), "plain", "utf-8")
                converted_message.add_header('Content-Disposition', 'attachment', filename="unknown_type.txt")
            return msg_type, converted_message

    def run_external_filter(self, message, msg_id, command, f_original_to_new_filename, maintype, subtype, shell=False):
        """
        Method calling external program used for filtering
        
        :param message: Non-multipart message to convert
        :param msg_id: Id of message for logging purposes
        :param command: List of strings (or Value) containing command and its arguments.
        :param f_original_to_new_filename: Function transforming original file name to converted file name
        :param maintype: Main MIME type of converted attachment
        :param subtype: Sybtype of converted attachment
        :param shell: Determines if command should be interpreted using shell
        :return: (Converted message, Error string)
        """

        converted_message = None
        msg_err = ""
        tmp_file = NamedTemporaryFile("wb", dir=self.tmp_path, delete=False, prefix="mf_" + msg_id + "_")
        tmp_file.write(message.get_payload(decode=True))
        tmp_file.close()
        logging.debug("Created temporary file " + str(tmp_file.name))
        for i in range(len(command)):
            if command[i] == Value.ORIGINAL_FILE_NAME:
                command[i] = tmp_file.name
            elif command[i] == Value.CONVERTED_FILE_NAME:
                command[i] = f_original_to_new_filename(tmp_file.name)
            elif command[i] == Value.TMP_PATH:
                command[i] = self.tmp_path
        if shell:  # if shell is used, command is converted to string
            command = " ".join(command)
        ended_process = subprocess.run(command, shell=shell, timeout=5, stdout=subprocess.DEVNULL)
        if ended_process.returncode == 0:
            logging.debug("Conversion of file " + str(tmp_file.name) + " successfull.")
            with open(f_original_to_new_filename(tmp_file.name), "rb") as converted_file:
                converted_message = MIMENonMultipart(maintype, subtype)
                converted_message.set_payload(converted_file.read())
                converted_message.add_header('Content-Disposition', 'attachment',
                                             filename=f_original_to_new_filename(message.get_filename()))
                encoders.encode_base64(converted_message)
        else:
            logging.error("Conversion of file " + str(tmp_file.name) + " failed with code " +
                          str(ended_process.returncode) + ".")
            msg_err += "Conversion failed.\nReturn code: "+str(ended_process.returncode)
        os.remove(tmp_file.name)
        logging.debug("Temporary file " + str(tmp_file.name) + " deleted.")
        return converted_message, msg_err


class Value(Enum):
    ORIGINAL_FILE_NAME = 1  # Substitute for name of original file
    CONVERTED_FILE_NAME = 2  # Substitute for name of converted file
    TMP_PATH = 3  # Path to temporary directory


server = SMTPfilter(('127.0.0.1', 10025), ("localhost", 10026), tmp_path="/tmp")
asyncore.loop()
