#include "CControlPreview.h"
#include "CMandelbrot.h"
#include "CJulia.h"
#include "stdio.h"

LRESULT CALLBACK CControlPreview::CustWndProcWrapper(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	CControlPreview *pThis = (CControlPreview*)GetProp(hwnd,"ClassPointer");
	if(pThis) 
		return pThis->CustWndProc(hwnd,msg,wParam,lParam);
	return DefWindowProc(hwnd,msg,wParam,lParam);
}

LRESULT CControlPreview::CustWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	PAINTSTRUCT ps; 
	HDC hdc; 

	switch(msg)
	{
	case WM_PAINT:
		if(wParam == 0)
			hdc = BeginPaint(hwnd, &ps);
		else
			hdc = (HDC)wParam;

		OnPaint(hdc);

		if(wParam == 0)
			EndPaint(hwnd, &ps);
	break;
	case WM_RBUTTONDOWN:
		OnRButtonDown(wParam, lParam);
	break;
	case WM_LBUTTONDOWN:
		RECT r;
		GetWindowRect(hwnd, &r);
		r.bottom--;
		r.top++;
		r.left++;
		r.right--;
		ClipCursor(&r);
		OnLButtonDown(wParam, lParam);
	break;
	case WM_LBUTTONUP:
		ClipCursor(NULL);
		OnLButtonUp(wParam, lParam);
	break;
	case WM_MOUSEMOVE:
		OnMouseMove(wParam, lParam);
	break;
	default:
        break;
    }

    return DefWindowProc(hwnd, msg, wParam, lParam);
}

CControlPreview::CControlPreview(HWND hwndParent, unsigned char *palette, int palSize, double indexVal, double indexInc, bool mand)
: CCustomControl(hwndParent),
  m_drag_x_e(0),m_drag_x_s(0),m_drag_y_e(0),m_drag_y_s(0),
  m_dragging(false)
{
	m_className = _T("ControlPreview");
	m_width = 320;
	m_height = 256;

	InitCustomControl();
	CreateCustomControl();

	if(mand)
	{
		m_fractalType = FRACTAL_TYPE_MAND;
		m_fractalObject = new CMandelbrot(m_width, m_height, hwndParent);
	}
	else
	{
		m_fractalType = FRACTAL_TYPE_JULIA;
		m_fractalObject = new CJulia(m_width, m_height, hwndParent);
	}

	m_bitmapInfo.biSize = sizeof(BITMAPINFOHEADER);
	m_bitmapInfo.biWidth = m_width;
	m_bitmapInfo.biHeight = m_height;
	m_bitmapInfo.biPlanes = 1;
	m_bitmapInfo.biBitCount = 24;
	m_bitmapInfo.biCompression = BI_RGB;
	m_bitmapInfo.biSizeImage = 0;
	m_bitmapInfo.biXPelsPerMeter = 0;
	m_bitmapInfo.biYPelsPerMeter = 0;
	m_bitmapInfo.biClrUsed = 0;
	m_bitmapInfo.biClrImportant = 0;

	HDC hdc = GetWindowDC(NULL);

	m_hBitmap = CreateDIBSection(hdc, (LPBITMAPINFO)&m_bitmapInfo,DIB_RGB_COLORS,(LPVOID *)&m_image,NULL,0);
	m_fractalObject->SetMemoryLocation(m_image);
	
	SetPalette(palette, palSize, indexVal, indexInc);

	ReleaseDC(NULL,hdc);
}

CControlPreview::~CControlPreview()
{
	DeleteObject(m_hBitmap);
}

void CControlPreview::InitCustomControl()
{
    WNDCLASSEX wc;
    
    wc.cbSize         = sizeof(wc);
    wc.lpszClassName  = m_className;
    wc.hInstance      = GetModuleHandle(0);
    wc.lpfnWndProc    = CustWndProcWrapper;
    wc.hCursor        = LoadCursor (NULL, IDC_ARROW);
    wc.hIcon          = 0;
    wc.lpszMenuName   = 0;
    wc.hbrBackground  = (HBRUSH)GetSysColorBrush(COLOR_BTNFACE);
    wc.style          = 0;
    wc.cbClsExtra     = 0;
    wc.cbWndExtra     = 0;
    wc.hIconSm        = 0;

    RegisterClassEx(&wc);
}


void CControlPreview::CreateCustomControl()
{
	m_hwndCtrl = CreateWindowEx(
                 WS_EX_STATICEDGE, // give it a standard border
                 m_className,
                 _T("Preview control"),
                 WS_VISIBLE | WS_CHILD,
                 320, 25, m_width + 2, m_height + 2,
                 m_hwndParent,
                 NULL, GetModuleHandle(0), NULL
               );

	SetProp(m_hwndCtrl,"ClassPointer",(HANDLE)this);
}

void CControlPreview::OnMouseMove(WPARAM wParam, LPARAM lParam)
{
	if(m_dragging)
	{
		m_drag_x_e = LOWORD(lParam);
		m_drag_y_e = HIWORD(lParam);
		InvalidateRect(m_hwndCtrl, NULL, false);
		UpdateWindow(m_hwndCtrl);
	}
}

void CControlPreview::OnLButtonDown(WPARAM wParam, LPARAM lParam) 
{
	m_drag_x_s = LOWORD(lParam);
	m_drag_y_s = HIWORD(lParam);
	m_drag_x_e = m_drag_x_s;
	m_drag_y_e = m_drag_y_s;
	m_dragging = true;
	
	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlPreview::OnLButtonUp(WPARAM wParam, LPARAM lParam) 
{
	m_dragging = false;
	if(m_fractalObject->ZoomIn(m_drag_x_s, m_drag_y_s, m_drag_x_e, m_drag_y_e))
	{
		m_fractalObject->Compute(0,1);
	}
	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlPreview::OnRButtonDown(WPARAM wParam, LPARAM lParam) 
{
	if(m_fractalObject->ZoomOut())
		m_fractalObject->Compute();

	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlPreview::OnKeyPress(WPARAM wParam){
	switch((int)wParam){
		case VK_SUBTRACT:
			DecreaseIterationCount();
		break;
		case VK_ADD:
			IncreaseIterationCount();
		break;
		case VK_BACK:
			m_fractalObject->Reset();
			m_fractalObject->Compute(0,1);
			InvalidateRect(m_hwndCtrl, NULL, false);
		break;
		case VK_ESCAPE:
//			m_fractalObject->Abort();
		break;
	}
}

void CControlPreview::IncreaseIterationCount(){
	m_fractalObject->SetIterationCount(int(m_fractalObject->GetIterationCount()*1.5));
	m_fractalObject->Compute();

	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlPreview::DecreaseIterationCount(){
	m_fractalObject->SetIterationCount(int(m_fractalObject->GetIterationCount()/1.5));
	m_fractalObject->Compute();

	InvalidateRect(m_hwndCtrl, NULL, false);
}

// Set Mandelbrot Set as current displayed fractal
void CControlPreview::SetCurrentMandelbrot(){
	delete m_fractalObject;
	m_fractalObject = new CMandelbrot(m_hwndParent);

	m_fractalObject->SetMemoryLocation(m_image);
	m_fractalObject->Compute(0,1);
	m_fractalType = FRACTAL_TYPE_MAND;
	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlPreview::SetCurrentJulia(float x, float y){
	delete m_fractalObject;
	m_fractalObject = new CJulia(m_hwndParent);
	((CJulia*)m_fractalObject)->SetSeed(x,y);

	m_fractalObject->SetMemoryLocation(m_image);
	m_fractalObject->Compute(0,1);
	m_fractalType = FRACTAL_TYPE_JULIA;
	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlPreview::DrawSelection(HDC hdc)
{
	if(!m_dragging)
		return;

	HPEN hPen, hpOld;
	hPen = CreatePen(PS_SOLID, 1, 0x00000000);
	hpOld = (HPEN)SelectObject(hdc, hPen);

	
	MoveToEx(hdc, m_drag_x_s, m_drag_y_s, NULL);
	LineTo(hdc, m_drag_x_e, m_drag_y_s);
	LineTo(hdc, m_drag_x_e, m_drag_y_e);
	LineTo(hdc, m_drag_x_s, m_drag_y_e);
	LineTo(hdc, m_drag_x_s, m_drag_y_s);

	SelectObject(hdc, hpOld);
	DeleteObject(hPen);  
}

void CControlPreview::SetData(CMandelbrot *data)
{
	m_fractalObject->SetTexture(data->GetImage(), data->GetSizeX()*data->GetSizeY()*sizeof(unsigned char)*3);
}

void CControlPreview::SetPalette(unsigned char *palette, int palSize, double indexVal, double indexInc)
{
	m_fractalObject->SetCustomPalette(palette, palSize);
	m_fractalObject->SetCustomIndexIncrement(indexInc);
	m_fractalObject->SetCustomIndexValue(indexVal);
	m_fractalObject->SetActualPalette(PALETTE_CUSTOM);
	m_fractalObject->Compute(0,1);
}

void CControlPreview::SetSmooth(bool smooth)
{
	if(smooth)
		m_fractalObject->SetColoring(COLORING_SMOOTH);
	else
		m_fractalObject->SetColoring(COLORING_NORMAL);
	m_fractalObject->Compute(0,1);
	InvalidateRect(m_hwndCtrl, NULL, false);
}

void CControlPreview::SetIterationCount(int iterations)
{
	m_fractalObject->SetIterationCount(iterations);
	m_fractalObject->Compute(0,1);
	InvalidateRect(m_hwndCtrl, NULL, false);
}

int CControlPreview::GetIterationCount()
{
	return m_fractalObject->GetIterationCount();
}