#include "NURBS.h"
#include "CFractal.h"			// for struct complex
#include <math.h>

#define NULL 0

CVertex::CVertex()
: m_x(0), m_y(0),
  m_next(NULL),
  m_selected(false),
  m_hover(false)
{}

CVertex::CVertex(double x, double y)
: m_x(x), m_y(y),
  m_next(NULL),
  m_selected(false),
  m_hover(false)
{}

CNURBS::CNURBS()
: m_firstVertex(NULL),
  m_vertexCount(0),
  m_d(2), m_m(m_d+m_vertexCount+1),
  m_stepFrames(24)
{
	m_tVector = new double[m_m + 1];
	for(int i = 0; i < m_m + 1; i++)
		m_tVector[i] = i;
}

double CNURBS::ComputeN(char i, char k, double t){
	if(k == 0)
		if(t >= m_tVector[i] && t < m_tVector[i+1])
			return 1;
		else 
			return 0;
	else{
		double firstpart, secondpart;
		if (m_tVector[i+k] - m_tVector[i] == 0)
			firstpart = 0;
		else
			firstpart = (t - m_tVector[i])/(m_tVector[i+k] - m_tVector[i]); 
		if(m_tVector[i+k+1] - m_tVector[i+1] == 0)
			secondpart = 0;
		else 
			secondpart = (m_tVector[i+k+1] - t)/(m_tVector[i+k+1] - m_tVector[i+1]);
		return firstpart*ComputeN(i, k-1, t) + secondpart*ComputeN(i+1, k-1, t);
	}
}

void CNURBS::ComputeS(CVertex *v, double t){
	CVertex *result = new CVertex();
	CVertex *tmp = m_firstVertex;
	double divisor = 0;
	for(int i = 0; i < m_vertexCount; i++){
		result->SetVertex(result->GetX() + (tmp->GetX())*ComputeN(i, m_d, t), result->GetY() + (tmp->GetY())*ComputeN(i, m_d, t)); 
		tmp = tmp->GetNext();
	}
	for(int i = 0; i < m_vertexCount; i++)
		divisor += ComputeN(i, m_d, t);
	if(divisor != 0)
		v->SetVertex(result->GetX()/divisor, result->GetY()/divisor);
	else
		v->SetVertex(0,0);
	delete result;
}

void CNURBS::AddPoint(double x, double y)
{
	if(m_firstVertex == NULL)
	{
		m_firstVertex = new CVertex(x, y);
		m_vertexCount++;
	}
	else
	{
		CVertex *n = new CVertex(x, y);
		m_vertexCount++;
		CVertex *v = m_firstVertex;
		while(v->GetNext() != NULL)
			v = v->GetNext();
		v->SetNext(n);
		
		m_m = m_vertexCount + m_d + 1;
		delete [] m_tVector;
		m_tVector = new double[m_m+1];
		for(int i = 0; i < m_m+1; i++)
			m_tVector[i] = i;
	}
}

CVertex *CNURBS::GetNearestPoint(double x, double y)
{
	CVertex *n = m_firstVertex;
	int resultCount = 0;
	CVertex *resultVertex = NULL;
	while(n != NULL)
	{
		if(sqrt((n->GetX() - x)*(n->GetX() - x) + (n->GetY() - y)*(n->GetY() - y)) < 0.025)
		{
			resultCount++;
			resultVertex = n;
		}
		n = n->GetNext();
	}
	if(resultCount == 0)
		return NULL;
	if(resultCount == 1)
		return resultVertex;

	double minDistance = 0.025;
	double distance;
	n = m_firstVertex;
	while(n != NULL)
	{
		distance = sqrt((n->GetX() - x)*(n->GetX() - x) + (n->GetY() - y)*(n->GetY() - y));
		if(distance < minDistance)
		{
			resultVertex = n;
		}
		n = n->GetNext();
	}
	return resultVertex;
}

complex CNURBS::GetFrame(int i)
{
	if(m_vertexCount < 3)
		i *= 2;
	else
		i = int((i/(double)(m_stepFrames*(m_vertexCount-1)))*(m_stepFrames*m_vertexCount));

	CVertex *n = new CVertex();
	double u = double((i%m_stepFrames)/(double)m_stepFrames + i/m_stepFrames + m_tVector[m_d-1]);
	ComputeS(n, u);
	complex c;
	c.r = n->GetX();
	c.i = n->GetY();
	delete n;
	return c;
}

CVertex *CNURBS::GetVertex(int i)
{
	CVertex *n = m_firstVertex;
	for(int j = 0; j < i && j < m_vertexCount; j++)
		n = n->GetNext();
	return n;
}

bool CNURBS::ClearVerticesHover()
{
	bool ret = false;
	CVertex *n = m_firstVertex;
	while(n != NULL)
	{
		if(n->GetHover())
			ret = true;
		n->SetHover(false);
		n = n->GetNext();
	}
	return ret;
}

void CNURBS::DeleteVertex(CVertex *v)
{
	CVertex *n = m_firstVertex;
	CVertex *prev = n;

	if(m_firstVertex == v)
	{
		m_firstVertex = n->GetNext();
		delete n;
		m_vertexCount--;
		return;
	}
	else
		n = n->GetNext();

	while(n != NULL){
		if(n == v){
			prev->SetNext(n->GetNext());
			delete n;
			m_vertexCount--;
			n = prev->GetNext();
		}
		else{
			n = n->GetNext();
			prev = prev->GetNext();
		}
	}
}