import fp from 'lodash-fp'
import {
  startOfMonth,
  subMonths,
  addMonths,
  startOfDay,
  format
} from 'date-fns'

import request from '../helpers/request'
import requestBuilders from '../helpers/requestBuilders'
import { getTimestampsBetween } from '../helpers/utils'
import cacheMiddleware from '../helpers/cacheMiddleware'
import { BASE_URL, NEEDED_DATA_LENGTH, QUOTE_CURRENCY } from '../consts'

const fetchApi = {
  fetchSymbols: cacheMiddleware('symbols')(async () => {
    const symbolsData = await request({ uri: `${BASE_URL}/symbol`, json: true })
    return symbolsData
      .filter(t => t.quoteCurrency === QUOTE_CURRENCY)
      .map(t => t.id)
  }),

  fetchDataLength: cacheMiddleware('data-length')(async () => {
    const symbols = await fetchApi.fetchSymbols()
    const dataLength = await Promise.all(
      symbols.map(async id => ({
        id,
        months: await requestBuilders.symbolDataLength(id)
      }))
    )
    return dataLength
  }),

  getRelevantSymbols: async () => {
    const dataLength = await fetchApi.fetchDataLength()
    return dataLength.filter(t => t.months > NEEDED_DATA_LENGTH).map(t => t.id)
  },

  fetchPrices: cacheMiddleware('prices')(async symbolId => {
    const now = new Date()
    const getPrices = requestBuilders.priceData(symbolId)

    const monthsToGet = (await requestBuilders.symbolDataLength(symbolId)) - 1

    const startDate = startOfMonth(subMonths(now, monthsToGet))
    const endDate = startOfMonth(now)

    const data = await Promise.all(
      getTimestampsBetween(startDate, endDate).map(getPrices)
    )
    const res = {}
    fp.flatten(data).forEach(t => (res[t.timestamp] = t.close))

    const sorted = {}
    Object.keys(res)
      .sort()
      .forEach(key => (sorted[key] = res[key]))
    return sorted
  }),

  toArrayPrices: cacheMiddleware('array')(async symbolId => {
    const now = new Date()
    const startDate = startOfDay(
      startOfMonth(subMonths(now, NEEDED_DATA_LENGTH))
    )
    const endDate = addMonths(startDate, NEEDED_DATA_LENGTH)
    const timestamps = getTimestampsBetween(startDate, endDate, 1).map(
      t => `${format(t, 'YYYY-MM-DDTHH:mm:ss.000')}Z`
    )

    const prices = await fetchApi.fetchPrices(symbolId)
    const arrayPrices = timestamps.map(t => prices[t] || null)
    return arrayPrices
  }),

  fetchAll: async () => {
    const symbols = await fetchApi.getRelevantSymbols()
    for (const symbolId of symbols) {
      await fetchApi.toArrayPrices(symbolId)
    }
  }
}

export default fetchApi
