#!/usr/bin/python
# encoding: utf8
from __future__ import print_function, division
import random
from rules import GridFillingRules
from grid import Grid, val_to_output
from game import GridFillingGame
import main
import postscript
import math

class SudokuRules(GridFillingRules):
    def __init__(self, game):
	self.game = game

    # each column, row and 3x3 square contains
    # every number at least once
    # 3 * 9 * 9 clauses (of form 0 v 1 v .. v 8)
    def crs_least_1(self, dest):
	term = self.game.term
	colors = self.game.colors
	sw, sh = self.game.sw, self.game.sh
	for n in range(colors):
	    for i in range(colors):
		dest.append([term(i,y,n) for y in range(colors)])
		dest.append([term(x,i,n) for x in range(colors)])
		x,y = i // sw, i % sw
		dest.append([term(sw*x+dx,sh*y+dy,n) for dy in range(sh) for dx in range(sw)])

    def crs_most_1(self, dest):
	term = self.game.term
	colors = self.game.colors
	sw, sh = self.game.sw, self.game.sh
	for n in range(colors):
	    for x in range(colors):
		for y in range(colors):
		    for i in range(y):
			dest.append([-term(x,y,n), -term(x,i,n)])
		    for i in range(x):
			dest.append([-term(x,y,n), -term(i,y,n)])
	    for i in range(colors):
		x,y = i // sw, i % sw
		for d1 in range(colors):
		    x1,y1 = d1 // sh, d1 % sh
		    for d2 in range(d1):
			x2,y2 = d2 // sh, d2 % sh
			dest.append([-term(sw*x+x1,sh*y+y1,n), -term(sw*x+x2,sh*y+y2,n)])

    rule_list = GridFillingRules.rule_list + [crs_least_1, crs_most_1]


class SudokuGrid(Grid):
    def as_ascii_grid(self):
	data = self.data
	w,h = self.game.sw, self.game.sh
	n = w*h
	res = [[' ' for x in range(n+h+1)] for y in range(n+w+1)]
	for x in range(h+1):
	    for y in range(w+1):
		res[y*(h+1)][x*(w+1)] = '+'
	for x in range(h+1):
	    for y in range(w):
		for dy in range(h):
		    res[y*(h+1)+dy+1][x*(w+1)] = '|'
	for x in range(h):
	    for y in range(w+1):
		for dx in range(w):
		    res[y*(h+1)][x*(w+1)+dx+1] = '-'
	for x in range(h):
	    for y in range(w):
		for dx in range(w):
		    for dy in range(h):
			res[y*(h+1)+dy+1][x*(w+1)+dx+1] = val_to_output(data[x*w+dx][n-y*h-dy-1], ' ')
	return str.join('\n', map(''.join, res))

    def as_eps(self):
	data = self.data
	w,h = self.game.sw, self.game.sh
	n = w*h
	c = 47
	eps = postscript.EPS(-0.05 * c, -0.05 * c, (n+0.1)*c, (n+0.1)*c)
	eps.scale(c, c)
	# darker parts of grid
	eps.setlinewidth(0.05)
	eps.rectangle(0, 0, n, n)
	for x in range(h-1):
	    eps.line((x+1)*w, 0, 0, n)
	for y in range(w-1):
	    eps.line(0, (y+1)*h, n, 0)
	# lighter parts of grid
	eps.setlinewidth(0.02)
	for x in range(n):
	    if x % w:
		eps.line(x, 0, 0, n)
	for y in range(n):
	    if y % h:
		eps.line(0, y, n, 0)
	# prefilled numbers
	eps.setfont("Helvetica", 1)
	eps.translate(0.25, 0.16)
	for x in range(n):
	    for y in range(n):
		if data[x][y] is not None:
		    eps.text(x, y, val_to_output(data[x][y], ' '))
	return eps.as_string()

class Sudoku(GridFillingGame):
    name = "sudoku"
    grid_class = SudokuGrid
    output_formatter = { "draw": grid_class.as_ascii_grid,
			 "dot": grid_class.as_dot,
			 "eps": grid_class.as_eps }

    def __init__(self, opts):
	self.sw = opts.width if opts.width else 3
	self.sh = opts.height if opts.height else self.sw
	self.colors = self.sw * self.sh
	self.width = self.height = self.colors
	super(Sudoku, self)._init(self)
	self.baseterms = self.nterms
	self.rules = SudokuRules(self)
	if opts.fortyseven:
	    self.initial_random = self.random

    def random(self):
	assert(self.sw == self.sh == 3)
	res = []
	for x in range(3):
	    for y in range(3):
		if x + y == 2:
		    continue
		res.append([self.term((x+y)   % 3 + x * 3, x + y * 3, 3)])
		res.append([self.term((x+y+1) % 3 + x * 3, x + y * 3, 6)])
	return res

    def initial_random(self):
	res = []
	for i in range(int(math.sqrt(self.width * self.height))):
	    res.append([(random.randrange(self.baseterms)+1)])
	return res


if __name__ == "__main__":
    main.main(Sudoku)
