#!/usr/bin/python
# encoding: utf8
from __future__ import print_function, division
import math
import random
from rules import Rules
import game
import main
import postscript


class HamiltonRules(Rules):
    def __init__(self, game):
	self.game = game

    def cell_2(self, dest):
	term = self.game.term
	ends = self.game.ends
	for x in range(self.game.width):
	    for y in range(self.game.height):
		if (x,y) not in ends:
		    for r in range(4):
			ts = filter(None, (term(x, y, (r + i) % 4) for i in range(3)))
			dest.append(ts)
			if len(ts) == 3:
			    dest.append([-z for z in ts])
		else:
		    dest.append(filter(None, (term(x, y, r) for r in range(4))))
		    for r1 in range(4):
			for r2 in range(r1):
			    ts = filter(None, (term(x, y, r1), term(x, y, r2)))
			    nts = map(lambda x: -x, ts)
			    if len(nts) == 2:
				dest.append(nts)

    def counting(self, dest):
	term2 = self.game.term2
	w,h = self.game.width, self.game.height
	colors = self.game.colors
	for x in range(w):
	    for y in range(h):
		if (x,y) not in self.game.ends:
		    # kazde policko ma aspon 1 cislo
		    dest.append([term2(x,y,n) for n in range(1,colors-1)])
		    # ziadne nema 2
		    for n1 in range(1,colors-1):
			for n2 in range(1,n1):
			    dest.append((-term2(x,y,n1), -term2(x,y,n2)))
		    # nie je 1 ani wh
		    dest.append((-term2(x,y,0),))
		    dest.append((-term2(x,y,colors-1),))

    def continuity(self, dest):
	term, term2 = self.game.term, self.game.term2
	dirs = self.game.dirs
	w,h = self.game.width, self.game.height
	colors = self.game.colors
	for x in range(w):
	    for y in range(h):
		for n in range(colors-1):
		    neigh = []
		    for d in range(4):
			t = term(x,y,d)
			if t is not None:
			    x1,y1 = x + dirs[d][0], y + dirs[d][1]
			    # policka su spojene alebo nenasleduju za sebou
			    dest.append((t, -term2(x,y,n), -term2(x1,y1,n+1)))
			    neigh.append((x1,y1))
		    # pri policku sa nachadza cislo o 1 vacsie
		    dest.append([-term2(x,y,n)] + [term2(x1,y1,n+1) for (x1,y1) in neigh])

    def endpoints(self, dest):
	for ((x,y),c) in self.game.ends.items():
	    # (x,y) je c
	    dest.append([self.game.term2(x, y, c)])
	    # a nie je nic ine
	    for n in range(self.game.colors):
		if n != c:
		    dest.append((-self.game.term2(x, y, n),))

    rule_list = [cell_2, counting, continuity, endpoints]

def val_to_output(x, plus, minus, none):
    if x is None:
	return none
    return minus if x < 0 else plus

def output_to_val(x, plus, minus, none):
    return {none: None, plus: 1, minus: -1}[x]

class HamiltonGrid(object):
    def __init__(self, game):
	self.game = game
	self.data = [[[None for d in [0,1,2]] for y in range(game.height)] for x in range(game.width)]

    def set_from_terms(self, terms):
	for x in terms:
	    term = x[0]
	    if abs(term) <= self.game.baseterms:
		x, y, dir = self.game.decomp(abs(term))
		self.data[x][y][dir] = term // abs(term)
	    else:
		x, y, n = self.game.decomp2(abs(term))
		if self.data[x][y][2] is not None:
		    print("Hmm... self[{0}][{1}][2] = {2} and ({0},{1},{3})".format(x,y,self.data[x][y][2], n))
		self.data[x][y][2] = n * term // abs(term)
	return self

    def set_from_dot(self, dot):
	game = self.game
	for t in range(self.game.baseterms):
	    x,y,d = self.game.decomp(t+1)
	    self.data[x][y][d] = output_to_val(dot[t], '+', '-', '.')
	return self

    def as_terms(self):
	res = []
	for t in range(self.game.baseterms):
	    x,y,d = self.game.decomp(t+1)
	    if self.data[x][y][d] is not None:
		if self.data[x][y][d] < 0:
		    res.append((-t-1,))
		else:
		    res.append((t+1,))
	return res

    def as_dot(self):
	res = []
	for t in range(self.game.baseterms):
	    x,y,d = self.game.decomp(t+1)
	    res.append(val_to_output(self.data[x][y][d], '+', '-', '.'))
	return str.join('', res)

    def as_ascii(self):
	data = self.data
	w,h = self.game.width, self.game.height
	res = [['  ' for x in range(2 * w + 1)] for y in range(2 * h + 1)]
	for x in range(w+1):
	    for y in range(h+1):
		res[2*y+0][2*x+0] = '＋'
	for x in [0, 2*w]:
	    for y in range(2*h-1):
		res[y+1][x] = '｜'
	for x in range(2*w-1):
	    for y in [0, 2*h]:
		res[y][x+1] = '──'
	for x in range(w-1):
	    for y in range(h):
		res[2*y+1][2*x+2] = val_to_output(self.data[x][y][0], '──', '｜', '  ')
	for x in range(w):
	    for y in range(h-1):
		res[2*y+2][2*x+1] = val_to_output(self.data[x][y][1], '｜', '──', '  ')
	for x in range(w):
	    for y in range(h):
		if self.data[x][y][2] is not None:
		    res[2*y+1][2*x+1] = '{0:2}'.format(self.data[x][y][2])
	for (x,y) in self.game.ends:
	    res[2*y+1][2*x+1] = 'HH'
	return str.join('\n', map(''.join, res))

    def as_eps(self):
	data = self.data
	w, h = self.game.width, self.game.height
	c = 47
	eps = postscript.EPS(-0.05 * c, -0.05 * c, (w+0.1)*c, (h+0.1)*c)
	eps.scale(c, c)
	# grid
	eps.setlinecap(1)
	eps.setlinewidth(0.05)
	eps.rectangle(0, 0, w, h)
	eps.setlinewidth(0.04)
	e = 0.1
	for x in range(1, w):
	    eps.line(x, 0, 0, e)
	    eps.line(x, h, 0, -e)
	for y in range(1, h):
	    eps.line(0, y, e, 0)
	    eps.line(w, y, -e, 0)
	for x in range(1, w):
	    for y in range(1, h):
		eps.line(x - e, y, e+e, 0)
		eps.line(x, y - e, 0, e+e)
	# prefill vertical walls
	for x in range(w-1):
	    for y in range(h):
		if data[x][y][0] is not None and data[x][y][0] < 0:
		    eps.line(x + 1, y, 0, 1)
	# prefill horizontal walls
	for x in range(w):
	    for y in range(h-1):
		if data[x][y][1] is not None and data[x][y][1] < 0:
		    eps.line(x, y+1, 1, 0)
	# draw endpoints
	eps.setgray(0.5)
	eps.setlinewidth(0.7)
	for (x,y) in self.game.ends:
	    eps.line(x + 0.5, y + 0.5, 0, 0)
	# draw through vertical walls
	eps.setlinewidth(0.3)
	for x in range(w-1):
	    for y in range(h):
		if data[x][y][0] is not None and data[x][y][0] > 0:
		    eps.line(x + 0.5, y + 0.5, 1, 0)
	# draw through horizontal walls
	for x in range(w):
	    for y in range(h-1):
		if data[x][y][1] is not None and data[x][y][1] > 0:
		    eps.line(x + 0.5, y + 0.5, 0, 1)
	eps.setfont("Helvetica", 0.5)
	eps.translate(0.34, 0.33)
	eps.setgray(0)
	for ((x,y),c) in self.game.ends.items():
	    if c == 0:
		eps.text(x, y, 'Z')
	    else:
		eps.text(x, y, 'K')
	return eps.as_string()


class Hamilton(game.GridCrossingGame):
    name = "hamilton"
    grid_class = HamiltonGrid
    output_formatter = { "draw": grid_class.as_ascii,
			 "dot": grid_class.as_dot,
			 "eps": grid_class.as_eps }

    def __init__(self, opts):
	self.width = opts.width if opts.width else 6
	self.height = opts.height if opts.height else self.width
	w, h = self.width, self.height
	self.colors = w * h
	super(Hamilton, self)._init(opts)
	self.baseterms = self.nterms
	self.nterms = self.nterms + w * h * self.colors
	self.rules = HamiltonRules(self)
	self.ends = {(0,0) : 0, (1,0) : (self.colors-1)}

    def term2(self, x, y, n):
	return ((y * self.width) + x) * self.colors + n + 1 + self.crossingterms

    def decomp2(self, term):
	term -= 1 + self.baseterms
	term, n = divmod(term, self.colors)
	term, x = divmod(term, self.width)
	y = term
	return x, y, n

    def initial_random(self):
	res = []
	for i in range(int(math.sqrt(self.width * self.height))):
	    res.append([-(random.randrange(self.baseterms)+1)])
	return res

    def rule(self, term):
	return [-term]


if __name__ == "__main__":
    main.main(Hamilton)
