from __future__ import print_function, division
import os
import sys
import random
import shared

class result_set(object):
    """Class to parse the result of running the sat solver"""

    def __init__(self, filename):
	self.filename = filename

    def __enter__(self):
	return self

    def __exit__(self, *whatever):
	pass

    def dump(self):
	with open(self.filename) as file:
	    sys.stdout.write(file.read())
	return self

    def read(self, solutions, details = False):
	with open(self.filename) as file:
	    sat, count, sols = self._read(file, solutions, details)
	self.sat = sat
	self.count = count
	self.solutions = sols
	return self

class custom_result_set(result_set):
    sat = "UNKNOWN"
    count = 0

    def read(self, solutions, details = False):
	return self

    def __init__(self, *a, **b):
	super(custom_result_set, self).__init__(*a, **b)
	self.solutions = []


class relsat_result_set(result_set):
    """Class to parse relsat output"""

    def _read(self, file, solutions, details):
	sat = "UNKNOWN"
	count = 0
	sols = []
	for line in file:
	    if line.startswith("SAT"):
		sat = "SAT"
	    elif line.startswith("UNSAT"):
		sat = "UNSAT"
		break
	    elif line.startswith("TIME LIMIT EXPIRED"):
		sat = "TIMEOUT"
	    elif line.startswith("Solution"):
		count += 1
		solutions and sols.append(map(int,line.split()[2:]))
	return sat, count, sols

class v_result(result_set):
    """Class to parse general v-prefixed sat-solver output"""

    def _read(self, file, solutions, details):
	sat = "UNKNOWN"
	count = 0
	sols = []
	sol = []
	for line in file:
	    if line[0] == 's':
		if line[2:].startswith('SATISFIABLE'):
		    sat = "SAT"
		elif line[2:].startswith('UNSATISFIABLE'):
		    sat = "UNSAT"
		elif line[2:].startswith('TIMEOUT'):
		    sat = "TIMEOUT"
		else:
		    self.dump()
		    raise Error("unable to parse: ", repr(line[2:]))
	    elif solutions and line[0] == 'v':
		vs = map(int,line[2:].split())
		for v in vs:
		    if v > 0:
			sol.append(v)
		    elif v == 0:
			sols.append(sol)
			sol = []
			count += 1
	if sat == "SAT" and not solutions:
	    count = 1
	return sat, count, sols

def system(cmd):
    status = os.system(cmd)
    if os.WIFSIGNALED(status):
	exit()

class sat(object):
    """Sat solver encapsulation"""

    _privar = None
    keep = False
    timeout = 0

    def __init__(self, name, variables = 0):
	self._prefix = shared.temp_prefix(name)
	self.solname = self._prefix + ".sol"

    def __enter__(self):
	return self

    def __exit__(self, *whatever):
	if not self.keep:
	    try:
		if self._privar:
		    os.remove(self._privar)
		os.remove(self.solname)
	    except OSError:
		pass

    def set_primary_variables(self, vars):
	with open(self._prefix + ".privar", "w") as f:
	    for i in vars:
		print(int(i), file=f)
	    self._privar = f.name

    def run(self, rules, count, seed = None, solutions = True):
	rules.flush()
	if seed is None:
	    seed = random.randrange(1000000) + 1
	return self._run(rules, count, seed, solutions)


class relsat(sat):
    """Relsat encapsulation"""

    prog = "relsat -r3 -a3"
    has_native_multisolution = True
    def timeoutopt(self, timeout):
	return "-t" + str(timeout)

    def _run(self, rules, count, seed, solutions):
	privar = ""
	if self._privar:
	    privar = "-v " + self._privar
	timeout = self.timeoutopt(self.timeout) if self.timeout else ""
	cmd = "{0} -s{1} -#{2} {3} {4} {5} > {6}".format(
	    self.prog, seed, count, privar, timeout, rules.filename, self.solname)
	system(cmd)
	return relsat_result_set(self.solname).read(solutions)

class basicsat(sat):
    """Limited sat solver encapsulation"""

    has_native_multisolution = False
    def extraopts(self, count, seed, solutions):
	return ""

    def set_primary_variables(self, vars):
	"not supported"

    def _run(self, rules, count, seed, solutions):
	"""WARNING: count > 1 is guaranteed to work only if there is no solution
	     whose positive variables are a subset of anotherone's"""
	solname = self.solname
	seed = self.seedopt(seed)
	timeout = self.timeoutopt(self.timeout) if self.timeout else ""
	extra = self.extraopts(count, seed, solutions or count != 1)
	cmd = "{0} {1} {2} {3} {4} > {5}".format(
	    self.prog, seed, rules.filename, timeout, extra, solname)
	if count == 1:
	    system(cmd)
	    return v_result(solname).read(solutions)
	had_rules = len(rules)
	res = custom_result_set(solname)
	while res.count < count:
	    system(cmd)
	    with v_result(solname).read(True) as result:
		if result.sat == "SAT":
		    res.sat = "SAT"
		    res.count += result.count
		    if solutions:
			res.solutions.extend(result.solutions)
		    if res.count == count:
			break
		    for sol in result.solutions:
			rules.append([-x for x in sol])
		    rules.flush()
		elif result.sat == "UNSAT":
		    if res.sat == "UNKNOWN":
			res.sat = "UNSAT"
		    break
		elif result.sat == "TIMEOUT":
		    res.sat = "TIMEOUT"
		    res.count += result.count
		    if solutions:
			res.solutions.extend(result.solutions)
		    break
		else:
		    result.dump()
		    print("problem:", result.sat)
		    raise "Error"
	rules.truncate(had_rules)
	return res


class picosat(basicsat):
    """Picosat encapsulation"""

    prog = "picosat"
    def seedopt(self, seed):
	return "-s " + str(seed)
    def timeoutopt(self, timeout):
	return "--timeout " + str(timeout)


class precosat(basicsat):
    """Precosat encapsulation"""

    prog = "precosat -v"
    def seedopt(self, seed):
	return "--seed=" + str(seed)
    def timeoutopt(self, timeout):
	return "-t " + str(timeout)

class minisat(basicsat):
    """Precosat encapsulation"""

    prog = "minisat"

    def seedopt(self, seed):
	"Not supported"
	return ""
    def timeoutopt(self, timeout):
	"Not supported"
	return ""
    def extraopts(self, count, seed, solutions):
	return "-" if solutions else ""

solvers = {"relsat": relsat, "picosat": picosat, "precosat": precosat, "minisat": minisat}
