<?php

if (!defined('FAIL'))
    define('FAIL', false);

class MmImagick
{
    var $mConvertPath;			// Holds complete path to the Image Magick 'convert' binary
    var $mIdentifyPath;         // Holds complete path to the Image Magick 'identify' binary
    var $mCompositePath;       // Holds complete path to the Image Magick 'composite' binary
    var $mActualImage;          // complate path to the actual picture being processed
    var $mDebug;                // Turns on/off debugging
    var $mLastErrorText;        // Holds last error set by ThrowError()
    var $mImageType;            // JPEG, GIF, PNG,...
    var $mWidth;                // x-proportion of the image
    var $mHeight;               // y-proportion of the image
    
    function MmImagick($convertPath = "/usr/bin/convert", $identifyPath = "/usr/bin/identify", $compositePath = "/usr/bin/composite")
    {
        $this->mConvertPath     = $convertPath;
        $this->mIdentifyPath    = $identifyPath;
        $this->mCompositePath    = $compositePath;
        $this->mDebug           = false;
    }
    
    /*
        SetDebug() - Turns on/off debugging
            If turned on, method ThrowError() will print out all errors on the screen
        Params:
            $debug - true / false == on / off
        Returns:
            Success - true
            Failure - FAIL
    */
    function SetDebug($debug)
    {
        if (false != is_bool($debug))
        {
            $this->mDebug = $debug;
            return true;
        }
        return FAIL;
    }

    /*
        ThrowError() - Sets the LastErrorText property, and displayes error if debugging is turned on
        Params:
            $errorText - string containing error
        Returns:
            Always returns: true
    */
    function ThrowError($errorText)
    {
        $this->mLastErrorText = $errorText;
        if (false != $this->mDebug)
            echo $this->mLastErrorText;
        return true;
    }
    
    /*
        CheckImInstall() - Verifies ImageMagick installation
        Params:
        Returns:
            Success - true
            Failure - FAIL
    */    
    function CheckImInstall()
    {
    	if (!is_executable($this->mIdentifyPath))
    	{
    		$this->ThrowError("Error in MmImagick::CheckImInstall(): ".$this->mIdentifyPath." is not executable.");
    		return FAIL;
    	}
    	if (!is_executable($this->mConvertPath))
    	{
    		$this->ThrowError("Error in MmImagick::CheckImInstall(): ".$this->mConvertPath." is not executable.");
    		return FAIL;
    	}
    	if (!is_executable($this->mCompositePath))
    	{
    		$this->ThrowError("Error in MmImagick::CheckImInstall(): ".$this->mCompositePath." is not executable.");
    		return FAIL;
    	}
    	return true;
    }
    
    /*
        OpenImage() - Sets the image to process, gets image type and proportions
        Params:
            $imagePath - complete absolut path to the image including the file name
        Returns:
            Success - true
            Failure - FAIL
    */
    function OpenImage($imagePath)
    {
        if (false != is_readable($imagePath))
        {
            $this->mActualImage = $imagePath;
        }
        else
        {
            $this->ThrowError("Error in MmImagick::OpenImage(): '$imagePath' is unreadable.");
            return FAIL;
        }
        
        // open a process to the Image Magick identify binary to get information about the image
        $handle = popen($this->mIdentifyPath." ".$this->mActualImage, 'r');
        if (false == $handle)
        {
            $this->ThrowError("Error opening process in MmImagick::OpenImage()");
            return FAIL;
        }
        $data = fread($handle, 512);
        if (empty($data))
        {
            $this->ThrowError("Error in MmImagick::OpenImage(): Unsupported file type or error reading '".$this->mActualImage."'");
            return FAIL;
        }

        // data result looks like this:
        //  image.gif GIF 46x26+0+0 PseudoClass 8c 8-bit 183.0 0.000u 0:01
        $parsed_data = explode(" ", $data);         // first split it by spaces
        $this->mImageType   = $parsed_data[1];      // image type is second parameter...
        $geometry = $parsed_data[2];
        $parsed_geometry = explode("x", $geometry); // now we split geometry by 'x'...
        $this->mWidth = $parsed_geometry[0];        // width is the first one...
        $geometry = $parsed_geometry[1];
        
        $height = "";
        // now we have to prevent height to contain non-numeric characters...
        for ($i = 0; $i < strlen($geometry); $i++)
        {
            if (false != is_numeric($geometry[$i]))
            {
                $height .= $geometry[$i];
            }
            else
            {
                break;
            }
        }
        $this->mHeight = $height;
        pclose($handle);
        
        return true;
    }
    
    function Width()
    {
        return $this->mWidth;
    }
    
    function Height()
    {
        return $this->mHeight;
    }
    
    function ImageType()
    {
        return $this->mImageType;
    }

    /*
        GetSize() - return size of the image file in bytes
        Params:
        Returns:
            Success - integer number expressing byte size of the image file
            Failure - FAIL
    */
    function GetSize()
    {
        $size = @filesize($this->mActualImage);
        if (false != $size)
        {
            return $size;
        }
        $this->ThrowError("Error in MmImagick::GetSize(): Unable to get size of '".$this->mActualImage."'");
        return FAIL;
    }
    
    /*
        GenerateThumbnail() - resizes the image into given maximum proportions
            Only resizes if one of its proportions exceedes given max interval.
            It writes the resized image into targetFile param.
        Params:
            $maxWidth       - maximum thumbnail width
            $maxHeight      - maximum thumbnal height
            $targetFile     - complete file path where the resized image shall be written
        Returns:
            Success - true
            Failure - FAIL
    */
    function GenerateThumbnail($maxWidth, $maxHeight, $targetFile)
    {
        // open a process to the Image Magick convert binary to generate the thumbnail
        $command = $this->mConvertPath." ".$this->mActualImage." -thumbnail '".$maxWidth."x".$maxHeight.">' ".$targetFile;
        $ret_val = null;
        passthru($command, $ret_val);
        if (0 != $ret_val)      // error occured
        {
            $this->ThrowError("Error generating thumbnail in MmImagick::GenerateThumbnail() Code: ".$ret_val);
            return FAIL;
        }
        return true;
    }

    /*
        PutStamp() - puts a stamp image on the top of original image, using composite

        Params:
            $stampFile - complete path to stamp file to be used
            $targetFile - complete file path where the result stamped image shall be written
        Returns:
            Success - true
            Failure - FAIL
    */
    function PutStamp($stampFile, $targetFile)
    {
        //open process to ImageMagick composite binary to put stamp
        $command = $this->mCompositePath . " -quality 70 -compose atop -gravity center " . $stampFile . " " . $this->mActualImage . " " . $targetFile;
        $ret_val = null;
        passthru($command, $ret_val);
        if (0 != $ret_val)      // error occured
        {
            $this->ThrowError("Error putting stamp in MmImagick::PutStamp() Code: ".$ret_val);
            return FAIL;
        }
        return true;
    }
    
    
    /*
        Annotate() - puts an anotation text to the image

        Params:
            $file - image file to annotate
            $text - anotation text
        Returns:
            Success - true
            Failure - FAIL
    */
    function Annotate($file, $text)
    {
        //open process to ImageMagick composite binary to put stamp
        $command = $this->mConvertPath." -font helvetica -fill white -pointsize 12 -draw 'text 10,10 \"$text\"' ".$file." ".$file;
        echo $command;
        $ret_val = null;
        passthru($command, $ret_val);
        if (0 != $ret_val)      // error occured
        {
            $this->ThrowError("Error putting annotating in MmImagick::Annotate() Code: ".$ret_val);
            return FAIL;
        }
        return true;
    }
}

?>
