from app import db
from datetime import datetime
from .constant import TestingStatus, Language, DEFAULT_SUBMIT_PRIORITY


def generate_password_hash(password):
    return 'sRNka' + password


def check_password_hash(password_hash, password):
    return password_hash == 'sRNka' + password


class Admin(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(30), unique=True, nullable=False)
    password_hash = db.Column(db.String(128))

    def set_password(self, password):
        self.password_hash = generate_password_hash(password)

    def check_password(self, password):
        return check_password_hash(self.password_hash, password)

    def __repr__(self):
        return f'<Admin {self.username}>'


class Client(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(20), unique=True, nullable=False)
    api_key = db.Column(db.String(36), unique=True, nullable=False)
    notification_url = db.Column(db.String(36))
    problems = db.relationship('Problem', backref='client')
    submits = db.relationship('Submit', backref='client')

    def __repr__(self):
        return f'<Client {self.name}, id={self.id}>'


class Problem(db.Model):
    id = db.Column(db.String(30), primary_key=True)
    client_id = db.Column(
        db.Integer, db.ForeignKey('client.id'), nullable=False, primary_key=True
    )
    submits = db.relationship('Submit', backref='problem')

    def __repr__(self):
        return f'<Problem {self.id}>'


class Submit(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    uid = db.Column(db.String(36))
    client_id = db.Column(
        db.Integer, db.ForeignKey('client.id'), nullable=False
    )
    problem_id = db.Column(
        db.String(30), db.ForeignKey('problem.id'), nullable=False
    )
    datetime = db.Column(db.DateTime, nullable=False, default=datetime.utcnow)
    language = db.Column(db.Enum(Language, values_callable=lambda x: [e.value for e in x]))
    priority = db.Column(db.Integer, default=DEFAULT_SUBMIT_PRIORITY)
    status = db.Column(db.Enum(TestingStatus, values_callable=lambda x: [e.value for e in x]), default=TestingStatus.IN_QUEUE)

    def __repr__(self):
        return (
            f'<Submit\n{self.id}\n{self.uid}\nclient {self.client}\n'
            f'problem {self.problem}, status={self.status}>'
        )
