import configparser
import glob
import imp
import os
import shutil

import isolate
from isolate import IsolateStatus
from languages import languages
from utils import (
    read_config,
    read_execution_metadata,
    execute_external,
    pretty_diff_files,
)
from constants import (
    TestingMode,
    TestingStatus,
    DEFAULT_EXECUTION_MEMORY_LIMIT_KILOBYTES,
    EXECUTION_TEMP_METADATA_PATH,
    EXECUTION_INPUT_FILE_NAME,
    EXECUTION_OUTPUT_FILE_NAME,
    TEST_PROTOCOL_PATH,
    TEST_RESULT_STATUS_PATH,
    BATCH_RESULTS_PATH,
)
from settings import (
    TASK_DIRECTORY,
    SUBMIT_LANGUAGE,
)


def task_time_limits_seconds(task_config, language):
    # This is a fallback for older task.configs which use the old format
    # It will hopefully be phased out when we deploy
    base_miliseconds = task_config.getint('task', 'timelimit')
    base_miliseconds = task_config.getint(
        'time_limits', SUBMIT_LANGUAGE, fallback=base_miliseconds
    )
    base = base_miliseconds / 1000.0
    time_limits = {'base': base}
    time_limits['wall'] = min(1.7 * base, base + 3)
    time_limits['extra'] = max(0.1, min(base / 10.0, 0.5))
    time_limits = language.normalise_time(time_limits)
    return time_limits


def task_memory_limit_kilobytes(task_config, language):
    base_memory_limit = task_config.getint(
        'task', 'memlimit', fallback=DEFAULT_EXECUTION_MEMORY_LIMIT_KILOBYTES
    )
    memory_limit = language.normalise_memory(base_memory_limit)
    return memory_limit


def custom_execute(task_config):
    module_name = task_config['task']['custom_execute']
    if module_name.endswith('.py'):
        module_name = module_name[:-3]
    mfile, mpath, mdesc = imp.find_module(module_name, [TASK_DIRECTORY])
    print(mpath)
    modul = imp.load_module('custom_execute', mfile, mpath, mdesc)
    return modul.custom_process()


def testing_mode(task_config):
    try:
        mode = TestingMode(task_config['task']['mode'])
    except (ValueError, KeyError):
        mode = TestingMode.ACM
    finally:
        return mode


def is_file_sample(file_name):
    return ('sample' in file_name) or ('example' in file_name)


def compare_outputs(sandbox_path, input_path, correct_output_path, tester):
    execution_output_path = os.path.join(
        sandbox_path, EXECUTION_OUTPUT_FILE_NAME
    )
    tester_out = None
    if tester:
        tester_path = os.path.join(TASK_DIRECTORY, tester)
        tester_out, error = execute_external([
            tester_path,
            TASK_DIRECTORY,
            input_path,
            correct_output_path,
            execution_output_path,
        ])
    else:
        diff_command = [
            'diff', '-q', execution_output_path, correct_output_path
        ]
        diff_output, error = execute_external(diff_command)
        status = TestingStatus.WRONG_ANSWER
        if error != 0:
            tester_out, _ = pretty_diff_files(
                correct_output_path, execution_output_path
            )

    status = TestingStatus.OK if error == 0 else TestingStatus.WRONG_ANSWER
    return status, tester_out


def batch_already_failed(batch_result_status):
    return batch_result_status is not TestingStatus.OK


def copy_input_into_sandbox(sandbox_path, input_path):
    sandbox_input_path = os.path.join(sandbox_path, EXECUTION_INPUT_FILE_NAME)
    shutil.copy(input_path, sandbox_input_path)


def execute_test_case(task_config):
    language = languages[SUBMIT_LANGUAGE]
    time_limits = task_time_limits_seconds(task_config, language)
    memory_limit = task_memory_limit_kilobytes(task_config, language)
    command = isolate.execution_command(
        language.run_command, time_limits, memory_limit
    )

    execute_external(command)


def evaluate_test_case(sandbox_path, task_config, input_file_name):
    input_path = os.path.join(TASK_DIRECTORY, input_file_name)

    copy_input_into_sandbox(sandbox_path, input_path)
    execute_test_case(task_config)

    execution_metadata = read_execution_metadata(EXECUTION_TEMP_METADATA_PATH)

    isolate_status = IsolateStatus(
        execution_metadata.get('test', 'status', fallback=IsolateStatus.OK)
    )

    if isolate_status == IsolateStatus.OK:
        tester = task_config.get('task', 'tester', fallback=None)
        correct_output_path = f'{input_path[:-3]}.out'
        status, tester_out = compare_outputs(
            sandbox_path, input_path, correct_output_path, tester
        )
        print(status, tester_out)
    elif isolate_status == IsolateStatus.RUNTIME_EXCEPTION:
        status = TestingStatus.RUNTIME_EXCEPTION
    elif isolate_status == IsolateStatus.SIGNAL:
        status = TestingStatus.RUNTIME_EXCEPTION
    elif isolate_status == IsolateStatus.TIME_LIMIT_EXCEEDED:
        status = TestingStatus.TIME_LIMIT_EXCEEDED

    execution_metadata['test']['status'] = str(status)
    if status == TestingStatus.WRONG_ANSWER:
        tester_out = str(tester_out, 'utf-8', errors='replace')
        execution_metadata['test']['details'] = '\n'.join([
            ' ' + x for x in tester_out.splitlines()
        ])

    return status, execution_metadata


def evaluate_batch(sandbox_path, task_config, batch):
    mode = testing_mode(task_config)
    batch_protocol = configparser.ConfigParser()
    batch_status = TestingStatus.OK

    for input_name in batch:
        if batch_already_failed(batch_status):
            batch_protocol.add_section(input_name)
            batch_protocol[input_name]['status'] = str(TestingStatus.IGNORED)
            continue

        status, execution_metadata = evaluate_test_case(
            sandbox_path, task_config, input_name
        )
        print(status)
        batch_protocol[input_name] = execution_metadata['test']

        if status != TestingStatus.OK:
            print('BATCH NOT OK')
            if batch_status == TestingStatus.OK:
                batch_status = status
            if mode == TestingMode.BATCH or mode == TestingMode.ACM:
                break

    return batch_status, batch_protocol


def input_file_list(path):
    return [os.path.basename(x) for x in glob.glob(os.path.join(path, '*.in'))]


def write_results_to_files(status, testing_protocol, batch_results):
    with open(TEST_PROTOCOL_PATH, 'w') as test_log_file:
        testing_protocol.write(test_log_file, space_around_delimiters=False)

    with open(TEST_RESULT_STATUS_PATH, 'w') as result_status_file:
        result_status_file.write(str(status))

    with open(BATCH_RESULTS_PATH, 'w') as batch_results_file:
        for key in sorted(batch_results.keys()):
            batch_results_file.write(f'{batch_results[key]}\n')


def split_into_batches(input_files):
    batches = {}
    for input_file in sorted(input_files):
        batch = input_file[:input_file.index('.')]
        if batch not in batches:
            batches[batch] = []
        batches[batch].append(input_file)
    for batch in batches.values():
        batch.sort()
    return batches


def evaluate_submit(sandbox_path, task_config):
    input_files = input_file_list(TASK_DIRECTORY)
    batches = split_into_batches(input_files)

    final_status = TestingStatus.OK
    batch_results = {}
    testing_protocol = configparser.ConfigParser()

    for batch_key, batch in batches.items():
        batch_status, batch_protocol = evaluate_batch(
            sandbox_path, task_config, batch
        )
        if batch_status is not TestingStatus.OK:
            final_status = batch_status

        batch_results[batch_key] = batch_status
        for input_name in batch:
            testing_protocol[input_name] = batch_protocol[input_name]

    write_results_to_files(final_status, testing_protocol, batch_results)
    return final_status


def execute_submit(sandbox_path):
    task_config_path = os.path.join(TASK_DIRECTORY, 'task.config')
    task_config = read_config(task_config_path)

    if 'custom_execute' in task_config['task']:
        return custom_execute(task_config)

    final_status = evaluate_submit(sandbox_path, task_config)
    return final_status
