#!/usr/bin/python3
import sys, os, subprocess, configparser, shutil, imp
from collections import defaultdict

def report_message(message):
    print(message)
    # FIXME
    pass

def execute_external( command_line, what_with_stderr = None ):
    try:
        out = subprocess.check_output( command_line, stderr=what_with_stderr )
        err = 0
    except subprocess.CalledProcessError as exc:
        out = exc.output
        err = exc.returncode
    return out, err

def colorize(text):
    red, green, gray = '\033[1;31m', '\033[1;32m', '\033[0;37m'
    if text=='OK': return green+text+gray
    return red+text+gray

def enum(**enums): return type('Enum', (), enums)
testing_mode = enum(ACM=0, ALL=1, BATCH=2)

def process(rootdir,submit_basename):
    bindir, tasksdir, spooldir, tmpdir = rootdir+'/bin/', rootdir+'/tasks/', rootdir+'/spool/', rootdir+'/tmp/'

    cfg = configparser.RawConfigParser()
    try:
        tmp = cfg.read(spooldir+'/'+submit_basename+'.config')
        assert len(tmp)==1
        submit = dict( cfg.items('submit') )
    except:
        report_message('INTERNAL ERROR: failed to parse the config file of the submission.')
        return 'IN'

    if submit['contest'] == 'LIAHEN3':
        tasksdir = rootdir+'/tasks-liahen/'

    taskdir = tasksdir+'/'+submit['task']+'/'
    if not os.path.isdir(taskdir):
        report_message('INTERNAL ERROR: task directory missing.')
        return 'IN'

    taskcfg = taskdir+'metadata.config'
    if not os.path.isfile(taskcfg):
        report_message('INTERNAL ERROR: task config missing.')
        return 'IN'

    cfg = configparser.RawConfigParser()
    try:
        tmp = cfg.read(taskcfg)
        assert len(tmp)==1
        task = dict( cfg.items('task') )
    except:
        report_message('INTERNAL ERROR: failed to parse the config file of the task.')
        return 'IN'
    
    if 'custom_execute' in task:
        module_name = task['custom_execute']
        if module_name.endswith('.py'): module_name = module_name[:-3]
        mfile, mpath, mdesc = imp.find_module( module_name, [taskdir] )
        print(mpath)
        modul = imp.load_module('custom_execute', mfile, mpath, mdesc)
        return modul.custom_process(rootdir,submit_basename)

    timelimit = float(task['timelimit']) / 1000.
    walllimit = min( 1.7*timelimit, timelimit+3 )
    extratime = max( 0.1, min( timelimit/10., 0.5 ) )
    
    memlimit = int( task['memlimit'] )

    tester = cfg.get( 'task', 'tester', fallback='' )

    mode = testing_mode.ACM
    if 'mode' in cfg['task']:
        if cfg['task']['mode'].lower() == 'acm':     mode = testing_mode.ACM
        if cfg['task']['mode'].lower() == 'all':     mode = testing_mode.ALL
        if cfg['task']['mode'].lower() == 'testall': mode = testing_mode.ALL
        if cfg['task']['mode'].lower() == 'batch':   mode = testing_mode.BATCH
    # for backwards compatibility:
    if 'testall' in cfg['task']:
        if cfg['task']['testall'].lower() in ['t','true','yes','1']: mode = testing_mode.ALL

    # cmdline.append( bindir+'/isolate' )

    isolate_args = []
    isolate_args.append( '--meta='+tmpdir+'/'+submit_basename+'.hmeta' )
    
    isolate_args.append( '--mem={}'.format( memlimit ) )
    if submit['language'] == 'java': isolate_args[-1] = '--mem={}'.format( memlimit+1912345 )
    if submit['language'] == 'clj':
        isolate_args[-1] = '--mem={}'.format( memlimit+1912345 )
        timelimit += 1.5 # startup enginu :(
        walllimit = min( 1.7*timelimit, timelimit+3 )
        extratime = max( 0.1, min( timelimit/10., 0.5 ) )

    if submit['language'] == 'cs':   isolate_args[-1] = '--mem={}'.format( memlimit+145678 )
    if submit['language'] == 'py':   isolate_args[-1] = '--mem={}'.format( memlimit+33000 )
    
    isolate_args.append( '--time={0:.3f}'.format( timelimit ) )
    isolate_args.append( '--wall-time={0:.3f}'.format( walllimit ) )
    isolate_args.append( '--extra-time={0:.3f}'.format( extratime ) )
    isolate_args.append( '--stdin=test.in' )
    isolate_args.append( '--stdout=test.out' )
    # isolate_args.append( '--dir=/etc' )
    # isolate_args.append( '--processes' )
    # FIXME tu by este mali ist control groups
    isolate_args.append( '--env=PATH=/usr/bin' )
    isolate_args.append( '--env=MONO_SHARED_DIR=/tmp' )
    isolate_args.append( '--env=LANG=en_US.UTF-8' )
    isolate_args.append( '--env=MONO_GC_PARAMS=max-heap-size={}k'.format(memlimit) )
    # isolate_args.append( '--run' )
    # isolate_args.append( '--' )
    
    all_tests_cfg = configparser.RawConfigParser()
    final_status = 'OK'

    maxlen = 0
    for vstup in sorted(os.listdir(taskdir)):
        if vstup.endswith('.in'): maxlen = max( maxlen, len(vstup) )

    batch_results = defaultdict(lambda:'OK')

    lastgroup = ''
    for vstup in sorted(os.listdir(taskdir)):
        if not (vstup.endswith('.in')): continue
        # print( ' {:>{}}: '.format(vstup,maxlen), end='')

        group = vstup[ : vstup.index('.') ]
        if group != lastgroup:
            if lastgroup != '' and mode != testing_mode.ACM: print()
            if lastgroup != '' and mode == testing_mode.ACM: print('  ',end='')
            lastgroup = group
            print('{}:'.format(group), end='')

        subgroup = vstup[ vstup.index('.')+1 : -3 ]
        print( ' '+subgroup, end='' )
        sys.stdout.flush()

        vystup = vstup[:-3] + '.out'
        batch = vstup.split('.')[0]
        if mode == testing_mode.BATCH and batch_results[batch] != 'OK' and vstup.count('.sample.')==0 and vstup.count('.example.')==0:
            # print( 'skipped (batch already failed)' )
            print( ' skip ;', end='' )
            sys.stdout.flush()
            all_tests_cfg.add_section(vstup)
            all_tests_cfg[vstup]['status'] = 'IG'
            continue

        box, err = execute_external( [ bindir+'/isolate', '--init' ] )
        box = str( box.strip(), 'utf-8', errors='ignore' )
        shutil.copy( taskdir+'/'+vstup, box+'/box/test.in' )

        if submit['language'] in ['c','cc','cpp','d','pas','dpr']:
            shutil.copy( tmpdir+'/binarka', box+'/box/' )
            out, err = execute_external( [bindir+'/isolate'] + isolate_args + ['--run','--','./binarka'], what_with_stderr=subprocess.STDOUT )
        
        if submit['language'] == 'hs':
            shutil.copy( tmpdir+'/binarka', box+'/box/' )
            out, err = execute_external( [bindir+'/isolate'] + isolate_args + ['--run','--','./binarka','+RTS','-K1G','-RTS'], what_with_stderr=subprocess.STDOUT )
        
        if submit['language'] == 'py':
            shutil.copy( spooldir+'/'+submit_basename+'.data', box+'/box/src.py' )
            out, err = execute_external( [bindir+'/isolate'] + isolate_args + ['--dir=/etc','--run','--','/usr/bin/python3','src.py'], what_with_stderr=subprocess.STDOUT )
        
        if submit['language'] == 'js':
            shutil.copy( spooldir+'/'+submit_basename+'.data', box+'/box/src.js' )
            out, err = execute_external( [bindir+'/isolate'] + isolate_args + ['--dir=/etc','-p','--run','--','/usr/bin/nodejs','src.js'], what_with_stderr=subprocess.STDOUT )
        
        if submit['language'] == 'lisp':
            shutil.copy( spooldir+'/'+submit_basename+'.data', box+'/box/src.lisp' )
            out, err = execute_external( [bindir+'/isolate'] + isolate_args + ['--dir=/etc','--run','--','/usr/bin/sbcl','--noinform','--dynamic-space-size','800','--load','src.lisp','--quit'], what_with_stderr=subprocess.STDOUT )
        
        if submit['language'] == 'pl':
            shutil.copy( spooldir+'/'+submit_basename+'.data', box+'/box/src.pl' )
            out, err = execute_external( [bindir+'/isolate'] + isolate_args + ['--run','--','/usr/bin/perl','src.pl'], what_with_stderr=subprocess.STDOUT )
        
        if submit['language'] == 'clj':
            shutil.copy( spooldir+'/'+submit_basename+'.data', box+'/box/src.clj' )
            shutil.copy( bindir+'/clojure/clojure-1.8.0.jar', box+'/box/clojure.jar' )
            out, err = execute_external( [bindir+'/isolate'] + isolate_args + ['--dir=/etc','-p','--run','--','/usr/bin/java','-Xmx{}k'.format(memlimit),'-Xss256m','-cp','clojure.jar','clojure.main','src.clj'], what_with_stderr=subprocess.STDOUT )
        
        if submit['language'] == 'cs':
            shutil.copy( tmpdir+'/binarka.exe', box+'/box/binarka.exe' )
            # print(isolate_args)
            out, err = execute_external( [bindir+'/isolate'] + isolate_args + ['-p','--run','--','/usr/bin/mono','binarka.exe'], what_with_stderr=subprocess.STDOUT )
        
        if submit['language'] == 'java':
            main='zrubem_sa_lebo_takuto_classu_nemam'
            for classa in sorted(os.listdir(tmpdir)):
                if not (classa.endswith('.class')): continue
                try: shutil.copy( tmpdir+'/'+classa, box+'/box/' )
                except: pass # nemalo by nastat
                if classa.count('$')==0: main = classa[:-6]
            # print(isolate_args)
            out, err = execute_external( [bindir+'/isolate'] + isolate_args + ['--dir=/etc','-p','--run','--','/usr/bin/java','-Xmx{}k'.format(memlimit),'-Xss256m',main], what_with_stderr=subprocess.STDOUT )
    
        this_test_cfg = configparser.RawConfigParser()
        tmp = this_test_cfg.read( tmpdir+'/'+submit_basename+'.hmeta' )
        if len(tmp)==0:
            # FIXME nieco sa dokaslalo, internal error treba hodit
            status = 'IN'
            runtime = 0
        else:
            status = this_test_cfg.get('test','status',fallback='OK')
            runtime = float( this_test_cfg.get('test','time',fallback='0') )
       
        if status=='OK':
            # treba otestovat ci OK alebo WA
            if tester=='':
                out, err = execute_external( [ 'diff', '-q', box+'/box/test.out', taskdir+'/'+vystup ], what_with_stderr=subprocess.STDOUT )
                if err != 0:
                    status = 'WA'
                    #tester_out, err = execute_external( [ bindir+'/progdiff', taskdir+'/'+vystup, box+'/box/test.out', 'nas ', 'tvoj' ], what_with_stderr=subprocess.STDOUT )
                    tester_out, err = execute_external( [ bindir+'/progdiff', taskdir+'/'+vystup, box+'/box/test.out', 'our ', 'your' ], what_with_stderr=subprocess.STDOUT )
            else:
                tester_out, err = execute_external( [ taskdir+'/'+tester, tasksdir, submit['task'], taskdir+'/'+vstup, taskdir+'/'+vystup, box+'/box/test.out' ], what_with_stderr=subprocess.STDOUT )
                if err!=0: status='WA' # tester vratil nenulovy exit code => rejectol

            this_test_cfg['test']['status'] = status
            if status=='WA':
                tester_out = str( tester_out, 'utf-8', errors='replace' )
                this_test_cfg['test']['details'] = '\n'.join( [ ' '+x for x in tester_out.splitlines() ] )

        all_tests_cfg[vstup] = this_test_cfg['test']

        # subprocess.call( [ bindir+'/isolate', '--cleanup' ] )
       
        batch_results[batch] = status
        print( ' '+colorize(status), runtime, ';', end='' )
        sys.stdout.flush()
        if final_status=='OK' and status!='OK': final_status = status
        if (mode == testing_mode.ACM) and (status!='OK'): break
    
    # dotestovane
    f = open( tmpdir+'/'+submit_basename+'.tlog', 'w' )
    all_tests_cfg.write( f, space_around_delimiters=False)
    f.close()

    f = open( tmpdir+'/'+submit_basename+'.tresult', 'w' )
    f.write(final_status)
    f.close()
    
    if mode == testing_mode.BATCH:
        # nechame v batch_results len tie batche ktore nemaju v sebe ziaden sample
        for vstup in sorted(os.listdir(taskdir)):
            if not (vstup.endswith('.in')): continue
            batch = vstup.split('.')[0]
            if vstup.count('.sample.')>0 or vstup.count('.example.')>0:
                try:
                    del batch_results[batch]
                except:
                    pass
        # vypiseme ich vysledky do suboru
        f = open( tmpdir+'/'+submit_basename+'.bresults', 'w' )
        for key in sorted(batch_results.keys()): f.write('{}\n'.format(batch_results[key]))
        f.close()
    
    print()
    print('Final result: --> {} <--'.format(colorize(final_status)))

    # FIXME debug
    # f = open('/home/testovac/debuglog','a')
    # f.write(submit_basename+' '+final_status+'\n')
    # f.close()

    return final_status

###     
###     if submit['language'] == 'py':
###         unknown_language = False
###         cmdline.append( '/usr/bin/python3' )
###         cmdline.append( '-m' )
###         cmdline.append( 'py_compile' )
###         cmdline.append( submit_filename )
###      
###     if submit['language'] == 'java':
###         unknown_language = False
###         cmdline.append( '/usr/bin/javac' )
###         cmdline.append( submit_filename )
###     
###     if submit['language'] == 'cs':
###         unknown_language = False
###         subprocess.call( [bindir+'/kill-mono-semaphores.sh'] )
###         cmdline.append( '/usr/bin/gmcs' )
###         cmdline.append( submit_filename )
### 

############################################################################################

if __name__=="__main__":
    # toto sa normalne nevykonava, je to tu len na testovanie
    if len(sys.argv)>1:
        res = process('/home/testovac',sys.argv[1])
        print('done: {}'.format(res))

