from enum import Enum
import subprocess

from constants import (
    COMPILATION_METADATA_PATH,
    COMPILATION_CPU_TIME_LIMIT_SECONDS,
    COMPILATION_WALL_TIME_LIMIT_SECONDS,
    COMPILATION_MAX_NUMBER_OF_PROCESSES,
    COMPILATION_MEMORY_LIMIT_KILOBYTES,
    EXECUTION_MAX_NUMBER_OF_PROCESSES,
    EXECUTION_TEMP_METADATA_PATH,
)


class IsolateStatus(Enum):
    OK = 'OK'
    TIME_LIMIT_EXCEEDED = 'TO'
    INTERNAL_ERROR = 'XX'
    RUNTIME_EXCEPTION = 'RE'
    SIGNAL = 'SG'


def initialise_sandbox():
    sandbox_path_bytes = subprocess.check_output(['isolate', '--init'])
    sandbox_path = str(sandbox_path_bytes.strip(), 'utf-8', errors='ignore')
    return sandbox_path


def compilation_command(language):
    isolate_run = [
        'isolate',
        f'--meta={COMPILATION_METADATA_PATH}',
        f'--mem={COMPILATION_MEMORY_LIMIT_KILOBYTES}',
        f'--time={COMPILATION_CPU_TIME_LIMIT_SECONDS}',
        f'--wall-time={COMPILATION_WALL_TIME_LIMIT_SECONDS}',
        '--dir=/etc',
        # '--dir=/usr/lib/ghc',
        # '--dir=/var/lib/ghc',
        f'--processes={COMPILATION_MAX_NUMBER_OF_PROCESSES}',
        # FIXME tu by este mali ist control groups
        '--env=PATH=/usr/bin',
        '--env=MONO_SHARED_DIR=/tmp',
        '--run',
        '--',
    ]
    isolate_run.extend(language.compile_command)
    return isolate_run


def execution_command(command, time_limits, memory_limit):
    isolate_run = [
        'isolate',
        f'--meta={EXECUTION_TEMP_METADATA_PATH}',
        f'--time={time_limits["base"]:.3f}',
        f'--wall-time={time_limits["wall"]:.3f}',
        f'--extra-time={time_limits["extra"]:.3f}',
        '--stdin=test.in',
        '--stdout=test.out',
        # '--dir=/etc',
        f'--processes={EXECUTION_MAX_NUMBER_OF_PROCESSES}',
        '--env=PATH=/usr/bin',
        '--env=MONO_SHARED_DIR=/tmp',
        '--env=LANG=en_US.UTF-8',
        f'--env=MONO_GC_PARAMS=max-heap-size={memory_limit}k',
        '--run',
        '--',
    ]
    return isolate_run + command


def cleanup_sandbox():
    return subprocess.call(['isolate', '--cleanup'])
