#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "util.h"

#include <unistd.h>
#include <getopt.h>

#include <cstdio>
#include <iostream>
#include <fstream>
#include <string>
#include <vector>

#include "regex.hh"

using std::ofstream;
using std::auto_ptr;
using std::string;
using std::cerr;

const char *nfa_file;
const char *dfa_file;
std::vector<const char *> input_files;
string regex_pat;
uint32_t flags=0;

int parse_args(int argc, char *argv[])
{
    static const char opts[]="hvxi";
    static const struct option opts_long[]={
        {"nfa",         required_argument,  NULL, 2    },
        {"dfa",         required_argument,  NULL, 3    },
        {"line-regexp", no_argument,        NULL, 'x'  },
        {"ignore-case", no_argument,        NULL, 'i'  },
        {"help",        no_argument,        NULL, 'h'  },
        {"version",     no_argument,        NULL, 'v'  },
        {NULL,          0,                  NULL, 0    } // sentinel
    };
    static const char multiple_graphs_msg[]=
        "Writing graphs to multiple files not supported. Stop.\n";

    for(;;) {
        int t=getopt_long(argc, argv, opts, opts_long, NULL);
        switch(t) {
            case -1:
                t=optind;
                if(!argv[t]) {
                    fputs("Pattern required. Stop.\n", stderr);
                    return 2;
                }
                regex_pat=argv[t++];
                while(argv[t])
                    input_files.push_back(argv[t++]);
                return 0;
            case 'v':
                puts(PACKAGE_STRING " " __DATE__ " " __TIME__ "\n"
                        "This program is free software; you may redistribute it under the terms of\n"
                        "the GNU General Public License <http://www.gnu.org/licenses/gpl.html>.\n"
                        "This program has absolutely no warranty.");
                return 1;
            case 'h':
                printf("usage: %s [options] <pattern> [<filename>]...\n"
                        "Options:\n"
                        "  -h, --help\t\t\tDisplay this information\n"
                        "  -v, --version\t\t\tDisplay this program's version number\n"
                        "  -x, --line-regexp\t\tSelect only those matches that exactly match "
                                                "the whole line.\n"
                        "  -i, --ignore-case\t\tIgnore case distinctions in both the <pattern> "
                                                "and the input files.\n"
                        "      --nfa=<filename>\t\tWrite nfa graph to <filename>\n"
                        "      --dfa=<filename>\t\tWrite dfa graph to <filename>\n",
                        argv[0]);
                return 1;
            case 'x':
                flags|=regex::MATCH_FULL;
                break;
            case 'i':
                flags|=regex::MATCH_CASE_INSENSITIVE;
                break;
            case 2:
                if(nfa_file) {
                    fputs(multiple_graphs_msg, stderr);
                    return 2;
                }
                nfa_file=optarg;
                break;
            case 3:
                if(dfa_file) {
                    fputs(multiple_graphs_msg, stderr);
                    return 2;
                }
                dfa_file=optarg;
                break;
            case '?':
            default:
                return 2;
        }
    }
}

int main(int argc, char *argv[])
{
    int ret=parse_args(argc, argv);
    if(ret==1)
        return 0;
    else if(ret)
        return 1;

    try {
        auto_ptr<ofstream> nfa_str(nfa_file?new ofstream(nfa_file):NULL);
        auto_ptr<ofstream> dfa_str(dfa_file?new ofstream(dfa_file):NULL);


        regex r(regex_pat, flags, nfa_str.get(), dfa_str.get());

        size_t i=0;
        do {
            auto_ptr<std::ifstream> input_str;
            std::istream *input;
            if(input_files.empty())
                input=&std::cin;
            else {
                input_str.reset(new std::ifstream(input_files[i]));
                if(!*input_str) {
                    string t=input_files[i];
                    string::size_type j=0;
                    while((j=t.find_first_of("'\\", j))!=string::npos) {
                        t.insert(j, "\\");
                        j+=2;
                    }
                    fprintf(stderr, "Unable to open file `%s' for reading.\n", t.c_str());
                }
                input=input_str.get();
            }

            string s;
            while(getline(*input, s))
                if(r.match(s))
                    std::cout << s << std::endl;
        } while(++i<input_files.size());
    }
    catch(regex::exception_parse &e) {
        e.print(cerr, argv[1]);
    }
#if DEBUGLEVEL >= 2
    catch(std::exception &e) {
        cerr << "Caught exception: " << e.what() << std::endl;
    }
    catch(...) {
        cerr << "Unexpected exception.\n";
    }
#endif
    return 0;
}
