package client;


import common.StaticValues;
import exceptions.DBException;
import exceptions.FileNotSharedException;
import exceptions.GroupUsedInFilesException;
import exceptions.MyException;
import exceptions.SharingNonExistingFileException;
import exceptions.UserNotRegisteredException;
import java.io.File;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;

public class ShareManager implements InfoProvider, StaticValues {
    
    public static final String STANDARD_GROUP = "general";
    public static final short STANDARD_OTHER_MODE = StaticValues.VISIBLE_FILE;
    public static final short STANDARD_GROUP_MODE = StaticValues.DOWNLOADABLE_FILE;
    public static final int ROOT_ID = 0;
    
    private static final String REG_USERS_TABLE = "reg_users";
    private static final String USER_GROUPS_TABLE = "user_groups";
    private static final String GROUP_MODS_TABLE = "groups";
    private static final String FILES_TABLE = "files";
    private static final String DIRS_TABLE = "dirs";
    
    private ArrayList<RegisteredUserListener> registeredUsersListeners = new ArrayList<RegisteredUserListener>();
    private ArrayList<GroupRegisterListener> groupRegisterListenrs = new ArrayList<GroupRegisterListener>();
    
    private PreparedStatement addUserStatement;
    private PreparedStatement isCorrectLoginSt;
    private PreparedStatement isRegisteredSt;
    private PreparedStatement removeUserFromAllGroupsStatement;
    private PreparedStatement removeUserStatement;
    private PreparedStatement assocUserWithGroupStatement;
    private PreparedStatement removeUserFromGroupStatement;
    
    private PreparedStatement getUsersStatemnet;
    private PreparedStatement getGroupsStatement;
    private PreparedStatement getGroupsForUserStatement;
    private PreparedStatement addGroupStatemnet;
    
    private PreparedStatement removeGroupStatemnet;
    private PreparedStatement removeAllUsersFromGroupStatement;
    /** is group used in file accessibility */
    private PreparedStatement isUsedGroupStatement;
    
    private PreparedStatement updateFileGroupsStatement;
    private PreparedStatement updateDirGroupsStatement;
    private PreparedStatement usersInGroupNotInOtherSt;
    
//    private PreparedStatement maxFileIdsStatement;
    private PreparedStatement idOfDirStatement;
    private PreparedStatement shareDirStatement;
    private PreparedStatement idOfFileStatement;
    private PreparedStatement shareFileStatement;
    
    private PreparedStatement isAlredyRootDirSt;
    private PreparedStatement isAlredyRootFileSt;
    
    private PreparedStatement removeFileFromRoots;
    private PreparedStatement removeDirFromRoots;
    private PreparedStatement setFileAsRootSt;
    private PreparedStatement setDirdAsRootSt;
    
    private PreparedStatement removeFileStatement;
    private PreparedStatement removeDirStatement;
    private PreparedStatement removeFilesWithThisParentSt;
    private PreparedStatement getDirsWithThisParentSt;
    
    private PreparedStatement getTopLevelDirsSt;
    private PreparedStatement getRootDirsStatement;
    private PreparedStatement getRootFilesStatement;
    private PreparedStatement getFileNameSt;
    
    private PreparedStatement setFilePubAccessSt;
    private PreparedStatement setDirPubAccessSt;
    private PreparedStatement setFileGroupAccessSt;
    private PreparedStatement setDirGroupAccessSt;
    
    private PreparedStatement setDirGroupSt;
    private PreparedStatement setFileGroupSt;
    
    private PreparedStatement setUserPasswordSt;
    
    private PreparedStatement getFileAccessSt;
    private PreparedStatement getParentNameIDSt;
    private PreparedStatement getShareAmountSt;
    
    /** sets the atributes as path dir*/
    private PreparedStatement setDirAsPathDirSt;
    private PreparedStatement setFileAttributesSt;
    private PreparedStatement setDirAttributesSt;
    
    private Connection con;
    
    public ShareManager(Connection con) throws SQLException {
        this.con = con;
        isCorrectLoginSt = con.prepareStatement("SELECT * FROM " + REG_USERS_TABLE + " WHERE username = ? AND passwd = ?" );
        isRegisteredSt = con.prepareStatement("SELECT * FROM " + REG_USERS_TABLE + " WHERE username = ?");
        addUserStatement = con.prepareStatement("INSERT INTO " + REG_USERS_TABLE + " VALUES (?,?)");
        removeUserStatement = con.prepareStatement("DELETE FROM "+REG_USERS_TABLE +" WHERE username = ?");
        removeUserFromAllGroupsStatement = con.prepareStatement("DELETE FROM " + USER_GROUPS_TABLE + " WHERE username = ?");
        assocUserWithGroupStatement = con.prepareStatement("INSERT INTO " + USER_GROUPS_TABLE + " VALUES (?,?)");
        removeUserFromGroupStatement = con.prepareStatement("DELETE FROM " + USER_GROUPS_TABLE + " WHERE username = ? AND" +
                " groupname = ?");
        
        getUsersStatemnet = con.prepareStatement("SELECT username FROM " + REG_USERS_TABLE + " ORDER BY username");
        getGroupsStatement = con.prepareStatement("SELECT groupname FROM " + GROUP_MODS_TABLE + " ORDER BY groupname");
        getGroupsForUserStatement = con.prepareStatement("SELECT groupname FROM " + USER_GROUPS_TABLE + " WHERE username = ?");
        addGroupStatemnet = con.prepareStatement("INSERT INTO " + GROUP_MODS_TABLE + " VALUES (?)");
        
        removeGroupStatemnet = con.prepareStatement("DELETE FROM " + GROUP_MODS_TABLE + " WHERE groupname = ?");
        isUsedGroupStatement = con.prepareStatement("SELECT file_id FROM " + FILES_TABLE + " WHERE groupname = ? UNION " +
                "SELECT dir_id FROM " + DIRS_TABLE +" WHERE groupname = ?");
        removeAllUsersFromGroupStatement = con.prepareStatement("DELETE FROM " + USER_GROUPS_TABLE + " WHERE groupname = ?");
        updateFileGroupsStatement = con.prepareStatement("UPDATE " + FILES_TABLE + " SET groupname = ? WHERE groupname = ?");
        updateDirGroupsStatement = con.prepareStatement("UPDATE " + DIRS_TABLE + " SET groupname = ? WHERE groupname = ?");
        
        usersInGroupNotInOtherSt = con.prepareStatement("SELECT U.username FROM " + USER_GROUPS_TABLE + " U WHERE groupname = ? AND " +
                " ? NOT IN (SELECT groupname FROM " + USER_GROUPS_TABLE +" U2 WHERE U.username = U2.username )");
        idOfDirStatement = con.prepareStatement("SELECT dir_id FROM " + DIRS_TABLE + " WHERE parent_id = ? AND dir_name = ?");
        idOfFileStatement = con.prepareStatement("SELECT file_id FROM " + FILES_TABLE + " WHERE parent_id = ? AND file_name = ?" +
                " AND extension = ?");
        shareDirStatement = con.prepareStatement("INSERT INTO " + DIRS_TABLE + " VALUES (?,DEFAULT,?,?,?,?,?)");
        shareFileStatement = con.prepareStatement("INSERT INTO " + FILES_TABLE + " VALUES (?,DEFAULT,?,?,?,?,?,?,?)");
        
        isAlredyRootDirSt = con.prepareStatement("SELECT 1 FROM " + DIRS_TABLE +  " WHERE dir_id = ? and isroot <> 0");
        isAlredyRootFileSt = con.prepareStatement("SELECT 1 FROM " + FILES_TABLE + " WHERE file_id = ? and isroot <> 0");
        
        removeDirStatement = con.prepareStatement("DELETE FROM "+ DIRS_TABLE + " WHERE dir_id = ? ");
        removeFileStatement = con.prepareStatement("DELETE FROM "+ FILES_TABLE + " WHERE file_id = ?");
        
        removeDirFromRoots = con.prepareStatement("UPDATE " + DIRS_TABLE + " SET isroot = 0 WHERE dir_id = ?");
        removeFileFromRoots = con.prepareStatement("UPDATE " + FILES_TABLE + " SET isroot = 0 WHERE file_id = ?");
        
        setDirdAsRootSt = con.prepareStatement("UPDATE " + DIRS_TABLE + " SET isroot = 1 WHERE dir_id = ?");
        setFileAsRootSt = con.prepareStatement("UPDATE " + FILES_TABLE + " SET isroot = 1 WHERE file_id = ?");
        
        removeFilesWithThisParentSt = con.prepareStatement("DELETE FROM " + FILES_TABLE + " WHERE parent_id = ?");
        getDirsWithThisParentSt = con.prepareStatement("SELECT dir_id, isroot FROM " + DIRS_TABLE + " WHERE parent_id = ?");
        
        
        getTopLevelDirsSt = con.prepareStatement("SELECT dir_id,dir_name,groupname,other_mode, group_mode FROM dirs WHERE parent_id = 0");
        
        getRootDirsStatement = con.prepareStatement("SELECT * FROM " + DIRS_TABLE + " WHERE isroot <> 0");
        getRootFilesStatement = con.prepareStatement("SELECT * FROM " + FILES_TABLE + " WHERE isroot <> 0");
        
        getFileNameSt = con.prepareStatement("SELECT file_name, extension, parent_id FROM files WHERE file_id = ?");
        
        setFilePubAccessSt = con.prepareStatement("UPDATE " +  FILES_TABLE + " SET other_mode = ? WHERE file_id = ?");
        setDirPubAccessSt = con.prepareStatement("UPDATE " + DIRS_TABLE + " SET other_mode = ? WHERE dir_id = ?");
        setFileGroupAccessSt = con.prepareStatement("UPDATE " +  FILES_TABLE + " SET group_mode = ? WHERE file_id = ?");
        setDirGroupAccessSt = con.prepareStatement("UPDATE " + DIRS_TABLE + " SET group_mode = ? WHERE dir_id = ?");
        
        
        setDirGroupSt = con.prepareStatement("UPDATE " + DIRS_TABLE + " SET groupname = ? where dir_id = ?");
        setFileGroupSt = con.prepareStatement("UPDATE " + FILES_TABLE + " SET groupname = ? where file_id = ?");
        
        getFileAccessSt = con.prepareStatement("SELECT groupname, group_mode, other_mode FROM files WHERE file_id = ?");
        
        getParentNameIDSt = con.prepareStatement("SELECT dir_name,parent_id FROM dirs WHERE dir_id = ?");
        setUserPasswordSt = con.prepareStatement("UPDATE reg_users SET passwd = ? WHERE username = ?");
        getShareAmountSt = con.prepareStatement("SELECT sum(size) FROM files");
        
        setDirAsPathDirSt = con.prepareStatement("UPDATE dirs SET group_mode = " + PATH_FILE + " ,other_mode = " +
                PATH_FILE + " WHERE dir_id = ?");
        
        setFileAttributesSt = con.prepareStatement("UPDATE files SET groupname = ?, group_mode = ?, other_mode = ? WHERE file_id = ?");
        setDirAttributesSt = con.prepareStatement("UPDATE dirs SET groupname = ?, group_mode = ?, other_mode = ? WHERE dir_id = ?");
    }
    
    public void addGroup(String groupName) throws SQLException {
        addGroupStatemnet.setString(1,groupName);
        addGroupStatemnet.executeUpdate();
        fireGroupRegistered(groupName);
    }
    
    public void addUser(String userName,String passwd) throws SQLException {
        addUserStatement.setString(1,userName);
        addUserStatement.setString(2,passwd);
        addUserStatement.executeUpdate();
        fireUserRegistered(userName);
    }
    
    public void removeUser(String userName) throws SQLException {
        removeUserFromAllGroupsStatement.setString(1,userName);
        removeUserFromAllGroupsStatement.executeUpdate();
        removeUserStatement.setString(1,userName);
        removeUserStatement.executeUpdate();
        fireUserUnRegistered(userName);
    }
    
    public void associateUserWithGroup(String username,String group) throws SQLException {
        assocUserWithGroupStatement.setString(1,username);
        assocUserWithGroupStatement.setString(2,group);
        assocUserWithGroupStatement.executeUpdate();
    }
    
    public void removeUserFromGroup(String user,String group) throws SQLException {
        removeUserFromGroupStatement.setString(1,user);
        removeUserFromGroupStatement.setString(2,group);
        removeUserFromGroupStatement.executeUpdate();
    }
    
    public ArrayList<String> getUsers() throws SQLException {
        ArrayList<String> result = new ArrayList<String>(50);
        
        ResultSet rs = getUsersStatemnet.executeQuery();
        while (rs.next()) {
            result.add(rs.getString(1));
        }
        return result;
    }
    
    public ArrayList<String> getGroups() {
        
        ArrayList<String> result = new ArrayList<String>(30);
        try {
            
            ResultSet rs = getGroupsStatement.executeQuery();
            
            while (rs.next()) {
                result.add(rs.getString(1));
            }
        } catch (SQLException ex) {
            ex.printStackTrace();
        }
        return result;
    }
    
    public ArrayList<String> getGroupsForUser(String userName) throws SQLException, UserNotRegisteredException {
        if (getUsers().indexOf(userName) == -1) throw new UserNotRegisteredException(userName);
        
        ArrayList<String> result = new ArrayList<String>();
        getGroupsForUserStatement.setString(1,userName);
        ResultSet rs = getGroupsForUserStatement.executeQuery();
        
        while (rs.next()) {
            result.add(rs.getString(1));
        }
        
        return result;
    }
    
    /**
     *Test what happens when a file is in this group
     */
    public void removeGroup(String groupName) throws SQLException, GroupUsedInFilesException {
        isUsedGroupStatement.setString(1,groupName);
        isUsedGroupStatement.setString(2,groupName);
        ResultSet rs = isUsedGroupStatement.executeQuery();
        if (rs.next()) throw new GroupUsedInFilesException(groupName);
        removeGroupStatemnet.setString(1,groupName);
        removeGroupStatemnet.executeUpdate();
        
        removeAllUsersFromGroupStatement.setString(1,groupName);
        removeAllUsersFromGroupStatement.executeUpdate();
        fireGroupUnRegistered(groupName);
    }
    
    /**
     *Substitute group must exist in database
     */
    public void removeGroup(String groupName,String substituteGroup) throws SQLException {
        updateDirGroupsStatement.setString(1,substituteGroup);
        updateDirGroupsStatement.setString(2,groupName);
        updateDirGroupsStatement.executeUpdate();
        
        updateFileGroupsStatement.setString(1,substituteGroup);
        updateFileGroupsStatement.setString(2,groupName);
        updateFileGroupsStatement.executeUpdate();
        
        usersInGroupNotInOtherSt.setString(1,groupName);
        usersInGroupNotInOtherSt.setString(2,substituteGroup);
        ResultSet rs = usersInGroupNotInOtherSt.executeQuery();
        
        String un;
        while (rs.next()) {
            un = rs.getString(1);
            assocUserWithGroupStatement.setString(1,un);
            assocUserWithGroupStatement.setString(2,substituteGroup);
            assocUserWithGroupStatement.executeUpdate();
        }
        
        removeAllUsersFromGroupStatement.setString(1,groupName);
        removeAllUsersFromGroupStatement.executeUpdate();
        
        removeGroupStatemnet.setString(1,groupName);
        removeGroupStatemnet.executeUpdate();
    }
    
    private int getDirID(int parent_id,String dir_name) throws SQLException {
        idOfDirStatement.setInt(1,parent_id);
        idOfDirStatement.setString(2,dir_name);
        ResultSet rs = idOfDirStatement.executeQuery();
        if (!rs.next()) return -1;
        return rs.getInt(1);
    }
    
    private int getFileID(int parent_id,String file_name,String ext) throws SQLException {
        idOfFileStatement.setInt(1,parent_id);
        idOfFileStatement.setString(2,file_name);
        idOfFileStatement.setString(3,ext);
        ResultSet rs = idOfFileStatement.executeQuery();
        if(!rs.next()) return -1;
        return rs.getInt(1);
    }
    
    private int sharePathParts(String[] pathParts,String groupname,short otherMode) throws SQLException {
        //  int nextID = getNextFileID();
        int i;
        int parent_id = 0;
        for (i = 0;i<pathParts.length;i++) {
            idOfDirStatement.setInt(1,parent_id);
            idOfDirStatement.setString(2,pathParts[i]);
            ResultSet rs = idOfDirStatement.executeQuery();
            if (!rs.next()) {
//                shareDirStatement.setInt(1,parent_id);
//                shareDirStatement.setString(2,pathParts[i]);
//                shareDirStatement.setString(3,STANDARD_GROUP);
//                shareDirStatement.setShort(4,StaticValues.VISIBLE_FILE);
//                shareDirStatement.setShort(5,StaticValues.VISIBLE_FILE);
//                shareDirStatement.setShort(6,StaticValues.NOT_ROOT_FILE);
//                shareDirStatement.executeUpdate();
//                parent_id = getDirID(parent_id,pathParts[i]);
//                i++;
                break;
            } else {
                parent_id = rs.getInt(1);
            }
        }
        
        if (i < pathParts.length) {
            for (int k = i;k<pathParts.length;k++) {
                shareDirStatement.setInt(1,parent_id);
                shareDirStatement.setString(2,pathParts[k]);
                shareDirStatement.setString(3,STANDARD_GROUP);
                shareDirStatement.setInt(4,StaticValues.VISIBLE_FILE);
                shareDirStatement.setShort(5,StaticValues.VISIBLE_FILE);
                shareDirStatement.setShort(6,StaticValues.NOT_ROOT_FILE);
                shareDirStatement.executeUpdate();
                parent_id = getDirID(parent_id,pathParts[k]);
            }
        }
        return parent_id;
    }
    
    public void setUserPassword(String username,String password) throws SQLException {
        setUserPasswordSt.setString(1,password);
        setUserPasswordSt.setString(2,username);
        setUserPasswordSt.executeUpdate();
    }
    
    
    private boolean isAlredyRoot(File f) throws SQLException {
        if (f.isDirectory()) {
            int id = getDirID(f);
            isAlredyRootDirSt.setInt(1,id);
            ResultSet rs = isAlredyRootDirSt.executeQuery();
            return rs.next();
        } else {
            int id = getFileID(f);
            isAlredyRootFileSt.setInt(1,id);
            ResultSet rs = isAlredyRootFileSt.executeQuery();
            return rs.next();
        }
        
    }
    
    public void addToRootFiles(File f) throws SQLException, FileNotSharedException {
        if (f.isDirectory()) {
            int id = getDirID(f);
            if (id == -1 ) throw new FileNotSharedException(f);
            
            setDirdAsRootSt.setInt(1,id);
            setDirdAsRootSt.executeUpdate();
        } else {
            int id = getFileID(f);
            if (id == -1) throw new FileNotSharedException(f);
            
            setFileAsRootSt.setInt(1,id);
            setFileAsRootSt.executeUpdate();
        }
    }
    
    public void addToRootFiles(OwnFileDescriptor ofd) throws SQLException {
        if (ofd.isDirectory()) {
            setDirdAsRootSt.setInt(1,ofd.getFile_id());
            setDirdAsRootSt.executeUpdate();
        } else {
            setFileAsRootSt.setInt(1,ofd.getFile_id());
            setFileAsRootSt.executeUpdate();
        }
    }
    
    public void removeFromRoots(OwnFileDescriptor ofd) throws SQLException {
        if (ofd.isDirectory()) {
            removeDirFromRoots.setInt(1,ofd.getFile_id());
            removeDirFromRoots.executeUpdate();
        } else {
            removeFileFromRoots.setInt(1,ofd.getFile_id());
            removeFileFromRoots.executeUpdate();
        }
    }
    
    private void setFileAttributes(int file_id,String groupname,short group_mode,short other_mode) throws SQLException {
        setFileAttributesSt.setString(1,groupname);
        setFileAttributesSt.setShort(2,group_mode);
        setFileAttributesSt.setShort(3,other_mode);
        setFileAttributesSt.setInt(4,file_id);
        setFileAttributesSt.executeUpdate();
    }
    
    private void setDirAttributes(int dir_id,String groupname,short group_mode,short other_mode) throws SQLException {
        setDirAttributesSt.setString(1,groupname);
        setDirAttributesSt.setShort(2,group_mode);
        setDirAttributesSt.setShort(3,other_mode);
        setDirAttributesSt.setInt(4,dir_id);
        setDirAttributesSt.executeUpdate();
    }
    
    public void shareFileAsRoot(File f) throws SharingNonExistingFileException, SQLException {
        try {
            shareFile(f);
            addToRootFiles(f);
        } catch (FileNotSharedException ex) {
            throw new SQLException(ex);
        }
    }
    
    public void shareFileAsRoot(File f,String group,short otherMode,short groupMod) throws SharingNonExistingFileException, SQLException {
        try {
            shareFile(f,group,otherMode,groupMod);
            addToRootFiles(f);
        } catch (FileNotSharedException ex) {
            throw new SQLException(ex);
        }
    }
    
    public void shareFileWithParentAsRoot(File f,String group,short otherMode,short groupMod) throws SharingNonExistingFileException, SQLException {
        shareFile(f,group,otherMode,groupMod);
        File parent = new File(f.getParent());
        try {
            addToRootFiles(parent);
        } catch (FileNotSharedException ex) {
            ex.printStackTrace();
        }
    }
    
    
    
    public void shareFile(File f) throws SharingNonExistingFileException, SQLException {
        shareFile(f,STANDARD_GROUP,STANDARD_OTHER_MODE,STANDARD_GROUP_MODE);
    }
    
    public void shareFile(File f, String group,short otherMode,short groupMod) throws SharingNonExistingFileException, SQLException {
        if (!f.exists()) throw new SharingNonExistingFileException(f);
        String sep = File.separator;
        if (sep.equals("\\")) sep += "\\";
        
        String[] pathParts = f.getPath().split(sep);
        int parent_id = 0;
        if (pathParts.length>1)
            parent_id = sharePathParts(Arrays.copyOfRange(pathParts,0,pathParts.length-1),group,otherMode);
        shareFileRec(parent_id,f,group,otherMode,groupMod);
        
//        System.out.println(f.getPath());
//        System.out.println(f.getName());
    }
    
    
    
    private void shareFile(int parent_id,File f,String group,short otherMod,short groupMod) throws SQLException, SharingNonExistingFileException {
        if (!f.exists()) throw new SharingNonExistingFileException(f);
        
        int indexOfLastDot = f.getName().lastIndexOf('.');
        String name,ext;
        if (indexOfLastDot != -1) {
            name = f.getName().substring(0,indexOfLastDot);
            ext = f.getName().substring(indexOfLastDot+1,f.getName().length());
        } else {
            name = f.getName();
            ext = "";
        }
        
        
        int file_id = getFileID(parent_id,name,ext);
        
        if (file_id == -1) {
            
            shareFileStatement.setInt(1,parent_id);
            shareFileStatement.setString(2,name);
            shareFileStatement.setString(3,ext);
            shareFileStatement.setLong(4,f.length());
            shareFileStatement.setString(5,group);
            shareFileStatement.setShort(6,otherMod);
            shareFileStatement.setShort(7,groupMod);
            shareFileStatement.setShort(8,StaticValues.NOT_ROOT_FILE);
            shareFileStatement.executeUpdate();
            
        } else {
            setFileAttributes(file_id,group,groupMod,otherMod);
        }
        
    }
    
    private void shareFileRec(int parent_id,File f,String group,short otherMod,short groupMode) throws SQLException, SharingNonExistingFileException {
        //   System.out.println(f.getAbsolutePath());
        if (!f.exists()) throw new SharingNonExistingFileException(f);
        if (f.isFile()) {
            shareFile(parent_id,f,group,otherMod,groupMode);
        } else {
            int this_id = getDirID(parent_id,f.getName());
            
            if (this_id == -1) {
                shareDirStatement.setInt(1,parent_id);
                shareDirStatement.setString(2,f.getName());
                shareDirStatement.setString(3,group);
                shareDirStatement.setShort(4,otherMod);
                shareDirStatement.setShort(5,groupMode);
                shareDirStatement.setShort(6,StaticValues.NOT_ROOT_FILE);
                shareDirStatement.executeUpdate();
                this_id = getDirID(parent_id,f.getName());
                
            } else {
                setDirAttributes(this_id,group,groupMode,otherMod);
            }
            
            
            File[] files = f.listFiles();
            if (files != null)
                for (int i = 0;i<files.length;i++) {
                shareFileRec(this_id,files[i],group,otherMod,groupMode);
                }
        }
    }
    
    public boolean isSharedFile(File f) throws SQLException {
        if (f.isDirectory()) {
            return !(getDirID(f) == -1);
        } else {
            return !(getFileID(f) == -1);
        }
    }
    
    /**
     *true if one of the files ancestor is shared;
     */
    public boolean isAncSharedAsRoot(File f) throws SQLException {
        String sep = File.separator;
        if (sep.equals("\\")) sep += "\\";
        String[] parts = f.getPath().split(sep);
        int temp_dir_id = 0;
        for (int i = 0;i<parts.length -1;i++) {
            temp_dir_id = getDirID(temp_dir_id,parts[i]);
            if (temp_dir_id != -1 && isRootDir(temp_dir_id)) return true;
        }
        return false;
    }
    
    private int getParentID(File f) throws SQLException {
        String sep = File.separator;
        if (sep.equals("\\")) sep += "\\";
        String[] parts = f.getPath().split(sep);
        int temp_dir_id = 0;
        for (int i = 0;i<parts.length -1;i++) {
            temp_dir_id = getDirID(temp_dir_id,parts[i]);
            if (temp_dir_id == -1) return -1;
        }
        
        return temp_dir_id;
    }
    
    /**
     * @return the file ID (not directory) in database,if file not shared the result is -1
     */
    
    private int getFileID(File f) throws SQLException {
        int temp_dir_id = getParentID(f);
        
        if (f.isFile()) {
            return getFileID(temp_dir_id,getFileNameWithoutExt(f.getName()),getFileExtension(f.getName()));
        } else {
            throw new RuntimeException("parameter is not normfile");
        }
    }
    
    private int getDirID(File f) throws SQLException {
        int temp_dir_id = getParentID(f);
        if (!f.isFile()) {
            return getDirID(temp_dir_id,f.getName());
        } else {
            throw new RuntimeException("parameter is not dir");
        }
    }
    
    private void setDirAsPathDir(int dir_id) throws SQLException {
        setDirAsPathDirSt.setInt(1,dir_id);
        setDirAsPathDirSt.executeUpdate();
    }
    
    public void unShare(File f) throws SQLException {
        
        if (f.isDirectory()) {
            
            unShareDir(getDirID(f));
        } else {
            unShareFile(getFileID(f));
        }
    }
    
    public void unShareDir(int dir_id) throws SQLException {
        unShareDirHlp(dir_id,true);
    }
    
    public void unShareFile(int file_id) throws SQLException {
        unShareFileHlp(file_id);
    }
    
    
    private void unShareDirHlp(int dir_id,boolean delThis) throws SQLException {
        removeFilesWithThisParentSt.setInt(1,dir_id);
        removeFilesWithThisParentSt.executeUpdate();
        
        getDirsWithThisParentSt.setInt(1,dir_id);
        ResultSet rs = getDirsWithThisParentSt.executeQuery();
        
        ArrayList<Integer> ids = new ArrayList<Integer>(100);
        
        boolean isRootSubfile = false;
        
        while (rs.next()) {
            boolean isRoot = rs.getShort("isroot") == StaticValues.ROOT_FILE;
            if (isRoot) isRootSubfile = true;
            
            if (!isRoot)
                ids.add(rs.getInt(1));
        }
        
        for (int id : ids) {
            unShareDirHlp(id,false);
        }
        
        if (isRootSubfile && !delThis) {
            setDirAsPathDir(dir_id);
        } else {
            removeDirStatement.setInt(1,dir_id);
            removeDirStatement.executeUpdate();
        }
    }
    
    private void unShareFileHlp(int file_Id) throws SQLException {
        removeFileStatement.setInt(1,file_Id);
        removeFileStatement.executeUpdate();
    }
    
    
    public void setFilePublicAccess(int file_id,short mode) throws SQLException {
        setFilePubAccessSt.setShort(1,mode);
        setFilePubAccessSt.setInt(2,file_id);
        setFilePubAccessSt.executeUpdate();
    }
    
    public void setDirPublicAccess(int dir_id,short mode) throws SQLException {
        setDirPubAccessSt.setShort(1,mode);
        setDirPubAccessSt.setInt(2,dir_id);
        setDirPubAccessSt.executeUpdate();
    }
    
    public void setFileGroupAccess(int file_id,short mode) throws SQLException {
        setFileGroupAccessSt.setShort(1,mode);
        setFileGroupAccessSt.setInt(2,file_id);
        setFileGroupAccessSt.executeUpdate();
    }
    
    public void setDirGroupAccess(int dir_id,short mode) throws SQLException {
        setDirGroupAccessSt.setShort(1,mode);
        setDirGroupAccessSt.setInt(2,dir_id);
        setDirGroupAccessSt.executeUpdate();
    }
    
    public void setDirGroup(int dir_id,String group) throws SQLException {
        setDirGroupSt.setString(1,group);
        setDirGroupSt.setInt(2,dir_id);
        setDirGroupSt.executeUpdate();
    }
    
    public void setFileGroup(int file_id,String group) throws SQLException {
        setFileGroupSt.setString(1,group);
        setFileGroupSt.setInt(2,file_id);
        setFileGroupSt.executeUpdate();
    }
    

    public void fireUserRegistered(String userName){
        for (RegisteredUserListener l : registeredUsersListeners )
            l.userRegistered(userName);
    }
    
    public void fireUserUnRegistered(String userName){
        for (RegisteredUserListener l : registeredUsersListeners )
            l.userUnRegistered(userName);
    }
    
    public void addGroupRegisterListener(GroupRegisterListener l) {
        groupRegisterListenrs.add(l);
    }
    
    public void removeGroupRegisterListener(GroupRegisterListener l) {
        groupRegisterListenrs.remove(l);
    }
    
    public void fireGroupRegistered(String groupname) {
        for (GroupRegisterListener l : groupRegisterListenrs) {
            l.groupRegistered(groupname);
        }
    }
    
    public void fireGroupUnRegistered(String groupname) {
        for (GroupRegisterListener l : groupRegisterListenrs) {
            l.groupUnRegistered(groupname);
        }
    }
    
    private String getFileExtension(String fileName) {
        int indexOfLastDot = fileName.lastIndexOf('.');
        String name,ext;
        if (indexOfLastDot != -1) {
            return  fileName.substring(indexOfLastDot+1,fileName.length());
        } else {
            return ext = "";
        }
    }
    
    private String getFileNameWithoutExt(String fileName) {
        int indexOfLastDot = fileName.lastIndexOf('.');
        String name,ext;
        if (indexOfLastDot != -1) {
            return  fileName.substring(0,indexOfLastDot);
        } else {
            return  fileName;
        }
    }
    
    public boolean isRootFile(int file_id) {
        try {
            isAlredyRootFileSt.setInt(1,file_id);
            ResultSet rs = isAlredyRootFileSt.executeQuery();
            return rs.next();
            
        } catch (SQLException ex) {
            ex.printStackTrace();
            throw new DBException(ex);
        }
    }
    
    public boolean isRootDir(int dir_id) {
        try {
            isAlredyRootDirSt.setInt(1,dir_id);
            ResultSet rs = isAlredyRootDirSt.executeQuery();
            return rs.next();
            
        } catch (SQLException ex) {
            ex.printStackTrace();
            throw new DBException(ex);
        }
    }
    
    public boolean isRoot(int id,boolean dir) {
        if (dir) {
            return isRootDir(id);
        } else {
            return isRootFile(id);
        }
    }
    
    public ArrayList<OwnFileDescriptor> getTopLevelFiles() {
        try {
            ResultSet rs = getTopLevelDirsSt.executeQuery();
            ArrayList<OwnFileDescriptor> result = OwnFileDescriptor.getDirFileDescriptors(rs);
//            rs = getTopLevelFilesSt.executeQuery();
//            result.addAll(OwnFileDescriptor.getNormFileDescriptors(rs));
            return result;
        } catch (SQLException ex) {
            ex.printStackTrace();
            throw new DBException(ex);
        }
    }
    
    public ArrayList<OwnFileDescriptor> getRootFiles() {
        try {
            ArrayList<OwnFileDescriptor> result;
            ResultSet rs = getRootDirsStatement.executeQuery();
            result = OwnFileDescriptor.getDirFileDescriptors(rs);
            
            rs = getRootFilesStatement.executeQuery();
            result.addAll(OwnFileDescriptor.getNormFileDescriptors(rs));
            return result;
        } catch (SQLException ex) {
            throw new DBException(ex);
        }
    }
    
/**if not registered returns null*/
    
    public ArrayList<String> getGroups(String username, String password) {
        try {
            isCorrectLoginSt.setString(1,username);
            isCorrectLoginSt.setString(2,password);
            ResultSet rs = isCorrectLoginSt.executeQuery();
            if (!rs.next()) return new ArrayList<String>();
            
            return getGroupsForUser(username);
        } catch (SQLException ex) {
            ex.printStackTrace();
            throw new DBException(ex);
        } catch (UserNotRegisteredException ex) {
            return new ArrayList<String>();
        }
    }
    
    public boolean isRegistered(String username) {
        
        try {
            isRegisteredSt.setString(1,username);
            ResultSet rs = isRegisteredSt.executeQuery();
            return rs.next();
        } catch (SQLException ex) {
            ex.printStackTrace();
        }
        return false;
    }
    
    public boolean canDownload(Profile profile, int fileId) {
        try {
            getFileAccessSt.setInt(1,fileId);
            ResultSet rs = getFileAccessSt.executeQuery();
            
            boolean isrow = rs.next();
            if (!isrow) return false;
            short access = rs.getShort("other_mode");
            if (access == StaticValues.DOWNLOADABLE_FILE) return true;
            ArrayList<String> groups = getGroups(profile.getName(),profile.getPassword());
            String group = rs.getString("groupname");
            if (!groups.contains(group)) return false;
            
            access = rs.getShort("group_mode");
            if (access == StaticValues.DOWNLOADABLE_FILE) return true;
            else return false;
            
        } catch (SQLException ex) {
            ex.printStackTrace();
        }
        return false;
    }
    
    /** returns the path of file with this id, if not in db resutl is null */
    
    public String getFilePath(int file_id) {
        StringBuilder res = new StringBuilder(30);
        
        try {
            getFileNameSt.setInt(1,file_id);
            ResultSet rs = getFileNameSt.executeQuery();
            if (!rs.next()) return null;
            String name = rs.getString(1);
            String ext = rs.getString(2);
            int parent = rs.getInt(3);
            res.append(name+"."+ext);
            
            getParentNameIDSt.setInt(1,parent);
            rs = getParentNameIDSt.executeQuery();
            
            while (rs.next()) {
                res.insert(0,rs.getString(1)+File.separator);
                parent = rs.getInt(2);
                getParentNameIDSt.setInt(1,parent);
                rs = getParentNameIDSt.executeQuery();
            }
            
            return res.toString();
        } catch (SQLException ex) {
            ex.printStackTrace();
        }
        return null;
    }
    
    public long getShareAmount() {
        ResultSet rs;
        try {
            rs = getShareAmountSt.executeQuery();
            rs.next();
            return rs.getLong(1);
        } catch (SQLException ex) {
            ex.printStackTrace();
        }
        return -1;
    }
    
    
    public void addRegisteredUserListener(RegisteredUserListener l) {
        registeredUsersListeners.add(l);
    }
    
    public void removeRegisteredUserListener(RegisteredUserListener l) {
        registeredUsersListeners.remove(l);
    }
    
    
}
