#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <pthread.h>
#include <stdbool.h>
#include <unistd.h>
#include <libxml/xmlreader.h>

#include "mikes.h"
#include "mikes_logs.h"
#include "util.h"
#include "mcl.h"
#include "functions.h"
#include "astar.h"
#include "base_module.h"
#include "pose.h"

#define POSTERIOR_CONST 0.95


pthread_mutex_t lidar_mcl_lock;

hypo_t hypo[2][HYPO_COUNT];
hypo_t current_hypo[HYPO_COUNT];
point poly_i[NUMBER_OF_VERTICES_I]; // pavilon I
point poly_a[NUMBER_OF_VERTICES_A]; // atrium
point poly_h[NUMBER_OF_VERTICES_H]; // miestnosti H3 a H6

line lines[NUMBER_OF_VERTICES]; //vsetky ciary z mapy

int world[STATES_H * STATES_W];

int activeHypo = 0;

int WorldAt(int col, int row)
{
	if (row >= 0 && row < STATES_H && col >= 0 && col < STATES_W) {
		return world[row*STATES_W + col];
	} else {
		return -1;
	}
}

int point_in_polygon(int n_vert, point *vertices, double test_x, double test_y)
{
   int i, j = 0;
   bool c = false;
   for (i = 0, j = n_vert - 1; i < n_vert; j = i++) {
      if ( ((vertices[i].y > test_y) != (vertices[j].y > test_y))
          && (test_x < (vertices[j].x - vertices[i].x) * (test_y - vertices[i].y) / (vertices[j].y - vertices[i].y) + vertices[i].x) )
      {
         c = !c;
      }
   }
   return c;
}

int is_in_corridor(int cx, int cy) {
   if (!point_in_polygon(NUMBER_OF_VERTICES_A, poly_a, cx, cy)
       && !point_in_polygon(NUMBER_OF_VERTICES_H, poly_h, cx, cy) 
       && point_in_polygon(NUMBER_OF_VERTICES_I, poly_i, cx, cy))
      return 1;
   else
      return 0;
}

const double epsilon = 0.000001;

void swap(double *a, double *b)
{
	double p = *a;
	*a = *b;
	*b = p;
}

void reduce(double x1, double x2, double *x3, double *x4)
{
	if (*x3 > x1)
	{
		if (*x3 < x2) *x3 = x1;
		else *x3 -= (x2 - x1);
	}
	if (*x4 > x1)
	{
		if (*x4 < x2) *x4 = x1;
		else *x4 -= (x2 - x1);
	}
}

int check_bouding_box_intersect(double x1, double y1, double x2, double y2, double x3, double y3, double x4, double y4)
{
	if (x2 < x1) swap(&x1, &x2);
	if (y2 < y1) swap(&y1, &y2);
	if (x4 < x3) swap(&x3, &x4);
	if (y4 < y3) swap(&y3, &y4);
	
	reduce(x1, x2, &x3, &x4);
	reduce(y1, y2, &y3, &y4);
	
	if((x3 <= x1) && (x4 >= x1) &&
	   (y3 <= y1) && (y4 >= y1))
		return 1;
	else
		return 0;
}

double line_intersection(
                         double x1, double y1,
                         double x2, double y2,
                         double x3, double y3,
                         double x4, double y4,
                         double *X, double *Y)
{	

   if (!check_bouding_box_intersect(x1,y1,x2,y2,x3,y3,x4,y4))
      return -3;
      
   double A1 = y2 - y1;
   double B1 = x1 - x2;
   double C1 = A1 * x1 + B1 * y1;
   
   double A2 = y4 - y3;
   double B2 = x3 - x4;
   double C2 = A2 * x3 + B2 * y3;
   
   double det = A1 * B2 - A2 * B1;
   if (det == 0) {
      //Lines are parallel
      return -1;
   } else {
      double x = (B2*C1 - B1*C2)/det;
      double y = (A1*C2 - A2*C1)/det;
      if ((fmin(x1,x2) < (x + epsilon))
          && (x < (fmax(x1,x2) + epsilon))
          && (fmin(y1,y2) < (y + epsilon))
          && (y < (fmax(y1,y2) + epsilon))
          && (fmin(x3,x4) < (x + epsilon))
          && (x < (fmax(x3,x4) + epsilon))
          && (fmin(y3,y4) < (y + epsilon))
          && (y < (fmax(y3,y4) + epsilon)))
      {
         *X = x;
         *Y = y;
         return 1;
      } else
         return -2;
   }
}

int get_line_intersection(double x1, double y1, double x2, double y2) {
   
   double nx = 0;
   double ny = 0;
   
   for (int i = 0; i < NUMBER_OF_VERTICES; ++i)
   {
		int li = line_intersection(x1, y1, x2, y2, lines[i].x1, lines[i].y1, lines[i].x2, lines[i].y2, &nx, &ny);
		if (li > 0)
			return 1;
   }
   
   return 0;
}

static double sqrt_2PI = sqrt(2 * M_PI);

double get_sensor_model(double d, double x) {
   
   double sigma = sqrt(2); 
   double scale_input = 40;  // in cm
   
   d /= scale_input;
   x /= scale_input;
   
   double exponent = (x - d) * (x - d) / (2 * sigma * sigma);
   double result = exp(-exponent) / (sigma * sqrt_2PI);
   
   return result;
}

void get_mcl_data(hypo_t *buffer)
{
   pthread_mutex_lock(&lidar_mcl_lock);
   memcpy(buffer, current_hypo, sizeof(hypo_t) * HYPO_COUNT);
   pthread_mutex_unlock(&lidar_mcl_lock);
}

double normAlpha(double alpha){
   if(alpha < 0){
      while(alpha < 0)
         alpha += 360;
   }
   else
      while(alpha >= 360)
         alpha -= 360;
   return alpha;
}


double generateGaussianNoise(double mu, double sigma)
{
   const double epsilon = 0.000000001;
   const double two_pi = 2.0*3.14159265358979323846;
   
   static double z0, z1;
   static int generate;
   generate = 1-generate;
   
   if (!generate)
      return z1 * sigma + mu;
   
   double u1, u2;
   do
   {
      u1 = rand() * (1.0 / RAND_MAX);
      u2 = rand() * (1.0 / RAND_MAX);
   }
   while ( u1 <= epsilon );
   
   z0 = sqrt(-2.0 * log(u1)) * cos(two_pi * u2);
   z1 = sqrt(-2.0 * log(u1)) * sin(two_pi * u2);
   return z0 * sigma + mu;
}

static double max_dist = 600*600;

double get_min_intersection_dist(double x1, double y1, double alpha) {
   
   double lm = max_dist;
   double min_dist = max_dist;
   double dist = max_dist;
   
   double x2 = x1 + lm*sin(alpha * M_PI / 180.0);
   double y2 = y1 - lm*cos(alpha * M_PI / 180.0);
   
   double nx = x1;
   double ny = y1;
   
   for (int i = 0; i < NUMBER_OF_VERTICES; ++i) {
	   if (line_intersection(x1, y1, x2, y2, lines[i].x1, lines[i].y1, lines[i].x2, lines[i].y2, &nx, &ny) > 0)
	   {
		   dist = (nx - x1)*(nx - x1) + (ny - y1)*(ny - y1);
		   if (dist < min_dist)
			   min_dist = dist;
	   }
	   
   }

   return sqrt(min_dist);
   
}

int init_world()
{
   // iniciovanie grid mapy => ak stvorcek je pretaty nejakou stenou, tak ho oznacime 0 (vykreslene cervenou) 
   for (int row = 0; row < STATES_H; ++row) {
      for (int col = 0; col < STATES_W; ++col) {

         int wrow = row*STATE_WIDTH;
         int wcol = col*STATE_WIDTH;
         if (get_line_intersection(wcol, wrow, wcol + STATE_WIDTH, wrow) > 0
             || get_line_intersection(wcol + STATE_WIDTH, wrow, wcol + STATE_WIDTH, wrow + STATE_WIDTH) > 0
             || get_line_intersection(wcol + STATE_WIDTH, wrow + STATE_WIDTH, wcol, wrow + STATE_WIDTH) > 0
             || get_line_intersection(wcol, wrow + STATE_WIDTH, wcol, wrow) > 0)
         {
            world[row*STATES_W + col] = 1;
         } else {
            world[row*STATES_W + col] = 0;
         }
         
      }
   }
   
   for (int row = 1; row < STATES_H-1; ++row) {
      for (int col = 1; col < STATES_W-1; ++col) {
		  if (world[row*STATES_W + col] == 0 && 
		      (world[(row-1)*STATES_W + col] == 1
		      || world[(row+1)*STATES_W + col] == 1
		      || world[(row)*STATES_W + col-1] == 1
		      || world[(row)*STATES_W + col+1] == 1
		      || world[(row-1)*STATES_W + col-1] == 1
		      || world[(row-1)*STATES_W + col+1] == 1
		      || world[(row+1)*STATES_W + col-1] == 1
		      || world[(row+1)*STATES_W + col+1] == 1))
		  {
		      world[(row)*STATES_W + col] = 2;
		  }
	  }
   }
   for (int row = 1; row < STATES_H-1; ++row) {
      for (int col = 1; col < STATES_W-1; ++col) {
		  if (world[row*STATES_W + col] == 2)
		      world[row*STATES_W + col] = 1;
	  }
   }

	//// write world to console
	//for (int row = 0; row < STATES_H; ++row) {
		//printf("%.3d ", row);
		//for (int col = 0; col < STATES_W; ++col) {
			//if (col % 20 == 0)
				//printf("|");
			//printf("%d", WorldAt(col, row));
		//}
		//printf("\n"); 
	//}
	
	
	return 1;
}

int mcl_update(double traveled, int heading, lidar_data_type liddata){
   mikes_log_double(ML_INFO, "MCL New data - traveled:", traveled);
   mikes_log_val(ML_INFO, "MCL New data - heading:", heading);

   long start_mcl_timestamp = usec_time();
   
   activeHypo = 1-activeHypo;
   
//   for(int i = 0; i < HYPO_COUNT; i++){
//      mikes_log_val(ML_DEBUG, "hypo id premove: ", i);
//      mikes_log_val2(ML_DEBUG, "hypo pos: ", hypo[1-activeHypo][i].x,hypo[1-activeHypo][i].y);
//      mikes_log_val2(ML_DEBUG, "hypo v&a: ", hypo[1-activeHypo][i].w*100,hypo[1-activeHypo][i].alpha);
//   }
   
   double max_hypo_prob = 0;
   
   for (int i = 0; i < HYPO_COUNT; i++)
   {
      
      double alpha_h = normAlpha(hypo[1-activeHypo][i].alpha+heading);
      hypo[1-activeHypo][i].x += traveled * sin(alpha_h * M_PI/180.0);
      hypo[1-activeHypo][i].y -= traveled * cos(alpha_h * M_PI/180.0);
      hypo[1-activeHypo][i].alpha = alpha_h;
         
      // point outside of the corridor
      if(!is_in_corridor(hypo[1-activeHypo][i].x, hypo[1-activeHypo][i].y)) {
         hypo[1-activeHypo][i].w = 0;
         continue;
      }
      
      
      //sensor position
      double possx = hypo[1-activeHypo][i].x + sin(hypo[1-activeHypo][i].alpha*M_PI/180.0) * 7;
      double possy = hypo[1-activeHypo][i].y - cos(hypo[1-activeHypo][i].alpha*M_PI/180.0) * 7;
      

      {
         double w_post = hypo[1-activeHypo][i].w;
         for (int j = 0; j < liddata.count; ++j) {
            uint16_t measured_distance = liddata.distance[j] / 40; // Actual distance = distance_q2 / 4 mm // but we want distance in cm
            if (measured_distance == 0 || (liddata.angle[j] > (120 * 64) && liddata.angle[j] < (240 * 64)))
				continue;
			if (measured_distance > 600)
				measured_distance = 600;
            
            uint16_t angle_64 = liddata.angle[j];
            double angle = angle_64 / 64.0;
            
            uint16_t computed_distance = (uint16_t) get_min_intersection_dist(possx, possy, angle);
            double smodel = get_sensor_model(computed_distance, measured_distance);
            
      //      if (abs(computed_distance - measured_distance) < 50)
    
            {
				double w_post_new = w_post * smodel;
				w_post_new = w_post_new / (double) NORM_PROB_CONST;
				w_post = POSTERIOR_CONST*w_post + (1-POSTERIOR_CONST)*w_post_new;
			}
			
            
            
         }
         
         mikes_log_double2(ML_DEBUG, "apriori | posterior ", hypo[1-activeHypo][i].w, w_post);
         
		 if (w_post > max_hypo_prob)
			 max_hypo_prob = w_post;
			
         hypo[1-activeHypo][i].w = w_post;
         
      }
   }
   
   for (int i = 0; i < HYPO_COUNT; i++)
   {
	   hypo[1-activeHypo][i].w /= max_hypo_prob;
   }
   
   
//   for(int i = 0; i< HYPO_COUNT; i++){
//      mikes_log_val(ML_DEBUG, "hypo id postmove: ", i);
//      mikes_log_val2(ML_DEBUG, "hypo pos: ", hypo[1-activeHypo][i].x,hypo[1-activeHypo][i].y);
//      mikes_log_val2(ML_DEBUG, "hypo v&a: ", hypo[1-activeHypo][i].w*100,hypo[1-activeHypo][i].alpha);
//   }
   
   // cumulative probability
   double cumP[HYPO_COUNT];
   double last = 0;
   for(int i = 0; i < HYPO_COUNT; i++){
      last += hypo[1-activeHypo][i].w;
      cumP[i] = last;
   }
   
   // generate new particle set
   int i;
   for(i = 0; i < HYPO_COUNT*0.9; i++)
   {
      double next = (double)rand() / (double)RAND_MAX * last;
      for(int j = 0; j < HYPO_COUNT; j++){
         if( next <= cumP[j]){
            hypo[activeHypo][i].x = hypo[1-activeHypo][j].x + generateGaussianNoise(0, 0.03*traveled);
            hypo[activeHypo][i].y = hypo[1-activeHypo][j].y + generateGaussianNoise(0, 0.03*traveled);
            hypo[activeHypo][i].alpha = normAlpha(hypo[1-activeHypo][j].alpha + generateGaussianNoise(0, heading*0.05));
            hypo[activeHypo][i].w = hypo[1-activeHypo][j].w;
            break;
         }
      }
   }
   
   // generate random particles
   for(; i< HYPO_COUNT; i++)
   {
      
      double rand_x = 0;
      double rand_y = 0;
      
      do {
         rand_x = rand() % MAP_W;
         rand_y = rand() % MAP_H;
      } while (!is_in_corridor(rand_x, rand_y));
      
      hypo[0][i].x = hypo[1][i].x = rand_x;
      hypo[0][i].y = hypo[1][i].y = rand_y;
      hypo[0][i].alpha = hypo[1][i].alpha = rand() % 360;
      hypo[0][i].w = hypo[1][i].w = 0.01;
      
   }
   
   pthread_mutex_lock(&lidar_mcl_lock);
   memcpy(current_hypo, hypo[activeHypo], sizeof(hypo_t) * HYPO_COUNT);
   pthread_mutex_unlock(&lidar_mcl_lock);
   
   long end_mcl_timestamp = usec_time();
   mikes_log_val(ML_DEBUG, "mcl time: ", (end_mcl_timestamp - start_mcl_timestamp) / 1000);
   return 0;
}


void *mcl_thread(void *arg)
{
	int mcl_update_threshold = 100; //minimal number of ticks to make mcl_update
	base_data_type base_data;
	lidar_data_type lidar_data;
	pose_type pose;
	sleep(3);
	
	get_base_data(&base_data);
	get_lidar_data(&lidar_data);
		
	base_data_type old_base_data;
	
	old_base_data = base_data;
	
	pose_type old_pose;
	
	
	int startx = 4;
	int starty = 22;
	int endx = 10;
	int endy = 24;
	int astar_result = astar(world, STATES_W, STATES_H, startx, starty, endx, endy, rectilinear);
	mikes_log_val(ML_INFO, "astar result:", astar_result);
	set_pose((startx*STATE_WIDTH + STATE_WIDTH / 2.0)*10, (starty*STATE_WIDTH + STATE_WIDTH / 2.0)*10, M_PI);
	
	get_pose(&pose);
	old_pose = pose;

	// mcl computes in cm
	while (program_runs)
	{
		
		get_base_data(&base_data);
		get_lidar_data(&lidar_data);
		get_pose(&pose);
		
		double vect_x = fabs(pose.x - old_pose.x);
		double vect_y = fabs(pose.y - old_pose.y);
		double dist =  vect_x*vect_x + vect_y*vect_y;
		//printf("mcl: counterA %d counterB %d, heading %d\n", base_data.counterA, base_data.counterB, base_data.heading);
		
		
		if (abs(old_base_data.counterA + old_base_data.counterB) - (base_data.counterA + base_data.counterB) < mcl_update_threshold)
		{
			
			double traveled =  sqrt(dist / 10);
			double heading = pose.heading * 180 / M_PI;
			
			mcl_update(traveled, heading, lidar_data);
			
			old_base_data = base_data;
			old_pose = pose;
		}
		
		
		sleep(3);
	}
	
	mikes_log(ML_INFO, "mcl quits.");
	threads_running_add(-1);
	return 0;
}

int init_mcl()
{
   pthread_t t;
   if (pthread_create(&t, 0, mcl_thread, 0) != 0)
   {
      perror("mikes:mcl");
      mikes_log(ML_ERR, "creating mcl thread");
   }
   else threads_running_add(1);
   
   pthread_mutex_init(&lidar_mcl_lock, 0);
   
   // seed random generator
   time_t tm;
   srand((unsigned) time(&tm));
   
   // get map
   get_lines_from_file("mapa_pavilonu_I.svg");
   get_polygons(poly_i, poly_a, poly_h); 
   init_world();
   
   base_data_type base_data;
   get_base_data(&base_data);
   int heading = base_data.heading;
   
   // init primary mcl particles
   for (int i = 0; i < HYPO_COUNT; i++){
      
      double rand_x = 0;
      double rand_y = 0;
      
      do {
         rand_x = rand() % MAP_W;
         rand_y = rand() % MAP_H;
      } while (!is_in_corridor(rand_x, rand_y));
      
      
      hypo[0][i].x = hypo[1][i].x = rand_x;
      hypo[0][i].y = hypo[1][i].y = rand_y;
      hypo[0][i].alpha = hypo[1][i].alpha = rand() % 360;
      
      int angle_diff = abs(heading - hypo[0][i].alpha);
      
      // incorporationg actual robot heading into primary mcl particles weight
      hypo[0][i].w = hypo[1][i].w = 0.05 + 0.3 * ((360 - angle_diff) / 360);
      
      //mikes_log_val(ML_INFO, "hypo id: ", i);
      //mikes_log_double2(ML_INFO, "hypo pos: ", hypo[0][i].x,hypo[0][i].y);
      //mikes_log_double2(ML_INFO, "hypo v&a: ", hypo[0][i].w,hypo[0][i].alpha);
   }
   
   return 0;
}
