import json

import requests
import typer

from ct_logs import CT_LOGS

ADD_CHAIN_SUFFIX = "/ct/v1/add-chain"


def main(
    chain_file: str = typer.Argument(
        help="Path to the .pem file with chain of certificates. "
        "Typically, the file is located at /etc/letsencrypt/live/.../fullchain.pem",
    ),
    log_name: str = typer.Option(
        help="The CT log in which the certificate should be inserted.",
        default="oak2023",
    ),
    verbose: bool = typer.Option(
        help="If true, it prints additional information about extracted chain and response.",
        default=False,
    ),
):
    if log_name not in CT_LOGS.keys():
        print(f"Error: {log_name} is not a valid/supported CT log name.")
        raise typer.Abort()

    with open(chain_file, "r") as f:
        chain = f.read()

    chain = chain.replace("\n", "")
    cert_begin = "-----BEGIN CERTIFICATE-----"
    cert_end = "-----END CERTIFICATE-----"
    certs = [x.replace(cert_begin, "") for x in chain.split(cert_end)[:-1]]
    chain_json = {"chain": [x for x in certs]}

    log_url = f"{CT_LOGS[log_name]}{ADD_CHAIN_SUFFIX}"
    headers = {
        "Content-Type": "application/json",
    }
    response = requests.post(log_url, json=chain_json, headers=headers)

    if verbose:
        print(f"Chain: {chain_json}")

    if response.status_code == 200:
        print(json.dumps(response.json(), indent=4))
        styled_text = typer.style("successfully", fg=typer.colors.GREEN, bold=True)
        bold_log_name = typer.style(log_name, bold=True)
        print(f"Certificate was {styled_text} inserted inside {bold_log_name} CT log.")
    else:
        print(response.status_code)
        print(response.json())


if __name__ == "__main__":
    typer.run(main)
