import string
import subprocess
from base64 import b32encode

import typer
from datetime import datetime, timezone

LABELS_IN_DOMAIN_LIMIT = (
    9  # 10, but because of DNS challenge, there is only 9 available
)
LABEL_LENGTH_LIMIT = 63
DOMAIN_BYTES_LENGTH_LIMIT = 253
SAN_PER_CERT_LIMIT = 100


def base32_safe_encode(data: bytes) -> bytes:
    return b32encode(data).rstrip(b"=")


def count_labels(url: str) -> int:
    return url.count(".") + 1


def is_correct(domain_name: str) -> bool:
    domain_name_bytes = len(domain_name.encode("utf-8"))
    labels_number = len(domain_name.split("."))

    if labels_number < 2:
        raise ValueError(f"Domain name must have at least two labels: '{domain_name}'!")

    if domain_name_bytes > DOMAIN_BYTES_LENGTH_LIMIT:
        raise ValueError(
            f"Domain name has {domain_name_bytes} bytes, but limit is {DOMAIN_BYTES_LENGTH_LIMIT}: {domain_name}"
        )

    if labels_number > LABELS_IN_DOMAIN_LIMIT:
        raise ValueError(
            f"Domain name has {labels_number} labels, but limit is {LABELS_IN_DOMAIN_LIMIT} labels: {domain_name}"
        )

    for i, label in enumerate(domain_name.split(".")):
        label_length = len(label)

        if label_length > LABEL_LENGTH_LIMIT:
            raise ValueError(
                f"Domain name has {label_length} characters, but limit is {LABEL_LENGTH_LIMIT} characters: {label}"
            )

    return True


def split_message(
    msg: str, remaining_bytes_per_san: int, remaining_labels: int
) -> list:
    start = 0
    san_prefixes = []

    while True:
        labels = []
        remaining_bytes_per_curent_san = remaining_bytes_per_san
        for i in range(0, remaining_labels):
            label_length = LABEL_LENGTH_LIMIT - 1
            if i == 0:
                label_length -= 1
            offset = min(label_length, remaining_bytes_per_curent_san)
            end = min(start + offset, len(msg))
            if i == 0:
                labels.append(f"{msg[start:end]}")
            else:
                labels.append(f"x{msg[start:end]}")
            remaining_bytes_per_curent_san -= end - start
            start += offset
            if start >= len(msg) or remaining_bytes_per_curent_san < 0:
                break
        san_prefixes.append(".".join(labels))
        if start >= len(msg) or remaining_bytes_per_curent_san < 0:
            break

    return san_prefixes


def get_sans_with_ordering(message_labels_parts: list) -> list:
    prefixes = []
    i = 0
    stop = False
    for c in string.ascii_lowercase:
        for n in range(0, 10):
            s = f"{c}{n}{message_labels_parts[i]}"
            prefixes.append(s)
            i += 1
            if i == len(message_labels_parts):
                stop = True
                break
        if stop:
            break
    return prefixes


def get_san(domain_prefix: str, subject_name: str) -> str:
    return f"{domain_prefix}.{subject_name}"


def main(
    message: str = typer.Argument(
        help="Message that will be encoded in the certificate.",
        # default="I'm sending a message through the CT channel. I will send another one on 30.6.2023 at 9:00:00 UTC.",
    ),
    subject_name: str = typer.Argument(
        help="Subject Name (SN) of the requested certificate, e.g. example.com.",
        default="ctlr.seclab.dcs.fmph.uniba.sk",
    ),
    production: bool = typer.Option(
        help="If true, it generates production certificate instead of testing certificate.",
        default=True,
    ),
    verbose: bool = typer.Option(
        help="If true, it prints additional information about encoding and content of SANs.",
        default=False,
    ),
    dry: bool = typer.Option(
        help="If true, it just mimics running a certbot script.",
        default=False,
    ),
):
    is_correct(subject_name)

    remaining_labels = LABELS_IN_DOMAIN_LIMIT - count_labels(subject_name)
    remaining_bytes_per_san_without_labels_limit = (
        DOMAIN_BYTES_LENGTH_LIMIT
        - len(subject_name)
        - remaining_labels  # dots between labels
        - remaining_labels  # safe character at the beginning of the label
        - 1  # prefix ordering
    )
    remaining_bytes_per_san = min(
        remaining_bytes_per_san_without_labels_limit,
        remaining_labels * (LABEL_LENGTH_LIMIT - 2) - 1,
    )
    remaining_bytes_msg = SAN_PER_CERT_LIMIT * remaining_bytes_per_san

    message_bytes = message.encode("utf-8")
    base32_url_message_bytes = base32_safe_encode(message_bytes)
    base32_url_message = base32_url_message_bytes.decode("utf-8")

    print(remaining_bytes_msg)

    if verbose:
        print(f"Subject name: {subject_name}")
        print(f"Message length: {len(message)}")
        print(f"Message bytes length: {len(message_bytes)}")
        print(f"Base32 encoded message bytes length: {len(base32_url_message_bytes)}")

    if len(base32_url_message) > remaining_bytes_msg:
        print(
            f"Encoded message is too long ({len(base32_url_message)} bytes)! With domain '{subject_name}', you only have {remaining_bytes_msg} remaining byte/s for your message."
        )
        raise typer.Abort()

    message_parts = split_message(
        base32_url_message, remaining_bytes_per_san, remaining_labels
    )
    domain_prefixes = get_sans_with_ordering(message_parts)
    sans = [get_san(x, subject_name) for x in domain_prefixes]

    [is_correct(x) for x in sans]

    sans_as_str = ",".join(sans)
    cmd = f'sudo certbot certonly --test-cert --manual --reinstall --break-my-certs --renew-with-new-domains --expand --force-renewal --manual-auth-hook /home/jurcak/cert_dns_hook.sh --preferred-challenges dns -d "{subject_name},{sans_as_str}"'

    if production:
        cmd = cmd.replace(" --test-cert", "")
    if verbose:
        print(f"\nSANs:")
        [print(san) for san in sans]
        print(f"\nRunning command: {cmd}")
    if not dry:
        subprocess.run(cmd, shell=True)

    now_utc = datetime.now(timezone.utc)
    formatted_date = now_utc.strftime("%Y-%m-%d_%H.%M.%S")
    filename = f"logs/{formatted_date}.log"
    if not dry:
        with open(filename, "w") as f:
            f.write(f"Subject name: {subject_name}\n")
            f.write(f"Message: {message}\n")
            f.write(f"Base32: {base32_url_message}\n")
            f.write("\nSANs:\n")
            for san in sans:
                f.write(f"{san}\n")

    print("\nCertificate with encoded message was successfully issued.")
    print(
        f"You can find additional data about message encoding in the log file: {filename}"
    )


if __name__ == "__main__":
    typer.run(main)
