//---------------------------------------------------------------------------

#ifndef StructuresH
#define StructuresH

//#define NOCALLBACK

#include <time.h>



//***************************************************************************
//*********** HASH TABLES ***************************************************
//***************************************************************************
#define HASHBIT	    (5)
#define HASHSIZE    (size_t)(1<<(3*HASHBIT))   /* hash table size (32768) */
#define MASK	      ((1<<HASHBIT)-1)
#define HASH(p_i,p_j,p_k) ((((((p_i)&MASK)<<HASHBIT)|((p_j)&MASK))<<HASHBIT)|((p_k)&MASK))

// index for hash tables
typedef struct t3index T3Index;
typedef struct t3index {
  long int I, J, K;
  friend int operator ==(const T3Index id1, const T3Index id2);
} T3Index;


//***************************************************************************
//*********** DIRECTIONS ****************************************************
//***************************************************************************
#define BIT(p_i, p_bit) (((p_i)>>(p_bit))&1)
#define IS_RIGHT(p_i) BIT((p_i), 0)
#define IS_TOP(p_i)   BIT((p_i), 1)
#define IS_FRONT(p_i) BIT((p_i), 2)
//#define ALL_DIR     7
//#define BACK(p_dir)   (p_dir)^ALL_DIR
#define BACK(p_dir) (p_dir)^1

//***************************************************************************
//*********** STRUCTURES ****************************************************
//***************************************************************************
typedef struct t3dpoint T3DPoint;
// Point in 3D space
typedef struct t3dpoint {
        double X, Y, Z;
        friend T3DPoint operator +(const T3DPoint pA, const T3DPoint pB);
        friend T3DPoint operator -(const T3DPoint pA, const T3DPoint pB);
        friend double operator *(const T3DPoint pA, const T3DPoint pB);
        friend T3DPoint operator %(const T3DPoint pA, const T3DPoint pB);
        friend T3DPoint operator *(const double c, const T3DPoint pA);
        friend T3DPoint operator /(const T3DPoint pA, const double c);
        
} T3DPoint;

typedef struct tcubevertex {
  double Value;
  T3Index Index;
} TCubeVertex;

typedef struct tcubevertices TCubeVertices;
typedef struct tcube TCube;
typedef struct tcubeedge TCubeEdge;
typedef struct tcubes TCubes;
typedef struct tpatchedge TPatchEdge;


typedef struct tcubevertices {
  TCubeVertex Vertex;
  TCubeVertices *Next;
} TCubeVertices;

typedef struct tcubeedge {
  TCubeVertex *Vertices[2];
  unsigned char Status; // 0-not intersected, 1-intersected, 2-new
  unsigned char Subdivide; // 0-no, 1-yes
  unsigned char SubdivisionLevel;
  unsigned char NormalComputed; // 0-no, 1-yes
  T3DPoint Intersection;
  T3DPoint Normal;
  TCubeEdge *Neighbours[4];  // 4 neighbour edges
  TCubes *Cubes[4];
  TPatchEdge *PatchEdges[4];
} TCubeEdge;

typedef struct tcubeedges TCubeEdges;

typedef struct tcubeedges {
  TCubeEdge Edge;
  TCubeEdges *Next;
} TCubeEdges;


typedef struct tcube {
  unsigned char Subdivide;  // 0 - no, 1 - yes
  unsigned char Evaluate;   // 0 - no, 1 - yes
  unsigned char SubdivisionLevel; // 0 - parent cube
  unsigned char Configuration;
//  unsigned char InList;     // 0 - no, 1 - yes
  T3Index Index;

  TCubes *Parent;
  TCubes *Children[8];
  TCubeVertex *Vertices[8];

//  TCube *Neighbour[6];

  TCubeEdge *Edges[12];

} TCube;

typedef struct tcubes {
  TCube Cube;
  struct tcubes *Next;
} TCubes;

typedef struct tcubeshash {
  TCubes Cube;
  struct tcubeshash *Next;
} TCubesHash;


typedef struct tpatch TPatch;
typedef struct tpatchedge {
  TCubeEdge *CubeEdges[2];      // connects 2 CubeEdges
  TPatch *Patches[2];           // edge belongs to 2 Patches
  TCube *Cubes[2];              // it is between 2 cubes
} TPatchEdge;

typedef struct tpatch {
  TCubeEdge *CubeEdge;
  TPatchEdge *Edge;
  TPatch *Next;
} TPatch;

typedef struct t3dnormals T3DNormals;
typedef struct t3dnormals {
  int Index;
  T3DPoint Normal;
  T3DNormals *Next;
} T3DNormals;

typedef struct t3dpoints T3DPoints;
typedef struct t3dpoints {
  int Index;
  TPatch *Patch;
  T3DPoint Point;
  T3DPoints *Next;
} T3DPoints;

typedef struct ttriangleedge {
  TPatch *Patch;
  T3DPoints *Vertices[2];
  T3DNormals *Normals[2];
} TTriangleEdge;

typedef struct ttriangleedges TTriangleEdges;
typedef struct ttriangleedges {
  T3Index Index;
  TTriangleEdge Edge;
  double Inactive;
  TTriangleEdges *Left;
  TTriangleEdges *Right;
  TTriangleEdges *NextInList;
  TTriangleEdges *NextInHash;
  TTriangleEdges *PreviousInHash;
} TTriangleEdges;


typedef struct ttriangle {
  T3DPoints *Vertices[3];
  T3DNormals *Normals[3];
} TTriangle;

typedef struct ttriangles TTriangles;
typedef struct ttriangles {
  TTriangle Triangle;
  TTriangles *Next;
} TTriangles;


typedef double (*TImplicitFunction)(double, double, double);

typedef struct tconfig{
  double STEP;          // cube size at lowest subdivision level
  double BIG_STEP;          // cube size at lowest subdivision level
  int RC_RATIO;      // (root-cube)/(last child cube size)

  int SUBDIVIDE;        // 1 - do subdivision, 0 - don't subdivide
  int MAX_DEPTH;     // max subdivision depth

  double ALPHA_SUBDIVISON;  // if angle > alpha then subdivide  
  double MIN_VALUE;     // minimal positive value for configuration determination

  double MAX_DISTANCE;  // max allowed istance from tangent plane for triangle vertex

  double EDGE_SIZE;     // triangele edge size
  double STEP3;         // ideal height of triangle ((sqrt(3)/2) * EDGE_SIZE)
  double MAX_STEP3;     // max height of triangle (for adaptive method)
  double MAX_STEP3_QUAD;     // MAX_STEP3 * MAX_STEP3
  double MIN_STEP3;     // min height if triangle
  double MIN_STEP3_QUAD;     // MIN_STEP3 * MIN_STEP3

  double ALPHA_DIVIDE;
  double ALPHA_STEP;

  double MAX_TN;        // max difference between normals in adjecant triangles (if good vertex was found)
  int MAX_TR_COUNT;
  T3DPoint STARTING_POSITION;
} TConfig;

typedef struct tstatistic{
  int Count;
  clock_t Start;
  clock_t End;
  double Max;
  double Sum;
} TStatistic;

typedef struct tsatistics{
  TStatistic Cubes;

  TStatistic Triangles;
  TStatistic Vertices;    // regular vertices
  TStatistic Angles;
  TStatistic Edges;

  TStatistic Patches;
  TStatistic All;
} TStatistics;

typedef struct tprocess TProcess;
typedef void (*TCallback)(TProcess *p, int step, int action);

typedef struct tprocess {
  TCubeVertices **FoundVertices;  // hash table for found cube-vertices
  TCubeEdges **FoundEdges;        // hash table for found cube-edges
  TCubesHash **FoundCubes;        // hash table for cubes
  TCubes *ActiveCubes;            // list of not processed cubes
  TCubes *ActiveCube;             // active cube

  TTriangles *FoundTriangles;     // list of found triangles

  TTriangleEdges **TriangleEdges; // hash table for triangle edges
  TTriangleEdges *ActiveEdges;    // list of active edges
  TTriangleEdges  *ActiveEdge;     // active edge

  T3DPoints *Vertices;            // list of triangle vertices
  T3DNormals *Normals;             // list of vertices normals

  TImplicitFunction Function;
  TCallback Callback;

  TConfig Config;
  TStatistics Stats;

  T3DPoints *NewVertex;

  int quit;
} TProcess;



#define S_INIT      0
#define S_MC        1
#define S_MT        2

#define A_IN_START  0
#define A_IN_END    1

#define A_MC_FIRST_CUBE       0
#define A_MC_EVALUATE         1
#define A_MC_FIND_EDGES       2
#define A_MC_SUBDIVIDE        3

#define A_MT_FIRST_EDGE       0
#define A_MT_TRIANGLE_CREATED 1
#define A_MT_NEW_EDGE         2
#define A_MT_PATCH_FOR_EDGE   3
#define A_MT_NEW_VERTEX       4
#define A_MT_CYCLING          5
#define A_MT_MAX_COUNT        6


#ifndef NOCALLBACK
  #define MCALLBACK(func, p, step, action) if(func != NULL) func(p, step, action);
#else
  #define MCALLBACK(func, p, step, action)
#endif


// Directions by direction, sign and cube-vertex
// index 1: 0-lft 1-dwn 2-bck
// index 2: 1-reverse
// index 3: vertices
const int CDIRS[3][2][4] = {{{0,2,4,6}, {1,3,5,7}},
{{0,4,1,5}, {2,6,3,7}},
{{0,1,2,3}, {4,5,6,7}}};

const int CEDGES[3][2][2] = {{{0, 6}, {1, 7}},
{{2, 8}, {3, 9}},
{{4, 10}, {5, 11}}};



#define CLEFT 0
#define CRIGHT 1
#define NO_EDGE -1
// how are vertices in cube patch connected
// CCONF[configuration][edge][CLEFT/RIGHT] = edge/NO_EDGE
const int CCONF[23][12][2] = {
// C0
  {{NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}},
// C1 625
  {{NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {6, 5}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {2, 6}, {5, 2}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}},
// C2 6287
  {{NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {6, 8}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {7, 2}, {8, 6},
   {2, 7}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}},
// C3 625 11,7,9
  {{NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {6, 5}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {2, 6}, {5, 2}, {11, 9},
   {NO_EDGE, NO_EDGE}, {7, 11}, {NO_EDGE, NO_EDGE}, {9, 7}},
// C4 625 9,1,10
  {{NO_EDGE, NO_EDGE}, {9, 10}, {6, 5}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {2, 6}, {5, 2}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {10, 1}, {1, 9}, {NO_EDGE, NO_EDGE}},
// C5 52017
  {{2, 1}, {0, 7}, {5, 0}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {7, 2}, {NO_EDGE, NO_EDGE}, {1, 5},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}},
// C6 6287 10,9,1
  {{NO_EDGE, NO_EDGE}, {9, 10}, {6, 8}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {7, 2}, {8, 6},
   {2, 7}, {10, 1}, {1, 9}, {NO_EDGE, NO_EDGE}},
// C7 6,11,3 875 10,9,1
  {{NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {11, 6},
   {NO_EDGE, NO_EDGE}, {7, 8}, {3, 11}, {8, 5},
   {5, 7}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {6, 3}},
// C8 6017
  {{6, 1}, {0, 7}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {7, 0}, {1, 6},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}},
// C9 6,3,10,1,8,5
  {{NO_EDGE, NO_EDGE}, {10, 8}, {NO_EDGE, NO_EDGE}, {6, 10},
   {NO_EDGE, NO_EDGE}, {8, 6}, {5, 3}, {NO_EDGE, NO_EDGE},
   {1, 5}, {NO_EDGE, NO_EDGE}, {3, 1}, {NO_EDGE, NO_EDGE}},
// C10 5,11,3,2   4,10,9,8
  {{NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {3, 5}, {11, 2},
   {8, 10}, {2, 11}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE},
   {9, 4}, {10, 8}, {4, 10}, {5, 3}},
// C11  560,10,98
  {{6, 10}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {8, 6}, {5, 0}, {NO_EDGE, NO_EDGE},
   {9, 5}, {10, 8}, {0, 9}, {NO_EDGE, NO_EDGE}},
// C12  52017 6,11,3
  {{2, 1}, {0, 7}, {5, 0}, {11, 6},
   {NO_EDGE, NO_EDGE}, {7, 2}, {3, 11}, {1, 5},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {6, 3}},
// C13 79,11 625 03,10 418
  {{10, 3}, {4, 8}, {6, 5}, {0, 10},
   {8, 1}, {2, 6}, {5, 2}, {11, 9},
   {1, 4}, {7, 11}, {3, 0}, {9, 7}},
// C14 23,10,175
  {{NO_EDGE, NO_EDGE}, {10, 7}, {5, 3}, {2, 10},
   {NO_EDGE, NO_EDGE}, {7, 2}, {NO_EDGE, NO_EDGE}, {1, 5},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {3, 1}, {NO_EDGE, NO_EDGE}},
// C15 6,3,10,1,8,5  9,11,7
  {{NO_EDGE, NO_EDGE}, {10, 8}, {NO_EDGE, NO_EDGE}, {6, 10},
   {NO_EDGE, NO_EDGE}, {8, 6}, {5, 3}, {11, 9},
   {1, 5}, {7, 11}, {3, 1}, {9, 7}},
// C16 2679,10,18    
  {{NO_EDGE, NO_EDGE}, {10, 8}, {8, 6}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {2, 7}, {6, 9},
   {1, 2}, {7, 10}, {9, 1}, {NO_EDGE, NO_EDGE}},
// C17 02571
  {{1, 2}, {7, 0}, {0, 5}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {2, 7}, {NO_EDGE, NO_EDGE}, {5, 1},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}},
// C18
  {{NO_EDGE, NO_EDGE}, {10, 9}, {5, 6}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {6, 2}, {2, 5}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {1, 10}, {9, 1}, {NO_EDGE, NO_EDGE}},
// C19 526,11,97
  {{NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {5, 6}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {7, 2}, {2, 11}, {9, 5},
   {NO_EDGE, NO_EDGE}, {11, 7}, {NO_EDGE, NO_EDGE}, {6, 9}},
// C20 2678
  {{NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {8, 6}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {2, 7}, {6, 8},
   {7, 2}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}},
// C21 526
  {{NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {5, 6}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {6, 2}, {2, 5}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}},
// C22
  {{NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE},
   {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}, {NO_EDGE, NO_EDGE}}
};

const int CROT[24][12] = {
/* 0  */ { 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11},
/* 1  */ { 2,  8,  6,  0,  5,  11, 3,  9,  7,  1,  4,  10},
/* 2  */ { 6,  7,  3,  2,  11, 10, 0,  1,  9,  8,  5,  4},
/* 3  */ { 3,  9,  0,  6,  10, 4,  2,  8,  1,  7,  11, 5},
/* 4  */ { 1,  7,  4,  10, 8,  2,  0,  6,  5,  11, 9,  3},
/* 5  */ { 7,  6,  8,  9,  5,  4,  1,  0,  2,  3,  11, 10},
/* 6  */ { 6,  0,  5,  11, 2,  8,  7,  1,  4,  10, 3,  9},
/* 7  */ { 10, 4,  3,  9,  0,  6,  11, 5,  2,  8,  1,  7},
/* 8  */ { 1,  0,  9,  8,  10, 11, 7,  6,  3,  2,  4,  5},
/* 9  */ { 4, 10,  8,  2,  1,  7,  5,  11, 9,  3,  0,  6},
/* 10 */ { 8,  9,  5,  4,  7,  6,  2,  3,  11, 10, 1,  0},
/* 11 */ { 9,  3,  7,  1,  11, 5,  8,  2,  6,  0,  10, 4},
/* 12 */ { 3,  2,  11, 10, 6,  7,  9,  8,  5,  4,  0,  1},
/* 13 */ { 7,  1,  11, 5,  9,  3,  6,  0,  10, 4,  8,  2},
/* 14 */ { 0,  6,  10, 4,  3,  9,  1,  7,  11, 5,  2,  8},
/* 15 */ { 9,  8,  10, 11, 1,  0,  3,  2,  4,  5,  7,  6},
/* 16 */ { 8,  2,  1,  7,  4,  10, 9,  3,  0,  6,  5,  11},
/* 17 */ { 2,  3,  4,  5,  0,  1,  8,  9,  10, 11, 6,  7},
/* 18 */ { 4,  5,  0,  1,  2,  3,  10, 11, 6,  7,  8,  9},
/* 19 */ { 5,  11, 2,  8,  6,  0,  4,  10, 3,  9,  7,  1},
/* 20 */ { 11, 10, 6,  7,  3,  2,  5,  4,  0,  1,  9,  8},
/* 21 */ { 10, 11, 1,  0,  9,  8,  4,  5,  7,  6,  3,  2},
/* 22 */ { 11, 5,  9,  3,  7,  1,  10, 4,  8,  2,  6,  0},
/* 23 */ { 5,  4,  7,  6,  8,  9,  11, 10, 1,  0,  2,  3}
};



const int CIROT[24][12] = {
  { 0,  1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11},
//  { 2,  8,  6,  0,  5,  11, 3,  9,  7,  1,  4,  10},
  { 3,  9,  0,  6, 10,  4,  2,  8,  1,  7,  11, 5},
//  { 6,  7,  3,  2,  11, 10, 0,  1,  9,  8,  5,  4},
  { 6,  7,  3,  2,  11, 10, 0,  1,  9,  8,  5,  4},
//  { 3,  9,  0,  6,  10, 4,  2,  8,  1,  7,  11, 5},
  { 2,  8,  6,  0,  5,  11, 3,  9,  7,  1,  4,  10},
//  { 1,  7,  4,  10, 8,  2,  0,  6,  5,  11, 9,  3},
  { 6,  0,  5,  11, 2,  8,  7,  1,  4,  10, 3,  9},
//  { 7,  6,  8,  9,  5,  4,  1,  0,  2,  3,  11, 10},
  { 7,  6,  8,  9,  5,  4,  1,  0,  2,  3,  11, 10},
//  { 6,  0,  5,  11, 2,  8,  7,  1,  4,  10, 3,  9},
  { 1,  7,  4,  10, 8,  2,  0,  6,  5,  11, 9,  3},
//  { 10, 4,  3,  9,  0,  6,  11, 5,  2,  8,  1,  7},
  { 4,  10, 8,  2,  1,  7,  5,  11, 9,  3,  0,  6},
//  { 1,  0,  9,  8,  10, 11, 7,  6,  3,  2,  4,  5},
  { 1,  0,  9,  8,  10, 11, 7,  6,  3,  2,  4,  5},
//  { 4, 10,  8,  2,  1,  7,  5,  11, 9,  3,  0,  6},
  { 10, 4,  3,  9,  0,  6,  11, 5,  2,  8,  1,  7},
//  { 8,  9,  5,  4,  7,  6,  2,  3,  11, 10, 1,  0},
  { 11, 10, 6,  7,  3,  2,  5,  4,  0,  1,  9,  8},
//  { 9,  3,  7,  1,  11, 5,  8,  2,  6,  0,  10, 4},
  { 9,  3,  7,  1,  11, 5,  8,  2,  6,  0,  10, 4},
//  { 3,  2,  11, 10, 6,  7,  9,  8,  5,  4,  0,  1},
  { 10, 11, 1,  0,  9,  8,  4,  5,  7,  6,  3,  2},
//  { 7,  1,  11, 5,  9,  3,  6,  0,  10, 4,  8,  2},
  { 7,  1,  11, 5,  9,  3,  6,  0,  10, 4,  8,  2},
//  { 0,  6,  10, 4,  3,  9,  1,  7,  11, 5,  2,  8},
  { 0,  6,  10, 4,  3,  9,  1,  7,  11, 5,  2,  8},
//  { 9,  8,  10, 11, 1,  0,  3,  2,  4,  5,  7,  6},
  { 5,  4,  7,  6,  8,  9,  11, 10, 1,  0,  2,  3},
//  { 8,  2,  1,  7,  4,  10, 9,  3,  0,  6,  5,  11},
  { 8,  2,  1,  7,  4,  10, 9,  3,  0,  6,  5,  11},
//  { 2,  3,  4,  5,  0,  1,  8,  9,  10, 11, 6,  7},
  { 4,  5,  0,  1,  2,  3,  10, 11, 6,  7,  8,  9},
//  { 4,  5,  0,  1,  2,  3,  10, 11, 6,  7,  8,  9},
  { 2,  3,  4,  5,  0,  1,  8,  9,  10, 11, 6,  7},
//  { 5,  11, 2,  8,  6,  0,  4,  10, 3,  9,  7,  1},
  { 5,  11, 2,  8,  6,  0,  4,  10, 3,  9,  7,  1},
//  { 11, 10, 6,  7,  3,  2,  5,  4,  0,  1,  9,  8},
  { 8,  9,  5,  4,  7,  6,  2,  3,  11, 10, 1,  0},
//  { 10, 11, 1,  0,  9,  8,  4,  5,  7,  6,  3,  2},
  { 3,  2,  11, 10, 6,  7,  9,  8,  5,  4,  0,  1},
//  { 11, 5,  9,  3,  7,  1,  10, 4,  8,  2,  6,  0},
  { 11, 5,  9,  3,  7,  1,  10, 4,  8,  2,  6,  0},
//  { 5,  4,  7,  6,  8,  9,  11, 10, 1,  0,  2,  3},
  { 9,  8,  10, 11, 1,  0,  3,  2,  4,  5,  7,  6}
};


#define C_ROT 0
#define C_CON 1

// rotation, configuration
const int CCALL[256][2] = {
{0, 0}, {3, 1}, {5, 1}, {3, 2}, {2, 1}, {15, 2}, {2, 3}, {16, 5},
{14, 1}, {19, 3}, {17, 2}, {15, 5}, {2, 2}, {17, 5}, {3, 5}, {3, 8},

{0, 1}, {4, 2}, {20, 3}, {6, 5}, {10, 3}, {7, 5}, {11, 7}, {0, 9},
{0, 4}, {20, 6}, {22, 6}, {0, 11}, {16, 6}, {7, 14}, {3, 12}, {10, 17},

{6, 1}, {3, 3}, {6, 2}, {0, 5}, {2, 4}, {15, 6}, {6, 6}, {0, 14},
{6, 3}, {2, 7}, {23, 5}, {4, 9}, {2, 6}, {17, 12}, {9, 11}, {11, 17},

{0, 2}, {5, 5}, {4, 5}, {0, 8}, {11, 6}, {6, 11}, {4, 12}, {2, 17},
{0, 6}, {5, 12}, {4, 14}, {14, 17}, {9, 10}, {1, 16}, {8, 16}, {1, 20},

{1, 1}, {4, 3}, {1, 4}, {5, 6}, {13, 2}, {20, 5}, {13, 6}, {7, 11},
{1, 3}, {5, 7}, {17, 6}, {15, 12}, {8, 5}, {3, 9}, {3, 14}, {1, 17},

{7, 2}, {18, 5}, {7, 6}, {6, 14}, {19, 5}, {7, 8}, {19, 12}, {21, 17},
{10, 6}, {18, 12}, {0, 10}, {9, 16}, {14, 11}, {22, 17}, {12, 16}, {9, 20},

{7, 3}, {8, 7}, {21, 6}, {0, 12}, {18, 6}, {20, 12}, {1, 10}, {14, 16},
{0, 7}, {1, 13}, {23, 12}, {4, 15}, {8, 12}, {3, 15}, {4, 16}, {0, 19},

{12, 5}, {1, 9}, {1, 11}, {13, 17}, {12, 14}, {9, 17}, {23, 16}, {14, 20},
{12, 12}, {1, 15}, {19, 16}, {18, 19}, {3, 16}, {12, 19}, {3, 18}, {8, 21},

{8, 1}, {3, 4}, {12, 3}, {3, 6}, {18, 3}, {19, 6}, {1, 7}, {16, 12},
{14, 2}, {23, 6}, {9, 5}, {9, 14}, {13, 5}, {3, 11}, {8, 9}, {12, 17},

{0, 3}, {4, 6}, {3, 7}, {6, 12}, {4, 7}, {7, 12}, {0, 13}, {0, 15},
{14, 6}, {3, 10}, {9, 12}, {18, 16}, {13, 12}, {21, 16}, {8, 15}, {7, 19},

{9, 2}, {12, 6}, {22, 5}, {4, 11}, {9, 6}, {2, 10}, {22, 12}, {10, 16},
{21, 5}, {21, 12}, {9, 8}, {19, 17}, {13, 14}, {7, 16}, {18, 17}, {7, 20},

{1, 5}, {1, 14}, {5, 9}, {8, 17}, {1, 12}, {17, 16}, {5, 15}, {1, 19},
{10, 11}, {13, 16}, {20, 17}, {13, 20}, {5, 16}, {1, 18}, {4, 19}, {1, 21},

{1, 2}, {8, 6}, {1, 6}, {7, 10}, {14, 5}, {14, 14}, {14, 12}, {0, 16},
{2, 5}, {2, 12}, {13, 11}, {11, 16}, {2, 8}, {4, 17}, {5, 17}, {0, 20},

{11, 5}, {12, 11}, {11, 12}, {2, 16}, {2, 9}, {23, 17}, {2, 15}, {6, 19},
{2, 14}, {6, 16}, {15, 16}, {2, 18}, {0, 17}, {6, 20}, {3, 19}, {6, 21},

{10, 5}, {10, 12}, {10, 14}, {16, 16}, {2, 11}, {22, 16}, {20, 16}, {0, 18},
{11, 9}, {11, 15}, {7, 17}, {10, 19}, {6, 17}, {20, 19}, {4, 20}, {0, 21},

{1, 8}, {3, 17}, {17, 17}, {2, 20}, {15, 17}, {17, 20}, {19, 19}, {14, 21},
{16, 17}, {2, 19}, {15, 20}, {2, 21}, {3, 20}, {5, 21}, {3, 21}, {0, 22}
};


// C_DIR_EDGES[direction][0..3] = edge
// face 0-left
//      1-right
//      2-up
//      3-down
//      4-front
//      5-back
const int C_DIR_EDGES[6][4] = {
/*  0 */  { 0,  2,  3,  6},
/*  1 */  { 1,  7,  8,  9},
/*  2 */  { 3,  9, 10, 11},
/*  3 */  { 2,  4,  5,  8},
/*  4 */  { 5,  6,  7, 11},
/*  5 */  { 0,  1,  4, 10}
};

// C_EDGE_DIRS[edge][0..2] = direction
const int C_EDGE_DIRS[12][2] = {
/*  0 */  {0, 5}, //dir = 0,5
/*  1 */  {1, 5}, //dir = 1,5
/*  2 */  {0, 3}, //dir = 0,3
/*  3 */  {0, 2}, //dir = 0,2
/*  4 */  {3, 5}, //dir = 3,5
/*  5 */  {3, 4}, //dir = 3,4
/*  6 */  {0, 4}, //dir = 0,4
/*  7 */  {1, 4}, //dir = 1,4
/*  8 */  {1, 3}, //dir = 1,3
/*  9 */  {1, 2}, //dir = 1,2
/* 10 */  {2, 5}, //dir = 2,5
/* 11 */  {2, 4}  //dir = 2,4
};

// C_EDGES_DIRS[edge][secondedge] = direction
const int C_EDGES_DIRS[12][12] = {
/*            0   1   2   3   4   5   6   7   8   9  10  11   */
/* 0  */   { -1,  5,  0,  0,  5, -1,  0, -1, -1, -1,  5, -1},
/* 1  */   {  5, -1, -1, -1,  5, -1, -1,  1,  1,  1,  5, -1},
/* 2  */   {  0, -1, -1,  0,  3,  3,  0, -1,  3, -1, -1, -1},
/* 3  */   {  0, -1,  0, -1, -1, -1,  0, -1, -1,  2,  2,  2},
/* 4  */   {  5,  5,  3, -1, -1,  3, -1, -1,  3, -1,  5, -1},
/* 5  */   { -1, -1,  3, -1,  3, -1,  4,  4,  3, -1, -1,  4},
/* 6  */   {  0, -1,  0,  0, -1,  4, -1,  4, -1, -1, -1,  4},
/* 7  */   { -1,  1, -1, -1, -1,  4,  4, -1,  1,  1, -1,  4},
/* 8  */   { -1,  1,  3, -1,  3,  3, -1,  1, -1,  1, -1, -1},
/* 9  */   { -1,  1, -1,  2, -1, -1, -1,  1,  1, -1,  2,  2},
/* 10 */   {  5,  5, -1,  2,  5, -1, -1, -1, -1,  2, -1,  2},
/* 11 */   { -1, -1, -1,  2, -1,  4,  4,  4, -1,  2,  2, -1}
};

// C_EDGE_DELTA[edge][0..2][i=0,j=1,k=2] = -1/0/1
const int C_EDGE_DELTA[12][3][3] = {
/*  0 */  {{-1,  0,  0}, { 0,  0, -1}, {-1,  0, -1}}, //dir = 0,5
/*  1 */  {{ 1,  0,  0}, { 0,  0, -1}, { 1,  0, -1}}, //dir = 1,5
/*  2 */  {{-1,  0,  0}, { 0, -1,  0}, {-1, -1,  0}}, //dir = 0,3
/*  3 */  {{-1,  0,  0}, { 0,  1,  0}, {-1,  1,  0}}, //dir = 0,2
/*  4 */  {{ 0, -1,  0}, { 0,  0, -1}, { 0, -1, -1}}, //dir = 3,5
/*  5 */  {{ 0, -1,  0}, { 0,  0,  1}, { 0, -1,  1}}, //dir = 3,4
/*  6 */  {{-1,  0,  0}, { 0,  0,  1}, {-1,  0,  1}}, //dir = 0,4
/*  7 */  {{ 1,  0,  0}, { 0,  0,  1}, { 1,  0,  1}}, //dir = 1,4
/*  8 */  {{ 1,  0,  0}, { 0, -1,  0}, { 1, -1,  0}}, //dir = 1,3
/*  9 */  {{ 1,  0,  0}, { 0,  1,  0}, { 1,  1,  0}}, //dir = 1,2
/* 10 */  {{ 0,  1,  0}, { 0,  0, -1}, { 0,  1, -1}}, //dir = 2,5
/* 11 */  {{ 0,  1,  0}, { 0,  0,  1}, { 0,  1,  1}}  //dir = 2,4
};


#define NO_DIR -1
// C_CONF_DIR[conf][face] = dir/NO_DIR
// face 0-left
//      1-right
//      2-up
//      3-down
//      4-front
//      5-back
const int C_CONF_DIR[23][6] = {
/* 0  */  {-1, -1, -1, -1, -1, -1},
/* 1  */  { 0,  3,  4, -1, -1, -1},
/* 2  */  { 0,  1,  3,  4, -1, -1},
/* 3  */  { 0,  1,  2,  3,  4, -1},
/* 4  */  { 0,  1,  2,  3,  4,  5},
/* 5  */  { 0,  1,  3,  4,  5, -1},
/* 6  */  { 0,  1,  2,  3,  4,  5},
/* 7  */  { 0,  1,  2,  3,  4,  5},
/* 8  */  { 0,  1,  4,  5, -1, -1},
/* 9  */  { 0,  1,  2,  3,  4,  5},
/* 10 */  { 0,  1,  2,  3,  4,  5},
/* 11 */  { 0,  1,  2,  3,  4,  5},
/* 12 */  { 0,  1,  2,  3,  4,  5},
/* 13 */  { 0,  1,  2,  3,  4,  5},
/* 14 */  { 0,  1,  2,  3,  4,  5},
/* 15 */  { 0,  1,  2,  3,  4,  5},
/* 16 */  { 0,  1,  2,  3,  4,  5},
/* 17 */  { 0,  1,  3,  4,  5, -1},
/* 18 */  { 0,  1,  2,  3,  4,  5},
/* 19 */  { 0,  1,  2,  3,  4, -1},
/* 20 */  { 0,  1,  3,  4, -1, -1},
/* 21 */  { 0,  3,  4, -1, -1, -1},
/* 22 */  {-1, -1, -1, -1, -1, -1},
};

// C_CONF_IS_DIR[conf][face] = dir/NO_DIR
// face 0-left
//      1-right
//      2-up
//      3-down
//      4-front
//      5-back
const int C_CONF_IS_DIR[23][6] = {
/* 0  */  {-1, -1, -1, -1, -1, -1},
/* 1  */  { 1, -1, -1,  1,  1, -1},
/* 2  */  { 1,  1, -1,  1,  1, -1},
/* 3  */  { 1,  1,  1,  1,  1, -1},
/* 4  */  { 1,  1,  1,  1,  1,  1},
/* 5  */  { 1,  1, -1,  1,  1,  1},
/* 6  */  { 1,  1,  1,  1,  1,  1},
/* 7  */  { 1,  1,  1,  1,  1,  1},
/* 8  */  { 1,  1, -1, -1,  1,  1},
/* 9  */  { 1,  1,  1,  1,  1,  1},
/* 10 */  { 1,  1,  1,  1,  1,  1},
/* 11 */  { 1,  1,  1,  1,  1,  1},
/* 12 */  { 1,  1,  1,  1,  1,  1},
/* 13 */  { 1,  1,  1,  1,  1,  1},
/* 14 */  { 1,  1,  1,  1,  1,  1},
/* 15 */  { 1,  1,  1,  1,  1,  1},
/* 16 */  { 1,  1,  1,  1,  1,  1},
/* 17 */  { 1,  1, -1,  1,  1,  1},
/* 18 */  { 1,  1,  1,  1,  1,  1},
/* 19 */  { 1,  1,  1,  1,  1, -1},
/* 20 */  { 1,  1, -1,  1,  1, -1},
/* 21 */  { 1, -1, -1,  1,  1, -1},
/* 22 */  {-1, -1, -1, -1, -1, -1},
};



// Rotation for faces
// face 0-left
//      1-right
//      2-up
//      3-down
//      4-front
//      5-back
// C_ROT_DIR[rotation][face] = rotated_face
const int C_ROT_DIR[24][6] = {
/* 0  */  { 0, 1, 2, 3, 4, 5,},
/* 1  */  { 0, 1, 5, 4, 2, 3,},
/* 2  */  { 0, 1, 3, 2, 5, 4,},
/* 3  */  { 0, 1, 4, 5, 3, 2,},
/* 4  */  { 5, 4, 2, 3, 0, 1,},
/* 5  */  { 1, 0, 2, 3, 5, 4,},
/* 6  */  { 4, 5, 2, 3, 1, 0,},
/* 7  */  { 2, 3, 1, 0, 4, 5,},
/* 8  */  { 1, 0, 3, 2, 4, 5,},
/* 9  */  { 3, 2, 0, 1, 4, 5,},
/* 10 */  { 3, 2, 5, 4, 0, 1,},
/* 11 */  { 1, 0, 5, 4, 3, 2,},
/* 12 */  { 2, 3, 5, 4, 1, 0,},
/* 13 */  { 4, 5, 3, 2, 0, 1,},
/* 14 */  { 5, 4, 3, 2, 1, 0,},
/* 15 */  { 2, 3, 4, 5, 0, 1,},
/* 16 */  { 1, 0, 4, 5, 2, 3,},
/* 17 */  { 3, 2, 4, 5, 1, 0,},
/* 18 */  { 5, 4, 1, 0, 2, 3,},
/* 19 */  { 3, 2, 1, 0, 5, 4,},
/* 20 */  { 4, 5, 1, 0, 3, 2,},
/* 21 */  { 5, 4, 0, 1, 3, 2,},
/* 22 */  { 2, 3, 0, 1, 5, 4,},
/* 23 */  { 4, 5, 0, 1, 2, 3,},
};


// C_TRAVERSAL_EDGES[configuration][0..11] = edge/-1
const int C_TRAVERSAL_EDGES[23][12] = {
/*  0 */ { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
/*  1 */ {  2,  5,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1},
/*  2 */ {  2,  6,  7,  8, -1, -1, -1, -1, -1, -1, -1, -1},
/*  3 */ {  2,  5,  6,  7,  9, 11, -1, -1, -1, -1, -1, -1},
/*  4 */ {  1,  2,  5,  6,  9, 10, -1, -1, -1, -1, -1, -1},
/*  5 */ {  0,  1,  2,  5,  7, -1, -1, -1, -1, -1, -1, -1},
/*  6 */ {  1,  2,  6,  7,  8,  9, 10, -1, -1, -1, -1, -1},
/*  7 */ {  3,  5,  6,  7,  8, 11, -1, -1, -1, -1, -1, -1},
/*  8 */ {  0,  1,  6,  7, -1, -1, -1, -1, -1, -1, -1, -1},
/*  9 */ {  1,  3,  5,  6,  8, 10, -1, -1, -1, -1, -1, -1},
/* 10 */ {  2,  3,  4,  5,  8,  9, 10, 11, -1, -1, -1, -1},
/* 11 */ {  0,  5,  6,  8,  9, 10, -1, -1, -1, -1, -1, -1},
/* 12 */ {  0,  1,  2,  3,  5,  6,  7, 11, -1, -1, -1, -1},
/* 13 */ {  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11},
/* 14 */ {  1,  2,  3,  5,  7, 10, -1, -1, -1, -1, -1, -1},
/* 15 */ {  1,  3,  5,  6,  7,  8,  9, 10, 11, -1, -1, -1},
/* 16 */ {  1,  2,  6,  7,  8,  9, 10, -1, -1, -1, -1, -1},
/* 17 */ {  0,  1,  2,  5,  7, -1, -1, -1, -1, -1, -1, -1},
/* 18 */ {  1,  2,  5,  6,  9, 10, -1, -1, -1, -1, -1, -1},
/* 19 */ {  2,  5,  6,  7,  9, 11, -1, -1, -1, -1, -1, -1},
/* 20 */ {  2,  6,  7,  8, -1, -1, -1, -1, -1, -1, -1, -1},
/* 21 */ {  2,  5,  6, -1, -1, -1, -1, -1, -1, -1, -1, -1},
/* 22 */ { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}
};

//C_EDGE_VERTICES[edge][0..1] = vertex
const int C_EDGE_VERTICES[12][2] = {
/* 0  */  { 0, 2},
/* 1  */  { 1, 3},
/* 2  */  { 0, 4},
/* 3  */  { 2, 6},
/* 4  */  { 0, 1},
/* 5  */  { 4, 5},
/* 6  */  { 4, 6},
/* 7  */  { 5, 7},
/* 8  */  { 1, 5},
/* 9  */  { 3, 7},
/* 10 */  { 2, 3},
/* 11 */  { 6, 7}
};


#define EAST      0
#define NORTH     1
#define WEST      2
#define SOUTH     3
//EDGE_NEIGHBOUR[edge] = EAST, NORTH, WEST, SOUTH
const int EDGE_NEIGHBOUR[12] = {
/* 0  */   NORTH,
/* 1  */   EAST,
/* 2  */  SOUTH,
/* 3  */  WEST,
/* 4  */    EAST,
/* 5  */    SOUTH,
/* 6  */   WEST,
/* 7  */   SOUTH,
/* 8  */  EAST,
/* 9  */  NORTH,
/* 10 */    NORTH,
/* 11 */    WEST
};

//EDGE_BACK_NEIGHBOUR[edge][second_edge] = edge
const int EDGE_BACK_NEIGHBOUR[12][12] = {
/*            0   1   2   3   4   5   6   7   8   9  10  11   */
/* 0  */   { -1,  6,  1,  1,  6, -1,  1, -1, -1, -1,  6, -1},
/* 1  */   {  7, -1, -1, -1,  7, -1, -1,  0,  0,  0,  7, -1},
/* 2  */   {  8, -1, -1,  8,  3,  3,  8, -1,  3, -1, -1, -1},
/* 3  */   {  9, -1,  9, -1, -1, -1,  9, -1, -1,  2,  2,  2},
/* 4  */   {  5,  5, 10, -1, -1, 10, -1, -1, 10, -1,  5, -1},
/* 5  */   { -1, -1, 11, -1, 11, -1,  4,  4, 11, -1, -1,  4},
/* 6  */   {  7, -1,  7,  7, -1,  0, -1,  0, -1, -1, -1,  0},
/* 7  */   { -1,  6, -1, -1, -1,  1,  1, -1,  6,  6, -1,  1},
/* 8  */   { -1,  2,  9, -1,  9,  9, -1,  2, -1,  2, -1, -1},
/* 9  */   { -1,  3, -1,  8, -1, -1, -1,  3,  3, -1,  8,  8},
/* 10 */   { 11, 11, -1,  4, 11, -1, -1, -1, -1,  4, -1,  4},
/* 11 */   { -1, -1, -1,  5, -1, 10, 10, 10, -1,  5,  5, -1}
};

//DIR_VERTICES[direction][0..3] = vertex
// face 0-left
//      1-right
//      2-up
//      3-down
//      4-front
//      5-back
const int DIR_VERTICES[6][4] = {
/* 0  */   { 0, 2, 6, 4},
/* 1  */   { 1, 5, 7, 3},
/* 2  */   { 2, 3, 7, 6},
/* 3  */   { 0, 4, 5, 1},
/* 4  */   { 4, 6, 7, 5},
/* 5  */   { 0, 1, 3, 2}
};

//EDGE_CHILDREN[edge][0..1] = child
const int EDGE_CHILDREN[12][2] = {
/* 0  */   { 0, 2},
/* 1  */   { 1, 3},
/* 2  */   { 0, 4},
/* 3  */   { 2, 6},
/* 4  */   { 0, 1},
/* 5  */   { 4, 5},
/* 6  */   { 4, 6},
/* 7  */   { 5, 7},
/* 8  */   { 1, 5},
/* 9  */   { 3, 7},
/* 10 */   { 2, 3},
/* 11 */   { 6, 7}
};

//***************************************************************************
//*********** COMMON FUNCTIONS **********************************************
//***************************************************************************

void *mycalloc (int nitems, int nbytes);
void myfree (void *block);


//---------------------------------------------------------------------------
#endif




