// SavingsAlgorithm.cpp: implementation of the CSavingsAlgorithm class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "VRP.h"
#include "SavingsAlgorithm.h"
#include <algorithm>
using namespace std;

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif


bool operator<(const sSaving& a, const sSaving& b) {
    return a.value > b.value;
}


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CSavingsAlgorithm::CSavingsAlgorithm()
{
  
}

CSavingsAlgorithm::CSavingsAlgorithm(CVRPDoc* doc) 
	:CAlgorithm(doc)
{

}

CSavingsAlgorithm::~CSavingsAlgorithm()
{

}

// compute savings and sort them 
void CSavingsAlgorithm::ComputeSavings()
{
	
	// count the number of nodes - depot node   
	int CustomerCount = m_pDoc->m_Graph->GetNodeCount() - 1;
	// count the number of savings   
	int SavingsCount = ((CustomerCount*CustomerCount)-CustomerCount)/2;
	
	m_pSavingArray = new sSaving[SavingsCount];

	int index = 0;
	sSaving saving;
	
	for (int i=1;i <= CustomerCount;i++) {
		for (int j=i+1;j <= CustomerCount;j++) {
			saving.f_NodeId = i;
			saving.s_NodeId = j;
			saving.value = (m_pDoc->m_Graph->GetEdgeValue(0,i) + 
						m_pDoc->m_Graph->GetEdgeValue(j,0)) - 
						m_pDoc->m_Graph->GetEdgeValue(i,j);
			m_pSavingArray[index] = saving;
			index += 1;
		}
	}
	sort(m_pSavingArray,m_pSavingArray+SavingsCount);
}

// create and initialize route for each customer
void CSavingsAlgorithm::SetSimpleRoutes()
{
	// count the number of nodes - depot node
	int CustomerCount = m_pDoc->m_Graph->GetNodeCount() - 1;
	// for each customer create a simple route
	m_pDoc->m_Graph->CreateRoutes(CustomerCount);

	// initialize created routes
	for (int i=0; i < CustomerCount; i++) {
		m_pDoc->m_Graph->SetRouteId(i,i);
		m_pDoc->m_Graph->SetRouteState(i,1);
		m_pDoc->m_Graph->SetRouteCost(i,0);
		m_pDoc->m_Graph->SetRouteDemand(i,m_pDoc->m_Graph->GetNodeDemand(i+1));
		m_pDoc->m_Graph->AddNodeToRouteTail(i,m_pDoc->m_Graph->GetNode(0));
		m_pDoc->m_Graph->AddNodeToRouteTail(i,m_pDoc->m_Graph->GetNode(i+1));
		m_pDoc->m_Graph->AddNodeToRouteTail(i,m_pDoc->m_Graph->GetNode(0));
	}
}

void CSavingsAlgorithm::SolveSequntial()
{
	m_pDoc->m_Graph->UpdateEdgeValues();
	SetSimpleRoutes();
	ComputeSavings();
	ProcesSavingsSequential();
	m_pDoc->UpdateAllViews(NULL);
}

// go thrue savings and construct routes paralel
void CSavingsAlgorithm::ProcesSavingsParalel()
{
	CNode* fNode;
	CNode* sNode;

	CRoute* fRoute;
	CRoute* sRoute;
	// count the number of nodes - depot node   
	int CustomerCount = m_pDoc->m_Graph->GetNodeCount() - 1;
	// count the number of savings   
	int SavingsCount = ((CustomerCount*CustomerCount)-CustomerCount)/2;

	for (int i=0;i < SavingsCount;i++) {

		fNode  = m_pDoc->m_Graph->GetNode(m_pSavingArray[i].f_NodeId);
		sNode = m_pDoc->m_Graph->GetNode(m_pSavingArray[i].s_NodeId);
		
		fRoute = m_pDoc->m_Graph->GetRoute(fNode->getRouteId());
		sRoute = m_pDoc->m_Graph->GetRoute(sNode->getRouteId());

		int Capacity = m_pDoc->m_Graph->GetCapacity();
	
		if ((fRoute->getId() != sRoute->getId()) &&
			((fRoute->getDemand() + sRoute->getDemand()) <= Capacity)) {
			// choose proper merge for routes 
			ChooseMerge(fRoute,sRoute,fNode->getId(),sNode->getId());
		}
	} 
}

void CSavingsAlgorithm::ChooseMerge(CRoute* fr,CRoute* sr, int fn, int sn)
{
	if ((fr->GetLastCustId() == fn) && (sr->GetFirstCustId() == sn))
		m_pDoc->m_Graph->MergeRoutesFS(fr->getId(),sr->getId(),fn,sn);
	else if ((fr->GetFirstCustId() == fn) && (sr->GetLastCustId() == sn)) 
		m_pDoc->m_Graph->MergeRoutesSF(fr->getId(),sr->getId(),fn,sn);
	else if ((fr->GetFirstCustId() == fn) && (sr->GetFirstCustId() == sn)) 
		m_pDoc->m_Graph->MergeRoutesSS(fr->getId(),sr->getId(),fn,sn);
	else if ((fr->GetLastCustId() == fn) && (sr->GetLastCustId() == sn)) 
		m_pDoc->m_Graph->MergeRoutesFF(fr->getId(),sr->getId(),fn,sn);
}

void CSavingsAlgorithm::SolveParalel()
{
	m_pDoc->m_Graph->UpdateEdgeValues();
	SetSimpleRoutes();
	ComputeSavings();
	ProcesSavingsParalel();
	m_pDoc->UpdateAllViews(NULL);	
}

void CSavingsAlgorithm::ProcesSavingsSequential()
{
	int actRouteId = FindRouteToProces();
	while (actRouteId != -1) {
  		GenerateRoute(actRouteId);
	//	int dem = m_routeArray[actRouteId].getDemand();
		actRouteId = FindRouteToProces();
	}
}

// go thrue savings and return first possible route use
int CSavingsAlgorithm::FindRouteToProces()
{
	// count the number of nodes - depot node   
	int CustomerCount = m_pDoc->m_Graph->GetNodeCount() - 1;
	// count the number of savings   
	int SavingsCount = ((CustomerCount*CustomerCount)-CustomerCount)/2;

	CNode* fNode;
	CNode* sNode;

	CRoute* fRoute;
	CRoute* sRoute;

	int count = 0;
	int routeId = -1;

	// find savings with both nodes in unprocesed route
	while ((routeId == -1) && (count < SavingsCount)) {
		if (m_pSavingArray[count].value != -1) {
			fNode  = m_pDoc->m_Graph->GetNode(m_pSavingArray[count].f_NodeId);
			sNode = m_pDoc->m_Graph->GetNode(m_pSavingArray[count].s_NodeId);

			fRoute = m_pDoc->m_Graph->GetRoute(fNode->getRouteId());
			sRoute = m_pDoc->m_Graph->GetRoute(sNode->getRouteId());
			
			if (((fRoute->getState() == 1) && (sRoute->getState() == 1)) &&
				(fRoute->getId() != sRoute->getId()))
				routeId = fNode->getRouteId();
		}
		count++;
	}
	return routeId;
}

void CSavingsAlgorithm::GenerateRoute(int id)
{
	CNode* fNode;
	CNode* sNode;
	CNode* SavingNode;
	
	CRoute* fRoute = m_pDoc->m_Graph->GetRoute(id);
	CRoute* sRoute;
	int index = 0;
	
	while (index != -1) {
		// find a saving which can be proces to actual route return -1 if no exist
		fNode = m_pDoc->m_Graph->GetNode(fRoute->GetFirstCustId());
		sNode = m_pDoc->m_Graph->GetNode(fRoute->GetLastCustId());
		index = FindPossibleSaving(fNode,sNode);
		
		if (index != -1) {
			// check nodes to merge new route to our actual and no reverse
			// if savings first is in curect route
			if ((fNode->getId() == m_pSavingArray[index].f_NodeId) ||
				(sNode->getId() == m_pSavingArray[index].f_NodeId)) {

				SavingNode = m_pDoc->m_Graph->GetNode(m_pSavingArray[index].s_NodeId); 
				sRoute = m_pDoc->m_Graph->GetRoute(SavingNode->getRouteId());
				ChooseMerge(fRoute,sRoute,m_pSavingArray[index].f_NodeId,m_pSavingArray[index].s_NodeId);
			}
			else {
				SavingNode = m_pDoc->m_Graph->GetNode(m_pSavingArray[index].f_NodeId); 
				sRoute = m_pDoc->m_Graph->GetRoute(SavingNode->getRouteId());
				ChooseMerge(fRoute,sRoute,m_pSavingArray[index].s_NodeId,m_pSavingArray[index].f_NodeId);
			}
			m_pSavingArray[index].value = -1;
		}
	}
	fRoute = m_pDoc->m_Graph->GetRoute(id);
	fRoute->setState(0);
}

// find posiible saving which can enlarge our route
int CSavingsAlgorithm::FindPossibleSaving(CNode *f, CNode *s)
{
	// count the number of nodes - depot node   
	int CustomerCount = m_pDoc->m_Graph->GetNodeCount() - 1;
	// count the number of savings   
	int SavingsCount = ((CustomerCount*CustomerCount)-CustomerCount)/2;

	CNode* fNode;
	CNode* sNode;

	int index = -1;
	int count = 0;
	
	while ((index == -1) && (count < SavingsCount)) {
		// check if saving was not already used
		if (m_pSavingArray[count].value != -1) {
			// check if saving is not on the same route
			fNode  = m_pDoc->m_Graph->GetNode(m_pSavingArray[count].f_NodeId);
			sNode = m_pDoc->m_Graph->GetNode(m_pSavingArray[count].s_NodeId);
			if (fNode->getRouteId() != sNode->getRouteId())
			{
				// find possible saving 
				if ((f->getId() == fNode->getId())  || 
					(f->getId() == sNode->getId()) ||
					(s->getId() == fNode->getId())  ||
					(s->getId() == sNode->getId())) {
					// check if merging do not overload the vehicle
					if ((m_pDoc->m_Graph->GetRouteDemand(fNode->getRouteId()) 
						+ m_pDoc->m_Graph->GetRouteDemand(sNode->getRouteId())) 
						<= m_pDoc->m_Graph->GetCapacity())
						index = count;
				}
			}
		}
		count++;
	}
	return index;
}

void CSavingsAlgorithm::SolveSeqParameter(float lambda)
{
	m_pDoc->m_Graph->UpdateEdgeValues();
	SetSimpleRoutes();
	ComputeSavingsPar(lambda);
	ProcesSavingsSequential();
	m_pDoc->UpdateAllViews(NULL);
}

void CSavingsAlgorithm::ComputeSavingsPar(float lambda)
{
	// count the number of nodes - depot node   
	int CustomerCount = m_pDoc->m_Graph->GetNodeCount() - 1;
	// count the number of savings   
	int SavingsCount = ((CustomerCount*CustomerCount)-CustomerCount)/2;
	
	m_pSavingArray = new sSaving[SavingsCount];

	int index = 0;
	sSaving saving;
	
	for (int i=1;i <= CustomerCount;i++) {
		for (int j=i+1;j <= CustomerCount;j++) {
			saving.f_NodeId = i;
			saving.s_NodeId = j;
			saving.value = (m_pDoc->m_Graph->GetEdgeValue(0,i) + 
						m_pDoc->m_Graph->GetEdgeValue(j,0)) - 
						(m_pDoc->m_Graph->GetEdgeValue(i,j) * lambda);
			m_pSavingArray[index] = saving;
			index += 1;
		}
	}
	sort(m_pSavingArray,m_pSavingArray+SavingsCount);
}

void CSavingsAlgorithm::SolveParParameter(float lambda)
{
	m_pDoc->m_Graph->UpdateEdgeValues();
	SetSimpleRoutes();
	ComputeSavingsPar(lambda);
	ProcesSavingsParalel();
	m_pDoc->UpdateAllViews(NULL);	
}
