/*
 * IntArrayWrapper.java
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * 1.0.0 (2007-05-04):
 * - extracted from DlpTransformer
 */

package lp.trans;

/**
 * A class used for storing an increasing sequence of nonnegative integers.
 * It is very simple and error-prone. Designed only for internal use by the
 * {@link DlpTransformer} and {@link EvolpTransformer} classes.
 *
 * @author Martin Slota
 * @version 1.0.0
 */
class IntArrayWrapper {
	/**
	 * An array with the sequence's members.
	 */
	private final int[] members;
	
	/**
	 * Index of the getTop element.
	 */
	private int top;
	
	/**
	 * Constructs a new instance which can hold up to {@code maxCount}
	 * elements.
	 *
	 * @param maxCount capacity of this instance
	 */
	public IntArrayWrapper(int maxCount) {
		members = new int[maxCount];
		top = -1;
	}
	
	/**
	 * Adds a new element to the sequence. Doesn't check whether the
	 * capacity limit given in the constructor has been exceeded. The new
	 * member is only added in case it is bigger than the getTop element. If
	 * it's less or equal, nothing happens.
	 *
	 * @param newMember candidate for a new member of this sequence
	 */
	public void add(int newMember) {
		if (top == -1 || newMember > members[top])
			members[++top] = newMember;
	}
	
	/**
	 * Returns current number of elements in this sequence.
	 *
	 * @return current number of elements in this sequence
	 */
	public int length() {
		return top + 1;
	}
	
	/**
	 * Returns the index of the last element.
	 *
	 * @return the index of the last element.
	 */
	public int getTop() {
		return top;
	}
	
	/**
	 * Returns the i-th member of this sequence. DOESN'T CHECK whether
	 * {@code i} is a valid member index.
	 *
	 * @param i
	 * @return the i-th member of this sequence
	 */
	public int get(int i) {
		return members[i];
	}
	
	public int getMax() {
		return (top == -1) ? -1 : members[top];
	}
	
	/**
	 * Returns the maximum element of this sequence that is less or equal
	 * than {@code limit}, or -1 if no such element exists.
	 *
	 * @param limit upper limit for the element returned
	 * @return the maximum element of this sequence that is less or equal
	 * than {@code limit}, or -1 if no such element exists.
	 */
	public int getMaxLeqThan(int limit) {
		if (top == -1 || members[0] > limit)
			return -1;
		if (members[top] <= limit)
			return members[top];
		int i = top - 1;
		while (members[i] > limit)
			i--;
		return members[i];
	}
	
	/**
	 * Returns {@code true} iff this sequence contains {@code number}.
	 *
	 * @param number the element to search for
	 * @return {@code true} if this sequence contains {@code number},
	 * {@code false} otherwise
	 */
	public boolean contains(int number) {
		if (top == -1)
			return false;
		int i = 0;
		while (i < top && members[i] < number)
			i++;
		return members[i] == number;
	}
	
	/**
	 * Returns {@code true} iff this sequence has 0 elements.
	 *
	 * @return {@code true} if this sequence has 0 elements, {@code false}
	 * otherwise.
	 */
	public boolean isEmpty() {
		return top == -1;
	}
}