/*
 * LpSolver.java
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-02-14): initial version
 * v0.2 (2007-02-14):
 * - implementation
 * - 1 test
 * v0.2.1 (2007-02-17):
 * - setModelLimit added
 * v1.0.0 (2007-05-06):
 * - methods common to LpSolver, DlpSolver and EvolpSolver moved to the abstract
 *   class Solver
 * - documentation added
 */

package lp.trans;


import lp.parse.LpParser;
import lp.unit.StableModel;

import lp.wrap.LparseWrapper;
import lp.wrap.SmodelsWrapper;

/**
 * A class that computes stable models of a grounded normal logic program.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see LpGrounder
 */
public class LpSolver extends Solver<StableModel> {
	/**
	 * Parser used to convert the string representation of a model into object
	 * representations of the atoms in the model.
	 */
	private final LpParser parser;
	
	/**
	 * Creates a new instance of {@code LpSolver}. It will create new instances
	 * of {@link LparseWrapper} and {@link SmodelsWrapper} classes and use them
	 * to compute the stable models of grounded normal logic programs passed in
	 * through the {@link #solve(LogicProgram, Consumer)} method.
	 */
	public LpSolver() {
		this(new LparseWrapper(), new SmodelsWrapper());
	}
	
	/**
	 * Creates a new instance of {@code LpSolver}. It will use the given 
	 * instances of {@link LparseWrapper} and {@link SmodelsWrapper} classes and
	 * use them to compute the stable models of grounded normal logic programs
	 * passed in through the {@link #solve(LogicProgram, Consumer)} method.
	 *
	 * @param lparseWrapper the object used to execute lparse
	 * @param smodelsWrapper the object used to execute smodels
	 */
	public LpSolver(LparseWrapper lparseWrapper,
			SmodelsWrapper smodelsWrapper) {
		super(lparseWrapper, smodelsWrapper);
		parser = new LpParser();
	}
	
	/**
	 * {@inheritDoc}
	 */
	protected StableModel parseModel(String modelString) {
		StableModel result = new StableModel();
		parser.setInput(modelString);
		while (parser.hasMoreTokens()) {
			result.add(parser.parseAtom());
		}
		parser.close();
		return result;
	}
}