/*
 * DynamicLogicProgram.java
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-02-08): initial version
 * v0.1.1 (2007-02-11):
 * - isGround added
 * v0.1.2 (2007-02-12):
 * - moved to lp.unit package
 * v0.1.3 (2007-02-12):
 * - EMPTY_PROGRAM is added instead of null
 * v0.1.4 (2007-02-14):
 * - implementation of EMPTY_PROGRAM changed
 * v1.0.0 (2007-05-05):
 * - EmptyProgram moved to a separate file
 * - documentation added
 *
 * PENDING create a junit test
 */

package lp.unit;

import java.util.ArrayList;

/**
 * Represents a dynamic logic program, i.e. a sequence of logic programs.
 * In the semantics of dynamic logic programs, the rules from programs further
 * in the sequence are preferred over rules from preceding programs.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see LogicProgram
 * @see GeneralizedLogicProgram
 */
public class DynamicLogicProgram extends ArrayList<LogicProgram> {
	/**
	 * Creates a new instance of {@code DynamicLogicProgram}.
	 */
	public DynamicLogicProgram() {
		super();
	}
	
	/**
	 * Creates a new instance of {@code DynamicLogicProgram} with the specified
	 * initial capacity.
	 *
	 * @param initialCapacity the initial capacity of this {@link ArrayList}
	 */
	public DynamicLogicProgram(int initialCapacity) {
		super(initialCapacity);
	}
	
	/**
	 * Adds {@code program} to end of this dynamic logic program. In case it is
	 * {@code null}, an immutable empty program is added.
	 *
	 * @param program the program to add
	 * @return {@code true}
	 * @see EmptyProgram
	 */
	@Override
	public boolean add(LogicProgram program) {
		if (program == null)
			super.add(EmptyProgram.getInstance());
		else
			super.add(program);
		return true;
	}
	
	/**
	 * Returns {@code true} iff all programs in the sequence are also ground.
	 *
	 * @return as specified above
	 * @see LogicProgram#isGround()
	 */
	public boolean isGround() {
		for (LogicProgram prog : this)
			if (!prog.isGround())
				return false;
		return true;
	}
}