/*
 * TransformedDlp.java
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-02-14): initial version
 * v0.1.1 (2007-02-14):
 * - implementation finished
 * v1.0.0 (2007-05-05):
 * - documentation added
 *
 * PENDING create a junit test
 */

package lp.unit;

import java.util.EnumMap;
import java.util.Map;
import lp.struct.LpRule;

/**
 * Represents the normal logic program that was constructed from a dynamic
 * logic program. It can be used as an ordinary logic program but it also offers
 * the possibility to remember the types of the rules in it. A rule of a
 * specific type can be added by using the
 * {@link #add(LpRule, TransDlpRuleType)} method. The
 * {@link #getGroup(TransDlpRuleType)} will return all rules that were added
 * with a specific type.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see lp.trans.DlpTransformer#transform(DynamicLogicProgram)
 */
public class TransformedDlp extends GeneralizedLogicProgram {
	/**
	 * A mapping of types to logic programs.
	 */
	private final Map<TransDlpRuleType, LogicProgram> ruleGroups;
	
	/**
	 * Creates a new instance of {@code TransformedDlp}. It is intially empty.
	 */
	public TransformedDlp() {
		ruleGroups = new EnumMap<TransDlpRuleType, LogicProgram>(
				TransDlpRuleType.class);
		for (TransDlpRuleType t : TransDlpRuleType.values()) {
			ruleGroups.put(t, new GeneralizedLogicProgram());
		}
	}
	
	/**
	 * Adds the rule {@code rule} to this program and remembers it to be of type
	 * {@code type}.
	 *
	 * @param rule the rule to add
	 * @param type type of the rule
	 * @return {@code true} if this program did not already contain the
	 * specified rule
	 */
	public boolean add(LpRule rule, TransDlpRuleType type) {
		boolean result = super.add(rule);
		ruleGroups.get(type).add(rule);
		return result;
	}
	
	/**
	 * Returns a {@link LogicProgram} containing all rules of type {@code type}
	 * from this program.
	 *
	 * @param type type of the rules that should returned
	 * @return as specified above
	 */
	public LogicProgram getGroup(TransDlpRuleType type) {
		return ruleGroups.get(type);
	}
}