/*
 * TransformedEvolp.java
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-02-20): initial version
 * v1.0.0 (2007-05-05):
 * - documentation added
 *
 * PENDING create a junit test
 */

package lp.unit;

import java.util.HashMap;
import java.util.Map;
import lp.struct.LpRule;

/**
 * Represents the normal logic program that was constructed from an evolving
 * logic program. It can be used as an ordinary logic program but it also offers
 * the possibility to remember the types and group numbers of the rules in it. A
 * rule of a specific type can be added by using the
 * {@link #add(LpRule, int, TransEvolpRuleType)} method. The
 * {@link #getGroup(int, TransEvolpRuleType)} will return all rules that were
 * added to a specific group with a specific type.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see lp.trans.EvolpTransformer#transform(EvolpProgram)
 */
public class TransformedEvolp extends GeneralizedLogicProgram {
	/**
	 * A mapping of groups and types to logic programs.
	 */
	private final Map<String, LogicProgram> ruleGroups;
	
	/**
	 * Maximum group used in this transformed program.
	 */
	private int maxGroup;
	
	/**
	 * Creates a new instance of {@code TransformedEvolp}.  It is intially
	 * empty.
	 */
	public TransformedEvolp() {
		ruleGroups = new HashMap<String, LogicProgram>();
		maxGroup = 0;
	}
	
	/**
	 * Adds the rule {@code rule} to this program and remembers it to be in the
	 * group no. {@code group} and of type {@code type}.
	 *
	 * @param rule the rule to add
	 * @param group number of the group to which the rule belongs
	 * @param type type of the rule
	 * @return {@code true} if this program did not already contain the
	 * specified rule
	 */
	public boolean add(LpRule rule, int group, TransEvolpRuleType type) {
		if (group > maxGroup)
			maxGroup = group;
		boolean result = super.add(rule);
		String key = type.toString() + Integer.toString(group);
		LogicProgram p = ruleGroups.get(key);
		if (p == null) {
			p = new GeneralizedLogicProgram();
			ruleGroups.put(key, p);
		}
		p.add(rule);
		return result;
	}
	
	/**
	 * Returns a {@link LogicProgram} containing all rules from group no.
	 * {@code group} and of the type {@code type}.
	 *
	 * @param group number of the group from which rules should be returned
	 * @param type type of the rules that should returned
	 * @return as specified above
	 */
	public LogicProgram getGroup(int group, TransEvolpRuleType type) {
		String key = type.toString() + Integer.toString(group);
		LogicProgram result = ruleGroups.get(key);
		if (result == null) {
			result = new GeneralizedLogicProgram();
			ruleGroups.put(key, result);
		}
		return result;
	}
	
	/**
	 * Returns the maximum number of group used in this transformed program.
	 *
	 * @return as specified above
	 */
	public int getMaxGroup() {
		return maxGroup;
	}
}