/*
 * LparseMessage.java
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-01-20): initial version
 * v0.2 (2007-01-26):
 * - implementation and documentation
 * v0.2.1 (2007-01-28):
 * - changed to an unchecked exception (see
 *   http://www.mindview.net/Etc/Discussions/CheckedExceptions)
 * - documentation updated
 * v0.2.2 (2007-02-08):
 * - hashCode() fixed
 * v1.0.0 (2007-05-04):
 * - override annotations added
 * - promoted to version 1.0.0 :o)
 */

package lp.wrap;

/**
 * This class represents a warning or error message from lparse.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see LparseWrapper
 */
public class LparseMessage extends RuntimeException {
	/**
	 * Is {@code true} if this is a warning message and {@code false} if it is
	 * an error message.
	 */
	private final boolean warning;
	
	/**
	 * Contains the number of line in the lparse's input to which this message
	 * is related. If the number is unknown, it must be set to -1.
	 */
	private final int lineNumber;
	
	/**
	 * Creates a new lparse error that has no a detail message. Has the same
	 * result as calling {@code new LparseMessage(null, -1, false)}.
	 */
	public LparseMessage() {
		this(null, -1, false);
	}
	
	/**
	 * Creates a new lparse error with the specified detail message and line
	 * number. Has the same result as calling
	 * {@code new LparseMessage(message, lineNumber, false)}.
	 *
	 * @param message the detail message
	 * @param lineNumber the number of line in the lparse's input on which this
	 * error was found
	 */
	public LparseMessage(String message, int lineNumber) {
		this(message, lineNumber, false);
	}
	
	/**
	 * Creates a new lparse message with the specified detail message and line
	 * number. If {@code warning} is {@code true}, then the message is just a
	 * warning. Otherwise it is an lparse error.
	 *
	 * @param message the detail message
	 * @param lineNumber the number of line in the lparse's input to which this
	 * message is related
	 * @param isWarning determines if this is a warning or error message
	 */
	public LparseMessage(String message, int lineNumber, boolean isWarning) {
		super(message);
		this.warning = isWarning;
		this.lineNumber = lineNumber;
	}
	
	/**
	 * Returns {@code true} if this is a warning message and {@code false} if it
	 * is an error message.
	 *
	 * @return as specified above
	 */
	public boolean isWarning() {
		return warning;
	}
	
	/**
	 * Returns the number of line in the lparse's input to which this message
	 * is related. If the number is unknown, -1 is returned.
	 *
	 * @return as specified above
	 */
	public int getLineNumber() {
		return lineNumber;
	}
	
	/**
	 * Returns {@code true} iff {@code obj} is an instance of
	 * {@code LparseMessage}, is of the same type (as specified by
	 * {@link #isWarning()}) and contains the same message for the same
	 * line number.
	 *
	 * @return as specified above
	 */
	@Override
	public boolean equals(Object obj) {
		if (!(obj instanceof LparseMessage))
			return false;
		
		LparseMessage e = (LparseMessage) obj;
		return (isWarning() == e.isWarning() &&
				getLineNumber() == e.getLineNumber()) &&
				((getMessage() == null) ? (e.getMessage() == null) : getMessage().equals(e.getMessage()));
	}
	
	/**
	 * Overriden in order to maintain the general contract of
	 * {@link Object#hashCode()}.
	 *
	 * @return the hash of this object
	 */
	@Override
	public int hashCode() {
		int result = isWarning() ? 1 : 0;
		result += getLineNumber();
		result += (getMessage() == null) ? 0 : getMessage().hashCode();
		return result;
	}
}