/*
 * LpAtomTest.java
 * JUnit based test
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-02-11): initial version
 * v0.2 (2007-02-14):
 * - tests moved/copied from appropriate tests in LpLiteralTest
 * v1.0.0 (2007-05-05):
 * - promoted to version 1.0.0 :o)
 */

package lp.struct;

import java.util.ArrayList;
import java.util.List;

import junit.framework.*;

/**
 * Contains tests of the {@link LpAtom} class.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see LpAtom
 */
public class LpAtomTest extends LpStructureUnitTestUtils {
	/**
	 * A default test case constructor.
	 *
	 * @param testName the name of the test case
	 */
	public LpAtomTest(String testName) {
		super(testName);
	}
	
	/**
	 * Tests if the constructor throws the appropriate exceptions.
	 */
	public void testConstructorExceptions() {
		// predicate must not be null
		try {
			LpAtom.getInstance(null, null);
			fail("A IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// predicate must not be null
		try {
			LpAtom.getInstance(null, new ArrayList<LpTerm>());
			fail("A IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// arity doesn't match the number of arguments
		try {
			LpAtom.getInstance(new LpPredicate("p", 4), null);
			fail("An IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// arity doesn't match the number of arguments
		try {
			LpAtom.getInstance(new LpPredicate("green", 1),
					new ArrayList<LpTerm>());
			fail("An IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
	}
	
	/**
	 * Tests the {@link LpAtom#equals(Object)} and
	 * {@link LpAtom#hashCode()} methods.
	 */
	public void testEqualsAndHashCode() {
		LpPredicate predDay = new LpPredicate("day", 0);
		LpPredicate predNice = new LpPredicate("nice", 1);
		LpPredicate predLoves = new LpPredicate("loves", 2);
		
		LpConstant constQuasimodo = new LpConstant("quasimodo");
		LpConstant constEsmeralda = new LpConstant("esmeralda");
		
		List<LpTerm> argsNice = new ArrayList<LpTerm>();
		argsNice.add(constQuasimodo);
		List<LpTerm> argsLoves = new ArrayList<LpTerm>();
		argsLoves.add(constQuasimodo);
		argsLoves.add(constEsmeralda);
		
		// the atom "day"
		LpAtom atomDay1 = LpAtom.getInstance(predDay, null);
		// another atom "day"
		LpAtom atomDay2 = LpAtom.getInstance(predDay, new ArrayList<LpTerm>());
		
		// the atom "nice(quasimodo)"
		LpAtom atomNice1 = LpAtom.getInstance(predNice, argsNice);
		// another atom "nice(quasimodo)"
		LpAtom atomNice2 = LpAtom.getInstance(predNice, argsNice);
		
		// the atom "loves(quasimodo, esmeralda)"
		LpAtom atomLoves1 = LpAtom.getInstance(predLoves, argsLoves);
		// another atom "loves(quasimodo, esmeralda)"
		LpAtom atomLoves2 = LpAtom.getInstance(predLoves, argsLoves);
		
		testEqual(atomDay1, atomDay1);
		testEqual(atomDay1, atomDay2);
		testEqual(atomDay2, atomDay2);
		testEqual(atomDay2, atomDay1);
		
		testEqual(atomNice1, atomNice1);
		testEqual(atomNice1, atomNice2);
		testEqual(atomNice2, atomNice2);
		testEqual(atomNice2, atomNice1);
		
		testEqual(atomLoves1, atomLoves1);
		testEqual(atomLoves1, atomLoves2);
		testEqual(atomLoves2, atomLoves2);
		testEqual(atomLoves2, atomLoves1);
		
		testNotEqual(atomDay1, null);
		testNotEqual(atomDay2, null);
		testNotEqual(atomNice1, null);
		testNotEqual(atomLoves1, null);
		
		testNotEqual(atomDay1, atomNice1);
		testNotEqual(atomDay2, atomNice1);
	}
	
	/**
	 * Asserts that the given atoms are equal and that their hash codes are
	 * equal.
	 *
	 * @param a1 first tested atom
	 * @param a2 second tested atom
	 */
	private void testEqual(LpAtom a1, LpAtom a2) {
		assertEquals("Atoms '" + a1.toString() + "' and '" + a2.toString() +
				"' should be equal!", a1, a2);
		assertEquals("Equal atoms should have equal hash codes!",
				a1.hashCode(), a2.hashCode());
	}
	
	/**
	 * Asserts that the given atoms are not equal.
	 *
	 * @param a1 first tested atom
	 * @param a2 second tested atom
	 */
	private void testNotEqual(LpAtom a1, LpAtom a2) {
		assertFalse("Atoms '" + a1 + "' and '" + a2 +
				"' should not be equal!", a1.equals(a2));
	}
	
	/**
	 * Tests {@link LpAtom#toString()} method.
	 */
	public void testToString() {
		LpPredicate predDay = new LpPredicate("day", 0);
		LpPredicate predNice = new LpPredicate("nice", 1);
		LpPredicate predLoves = new LpPredicate("loves", 2);
		
		LpConstant constQuasimodo = new LpConstant("quasimodo");
		LpConstant constEsmeralda = new LpConstant("esmeralda");
		
		List<LpTerm> argsNice = new ArrayList<LpTerm>();
		argsNice.add(constQuasimodo);
		List<LpTerm> argsLoves = new ArrayList<LpTerm>();
		argsLoves.add(constQuasimodo);
		argsLoves.add(constEsmeralda);
		
		// the atom "day"
		LpAtom atomDay1 = LpAtom.getInstance(predDay, null);
		testToString("day", atomDay1);
		
		// another atom "day"
		LpAtom atomDay2 = LpAtom.getInstance(predDay, new ArrayList<LpTerm>());
		testToString("day", atomDay2);
		
		// the atom "nice(quasimodo)"
		LpAtom atomNice = LpAtom.getInstance(predNice, argsNice);
		testToString("nice(quasimodo)", atomNice);
		
		// the atom "loves(quasimodo, esmeralda)"
		LpAtom atomLoves = LpAtom.getInstance(predLoves, argsLoves);
		testToString("loves(quasimodo, esmeralda)", atomLoves);
	}
}