/*
 * LpCompoundTermTest.java
 * JUnit based test
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-01-02): initial version
 * v0.2 (2007-01-02): testConstructorExceptions added
 * v0.2.1 (2007-01-04): testEqualsAndHashCode added
 * v0.2.2 (2007-01-05): testNormalFormAndToString added
 * v0.2.3 (2007-01-12):
 * - testConstructorExceptions modified to expect IllegalArgumentException
 *   instead of NullPointerException
 * v0.2.4 (2007-01-18):
 * - tests of normal form moved to LpPrettyPrinterTest
 * - testNormalFormAndToString renamed to testToString
 * - documentation updated
 * v1.0.0 (2007-05-05):
 * - promoted to version 1.0.0 :o)
 */

package lp.struct;

import java.util.ArrayList;
import java.util.List;

import junit.framework.*;

/**
 * Contains tests of the {@link LpCompoundTerm} class.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see LpCompoundTerm
 */
public class LpCompoundTermTest extends LpStructureUnitTestUtils {
	/**
	 * A default test case constructor.
	 *
	 * @param testName the name of the test case
	 */
	public LpCompoundTermTest(String testName) {
		super(testName);
	}
	/**
	 * Tests if the constructor throws the appropriate exceptions.
	 */
	public void testConstructorExceptions() {
		// function must not be null
		try {
			LpCompoundTerm.getInstance(null, null);
			fail("A IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// function must not be null
		try {
			LpCompoundTerm.getInstance(null, new ArrayList<LpTerm>());
			fail("A IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// arity doesn't match the number of arguments
		try {
			LpCompoundTerm.getInstance(new LpFunction("f", 4), null);
			fail("An IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// arity doesn't match the number of arguments
		try {
			LpCompoundTerm.getInstance(new LpFunction("head", 1),
					new ArrayList<LpTerm>());
			fail("An IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
	}
	
	/**
	 * Tests the {@link LpCompoundTerm#equals(Object)} and
	 * {@link LpCompoundTerm#hashCode()} methods.
	 */
	public void testEqualsAndHashCode() {
		LpFunction funcBrother = new LpFunction("brother", 1);
		LpFunction funcChild = new LpFunction("child", 2);
		
		LpConstant constQuasimodo = new LpConstant("quasimodo");
		LpConstant constEsmeralda = new LpConstant("esmeralda");
		
		List<LpTerm> argsBrother = new ArrayList<LpTerm>();
		argsBrother.add(constQuasimodo);
		List<LpTerm> argsChild = new ArrayList<LpTerm>();
		argsChild.add(constQuasimodo);
		argsChild.add(constEsmeralda);
		
		// the term "brother(quasimodo)"
		LpCompoundTerm termBrother1 = LpCompoundTerm.getInstance(funcBrother, argsBrother);
		// another term "brother(quasimodo)"
		LpCompoundTerm termBrother2 = LpCompoundTerm.getInstance(funcBrother, argsBrother);
		
		// the term "child(quasimodo, esmeralda)"
		LpCompoundTerm termChild1 = LpCompoundTerm.getInstance(funcChild, argsChild);
		// another term "child(quasimodo, esmeralda)"
		LpCompoundTerm termChild2 = LpCompoundTerm.getInstance(funcChild, argsChild);
		
		testEqual(termBrother1, termBrother1);
		testEqual(termBrother1, termBrother2);
		testEqual(termBrother2, termBrother2);
		testEqual(termBrother2, termBrother1);
		
		testEqual(termChild1, termChild1);
		testEqual(termChild1, termChild2);
		testEqual(termChild2, termChild2);
		testEqual(termChild2, termChild1);
		
		testNotEqual(termBrother1, null);
		testNotEqual(termChild1, null);
		
		testNotEqual(termBrother1, termChild1);
		testNotEqual(termChild1, termBrother1);
	}
	
	/**
	 * Asserts that the given terms are equal and that their hash codes
	 * are equal.
	 *
	 * @param t1 first tested term
	 * @param t2 second tested term
	 */
	private void testEqual(LpCompoundTerm t1, LpCompoundTerm t2) {
		assertEquals("Terms '" + t1.toString() + "' and '" + t2.toString() +
				"' should be equal!", t1, t2);
		assertEquals("Equal terms should have equal hash codes!",
				t1.hashCode(), t2.hashCode());
	}
	
	/**
	 * Asserts that the given terms are not equal.
	 *
	 * @param t1 first tested term
	 * @param t2 second tested term
	 */
	private void testNotEqual(LpCompoundTerm t1, LpCompoundTerm t2) {
		assertFalse("Terms '" + t1 + "' and '" + t2 +
				"' should not be equal!", t1.equals(t2));
	}
	
	/**
	 * Tests the {@link LpCompoundTerm#toString()} method.
	 */
	public void testToString() {
		LpFunction funcBrother = new LpFunction("brother", 1);
		LpFunction funcSister = new LpFunction("sister", 1);
		LpFunction funcChild = new LpFunction("child", 2);
		
		LpConstant constQuasimodo = new LpConstant("quasimodo");
		LpConstant constEsmeralda = new LpConstant("esmeralda");
		LpVariable varPerson = new LpVariable("Person");
		
		List<LpTerm> argsBrother = new ArrayList<LpTerm>();
		argsBrother.add(constQuasimodo);
		List<LpTerm> argsSister = new ArrayList<LpTerm>();
		argsSister.add(constQuasimodo);
		List<LpTerm> argsChild1 = new ArrayList<LpTerm>();
		argsChild1.add(constQuasimodo);
		argsChild1.add(varPerson);
		List<LpTerm> argsChild2 = new ArrayList<LpTerm>();
		argsChild2.add(constQuasimodo);
		argsChild2.add(constEsmeralda);
		
		LpCompoundTerm termBrother = LpCompoundTerm.getInstance(
				funcBrother, argsBrother);
		testToString("brother(quasimodo)", termBrother);
		
		LpCompoundTerm termSister = LpCompoundTerm.getInstance(
				funcSister, argsSister);
		testToString("sister(quasimodo)", termSister);
		
		LpCompoundTerm termChild1 = LpCompoundTerm.getInstance(
				funcChild, argsChild1);
		testToString("child(quasimodo, Person)", termChild1);
		
		LpCompoundTerm termChild2 = LpCompoundTerm.getInstance(
				funcChild, argsChild2);
		testToString("child(quasimodo, esmeralda)", termChild2);
	}
}