/*
 * LpConstantTest.java
 * JUnit based test
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-01-02): initial version
 * v0.2 (2007-01-02): testConstructorExceptions added
 * v0.2.1 (2007-01-04): testEqualsAndHashCode added
 * v0.2.2 (2007-01-04): testNormalFormAndToString added
 * v0.2.3 (2007-01-12):
 * - testConstructorExceptions modified to expect IllegalArgumentException
 *   instead of NullPointerException
 * v0.2.4 (2007-01-18):
 * - tests of normal form moved to LpPrettyPrinterTest
 * - testNormalFormAndToString renamed to testToString
 * - documentation updated
 * v1.0.0 (2007-05-05):
 * - promoted to version 1.0.0 :o)
 */

package lp.struct;

import junit.framework.*;

/**
 * Contains tests of the {@link LpConstant} class.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see LpConstant
 */
public class LpConstantTest extends LpStructureUnitTestUtils {
	/**
	 * A default test case constructor.
	 *
	 * @param testName the name of the test case
	 */
	public LpConstantTest(String testName) {
		super(testName);
	}
	
	/**
	 * Tests if the constructor throws the appropriate exceptions.
	 */
	public void testConstructorExceptions() {
		// constant's name must not be null
		try {
			new LpConstant(null);
			fail("A IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// constant's name must not be empty
		try {
			new LpConstant("");
			fail("An IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
	}
	
	/**
	 * Tests the {@link LpConstant#equals(Object)} and
	 * {@link LpConstant#hashCode()} methods.
	 */
	public void testEqualsAndHashCode() {
		LpConstant constBob1 = new LpConstant("bob");
		LpConstant constBob2 = new LpConstant("bob");
		
		LpConstant constAlice1 = new LpConstant("alice");
		LpConstant constAlice2 = new LpConstant("alice");
		
		testEqual(constBob1, constBob1);
		testEqual(constBob1, constBob2);
		testEqual(constBob2, constBob2);
		testEqual(constBob2, constBob1);
		
		testEqual(constAlice1, constAlice1);
		testEqual(constAlice1, constAlice2);
		testEqual(constAlice2, constAlice2);
		testEqual(constAlice2, constAlice1);
		
		testNotEqual(constBob1, null);
		testNotEqual(constAlice1, null);
		
		testNotEqual(constBob1, constAlice1);
		testNotEqual(constAlice1, constBob1);
	}
	
	/**
	 * Asserts that the given constants are equal and that their hash codes
	 * are equal.
	 *
	 * @param c1 first tested constant
	 * @param c2 second tested constant
	 */
	private void testEqual(LpConstant c1, LpConstant c2) {
		assertEquals("Constants '" + c1.toString() + "' and '" + c2.toString() +
				"' should be equal!", c1, c2);
		assertEquals("Equal constants should have equal hash codes!",
				c1.hashCode(), c2.hashCode());
	}
	
	/**
	 * Asserts that the given constants are not equal.
	 *
	 * @param c1 first tested constant
	 * @param c2 second tested constant
	 */
	private void testNotEqual(LpConstant c1, LpConstant c2) {
		assertFalse("Constants '" + c1 + "' and '" + c2 +
				"' should not be equal!", c1.equals(c2));
	}
	
	/**
	 * Tests the {@link LpConstant#toString()} method.
	 */
	public void testToString() {
		LpConstant c1 = new LpConstant("someConstant");
		testToString("someConstant", c1);
		
		LpConstant c2 = new LpConstant("a_different_constant");
		testToString("a_different_constant", c2);
	}
}