/*
 * LpFunctionTest.java
 * JUnit based test
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-01-02): initial version
 * v0.2 (2007-01-02): testConstructorExceptions added
 * v0.2.1 (2007-01-04): testEqualsAndHashCode added
 * v0.2.2 (2007-01-04): testNormalFormAndToString added
 * v0.2.3 (2007-01-12):
 * - testConstructorExceptions modified to expect IllegalArgumentException
 *   instead of NullPointerException
 * v0.2.4 (2007-01-18):
 * - tests of normal form moved to LpPrettyPrinterTest
 * - testNormalFormAndToString renamed to testToString
 * - documentation updated
 * v1.0.0 (2007-05-05):
 * - promoted to version 1.0.0 :o)
 */

package lp.struct;

import junit.framework.*;

/**
 * Contains tests of the {@link LpFunction} class.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see LpFunction
 */
public class LpFunctionTest extends LpStructureUnitTestUtils {
	/**
	 * A default test case constructor.
	 *
	 * @param testName the name of the test case
	 */
	public LpFunctionTest(String testName) {
		super(testName);
	}
	
	/**
	 * Tests if the constructor throws the appropriate exceptions.
	 */
	public void testConstructorExceptions() {
		// function name must not be null
		try {
			new LpFunction(null, 1);
			fail("A IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// function name must not be null
		try {
			new LpFunction(null, -1);
			fail("A IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// function name must not be empty
		try {
			new LpFunction("", 1);
			fail("An IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// function arity must be positive
		try {
			new LpFunction("a_function", 0);
			fail("An IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// function arity must be positive
		try {
			new LpFunction("a_function", -1);
			fail("An IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
	}
	
	/**
	 * Test of {@link LpFunction#getArity()} method.
	 */
	public void testGetArity() {
		LpFunction function = new LpFunction("p", 7);
		assertEquals("Different function arity expected!", 7, function.getArity());
	}
	
	/**
	 * Tests the {@link LpFunction#equals(Object)} and
	 * {@link LpFunction#hashCode()} methods.
	 */
	public void testEqualsAndHashCode() {
		LpFunction funcF1 = new LpFunction("f", 1);
		LpFunction funcF2 = new LpFunction("f", 1);
		
		LpFunction funcF3 = new LpFunction("f", 2);
		
		LpFunction funcG1 = new LpFunction("plus", 3);
		LpFunction funcG2 = new LpFunction("plus", 3);
		
		LpFunction funcH = new LpFunction("minus", 3);
		
		testEquals(funcF1, funcF1);
		testEquals(funcF1, funcF2);
		testEquals(funcF2, funcF2);
		testEquals(funcF2, funcF1);
		
		testEquals(funcG1, funcG1);
		testEquals(funcG1, funcG2);
		testEquals(funcG2, funcG2);
		testEquals(funcG2, funcG1);
		
		testNotEquals(funcF1, null);
		testNotEquals(funcF3, null);
		testNotEquals(funcG1, null);
		testNotEquals(funcH, null);
		
		testNotEquals(funcF1, funcF3);
		testNotEquals(funcF3, funcF1);
		
		testNotEquals(funcF1, funcG1);
		testNotEquals(funcG1, funcF1);
		
		testNotEquals(funcG1, funcH);
		testNotEquals(funcH, funcG1);
	}
	
	/**
	 * Asserts that the given function symbols are equal and that their hash
	 * codes are equal.
	 *
	 * @param f1 first tested literal
	 * @param f2 second tested literal
	 */
	private void testEquals(LpFunction f1, LpFunction f2) {
		assertEquals("Functions '" + f1 + "' and '" + f2 +
				"' should be equal!", f1, f2);
		assertEquals("Equal Functions should have equal hash codes!",
				f1.hashCode(), f2.hashCode());
	}
	
	/**
	 * Asserts that the given function symbols are not equal.
	 *
	 * @param f1 first tested literal
	 * @param f2 second tested literal
	 */
	private void testNotEquals(LpFunction f1, LpFunction f2) {
		assertFalse("Functions '" + f1 + "' and '" + f2 +
				"' should not be equal!", f1.equals(f2));
	}
	
	/**
	 * Tests the {@link LpFunction#toString()} method.
	 */
	public void testToString() {
		LpFunction f1 = new LpFunction("times", 2);
		testToString("times/2", f1);
		
		LpFunction f2 = new LpFunction("divBy2", 1);
		testToString("divBy2/1", f2);
	}
}