/*
 * LpLiteralTest.java
 * JUnit based test
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-01-02): initial version
 * v0.2 (2007-01-02): testConstructorExceptions added
 * v0.2.1 (2007-01-03): testEqualsAndHashCode added
 * v0.2.2 (2007-01-05): testNormalFormAndToString added
 * v0.2.3 (2007-01-12):
 * - testConstructorExceptions modified to expect IllegalArgumentException
 *   instead of NullPointerException
 * v0.2.4 (2007-01-18):
 * - tests of normal form moved to LpPrettyPrinterTest
 * - testNormalFormAndToString renamed to testToString
 * - documentation updated
 * v0.2.5 (2007-02-11):
 * - changed to reflect the new LpAtom class
 * v0.2.6 (2007-02-14):
 * - review of all tests
 * v1.0.0 (2007-05-05):
 * - promoted to version 1.0.0 :o)
 */

package lp.struct;

import java.util.ArrayList;
import java.util.List;

import junit.framework.*;

/**
 * Contains tests of the {@link LpLiteral} class.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see LpLiteral
 */
public class LpLiteralTest extends LpStructureUnitTestUtils {
	/**
	 * A default test case constructor.
	 *
	 * @param testName the name of the test case
	 */
	public LpLiteralTest(String testName) {
		super(testName);
	}
	
	/**
	 * Tests if the constructor throws the appropriate exceptions.
	 */
	public void testConstructorExceptions() {
		// atom must not be null
		try {
			new LpLiteral(true, null);
			fail("A IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
	}
	
	/**
	 * Tests the {@link LpLiteral#equals(Object)} and
	 * {@link LpLiteral#hashCode()} methods.
	 */
	public void testEqualsAndHashCode() {
		LpPredicate predDay = new LpPredicate("day", 0);
		LpPredicate predNice = new LpPredicate("nice", 1);
		LpPredicate predLoves = new LpPredicate("loves", 2);
		
		LpConstant constQuasimodo = new LpConstant("quasimodo");
		LpConstant constEsmeralda = new LpConstant("esmeralda");
		
		List<LpTerm> argsNice = new ArrayList<LpTerm>();
		argsNice.add(constQuasimodo);
		List<LpTerm> argsLoves = new ArrayList<LpTerm>();
		argsLoves.add(constQuasimodo);
		argsLoves.add(constEsmeralda);
		
		// the literal "day"
		LpLiteral litDay1 = new LpLiteral(true,
				LpAtom.getInstance(predDay, null));
		// another literal "day"
		LpLiteral litDay2 = new LpLiteral(true,
				LpAtom.getInstance(predDay, null));
		// the literal "not day"
		LpLiteral litDay3 = new LpLiteral(false,
				LpAtom.getInstance(predDay, null));
		// another literal "not day"
		LpLiteral litDay4 = new LpLiteral(false,
				LpAtom.getInstance(predDay, null));
		
		// the literal "not nice(quasimodo)"
		LpLiteral litNice1 = new LpLiteral(false,
				LpAtom.getInstance(predNice, argsNice));
		// another literal "not nice(quasimodo)"
		LpLiteral litNice2 = new LpLiteral(false,
				LpAtom.getInstance(predNice, argsNice));
		
		// the literal "loves(quasimodo, esmeralda)"
		LpLiteral litLoves1 = new LpLiteral(true,
				LpAtom.getInstance(predLoves, argsLoves));
		// another literal "loves(quasimodo, esmeralda)"
		LpLiteral litLoves2 = new LpLiteral(true,
				LpAtom.getInstance(predLoves, argsLoves));
		
		testEqual(litDay1, litDay1);
		testEqual(litDay1, litDay2);
		testEqual(litDay2, litDay2);
		testEqual(litDay2, litDay1);
		
		testEqual(litDay3, litDay3);
		testEqual(litDay3, litDay4);
		testEqual(litDay4, litDay4);
		testEqual(litDay4, litDay3);
		
		testEqual(litNice1, litNice1);
		testEqual(litNice1, litNice2);
		testEqual(litNice2, litNice2);
		testEqual(litNice2, litNice1);
		
		testEqual(litLoves1, litLoves1);
		testEqual(litLoves1, litLoves2);
		testEqual(litLoves2, litLoves2);
		testEqual(litLoves2, litLoves1);
		
		testNotEqual(litDay1, null);
		testNotEqual(litDay2, null);
		testNotEqual(litDay3, null);
		testNotEqual(litDay4, null);
		testNotEqual(litNice1, null);
		testNotEqual(litLoves1, null);
		
		testNotEqual(litDay1, litDay3);
		testNotEqual(litDay1, litDay4);
		testNotEqual(litDay2, litDay3);
		testNotEqual(litDay2, litDay4);
		
		testNotEqual(litDay3, litDay1);
		testNotEqual(litDay3, litDay2);
		testNotEqual(litDay4, litDay1);
		testNotEqual(litDay4, litDay2);
		
		testNotEqual(litDay1, litNice1);
		testNotEqual(litDay2, litNice1);
		testNotEqual(litDay3, litLoves1);
		testNotEqual(litDay4, litLoves1);
	}
	
	/**
	 * Asserts that the given literals are equal and that their hash codes
	 * are equal.
	 *
	 * @param l1 first tested literal
	 * @param l2 second tested literal
	 */
	private void testEqual(LpLiteral l1, LpLiteral l2) {
		assertEquals("Literals '" + l1.toString() + "' and '" + l2.toString() +
				"' should be equal!", l1, l2);
		assertEquals("Equal literals should have equal hash codes!",
				l1.hashCode(), l2.hashCode());
	}
	
	/**
	 * Asserts that the given literals are not equal.
	 *
	 * @param l1 first tested literal
	 * @param l2 second tested literal
	 */
	private void testNotEqual(LpLiteral l1, LpLiteral l2) {
		assertFalse("Literals '" + l1 + "' and '" + l2 +
				"' should not be equal!", l1.equals(l2));
	}
	
	/**
	 * Tests {@link LpLiteral#toString()} method.
	 */
	public void testToString() {
		LpPredicate predDay = new LpPredicate("day", 0);
		LpPredicate predNice = new LpPredicate("nice", 1);
		LpPredicate predLoves = new LpPredicate("loves", 2);
		
		LpConstant constQuasimodo = new LpConstant("quasimodo");
		LpConstant constEsmeralda = new LpConstant("esmeralda");
		
		List<LpTerm> argsNice = new ArrayList<LpTerm>();
		argsNice.add(constQuasimodo);
		List<LpTerm> argsLoves = new ArrayList<LpTerm>();
		argsLoves.add(constQuasimodo);
		argsLoves.add(constEsmeralda);
		
		// the literal "day"
		LpLiteral litDay1 = new LpLiteral(true,
				LpAtom.getInstance(predDay, null));
		testToString("day", litDay1);
		
		// another literal "day"
		LpLiteral litDay2 = new LpLiteral(true,
				LpAtom.getInstance(predDay, new ArrayList<LpTerm>()));
		testToString("day", litDay2);
		
		// the literal "not nice(quasimodo)"
		LpLiteral litNice = new LpLiteral(false,
				LpAtom.getInstance(predNice, argsNice));
		testToString("not nice(quasimodo)", litNice);
		
		// the literal "loves(quasimodo, esmeralda)"
		LpLiteral litLoves = new LpLiteral(true,
				LpAtom.getInstance(predLoves, argsLoves));
		testToString("loves(quasimodo, esmeralda)", litLoves);
	}
}