/*
 * LpPredicateTest.java
 * JUnit based test
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-01-02): initial version
 * v0.2 (2007-01-02): testConstructorExceptions and testGetArity added
 * v0.2.1 (2007-01-04): testEqualsAndHashCode added
 * v0.2.2 (2007-01-04): testNormalFormAndToString added
 * v0.2.3 (2007-01-12):
 * - testConstructorExceptions modified to expect IllegalArgumentException
 *   instead of NullPointerException
 * v0.2.4 (2007-01-18):
 * - tests of normal form moved to LpPrettyPrinterTest
 * - testNormalFormAndToString renamed to testToString
 * - documentation updated
 * v1.0.0 (2007-05-05):
 * - promoted to version 1.0.0 :o)
 */

package lp.struct;

import junit.framework.*;

/**
 * Contains tests of the {@link LpPredicate} class.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see LpPredicate
 */
public class LpPredicateTest extends LpStructureUnitTestUtils {
	/**
	 * A default test case constructor.
	 *
	 * @param testName the name of the test case
	 */
	public LpPredicateTest(String testName) {
		super(testName);
	}
	
	/**
	 * Tests if the constructor throws the appropriate exceptions.
	 */
	public void testConstructorExceptions() {
		// predicate name must not be null
		try {
			new LpPredicate(null, 0);
			fail("A IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// predicate name must not be null
		try {
			new LpPredicate(null, -1);
			fail("A IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// predicate name must not be empty
		try {
			new LpPredicate("", 1);
			fail("An IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
		
		// predicate arity must not be negative
		try {
			new LpPredicate("a_predicate", -1);
			fail("An IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
	}
	
	/**
	 * Test of {@link LpPredicate#getArity()} method.
	 */
	public void testGetArity() {
		LpPredicate predicate = new LpPredicate("p", 7);
		assertEquals("Different predicate arity expected!", 7, predicate.getArity());
	}
	
	/**
	 * Tests the {@link LpPredicate#equals(Object)} and
	 * {@link LpPredicate#hashCode()} methods.
	 */
	public void testEqualsAndHashCode() {
		LpPredicate predBelongs1 = new LpPredicate("belongs", 1);
		LpPredicate predBelongs2 = new LpPredicate("belongs", 1);
		
		LpPredicate predBelongs3 = new LpPredicate("belongs", 2);
		
		LpPredicate predPlus1 = new LpPredicate("plus", 3);
		LpPredicate predPlus2 = new LpPredicate("plus", 3);
		
		LpPredicate predMinus = new LpPredicate("minus", 3);
		
		testEquals(predBelongs1, predBelongs1);
		testEquals(predBelongs1, predBelongs2);
		testEquals(predBelongs2, predBelongs2);
		testEquals(predBelongs2, predBelongs1);
		
		testEquals(predPlus1, predPlus1);
		testEquals(predPlus1, predPlus2);
		testEquals(predPlus2, predPlus2);
		testEquals(predPlus2, predPlus1);
		
		testNotEquals(predBelongs1, null);
		testNotEquals(predBelongs3, null);
		testNotEquals(predPlus1, null);
		testNotEquals(predMinus, null);
		
		testNotEquals(predBelongs1, predBelongs3);
		testNotEquals(predBelongs3, predBelongs1);
		
		testNotEquals(predBelongs1, predPlus1);
		testNotEquals(predPlus1, predBelongs1);
		
		testNotEquals(predPlus1, predMinus);
		testNotEquals(predMinus, predPlus1);
	}
	
	/**
	 * Asserts that the given predicate symbols are equal and that their hash
	 * codes are equal.
	 *
	 * @param p1 first tested literal
	 * @param p2 second tested literal
	 */
	private void testEquals(LpPredicate p1, LpPredicate p2) {
		assertEquals("Predicates '" + p1 + "' and '" + p2 +
				"' should be equal!", p1, p2);
		assertEquals("Equal predicates should have equal hash codes!",
				p1.hashCode(), p2.hashCode());
	}
	
	/**
	 * Asserts that the given predicate symbols are not equal.
	 *
	 * @param p1 first tested literal
	 * @param p2 second tested literal
	 */
	private void testNotEquals(LpPredicate p1, LpPredicate p2) {
		assertFalse("Predicates '" + p1 + "' and '" + p2 +
				"' should not be equal!", p1.equals(p2));
	}
	
	/**
	 * Tests {@link LpPredicate#toString()} method.
	 */
	public void testToString() {
		LpPredicate p1 = new LpPredicate("div", 3);
		testToString("div/3", p1);
		
		LpPredicate p2 = new LpPredicate("times", 3);
		testToString("times/3", p2);
	}
}