/*
 * LpRuleTest.java
 * JUnit based test
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-01-03): initial version
 * v0.2.1 (2007-01-03):
 * - testConstructorExceptions and testEqualsAndHashCode added
 * - documentation added
 * v0.2.2 (2007-01-05): testNormalFormAndToString added
 * v0.2.3 (2007-01-12):
 * - testConstructorExceptions modified to expect IllegalArgumentException
 *   instead of NullPointerException
 * v0.2.4 (2007-01-18):
 * - tests of normal form moved to LpPrettyPrinterTest
 * - testNormalFormAndToString renamed to testToString
 * - documentation updated
 * v0.2.5 (2007-02-08):
 * - testGetBodyCount added
 * v0.2.6 (2007-02-11):
 * - changed to reflect the new LpAtom class
 * v1.0.0 (2007-05-05):
 * - promoted to version 1.0.0 :o)
 */

package lp.struct;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

/**
 * Contains tests of the {@link LpRule} class.
 *
 * @author Martin Slota
 * @version 1.0.0
 * @see LpRule
 */
public class LpRuleTest extends LpStructureUnitTestUtils {
	/**
	 * A default test case constructor.
	 *
	 * @param testName the name of the test case
	 */
	public LpRuleTest(String testName) {
		super(testName);
	}
	
	/**
	 * Tests if the constructor throws the appropriate exceptions.
	 */
	public void testConstructorExceptions() {
		// rule must not have both a null head and a null body
		try {
			new LpRule(null, null);
			fail("A IllegalArgumentException expected!");
		} catch (IllegalArgumentException e) {
			// behaves as expected
		}
	}
	
	/**
	 * Tests the {@link LpRule#equals(Object)} and {@link LpRule#hashCode()}
	 * methods.
	 */
	public void testEqualsAndHashCode() {
		LpPredicate predP = new LpPredicate("p", 0);
		LpPredicate predQ = new LpPredicate("q", 2);
		
		LpConstant constA = new LpConstant("a");
		LpVariable varX = new LpVariable("X");
		List<LpTerm> args = new ArrayList<LpTerm>();
		args.add(constA);
		args.add(varX);
		
		LpLiteral litP = LpAtom.getInstance(predP, null).getLiteral(false);
		LpLiteral litQ = LpAtom.getInstance(predQ, args).getLiteral(true);
		
		Set<LpLiteral> ruleBody = new LinkedHashSet<LpLiteral>();
		ruleBody.add(litP);
		
		// the rule "not p."
		LpRule fact1 = new LpRule(litP, null);
		// another rule "not p."
		LpRule fact2 = new LpRule(litP, new LinkedHashSet<LpLiteral>());
		// the rule "q(a, X) <- not p."
		LpRule rule1 = new LpRule(litQ, ruleBody);
		// another rule "q(a, X) <- not p."
		LpRule rule2 = new LpRule(litQ, ruleBody);
		// the rule "<- not p."
		LpRule constraint1 = new LpRule(null, ruleBody);
		// another rule "<- not p."
		LpRule constraint2 = new LpRule(null, ruleBody);
		
		testEqual(fact1, fact1);
		testEqual(fact1, fact2);
		testEqual(fact2, fact2);
		testEqual(fact2, fact1);
		
		testEqual(rule1, rule1);
		testEqual(rule1, rule2);
		
		testEqual(constraint1, constraint1);
		testEqual(constraint1, constraint2);
		
		testNotEqual(fact1, rule1);
		testNotEqual(fact2, rule1);
		testNotEqual(fact1, constraint1);
		testNotEqual(fact2, constraint1);
		
		testNotEqual(rule1, fact1);
		testNotEqual(rule1, fact2);
		testNotEqual(constraint1, fact1);
		testNotEqual(constraint1, fact2);
		
		testNotEqual(fact1, null);
		testNotEqual(fact2, null);
		testNotEqual(rule1, null);
		testNotEqual(constraint1, null);
	}
	
	/**
	 * Asserts that the given rules are equal and that their hash codes are
	 * equal.
	 *
	 * @param r1 first tested rule
	 * @param r2 second tested rule
	 */
	private void testEqual(LpRule r1, LpRule r2) {
		assertEquals("Rules '" + r1.toString() + "' and '" + r2.toString() +
				"' should be equal!", r1, r2);
		assertEquals("Equal rules should have equal hash codes!",
				r1.hashCode(), r2.hashCode());
	}
	
	/**
	 * Asserts that the given rules are not equal.
	 *
	 * @param r1 first tested rule
	 * @param r2 second tested rule
	 */
	private void testNotEqual(LpRule r1, LpRule r2) {
		assertFalse("Rules '" + r1 + "' and '" + r2 +
				"' should not be equal!", r1.equals(r2));
	}
	
	/**
	 * Tests the {@link LpRule#toString()} method.
	 */
	public void testToString() {
		LpPredicate predP = new LpPredicate("p", 0);
		LpPredicate predQ = new LpPredicate("q", 2);
		
		LpConstant constA = new LpConstant("a");
		LpVariable varX = new LpVariable("X");
		List<LpTerm> args = new ArrayList<LpTerm>();
		args.add(constA);
		args.add(varX);
		
		LpLiteral litP = LpAtom.getInstance(predP, null).getLiteral(false);
		LpLiteral litQ = LpAtom.getInstance(predQ, args).getLiteral(true);
		
		Set<LpLiteral> ruleBody = new LinkedHashSet<LpLiteral>();
		ruleBody.add(litP);
		
		// the rule "not p."
		LpRule fact1 = new LpRule(litP, null);
		testToString("not p.", fact1);
		
		// another rule "not p."
		LpRule fact2 = new LpRule(litP, new LinkedHashSet<LpLiteral>());
		testToString("not p.", fact2);
		
		// the rule "q(a, X) <- not p."
		LpRule rule = new LpRule(litQ, ruleBody);
		testToString("q(a, X) <- not p.", rule);
		
		// the rule "<- p."
		LpRule constraint = new LpRule(null, ruleBody);
		testToString("<- not p.", constraint);
	}
	
	/**
	 * Tests the {@link LpRule#getBodySize()} method.
	 */
	public void testGetBodyCount() {
		LpPredicate predP = new LpPredicate("p", 0);
		LpPredicate predQ = new LpPredicate("q", 2);
		
		LpConstant constA = new LpConstant("a");
		LpVariable varX = new LpVariable("X");
		List<LpTerm> args = new ArrayList<LpTerm>();
		args.add(constA);
		args.add(varX);
		
		LpLiteral litP = LpAtom.getInstance(predP, null).getLiteral(false);
		LpLiteral litQ = LpAtom.getInstance(predQ, args).getLiteral(true);
		
		Set<LpLiteral> ruleBody = new LinkedHashSet<LpLiteral>();
		ruleBody.add(litP);
		
		Set<LpLiteral> constraintBody = new LinkedHashSet<LpLiteral>();
		constraintBody.add(litQ);
		constraintBody.add(litP);
		
		// the fact "not p."
		LpRule fact1 = new LpRule(litP, null);
		assertEquals("Wrong number of literals in the body!",
				0, fact1.getBodySize());
		
		// another fact "not p."
		LpRule fact2 = new LpRule(litP, new LinkedHashSet<LpLiteral>());
		assertEquals("Wrong number of literals in the body!",
				0, fact2.getBodySize());
		
		// the rule "q(a, X) <- not p."
		LpRule rule = new LpRule(litQ, ruleBody);
		assertEquals("Wrong number of literals in the body!",
				1, rule.getBodySize());
		
		// the constraint "<- q(a, X), not p."
		LpRule constraint = new LpRule(null, constraintBody);
		assertEquals("Wrong number of literals in the body!",
				2, constraint.getBodySize());
	}
}