/*
 * FakeProcess.java
 *
 * Copyright (C) 2006 - 2007 Martin Slota
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/*
 * History:
 * v0.1 (2007-01-14): initial version
 * v0.2 (2007-01-14):
 * - initial implementation
 * - documentation added
 * v0.2.1 (2007-01-28):
 * - unnecessary throws clauses removed
 * - documentation updated
 * v1.0.0 (2007-05-05):
 * - promoted to version 1.0.0 :o)
 */

package lp.test.util;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * A fake process class. Was used before but now it is here just for fun :o)
 *
 * @author Martin Slota
 * @version 1.0.0
 */
public class FakeProcess extends Process {
	/**
	 * The standard input {@code OutputStream}.
	 */
	private final OutputStream stdin;
	
	/**
	 * The standard output {@code InputStream}.
	 */
	private final InputStream stdout;
	
	/**
	 * The standard error {@code InputStream}.
	 */
	private final InputStream stderr;
	
	/**
	 * The exit value.
	 */
	private final int exit;
	
	/**
	 * Creates a new instance with the given return value. Standard output and
	 * standard error stream contents are set to an empty string and standard
	 * input is set to a <q>black hole</q> {@code OutputStream}.
	 *
	 * @param exit the return value of the new fake process
	 */
	public FakeProcess(int exit) {
		this("", "", exit);
	}
	
	/**
	 * Creates a new instance with the given contents of standard output and
	 * and a given return value. Standard error contents are set to an empty
	 * string and standard input is set to a <q>black hole</q>
	 * {@code OutputStream}.
	 *
	 * @param stdout the contents of the standard output stream of the new
	 * process
	 * @param exit the return value of the new fake process
	 */
	public FakeProcess(String stdout, int exit) {
		this(stdout, "", exit);
	}
	
	/**
	 * Creates a new instance with the given contents of standard output and
	 * standard error streams and a given return value. Standard input is
	 * set to a <q>black hole</q> {@code OutputStream}.
	 *
	 * @param stdout the contents of the standard output stream of the new
	 * fake process
	 * @param stderr the contents of the error output stream of the new
	 * fake process
	 * @param exit the return value of the new fake process
	 */
	public FakeProcess(String stdout, String stderr, int exit) {
		this.stdout = new ByteArrayInputStream(stdout.getBytes());
		this.stderr = new ByteArrayInputStream(stderr.getBytes());
		this.exit = exit;
		stdin = new OutputStream() {
			/**
			 * Throws everything away.
			 */
			@Override
			public void write(int b) {
				// do nothing
			}
		};
	}
	
	/**
	 * Returns the standard input {@code OutputStream} of this fake process.
	 *
	 * @return the standard input {@code OutputStream} of this fake process
	 */
	@Override
	public OutputStream getOutputStream() {
		return stdin;
	}
	
	/**
	 * Returns the standard output {@code InputStream} of this fake process.
	 *
	 * @return the standard output {@code InputStream} of this fake process
	 */
	@Override
	public InputStream getInputStream() {
		return stdout;
	}
	
	/**
	 * Returns the standard error {@code InputStream} of this fake process.
	 *
	 * @return the standard error {@code InputStream} of this fake process
	 */
	@Override
	public InputStream getErrorStream() {
		return stderr;
	}
	
	/**
	 * Returns the exit value of this fake process.
	 *
	 * @return the exit value of this fake process
	 */
	@Override
	public int waitFor() {
		return exitValue();
	}
	
	/**
	 * Returns the exit value of this fake process.
	 *
	 * @return the exit value of this fake process
	 */
	@Override
	public int exitValue() {
		return exit;
	}
	
	/**
	 * Has no effect.
	 */
	@Override
	public void destroy() {
		// do nothing
	}
}