
from context import flatten
from toypython.lex import tokenizer
from toypython.ast import *

def set_context(expr, context):
    if isinstance(expr, (Attribute, Subscript, Name)):
        return expr._replace(ctx=context)

    if isinstance(expr, Starred):
        return Starred(expr.location, set_context(expr.value, context), context)

    if isinstance(expr, List) or (isinstance(expr, Tuple) and expr.elts):
        new_elts = [set_context(e, context) for e in expr.elts]
        return expr.__class__(expr.location, new_elts, context)

    print(expr)
    name = expr.__class__.__name__
    if context == Del():
        raise ParseError(expr.location, "can't delete %s" % name)
    else:
        raise ParseError(expr.location, "can't assign to %s" % name)

def ast_for_testlist(tests, commas):
    return tests[0] if not commas else Tuple(tests[0].location, tests, Load())

def ast_for_arguments(items):
    posargs = []
    vararg = None
    after_vararg = False
    kwonlyargs = []
    kwdefaults = []
    kwarg = None
    posdefaults = []

    if items[-1][0] and items[-1][0].type == '**':
        kwarg = items[-1][1]
        items = items[:-1]

    if items[-1][0] and items[-1][0].type == '*' and not items[-1][1]:
        raise ParseError(items[-1][0].location, "named arguments must follow bare *")

    for (star_token, fpdef, default) in items:
        if star_token and star_token.type == '**':
            raise ParseError(star_token.location, "**kwargs must be last in arguments list")

        if star_token and after_vararg:
            raise ParseError(star_token.location, "*args can only be given once")

        if star_token:
            after_vararg = True
            vararg = fpdef
        elif after_vararg:
            kwonlyargs.append(fpdef)
            kwdefaults.append(default)
        else:
            if default:
                posdefaults.append(default)
            elif posdefaults:
                raise ParseError(fpdef.location, "non-default argument follows default argument")
            posargs.append(fpdef)

    return arguments(posargs, vararg, kwonlyargs, kwdefaults, kwarg, posdefaults)

def ast_for_dotted_name(name_tokens):
    rv = None
    for name_token in name_tokens:
        if rv:
            rv = Attribute(rv.location, rv, name_token.value, Load())
        else:
            rv = Name(name_token.location, name_token.value, Load())
    return rv

def ast_for_decorator(loc, name_expr, have_parens, arglist):
    if not have_parens: return name_expr
    return ast_for_call(loc, name_expr, arglist)

def ast_for_comprehension(root_comp_for):
    comps = []

    for item in flatten(root_comp_for):
        if item[0].type == 'for':
            token, (exprlist, exprlist_has_comma), iter = item
            if exprlist_has_comma:
                target = Tuple(exprlist[0].location, exprlist, Store())
            else:
                target = exprlist[0]
            target = set_context(target, Store())
            comps.append(comprehension(target, iter, []))
        else:
            token, cond = item
            comps[-1].ifs.append(cond)

    return comps

token_type_to_operator_class = {
    '|': BitOr,
    '^': BitXor,
    '&': BitAnd,
    '<<': LShift,
    '>>': RShift,
    '+': Add,
    '-': Sub,
    '*': Mult,
    '/': Div,
    '//': FloorDiv,
    '%': Mod,
}

def ast_for_binop(symbols):
    symbols = iter(symbols)
    left = next(symbols)
    while True:
        try:
            op_token = next(symbols)
            op_class = token_type_to_operator_class[op_token.type]
            right = next(symbols)
            left = BinOp(op_token.location, left, op_class(), right)
        except StopIteration:
            return left

def ast_for_power(atom, trailers, factor):
    for trailer in trailers:
        atom = trailer(atom)._replace(location=atom.location)
    if factor:
        atom = BinOp(atom.location, atom, Pow(), factor)
    return atom

def ast_for_call(loc, func, arglist):
    if not arglist: return Call(loc, func, None, None, None, None)

    args = []
    keywords = []
    vararg = None
    kwarg = None
    keyword_names = set()

    star_token, test, root_comp_for, kwvalue = arglist[0]
    if len(arglist) == 1 and root_comp_for:
        value = GeneratorExp(test.location, test, ast_for_comprehension(root_comp_for))
        arglist = [(None, value, None, None)]

    for (star_token, test, root_comp_for, kwvalue) in arglist:
        if root_comp_for:
            raise ParseError(test, "Generator expression must be parenthesized "
                             "if not sole argument")

    if arglist[-1][0] and arglist[-1][0].type == '**':
        kwarg = arglist[-1][1]
        arglist = arglist[:-1]

    for (star_token, test, root_comp_for, kwvalue) in arglist:
        if star_token and star_token.type == '*':
            if vararg:
                raise ParseError(star_token.location, "*expression can only be given once")
            vararg = test
        elif star_token and star_token.type == '**':
            raise ParseError(star_token.location, "**expression must be last in arguments list")
        elif not kwvalue:
            if keywords:
                raise ParseError(test.location, "non-keyword arg after keyword arg")
            if vararg:
                raise ParseError(test.location, "only named arguments may follow *expression")
            args.append(test)
        elif isinstance(test, Lambda):
            raise ParseError(test.location, "lambda cannot contain assignment")
        elif not isinstance(test, Name):
            raise ParseError(test.location, "keyword can't be an expression")
        elif test.id in keyword_names:
            raise ParseError(test.location, "keyword argument repeated")
        else:
            keyword_names.add(test.id)
            keywords.append(keyword(test.id, kwvalue))

    return Call(loc, func, args, keywords, vararg, kwarg)

def ast_for_expr_stmt(base, augassign_op, augassign_value, assignments):
    if augassign_op:
        base = set_context(base, Store())
        if not isinstance(base, (Name, Attribute, Subscript)):
            raise ParseError(base.location, "illegal expression for augmented assignment")
        return AugAssign(base.location, base, augassign_op, augassign_value)

    if assignments:
        value = assignments[-1]
        targets = []
        for target in [base] + assignments[:-1]:
            if isinstance(target, (Yield, YieldFrom)):
                raise ParseError(target.location, "assignment to yield expression not possible")
            targets.append(set_context(target, Store()))
        return Assign(base.location, targets, value)

    return Expr(base.location, base)

def ast_for_if_stmt(tokens, conds, suites, else_suite):
    for token, cond, suite in reversed(list(zip(tokens, conds, suites))):
        else_suite = [If(token.location, cond, suite, else_suite or [])]
    return else_suite

def ast_for_classdef(loc, name, arglist, body):
    dummy_call = ast_for_call(loc, None, arglist)
    return ClassDef(loc, name, dummy_call.args, dummy_call.keywords,
                    dummy_call.starargs, dummy_call.kwargs, body, [])



from context import combine_action

from context import ParseError

def parse(ctx, start_nt=None, close_with=None):
    stack = [close_with or 'EOF', start_nt or 'file_input']
    results = []
    tok_iter = tokenizer(ctx, close_with)
    token = next(tok_iter)

    while True:
        stack_top = stack[-1]

        if isinstance(stack_top, int):
            rule, action = RULES[stack_top]
            stack.pop()
            n = len(rule)
            args = results[len(results) - n:]
            del results[len(results) - n:]
            results.append(action(ctx, *args))

        elif stack_top not in TABLE:
            if token.type != stack_top:
                raise ParseError(token.location, "Expected " + stack_top)
            stack.pop()
            if not stack: break
            results.append(token)
            token = next(tok_iter)

        else:
            row = TABLE[stack_top]
            if token.type not in row:
                raise ParseError(token.location, "Unexpected {}, expected {}".format(
                    repr(token.type), ', '.join(sorted(repr(k) for k in row))))
            rule_num = row[token.type]
            rule, action = RULES[rule_num]
            stack.pop()
            stack.append(rule_num)
            stack.extend(reversed(rule))

    return results[0]

RULES = [(('stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _ctx, _loc: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('import', 'dotted_as_names'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('n', v2)], user=lambda _all, n, _ctx, _loc: (
Import(_loc, n)
), normal_vars=['n'], list_vars=[]))
), (('decorator', 'decorators__1'), 
(combine_action(root=lambda ctx, v1, v2: [('d', v1), v2], user=lambda _all, d, _ctx, _loc: (
d
), normal_vars=[], list_vars=['d']))
), (('except_clause', ':', 'suite', 'try_stmt__2', 'try_stmt__3', 'try_stmt__4'), 
(lambda ctx, v1, v2, v3, v4, v5, v6: [('ec', v1), (None, v2), ('es', v3), v4, v5, v6])
), (('finally', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('f', v3)])
), ((',', 'subscriptlist__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('dictorsetmaker',), 
(lambda ctx, v1: [('d', v1)])
), ((), 
(lambda ctx: [])
), ((',', 'NAME', 'global_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('shift_expr__2', 'arith_expr', 'shift_expr__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
), ((':', 'sliceop__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, s, _ctx, _loc: (
s
), normal_vars=['s'], list_vars=[]))
), (('+', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _all, e, _ctx, _loc: (
UnaryOp(_loc, UAdd(), e)
), normal_vars=['e'], list_vars=[]))
), (('-', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _all, e, _ctx, _loc: (
UnaryOp(_loc, USub(), e)
), normal_vars=['e'], list_vars=[]))
), (('~', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _all, e, _ctx, _loc: (
UnaryOp(_loc, Invert(), e)
), normal_vars=['e'], list_vars=[]))
), (('power',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _ctx, _loc: (
e
), normal_vars=['e'], list_vars=[]))
), (('dotted_name',), 
(lambda ctx, v1: [('m', v1)])
), ((), 
(lambda ctx: [])
), (('augassign', 'expr_stmt__2'), 
(lambda ctx, v1, v2: [('aa', v1), v2])
), (('expr_stmt__3',), 
(lambda ctx, v1: [v1])
), (('(', 'atom__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _all, y, l, _ctx, _loc: (

      y if y else
      Tuple(_loc, None, Load()) if not l else
      GeneratorExp(_loc, l[0][0], ast_for_comprehension(l[2])) if l[2] else
      ast_for_testlist(l[0], l[1])
), normal_vars=['l', 'y'], list_vars=[]))
), (('[', 'atom__2', ']'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _all, l, _ctx, _loc: (

      List(_loc, None, Load()) if not l else
      ListComp(_loc, l[0][0], ast_for_comprehension(l[2])) if l[2] else
      List(_loc, l[0], Load())
), normal_vars=['l'], list_vars=[]))
), (('{', 'atom__3', '}'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _all, d, _ctx, _loc: (
d._replace(location=_loc) if d else Dict(_loc, None, None)
), normal_vars=['d'], list_vars=[]))
), (('NAME',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _all, n, _ctx, _loc: (
Name(_loc, n.value, Load())
), normal_vars=['n'], list_vars=[]))
), (('NUMBER',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _all, n, _ctx, _loc: (
Num(_loc, n.value)
), normal_vars=['n'], list_vars=[]))
), (('STRING', 'atom__4'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda _all, s, _ctx, _loc: (
Str(_loc, ''.join(t.value for t in s))
), normal_vars=[], list_vars=['s']))
), (('...',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Ellipsis(_loc)
), normal_vars=[], list_vars=[]))
), (('None',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
NameConstant(_loc, None)
), normal_vars=[], list_vars=[]))
), (('True',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
NameConstant(_loc, True)
), normal_vars=[], list_vars=[]))
), (('False',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
NameConstant(_loc, False)
), normal_vars=[], list_vars=[]))
), (('EMBEDEXPR',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _ctx, _loc: (
EmbedExp(_loc, e.value)
), normal_vars=['e'], list_vars=[]))
), ((',', 'exprlist__4'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('typedargslist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, l, _ctx, _loc: (
ast_for_arguments(l)
), normal_vars=[], list_vars=['l']))
), (('(', 'decorator__2', ')'), 
(lambda ctx, v1, v2, v3: [('b', v1), v2, (None, v3)])
), ((), 
(lambda ctx: [])
), (('varargslist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('sliceop',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), (('subscriptlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, s, _ctx, c, _loc: (

    s[0] if not c else
    ExtSlice(s) if any(isinstance(k, Slice) for k in s) else
    Index(Tuple(s[0].value.location, [k.value for k in s], Load()))
), normal_vars=[], list_vars=['c', 's']))
), ((':', 'subscript__3', 'subscript__4'), 
(lambda ctx, v1, v2, v3: [('d', v1), v2, v3])
), ((), 
(lambda ctx: [])
), (('dotted_name',), 
(lambda ctx, v1: [('m', v1)])
), (('import_from__3', 'import_from__4', 'import_from__5'), 
(lambda ctx, v1, v2, v3: [v1, v2, v3])
), (('exprlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('del', 'exprlist'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('l', v2)], user=lambda _all, l, _ctx, _loc: (
Delete(_loc, [set_context(e, Del()) for e in l[0]])
), normal_vars=['l'], list_vars=[]))
), ((',', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('b', v2)])
), ((), 
(lambda ctx: [])
), (('except', 'except_clause__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, n, _ctx, _loc, t: (
lambda body: ExceptHandler(_loc, t, n.value if n else None, body)
), normal_vars=['n', 't'], list_vars=[]))
), (('subscript', 'subscriptlist__2'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), (('comp_for',), 
(lambda ctx, v1: [('f', v1)])
), (('testlist_comp__3',), 
(lambda ctx, v1: [v1])
), (('@', 'dotted_name', 'decorator__1', 'NEWLINE'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), ('d', v2), v3, (None, v4)], user=lambda _all, d, _ctx, _loc, b, a: (
ast_for_decorator(_loc, ast_for_dotted_name(d), b, a)
), normal_vars=['a', 'b', 'd'], list_vars=[]))
), (('return', 'return_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _ctx, _loc, t: (
Return(_loc, t)
), normal_vars=['t'], list_vars=[]))
), (('|', 'xor_expr', 'expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), ((',', 'testlist__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('^', 'and_expr', 'xor_expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('+=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Add()
), normal_vars=[], list_vars=[]))
), (('-=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Sub()
), normal_vars=[], list_vars=[]))
), (('*=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Mul()
), normal_vars=[], list_vars=[]))
), (('/=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Div()
), normal_vars=[], list_vars=[]))
), (('%=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Mod()
), normal_vars=[], list_vars=[]))
), (('&=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
BitAnd()
), normal_vars=[], list_vars=[]))
), (('|=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
BitOr()
), normal_vars=[], list_vars=[]))
), (('^=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
BitXor()
), normal_vars=[], list_vars=[]))
), (('<<=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
LShift()
), normal_vars=[], list_vars=[]))
), (('>>=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
RShift()
), normal_vars=[], list_vars=[]))
), (('**=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Pow()
), normal_vars=[], list_vars=[]))
), (('//=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
FloorDiv()
), normal_vars=[], list_vars=[]))
), (('comp_for',), 
(lambda ctx, v1: [('c', v1)])
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('v', v2)])
), ((), 
(lambda ctx: [])
), (('vfpdef',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('from', 'import_from__2', 'import', 'import_from__6'), 
(lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), v4])
), (('test',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('d', v2)])
), ((), 
(lambda ctx: [])
), (('continue',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Continue(_loc)
), normal_vars=[], list_vars=[]))
), (('NAME', 'dotted_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda _all, _ctx, _loc, t: (
t
), normal_vars=[], list_vars=['t']))
), ((',', 'dictorsetmaker__5'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('small_stmt', 'simple_stmt__2'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), (('testlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('test',), 
(lambda ctx, v1: [('u', v1)])
), ((), 
(lambda ctx: [])
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('n', v2)])
), ((), 
(lambda ctx: [])
), ((':', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('t', v2)])
), ((), 
(lambda ctx: [])
), (('trailer', 'power__1'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), ((), 
(lambda ctx: [])
), (('*',), 
(lambda ctx, v1: [(None, v1)])
), (('/',), 
(lambda ctx, v1: [(None, v1)])
), (('%',), 
(lambda ctx, v1: [(None, v1)])
), (('//',), 
(lambda ctx, v1: [(None, v1)])
), (('not', 'not_test'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _all, e, _ctx, _loc: (
UnaryOp(_loc, Not(), e)
), normal_vars=['e'], list_vars=[]))
), (('comparison',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _ctx, _loc: (
e
), normal_vars=['e'], list_vars=[]))
), (('NEWLINE', 'file_input__1'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), (('stmt', 'file_input__1'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), ((',', 'with_item', 'with_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('w', v2), v3])
), ((), 
(lambda ctx: [])
), (('if', 'test', ':', 'suite', 'if_stmt__1', 'if_stmt__2'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6: [('t', v1), ('c', v2), (None, v3), ('s', v4), v5, v6], user=lambda _all, s, e, c, _ctx, t, _loc: (
ast_for_if_stmt(t, c, s, e)
), normal_vars=['e'], list_vars=['c', 's', 't']))
), (('flow_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _ctx, _loc: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('.',), 
(lambda ctx, v1: [('d', v1)])
), (('...',), 
(lambda ctx, v1: [('d', v1)])
), (('for', 'exprlist', 'in', 'testlist', ':', 'suite', 'for_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6, v7: [(None, v1), ('t', v2), (None, v3), ('c', v4), (None, v5), ('s', v6), v7], user=lambda _all, s, e, c, _ctx, t, _loc: (
For(_loc, set_context(Tuple(t[0][0].location, t[0], Store()) if t[1] else t[0][0], Store()), c, s, e)
), normal_vars=['c', 'e', 's', 't'], list_vars=[]))
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('a', v2)])
), ((), 
(lambda ctx: [])
), ((',', 'testlist_star_expr__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('elif', 'test', ':', 'suite', 'if_stmt__1'), 
(lambda ctx, v1, v2, v3, v4, v5: [('t', v1), ('c', v2), (None, v3), ('s', v4), v5])
), ((), 
(lambda ctx: [])
), (('or_test',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _ctx, _loc: (
e
), normal_vars=['e'], list_vars=[]))
), (('lambdef_nocond',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _ctx, _loc: (
e
), normal_vars=['e'], list_vars=[]))
), (('testlist_star_expr__1', 'testlist_star_expr__2'), 
(lambda ctx, v1, v2: [v1, v2])
), ((), 
(lambda ctx: [])
), (('(', 'trailer__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _all, _ctx, _loc, a: (
lambda left_expr: ast_for_call(_loc, left_expr, a)
), normal_vars=['a'], list_vars=[]))
), (('[', 'subscriptlist', ']'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('s', v2), (None, v3)], user=lambda _all, s, _ctx, _loc: (
lambda left_expr: Subscript(_loc, left_expr, s, Load())
), normal_vars=['s'], list_vars=[]))
), (('.', 'NAME'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('n', v2)], user=lambda _all, n, _ctx, _loc: (
lambda left_expr: Attribute(_loc, left_expr, n.value, Load())
), normal_vars=['n'], list_vars=[]))
), (('shift_expr', 'and_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _ctx, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('simple_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _all, s, _ctx, _loc: (
flatten(s)
), normal_vars=['s'], list_vars=[]))
), (('NEWLINE', 'INDENT', 'stmt', 'suite__1', 'DEDENT'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), (None, v2), ('s', v3), v4, (None, v5)], user=lambda _all, s, _ctx, _loc: (
flatten(s)
), normal_vars=[], list_vars=['s']))
), (('test', ':', 'test', 'dictorsetmaker__4'), 
(lambda ctx, v1, v2, v3, v4: [('k', v1), (None, v2), ('v', v3), v4])
), ((), 
(lambda ctx: [])
), (('yield_expr',), 
(lambda ctx, v1: [('v', v1)])
), (('testlist_star_expr',), 
(lambda ctx, v1: [('v', v1)])
), (('testlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _ctx, c, _loc, t: (
ast_for_testlist(t, c)
), normal_vars=[], list_vars=['c', 't']))
), (('decorator', 'decorators__1'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), ((), 
(lambda ctx: [])
), (('exprlist__2', 'exprlist__3'), 
(lambda ctx, v1, v2: [v1, v2])
), (('simple_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('compound_stmt',), 
(lambda ctx, v1: [(None, v1)])
), ((',', 'dotted_as_name', 'dotted_as_names__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('yield_expr',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _ctx, _loc: (
Expr(e.location, e)
), normal_vars=['e'], list_vars=[]))
), (('arith_expr', 'shift_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _ctx, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('and_test', 'or_test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda _all, l, _ctx, _loc: (
l[0] if len(l) == 1 else BoolOp(l[0].location, And(), l)
), normal_vars=[], list_vars=['l']))
), (('NEWLINE', 'eval_input__1'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('sliceop',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), (('&', 'shift_expr', 'and_expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('arglist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _ctx, _loc, a: (
a
), normal_vars=[], list_vars=['a']))
), (('comp_iter',), 
(lambda ctx, v1: [('r', v1)])
), ((), 
(lambda ctx: [])
), (('yield_expr',), 
(lambda ctx, v1: [('y', v1)])
), (('testlist_comp',), 
(lambda ctx, v1: [('l', v1)])
), ((), 
(lambda ctx: [])
), (('import_name',), 
(combine_action(root=lambda ctx, v1: [('i', v1)], user=lambda _all, i, _ctx, _loc: (
i
), normal_vars=['i'], list_vars=[]))
), (('import_from',), 
(combine_action(root=lambda ctx, v1: [('i', v1)], user=lambda _all, i, _ctx, _loc: (
i
), normal_vars=['i'], list_vars=[]))
), (('as', 'expr'), 
(lambda ctx, v1, v2: [(None, v1), ('e', v2)])
), ((), 
(lambda ctx: [])
), (('<<',), 
(lambda ctx, v1: [(None, v1)])
), (('>>',), 
(lambda ctx, v1: [(None, v1)])
), (('compound_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _ctx, _loc: (
_all[0]
), normal_vars=[], list_vars=[]))
), ((';', 'simple_stmt__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('comp_for',), 
(combine_action(root=lambda ctx, v1: [('c', v1)], user=lambda _all, _ctx, c, _loc: (
c
), normal_vars=['c'], list_vars=[]))
), (('comp_if',), 
(combine_action(root=lambda ctx, v1: [('c', v1)], user=lambda _all, _ctx, c, _loc: (
c
), normal_vars=['c'], list_vars=[]))
), ((',', 'testlist_comp__4'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('typedargslist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('expr', 'comparison__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda _all, _ctx, op, _loc, l, r: (
l if not op else Compare(l.location, l, op, r)
), normal_vars=['l'], list_vars=['op', 'r']))
), (('and_expr', 'xor_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _ctx, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('dotted_name', 'dotted_as_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _all, n, _ctx, _loc, a: (
alias('.'.join(t.value for t in n), a.value if a else None)
), normal_vars=['a', 'n'], list_vars=[]))
), (('arglist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('test', 'raise_stmt__2'), 
(lambda ctx, v1, v2: [('a', v1), v2])
), ((), 
(lambda ctx: [])
), (('**', 'factor'), 
(lambda ctx, v1, v2: [(None, v1), ('f', v2)])
), ((), 
(lambda ctx: [])
), (('(', 'classdef__2', ')'), 
(lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)])
), ((), 
(lambda ctx: [])
), (('pass',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Pass(_loc)
), normal_vars=[], list_vars=[]))
), (('factor', 'term__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _ctx, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('d', v2)])
), ((), 
(lambda ctx: [])
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('typedargslist_item', 'typedargslist__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), ((',', 'varargslist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('.', 'import_from__4'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), (('...', 'import_from__4'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), ((), 
(lambda ctx: [])
), (('comp_for',), 
(lambda ctx, v1: [('dc', v1)])
), (('dictorsetmaker__4',), 
(lambda ctx, v1: [v1])
), (('<',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Lt()
), normal_vars=[], list_vars=[]))
), (('>',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Gt()
), normal_vars=[], list_vars=[]))
), (('==',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Eq()
), normal_vars=[], list_vars=[]))
), (('>=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
GtE()
), normal_vars=[], list_vars=[]))
), (('<=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
LtE()
), normal_vars=[], list_vars=[]))
), (('!=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
NotEq()
), normal_vars=[], list_vars=[]))
), (('in',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
In()
), normal_vars=[], list_vars=[]))
), (('not', 'in'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), (None, v2)], user=lambda _all, _ctx, _loc: (
NotIn()
), normal_vars=[], list_vars=[]))
), (('is', 'comp_op__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, n, _ctx, _loc: (
IsNot() if n else Is()
), normal_vars=['n'], list_vars=[]))
), (('for', 'exprlist', 'in', 'or_test', 'comp_for__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [('f', v1), ('t', v2), (None, v3), ('i', v4), v5], user=lambda _all, i, f, _ctx, _loc, t, r: (
[(f, t, i), r or []]
), normal_vars=['f', 'i', 'r', 't'], list_vars=[]))
), (('nonlocal', 'NAME', 'nonlocal_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3], user=lambda _all, n, _ctx, _loc: (
Nonlocal(_loc, [t.value for t in n])
), normal_vars=[], list_vars=['n']))
), (('or_test', 'test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda _all, _ctx, c, _loc, b, a: (
IfExp(a.location, b, a, c) if b else a
), normal_vars=['a', 'b', 'c'], list_vars=[]))
), (('lambdef',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _ctx, _loc: (
e
), normal_vars=['e'], list_vars=[]))
), (('small_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _ctx, _loc: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('test',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), (('expr',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), (('decorators', 'decorated__1'), 
(combine_action(root=lambda ctx, v1, v2: [('d', v1), v2], user=lambda _all, d, cf, _ctx, _loc: (
cf._replace(decorator_list=d)
), normal_vars=['cf', 'd'], list_vars=[]))
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('file_input__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, s, _ctx, _loc: (
Module(flatten(s))
), normal_vars=[], list_vars=['s']))
), (('testlist_star_expr', 'expr_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda _all, _ctx, _loc, t, av, aa, v: (
ast_for_expr_stmt(t, aa, av, v)
), normal_vars=['aa', 'av', 't'], list_vars=['v']))
), (('tfpdef', 'typedargslist_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda _all, d, _ctx, _loc, a: (
(None, a, d)
), normal_vars=['a', 'd'], list_vars=[]))
), (('*', 'typedargslist_item__2'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda _all, s, _ctx, _loc, a: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'tfpdef'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _all, s, _ctx, _loc, a: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('test', 'dictorsetmaker__6'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('NAME', 'import_as_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _all, n, _ctx, _loc, a: (
alias(n.value, a.value if a else None)
), normal_vars=['a', 'n'], list_vars=[]))
), (('test', 'testlist__2'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), (('term__2', 'factor', 'term__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('test', 'with_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda _all, e, _ctx, _loc, t: (
withitem(t, set_context(e, Store()) if e else None)
), normal_vars=['e', 't'], list_vars=[]))
), (('yield_expr',), 
(lambda ctx, v1: [('av', v1)])
), (('testlist',), 
(lambda ctx, v1: [('av', v1)])
), (('while', 'test', ':', 'suite', 'while_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('c', v2), (None, v3), ('s', v4), v5], user=lambda _all, s, e, c, _ctx, _loc: (
While(_loc, c, s, e)
), normal_vars=['c', 'e', 's'], list_vars=[]))
), (('def', 'NAME', 'parameters', 'funcdef__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6: [(None, v1), ('n', v2), ('p', v3), v4, (None, v5), ('b', v6)], user=lambda _all, n, _ctx, _loc, p, r, b: (
FunctionDef(_loc, n.value, p, b, [], r)
), normal_vars=['b', 'n', 'p', 'r'], list_vars=[]))
), (('argument',), 
(combine_action(root=lambda ctx, v1: [('a', v1)], user=lambda _all, _ctx, _loc, a: (
a
), normal_vars=['a'], list_vars=[]))
), (('*', 'test'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _all, s, _ctx, _loc, a: (
(s, a, None, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'test'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _all, s, _ctx, _loc, a: (
(s, a, None, None)
), normal_vars=['a', 's'], list_vars=[]))
), ((',', 'dictorsetmaker__7'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('if', 'test_nocond', 'comp_if__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [('i', v1), ('c', v2), v3], user=lambda _all, i, _ctx, c, _loc, r: (
[(i, c), r or []]
), normal_vars=['c', 'i', 'r'], list_vars=[]))
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('l', v3)])
), ((), 
(lambda ctx: [])
), ((',', 'import_as_names__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('classdef',), 
(lambda ctx, v1: [('cf', v1)])
), (('funcdef',), 
(lambda ctx, v1: [('cf', v1)])
), (('testlist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('testlist_comp__1', 'testlist_comp__3'), 
(lambda ctx, v1, v2: [v1, v2])
), ((), 
(lambda ctx: [])
), (('*', 'expr'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _all, e, _ctx, _loc: (
Starred(_loc, e, Load())
), normal_vars=['e'], list_vars=[]))
), (('dictorsetmaker__2',), 
(lambda ctx, v1: [v1])
), (('comp_for',), 
(lambda ctx, v1: [('sc', v1)])
), (('dictorsetmaker__6',), 
(lambda ctx, v1: [v1])
), (('break_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('continue_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('return_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('raise_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('yield_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('and', 'not_test', 'and_test__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('l', v2), v3])
), ((), 
(lambda ctx: [])
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('varargslist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('or', 'and_test', 'or_test__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('l', v2), v3])
), ((), 
(lambda ctx: [])
), (('testlist_comp__1', 'testlist_comp__2'), 
(combine_action(root=lambda ctx, v1, v2: [v1, v2], user=lambda _all, f, c, _ctx, t, _loc: (
(t, c, f)
), normal_vars=['f'], list_vars=['c', 't']))
), (('tfpdef',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('testlist_comp',), 
(lambda ctx, v1: [('l', v1)])
), ((), 
(lambda ctx: [])
), (('test', 'subscript__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), ((':', 'subscript__5', 'subscript__6'), 
(lambda ctx, v1, v2, v3: [('d', v1), v2, v3])
), (('subscript__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, d, s, _ctx, _loc, l, u: (
Slice(l, u, s) if d else Index(l)
), normal_vars=['d', 'l', 's', 'u'], list_vars=[]))
), (('not_test', 'and_test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda _all, l, _ctx, _loc: (
l[0] if len(l) == 1 else BoolOp(l[0].location, Or(), l)
), normal_vars=[], list_vars=['l']))
), (('exprlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _ctx, c, _loc, t: (
(t, bool(c))
), normal_vars=[], list_vars=['c', 't']))
), (('typedargslist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('arglist_item', 'arglist__2'), 
(lambda ctx, v1, v2: [('a', v1), v2])
), (('test', 'dictorsetmaker__1'), 
(combine_action(root=lambda ctx, v1, v2: [('k1', v1), v2], user=lambda _all, s, v1, sc, _ctx, _loc, k1, dc, v, k: (

    DictComp(k1.location, k1, v1, ast_for_comprehension(dc)) if dc else
    SetComp(k1.location, k1, ast_for_comprehension(sc)) if sc else
    Dict(k1.location, [k1]+k, [v1]+v) if v1 else
    Set(k1.location, [k1]+s)
), normal_vars=['dc', 'k1', 'sc', 'v1'], list_vars=['k', 's', 'v']))
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('vfpdef', 'varargslist_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda _all, d, _ctx, _loc, a: (
(None, a, d)
), normal_vars=['a', 'd'], list_vars=[]))
), (('*', 'varargslist_item__2'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda _all, s, _ctx, _loc, a: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'vfpdef'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _all, s, _ctx, _loc, a: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('->', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('r', v2)])
), ((), 
(lambda ctx: [])
), (('break',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Break(_loc)
), normal_vars=[], list_vars=[]))
), (('import_as_names__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, n, _ctx, _loc: (
n
), normal_vars=[], list_vars=['n']))
), (('comp_op', 'expr', 'comparison__1'), 
(lambda ctx, v1, v2, v3: [('op', v1), ('r', v2), v3])
), ((), 
(lambda ctx: [])
), (('except_clause', ':', 'suite', 'try_stmt__2'), 
(lambda ctx, v1, v2, v3, v4: [('ec', v1), (None, v2), ('es', v3), v4])
), ((), 
(lambda ctx: [])
), (('subscriptlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('xor_expr', 'expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _ctx, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), ((',', 'arglist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('raise', 'raise_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _ctx, _loc, b, a: (
Raise(_loc, a, b)
), normal_vars=['a', 'b'], list_vars=[]))
), (('term', 'arith_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _ctx, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('STRING', 'atom__4'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('if', 'or_test', 'else', 'test'), 
(lambda ctx, v1, v2, v3, v4: [(None, v1), ('b', v2), (None, v3), ('c', v4)])
), ((), 
(lambda ctx: [])
), (('lambda', 'lambdef__1', ':', 'test'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), ('t', v4)], user=lambda t, _all, _ctx, _loc, a: (
Lambda(_loc, a or arguments(None, None, None, None, None, None), t)
), normal_vars=['a', 't'], list_vars=[]))
), (('from', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('f', v2)])
), (('testlist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('assert', 'test', 'assert_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('a', v2), v3], user=lambda _all, _ctx, _loc, b, a: (
Assert(_loc, a, b)
), normal_vars=['a', 'b'], list_vars=[]))
), (('NAME', 'tfpdef__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _all, n, _ctx, _loc, t: (
arg(_loc, n.value, t)
), normal_vars=['n', 't'], list_vars=[]))
), (('import_as_name', 'import_as_names__2'), 
(lambda ctx, v1, v2: [('n', v1), v2])
), (('testlist_star_expr__1', 'testlist_star_expr__2'), 
(combine_action(root=lambda ctx, v1, v2: [v1, v2], user=lambda _all, _ctx, c, _loc, t: (
ast_for_testlist(t, c)
), normal_vars=[], list_vars=['c', 't']))
), (('class', 'NAME', 'classdef__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('n', v2), v3, (None, v4), ('b', v5)], user=lambda _all, n, _ctx, _loc, b, a: (
ast_for_classdef(_loc, n.value, a, b)
), normal_vars=['a', 'b', 'n'], list_vars=[]))
), (('(', 'parameters__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _all, _ctx, _loc, t: (
t or arguments(None, None, None, None, None, None)
), normal_vars=['t'], list_vars=[]))
), (('lambda', 'lambdef_nocond__1', ':', 'test_nocond'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), ('t', v4)], user=lambda t, _all, _ctx, _loc, a: (
Lambda(_loc, a or arguments(None, None, None, None, None, None), t)
), normal_vars=['a', 't'], list_vars=[]))
), (('embed_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('expr_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('del_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('pass_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('flow_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('import_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('global_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('nonlocal_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('assert_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('try', ':', 'suite', 'try_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), (None, v2), ('b', v3), v4], user=lambda ec, _all, f, _loc, es, _ctx, l, b: (
Try(_loc, b, [kc(ks) for kc, ks in zip(ec, es)], l, f)
), normal_vars=['b', 'f', 'l'], list_vars=['ec', 'es']))
), (('test',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), (('finally', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('f', v3)])
), ((), 
(lambda ctx: [])
), (('from', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('b', v2)])
), ((), 
(lambda ctx: [])
), (('atom', 'power__1', 'power__2'), 
(combine_action(root=lambda ctx, v1, v2, v3: [('a', v1), v2, v3], user=lambda _all, f, _ctx, _loc, t, a: (
ast_for_power(a, t, f)
), normal_vars=['a', 'f'], list_vars=['t']))
), (('.', 'NAME', 'dotted_name__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('t', v2), v3])
), ((), 
(lambda ctx: [])
), (('simple_stmt__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('simple_stmt__1', 'NEWLINE'), 
(combine_action(root=lambda ctx, v1, v2: [v1, (None, v2)], user=lambda _all, s, _ctx, _loc: (
s
), normal_vars=[], list_vars=['s']))
), ((',', 'typedargslist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('+',), 
(lambda ctx, v1: [(None, v1)])
), (('-',), 
(lambda ctx, v1: [(None, v1)])
), (('NAME',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _all, n, _ctx, _loc: (
arg(_loc, n.value, None)
), normal_vars=['n'], list_vars=[]))
), (('with', 'with_item', 'with_stmt__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('w', v2), v3, (None, v4), ('b', v5)], user=lambda _all, w, _ctx, _loc, b: (
With(_loc, w, b)
), normal_vars=['b'], list_vars=['w']))
), (('=', 'expr_stmt__4', 'expr_stmt__3'), 
(lambda ctx, v1, v2, v3: [(None, v1), v2, v3])
), ((), 
(lambda ctx: [])
), (('import_from__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, s, d, n, _ctx, m, _loc: (
ImportFrom(_loc, '.'.join(t.value for t in m) if m else None, [alias('*', None)] if s else n, sum(len(t.type) for t in d))
), normal_vars=['m', 'n', 's'], list_vars=['d']))
), (('dotted_as_name', 'dotted_as_names__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _all, n, _ctx, _loc: (
n
), normal_vars=[], list_vars=['n']))
), (('*',), 
(lambda ctx, v1: [('s', v1)])
), (('(', 'import_as_names', ')'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), (None, v3)])
), (('import_as_names',), 
(lambda ctx, v1: [('n', v1)])
), (('not',), 
(lambda ctx, v1: [('n', v1)])
), ((), 
(lambda ctx: [])
), (('varargslist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('varargslist_item', 'varargslist__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), (('global', 'NAME', 'global_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3], user=lambda _all, n, _ctx, _loc: (
Global(_loc, [t.value for t in n])
), normal_vars=[], list_vars=['n']))
), (('EMBEDSTAT',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _ctx, _loc: (
EmbedStat(_loc, e.value)
), normal_vars=['e'], list_vars=[]))
), (('comp_iter',), 
(lambda ctx, v1: [('r', v1)])
), ((), 
(lambda ctx: [])
), (('test',), 
(lambda ctx, v1: [('u', v1)])
), ((), 
(lambda ctx: [])
), (('import_as_names__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('test', 'except_clause__2'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), ((), 
(lambda ctx: [])
), ((':', 'test', 'dictorsetmaker__3'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('v1', v2), v3])
), (('if_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('while_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('for_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('try_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('with_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('funcdef',), 
(lambda ctx, v1: [(None, v1)])
), (('classdef',), 
(lambda ctx, v1: [(None, v1)])
), (('decorated',), 
(lambda ctx, v1: [(None, v1)])
), (('test', 'argument__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda _all, _ctx, c, _loc, t, v: (
(None, t, c, v)
), normal_vars=['c', 't', 'v'], list_vars=[]))
), (('NEWLINE',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _ctx, _loc: (
Interactive([])
), normal_vars=[], list_vars=[]))
), (('simple_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _all, s, _ctx, _loc: (
Interactive(flatten(s))
), normal_vars=['s'], list_vars=[]))
), (('compound_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _all, s, _ctx, _loc: (
Interactive(flatten(s))
), normal_vars=['s'], list_vars=[]))
), (('yield', 'yield_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, f, _ctx, _loc, t: (
YieldFrom(_loc, f) if f else Yield(_loc, t)
), normal_vars=['f', 't'], list_vars=[]))
), (('testlist', 'eval_input__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda _all, _ctx, _loc, t: (
Expression(t)
), normal_vars=['t'], list_vars=[]))
), (('varargslist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, l, _ctx, _loc: (
ast_for_arguments(l)
), normal_vars=[], list_vars=['l']))
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('a', v2)])
), ((), 
(lambda ctx: [])
), ((',', 'NAME', 'nonlocal_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('stmt', 'suite__1'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('arith_expr__2', 'term', 'arith_expr__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
)]
TABLE = {'stmt': {'True': 0, '+': 0, 'break': 0, '{': 0, 'EMBEDEXPR': 0, 'from': 0, 'assert': 0, 'lambda': 0, 'EMBEDSTAT': 0, 'global': 0, 'yield': 0, '*': 0, 'while': 0, 'pass': 0, 'class': 0, 'if': 0, '~': 0, '@': 0, '-': 0, 'nonlocal': 0, 'def': 0, 'for': 0, 'try': 0, 'with': 0, 'continue': 0, 'return': 0, '[': 0, 'del': 0, 'NAME': 0, 'import': 0, '(': 0, 'False': 0, 'None': 0, 'not': 0, 'STRING': 0, 'raise': 0, '...': 0, 'NUMBER': 0}, 'import_name': {'import': 1}, 'decorators': {'@': 2}, 'try_stmt__1': {'except': 3, 'finally': 4}, 'atom__3': {'-': 7, 'True': 7, '+': 7, '~': 7, '{': 7, 'EMBEDEXPR': 7, 'lambda': 7, '[': 7, '}': 8, 'NAME': 7, '(': 7, 'False': 7, 'None': 7, 'NUMBER': 7, 'not': 7, '...': 7, 'STRING': 7}, 'global_stmt__1': {',': 9, 'NEWLINE': 10, ';': 10}, 'shift_expr__1': {'&': 12, '==': 12, '|': 12, 'if': 12, 'from': 12, '%=': 12, '!=': 12, 'NEWLINE': 12, '^': 12, '>=': 12, '|=': 12, '>>': 11, '//=': 12, '+=': 12, '<<=': 12, 'for': 12, 'and': 12, '&=': 12, '}': 12, ':': 12, ';': 12, ')': 12, 'is': 12, '**=': 12, '/=': 12, '=': 12, '>': 12, '>>=': 12, '*=': 12, '<': 12, '^=': 12, ',': 12, 'else': 12, 'or': 12, 'in': 12, '<<': 11, 'not': 12, '<=': 12, '-=': 12, 'as': 12, ']': 12}, 'sliceop': {':': 13}, 'factor': {'-': 15, 'True': 17, '+': 14, '{': 17, 'False': 17, '[': 17, 'NAME': 17, '(': 17, 'EMBEDEXPR': 17, '~': 16, 'STRING': 17, 'None': 17, '...': 17, 'NUMBER': 17}, 'import_from__5': {'NAME': 18, 'import': 19}, 'expr_stmt__1': {'//=': 20, '*=': 20, '**=': 20, '/=': 20, '&=': 20, ';': 21, '+=': 20, '^=': 20, '|=': 20, '>>=': 20, 'NEWLINE': 21, '-=': 20, '%=': 20, '<<=': 20, '=': 21}, 'atom': {'STRING': 27, 'NAME': 25, '(': 22, '{': 24, 'False': 31, 'None': 29, '[': 23, 'EMBEDEXPR': 32, 'True': 30, '...': 28, 'NUMBER': 26}, 'try_stmt': {'try': 332}, 'typedargslist': {'NAME': 35, '*': 35, '**': 35}, 'raise_stmt__2': {'from': 337, ';': 338, 'NEWLINE': 338}, 'varargslist__3': {'NAME': 38, ':': 39, '*': 38, '**': 38}, 'subscriptlist': {'-': 42, 'True': 42, '+': 42, ':': 42, '{': 42, 'EMBEDEXPR': 42, '~': 42, 'lambda': 42, '[': 42, 'NAME': 42, '(': 42, 'False': 42, 'None': 42, 'NUMBER': 42, 'not': 42, '...': 42, 'STRING': 42}, 'import_stmt': {'from': 165, 'import': 164}, 'import_from__2': {'.': 46, 'NAME': 45, '...': 46}, 'exprlist__4': {'-': 47, 'True': 47, '+': 47, '*': 47, ';': 48, '{': 47, 'False': 47, 'NEWLINE': 48, '[': 47, 'NAME': 47, '(': 47, 'EMBEDEXPR': 47, 'in': 48, 'None': 47, 'STRING': 47, '~': 47, '...': 47, 'NUMBER': 47}, 'del_stmt': {'del': 49}, 'assert_stmt__1': {',': 50, 'NEWLINE': 51, ';': 51}, 'except_clause': {'except': 52}, 'subscriptlist__1': {'-': 53, 'True': 53, '+': 53, ':': 53, '{': 53, 'EMBEDEXPR': 53, '~': 53, 'lambda': 53, '[': 53, 'NAME': 53, '(': 53, 'False': 53, 'None': 53, 'NUMBER': 53, 'not': 53, '...': 53, 'STRING': 53}, 'testlist_comp__2': {')': 55, 'for': 54, ',': 55, ']': 55}, 'decorator': {'@': 56}, 'return_stmt': {'return': 57}, 'expr__1': {'==': 59, '|': 58, 'if': 59, 'from': 59, '%=': 59, '!=': 59, 'NEWLINE': 59, '|=': 59, ']': 59, '//=': 59, 'or': 59, '+=': 59, '<<=': 59, 'for': 59, '<': 59, '&=': 59, ':': 59, ';': 59, ')': 59, 'is': 59, '**=': 59, '/=': 59, '=': 59, '>': 59, '>>=': 59, '*=': 59, 'and': 59, '^=': 59, ',': 59, 'else': 59, 'in': 59, '>=': 59, '}': 59, 'not': 59, '-=': 59, 'as': 59, '<=': 59}, 'testlist__2': {')': 61, ',': 60, ':': 61, ';': 61, '=': 61, 'NEWLINE': 61}, 'for_stmt__1': {'True': 63, '+': 63, 'break': 63, '?>': 63, 'DEDENT': 63, '{': 63, 'False': 63, 'from': 63, 'assert': 63, 'lambda': 63, 'EMBEDSTAT': 63, 'global': 63, 'yield': 63, 'NEWLINE': 63, '*': 63, 'while': 63, 'pass': 63, 'class': 63, 'if': 63, '~': 63, '@': 63, '-': 63, 'nonlocal': 63, 'def': 63, 'for': 63, 'try': 63, 'with': 63, 'continue': 63, 'return': 63, '[': 63, 'del': 63, '}': 63, 'NAME': 63, 'import': 63, '...': 63, '(': 63, 'EMBEDEXPR': 63, 'None': 63, 'not': 63, 'STRING': 63, 'raise': 63, 'else': 62, 'NUMBER': 63}, 'xor_expr__1': {'==': 65, '|': 65, 'in': 65, 'from': 65, '%=': 65, '!=': 65, 'NEWLINE': 65, '^': 64, '|=': 65, ']': 65, '//=': 65, '+=': 65, 'if': 65, '<<=': 65, 'for': 65, 'and': 65, '&=': 65, ':': 65, ';': 65, ')': 65, 'is': 65, '**=': 65, '/=': 65, '=': 65, '>': 65, '>>=': 65, '*=': 65, '<': 65, '^=': 65, ',': 65, 'else': 65, 'or': 65, '>=': 65, '}': 65, 'not': 65, '-=': 65, 'as': 65, '<=': 65}, 'augassign': {'&=': 71, '*=': 68, '**=': 76, '-=': 67, '/=': 69, '%=': 70, '>>=': 75, '^=': 73, '|=': 72, '//=': 77, '+=': 66, '<<=': 74}, 'argument__1': {')': 80, 'for': 78, ',': 80, '=': 79}, 'varargslist_item__2': {'NAME': 81, ',': 82, ':': 82}, 'import_from__1': {'from': 83}, 'varargslist_item__1': {',': 87, '=': 86, ':': 87}, 'continue_stmt': {'continue': 88}, 'dotted_name': {'NAME': 89}, 'dictorsetmaker__4': {',': 90, '}': 91}, 'simple_stmt__1': {'True': 92, '+': 92, 'break': 92, '{': 92, 'False': 92, 'from': 92, 'assert': 92, 'lambda': 92, 'EMBEDSTAT': 92, 'global': 92, '*': 92, 'pass': 92, 'None': 92, '~': 92, 'STRING': 92, '-': 92, 'nonlocal': 92, 'yield': 92, 'continue': 92, 'return': 92, '[': 92, 'del': 92, 'NAME': 92, '...': 92, '(': 92, 'EMBEDEXPR': 92, 'not': 92, 'raise': 92, 'import': 92, 'NUMBER': 92}, 'testlist__3': {'-': 93, 'True': 93, '+': 93, 'NEWLINE': 94, '~': 93, ';': 94, '{': 93, 'EMBEDEXPR': 93, '=': 94, 'lambda': 93, '[': 93, ')': 94, 'NAME': 93, ':': 94, '(': 93, 'False': 93, 'None': 93, 'NUMBER': 93, 'not': 93, '...': 93, 'STRING': 93}, 'subscript__5': {'-': 95, ',': 96, 'True': 95, '+': 95, '~': 95, '{': 95, 'EMBEDEXPR': 95, 'lambda': 95, '[': 95, 'NAME': 95, ']': 96, '(': 95, 'False': 95, ':': 96, 'None': 95, 'NUMBER': 95, 'not': 95, '...': 95, 'STRING': 95}, 'except_clause__2': {':': 98, 'as': 97}, 'tfpdef__1': {')': 100, ',': 100, '=': 100, ':': 99}, 'power__1': {'|': 102, 'in': 102, '^': 102, '&': 102, '>>': 102, '<<=': 102, '-': 102, '<': 102, '|=': 102, 'is': 102, 'or': 102, '/=': 102, '=': 102, '//': 102, '[': 101, 'and': 102, '^=': 102, ',': 102, '(': 101, 'if': 102, '>=': 102, '<<': 102, '%': 102, 'as': 102, '.': 101, '+': 102, '==': 102, 'from': 102, '%=': 102, '!=': 102, 'not': 102, 'NEWLINE': 102, '&=': 102, ']': 102, '*': 102, '**': 102, '+=': 102, '*=': 102, '//=': 102, ':': 102, 'for': 102, '/': 102, '>': 102, '-=': 102, ')': 102, ';': 102, '**=': 102, '}': 102, '>>=': 102, 'else': 102, '<=': 102}, 'exprlist': {'-': 283, 'True': 283, '+': 283, '*': 283, '{': 283, 'False': 283, '[': 283, 'NAME': 283, '(': 283, 'EMBEDEXPR': 283, 'None': 283, 'STRING': 283, '~': 283, '...': 283, 'NUMBER': 283}, 'testlist_star_expr__1': {'-': 84, 'True': 84, '+': 84, '~': 84, '{': 84, '*': 85, 'EMBEDEXPR': 84, 'lambda': 84, '[': 84, 'NAME': 84, '(': 84, 'False': 84, 'None': 84, 'NUMBER': 84, 'not': 84, '...': 84, 'STRING': 84}, 'not_test': {'-': 108, 'True': 108, '+': 108, '~': 108, '{': 108, 'False': 108, '[': 108, 'NAME': 108, '(': 108, 'EMBEDEXPR': 108, 'not': 107, 'STRING': 108, 'None': 108, '...': 108, 'NUMBER': 108}, 'file_input__1': {'True': 110, '+': 110, 'break': 110, '?>': 111, '{': 110, 'False': 110, 'from': 110, 'lambda': 110, 'NEWLINE': 109, 'EMBEDSTAT': 110, 'global': 110, 'yield': 110, '*': 110, 'while': 110, 'pass': 110, 'class': 110, 'if': 110, '~': 110, '@': 110, '-': 110, 'nonlocal': 110, 'assert': 110, 'def': 110, 'for': 110, 'try': 110, 'with': 110, 'continue': 110, 'return': 110, '[': 110, 'del': 110, '}': 111, 'NAME': 110, 'import': 110, '(': 110, 'EMBEDEXPR': 110, 'None': 110, 'not': 110, 'STRING': 110, 'raise': 110, '...': 110, 'NUMBER': 110}, 'testlist_comp__3': {')': 176, ',': 175, ']': 176}, 'if_stmt': {'if': 114}, 'flow_stmt': {'yield': 115, 'break': 115, 'raise': 115, 'return': 115, 'continue': 115}, 'import_from__3': {'.': 116, '...': 117}, 'single_input': {'class': 385, 'True': 384, '+': 384, 'break': 384, '{': 384, 'False': 384, 'from': 384, 'lambda': 384, 'NEWLINE': 383, 'EMBEDSTAT': 384, 'global': 384, '*': 384, 'while': 385, 'pass': 384, 'None': 384, 'if': 385, '~': 384, 'STRING': 384, '-': 384, 'nonlocal': 384, 'assert': 384, 'def': 385, 'yield': 384, 'try': 385, 'with': 385, 'continue': 384, 'return': 384, '[': 384, 'del': 384, 'for': 385, 'NAME': 384, '...': 384, '(': 384, 'EMBEDEXPR': 384, 'not': 384, '@': 385, 'raise': 384, 'import': 384, 'NUMBER': 384}, 'for_stmt': {'for': 118}, 'dotted_as_name__1': {',': 120, 'NEWLINE': 120, 'as': 119, ';': 120}, 'expr_stmt__2': {'-': 238, 'True': 238, '+': 238, '~': 238, '{': 238, 'False': 238, 'EMBEDEXPR': 238, 'lambda': 238, '[': 238, 'NAME': 238, '(': 238, 'yield': 237, 'None': 238, 'NUMBER': 238, 'not': 238, '...': 238, 'STRING': 238}, 'testlist_star_expr__2': {'&=': 122, '|=': 122, ';': 122, '**=': 122, '/=': 122, '%=': 122, 'NEWLINE': 122, '-=': 122, '<<=': 122, '^=': 122, ',': 121, '>>=': 122, '//=': 122, '+=': 122, '*=': 122, '=': 122}, 'if_stmt__1': {'True': 124, 'break': 124, 'False': 124, 'assert': 124, 'global': 124, 'pass': 124, 'while': 124, 'None': 124, '-': 124, 'def': 124, 'not': 124, 'with': 124, 'continue': 124, 'return': 124, '[': 124, 'del': 124, '}': 124, '(': 124, 'if': 124, 'NEWLINE': 124, 'STRING': 124, '...': 124, 'NUMBER': 124, '+': 124, '{': 124, 'from': 124, 'DEDENT': 124, 'lambda': 124, 'EMBEDSTAT': 124, '*': 124, 'class': 124, '~': 124, '@': 124, 'EMBEDEXPR': 124, 'nonlocal': 124, 'elif': 123, 'yield': 124, 'try': 124, 'NAME': 124, 'import': 124, '?>': 124, 'for': 124, 'raise': 124, 'else': 124}, 'test_nocond': {'-': 125, 'True': 125, '+': 125, '~': 125, '{': 125, 'False': 125, 'lambda': 126, '[': 125, 'NAME': 125, '(': 125, 'EMBEDEXPR': 125, 'None': 125, 'STRING': 125, 'not': 125, '...': 125, 'NUMBER': 125}, 'typedargslist_item__2': {')': 276, 'NAME': 275, ',': 276}, 'trailer': {'.': 131, '(': 129, '[': 130}, 'and_expr': {'-': 132, 'True': 132, '+': 132, '{': 132, 'False': 132, '[': 132, 'NAME': 132, '(': 132, 'EMBEDEXPR': 132, 'None': 132, 'STRING': 132, '~': 132, '...': 132, 'NUMBER': 132}, 'expr_stmt__4': {'-': 138, 'True': 138, '+': 138, '(': 138, '{': 138, 'False': 138, '~': 138, 'EMBEDEXPR': 138, 'lambda': 138, '[': 138, 'NAME': 138, '*': 138, 'yield': 137, 'None': 138, 'NUMBER': 138, 'not': 138, '...': 138, 'STRING': 138}, 'dictorsetmaker': {'-': 287, 'True': 287, '+': 287, '~': 287, '{': 287, 'EMBEDEXPR': 287, 'lambda': 287, '[': 287, 'NAME': 287, '(': 287, 'False': 287, 'None': 287, 'NUMBER': 287, 'not': 287, '...': 287, 'STRING': 287}, 'decorators__1': {'class': 141, 'def': 141, '@': 140}, 'exprlist__1': {'-': 142, 'True': 142, '+': 142, '*': 142, '{': 142, 'False': 142, '[': 142, 'NAME': 142, '(': 142, 'EMBEDEXPR': 142, 'None': 142, 'STRING': 142, '~': 142, '...': 142, 'NUMBER': 142}, 'term__2': {'%': 105, '/': 104, '//': 106, '*': 103}, 'dotted_as_names__1': {',': 145, 'NEWLINE': 146, ';': 146}, 'yield_stmt': {'yield': 147}, 'shift_expr': {'-': 148, 'True': 148, '+': 148, '{': 148, 'False': 148, '[': 148, 'NAME': 148, '(': 148, 'EMBEDEXPR': 148, 'None': 148, 'STRING': 148, '~': 148, '...': 148, 'NUMBER': 148}, 'or_test': {'-': 149, 'True': 149, '+': 149, '~': 149, '{': 149, 'False': 149, '[': 149, 'NAME': 149, '(': 149, 'EMBEDEXPR': 149, 'None': 149, 'STRING': 149, 'not': 149, '...': 149, 'NUMBER': 149}, 'eval_input__1': {')': 151, 'NEWLINE': 150}, 'while_stmt__1': {'True': 153, '+': 153, 'break': 153, '?>': 153, 'DEDENT': 153, '{': 153, 'False': 153, 'from': 153, 'assert': 153, 'lambda': 153, 'EMBEDSTAT': 153, 'global': 153, 'yield': 153, 'NEWLINE': 153, '*': 153, 'while': 153, 'pass': 153, 'class': 153, 'if': 153, '~': 153, '@': 153, '-': 153, 'nonlocal': 153, 'def': 153, 'for': 153, 'try': 153, 'with': 153, 'continue': 153, 'return': 153, '[': 153, 'del': 153, '}': 153, 'NAME': 153, 'import': 153, '...': 153, '(': 153, 'EMBEDEXPR': 153, 'None': 153, 'not': 153, 'STRING': 153, 'raise': 153, 'else': 152, 'NUMBER': 153}, 'classdef__2': {'-': 194, 'True': 194, '+': 194, '(': 194, '{': 194, 'EMBEDEXPR': 194, '~': 194, '**': 194, 'lambda': 194, '[': 194, ')': 195, 'NAME': 194, '*': 194, 'False': 194, 'None': 194, 'NUMBER': 194, 'not': 194, '...': 194, 'STRING': 194}, 'and_expr__1': {'&': 156, '==': 157, '|': 157, 'if': 157, 'from': 157, '%=': 157, '!=': 157, 'NEWLINE': 157, '^': 157, '|=': 157, ']': 157, '//=': 157, 'or': 157, '+=': 157, '<<=': 157, 'for': 157, 'and': 157, '&=': 157, ':': 157, ';': 157, ')': 157, 'is': 157, '**=': 157, '/=': 157, '=': 157, '>': 157, '>>=': 157, '*=': 157, '<': 157, '^=': 157, ',': 157, 'else': 157, 'in': 157, '>=': 157, '}': 157, 'not': 157, '-=': 157, 'as': 157, '<=': 157}, 'arglist': {'-': 158, 'True': 158, '+': 158, '(': 158, '{': 158, 'EMBEDEXPR': 158, '~': 158, '**': 158, 'lambda': 158, '[': 158, 'NAME': 158, '*': 158, 'False': 158, 'None': 158, 'NUMBER': 158, 'not': 158, '...': 158, 'STRING': 158}, 'atom__1': {'-': 162, 'True': 162, '+': 162, '(': 162, '{': 162, 'False': 162, '~': 162, 'EMBEDEXPR': 162, 'lambda': 162, '[': 162, ')': 163, 'NAME': 162, '*': 162, 'yield': 161, 'None': 162, 'NUMBER': 162, 'not': 162, '...': 162, 'STRING': 162}, 'subscript__2': {',': 44, ']': 44, ':': 43}, 'with_item__1': {',': 167, ':': 167, 'as': 166}, 'shift_expr__2': {'<<': 168, '>>': 169}, 'compound_stmt': {'while': 170, 'def': 170, 'for': 170, 'if': 170, 'class': 170, 'try': 170, 'with': 170, '@': 170}, 'simple_stmt__2': {';': 171, 'NEWLINE': 172}, 'exprlist__2': {'-': 220, 'True': 220, '+': 220, '*': 221, '{': 220, 'False': 220, '[': 220, 'NAME': 220, '(': 220, 'EMBEDEXPR': 220, 'None': 220, 'STRING': 220, '~': 220, '...': 220, 'NUMBER': 220}, 'comp_iter': {'for': 173, 'if': 174}, 'typedargslist__3': {')': 178, 'NAME': 177, '*': 177, '**': 177}, 'comparison': {'-': 179, 'True': 179, '+': 179, '{': 179, 'False': 179, '[': 179, 'NAME': 179, '(': 179, 'EMBEDEXPR': 179, 'None': 179, 'STRING': 179, '~': 179, '...': 179, 'NUMBER': 179}, 'xor_expr': {'-': 180, 'True': 180, '+': 180, '{': 180, 'False': 180, '[': 180, 'NAME': 180, '(': 180, 'EMBEDEXPR': 180, 'None': 180, 'STRING': 180, '~': 180, '...': 180, 'NUMBER': 180}, 'dotted_as_name': {'NAME': 181}, 'import_from__6': {'NAME': 357, '(': 356, '*': 355}, 'arglist__3': {'-': 182, 'True': 182, '+': 182, '(': 182, '{': 182, 'EMBEDEXPR': 182, '~': 182, '**': 182, 'lambda': 182, '[': 182, ')': 183, 'NAME': 182, '*': 182, 'False': 182, 'None': 182, 'NUMBER': 182, 'not': 182, '...': 182, 'STRING': 182}, 'raise_stmt__1': {'-': 184, 'True': 184, '+': 184, '~': 184, ';': 185, '{': 184, 'EMBEDEXPR': 184, 'lambda': 184, '[': 184, 'NAME': 184, 'NEWLINE': 185, '(': 184, 'False': 184, 'None': 184, 'NUMBER': 184, 'not': 184, '...': 184, 'STRING': 184}, 'power__2': {'|': 187, 'in': 187, '^': 187, '&': 187, '>>': 187, '<<=': 187, '-': 187, 'and': 187, '|=': 187, 'is': 187, 'or': 187, '/=': 187, '=': 187, '//': 187, '}': 187, '<': 187, '^=': 187, ',': 187, 'if': 187, '>=': 187, '<<': 187, '%': 187, 'as': 187, '+': 187, '==': 187, 'from': 187, '%=': 187, '!=': 187, 'not': 187, 'NEWLINE': 187, '&=': 187, ']': 187, '*': 187, '**': 186, '+=': 187, '*=': 187, '//=': 187, ':': 187, 'for': 187, '/': 187, '>': 187, '-=': 187, ')': 187, ';': 187, '**=': 187, '>>=': 187, 'else': 187, '<=': 187}, 'classdef__1': {':': 189, '(': 188}, 'pass_stmt': {'pass': 190}, 'term': {'-': 191, 'True': 191, '+': 191, '{': 191, 'False': 191, '[': 191, 'NAME': 191, '(': 191, 'EMBEDEXPR': 191, 'None': 191, 'STRING': 191, '~': 191, '...': 191, 'NUMBER': 191}, 'typedargslist_item__1': {')': 193, ',': 193, '=': 192}, 'typedargslist__1': {'NAME': 196, '*': 196, '**': 196}, 'varargslist__2': {',': 197, ':': 198}, 'import_from__4': {'.': 199, 'NAME': 201, 'import': 201, '...': 200}, 'dictorsetmaker__3': {'for': 202, ',': 203, '}': 203}, 'comp_op': {'<': 204, 'is': 212, '==': 206, '>=': 207, 'in': 210, 'not': 211, '>': 205, '!=': 209, '<=': 208}, 'comp_for': {'for': 213}, 'nonlocal_stmt': {'nonlocal': 214}, 'test': {'-': 215, 'True': 215, '+': 215, '~': 215, '{': 215, 'False': 215, 'lambda': 216, '[': 215, 'NAME': 215, '(': 215, 'EMBEDEXPR': 215, 'None': 215, 'STRING': 215, 'not': 215, '...': 215, 'NUMBER': 215}, 'small_stmt': {'True': 217, '+': 217, 'break': 217, '{': 217, 'False': 217, 'from': 217, 'assert': 217, 'lambda': 217, 'EMBEDSTAT': 217, 'global': 217, '*': 217, 'pass': 217, 'None': 217, '~': 217, 'STRING': 217, '-': 217, 'nonlocal': 217, 'yield': 217, 'continue': 217, 'return': 217, '[': 217, 'del': 217, 'NAME': 217, '...': 217, '(': 217, 'EMBEDEXPR': 217, 'not': 217, 'raise': 217, 'import': 217, 'NUMBER': 217}, 'testlist_comp__1': {'-': 218, 'True': 218, '+': 218, '~': 218, '{': 218, '*': 219, 'EMBEDEXPR': 218, 'lambda': 218, '[': 218, 'NAME': 218, '(': 218, 'False': 218, 'None': 218, 'NUMBER': 218, 'not': 218, '...': 218, 'STRING': 218}, 'comp_for__1': {')': 160, ',': 160, ']': 160, 'if': 159, 'for': 159, '}': 160}, 'decorated': {'@': 222}, 'trailer__1': {'-': 223, 'True': 223, '+': 223, '(': 223, '{': 223, 'EMBEDEXPR': 223, '~': 223, '**': 223, 'lambda': 223, '[': 223, ')': 224, 'NAME': 223, '*': 223, 'False': 223, 'None': 223, 'NUMBER': 223, 'not': 223, '...': 223, 'STRING': 223}, 'file_input': {'True': 225, '+': 225, 'break': 225, '?>': 225, '{': 225, 'EMBEDEXPR': 225, 'from': 225, 'assert': 225, 'lambda': 225, 'EMBEDSTAT': 225, 'global': 225, 'yield': 225, 'NEWLINE': 225, '*': 225, 'while': 225, 'pass': 225, 'class': 225, 'if': 225, '~': 225, '@': 225, '-': 225, 'nonlocal': 225, 'def': 225, 'for': 225, 'try': 225, 'with': 225, 'continue': 225, 'return': 225, '[': 225, 'del': 225, '}': 225, 'NAME': 225, 'import': 225, '(': 225, 'False': 225, 'None': 225, 'not': 225, 'STRING': 225, 'raise': 225, '...': 225, 'NUMBER': 225}, 'dictorsetmaker__5': {'-': 135, 'True': 135, '+': 135, '~': 135, '{': 135, 'EMBEDEXPR': 135, 'lambda': 135, '[': 135, '}': 136, 'NAME': 135, '(': 135, 'False': 135, 'None': 135, 'NUMBER': 135, 'not': 135, '...': 135, 'STRING': 135}, 'typedargslist_item': {'NAME': 227, '*': 228, '**': 229}, 'dictorsetmaker__7': {'-': 230, 'True': 230, '+': 230, '~': 230, '{': 230, 'EMBEDEXPR': 230, 'lambda': 230, '[': 230, '}': 231, 'NAME': 230, '(': 230, 'False': 230, 'None': 230, 'NUMBER': 230, 'not': 230, '...': 230, 'STRING': 230}, 'or_test__1': {',': 273, ';': 273, 'if': 273, 'from': 273, '%=': 273, 'NEWLINE': 273, '+=': 273, '<<=': 273, '|=': 273, ']': 273, '-=': 273, '*=': 273, 'for': 273, '&=': 273, ':': 273, 'or': 272, '/=': 273, '=': 273, '}': 273, ')': 273, '^=': 273, '//=': 273, 'else': 273, '**=': 273, '>>=': 273, 'as': 273}, 'compound_stmt__1': {'while': 375, 'def': 379, 'for': 376, 'if': 374, 'class': 380, 'try': 377, 'with': 378, '@': 381}, 'term__1': {'|': 235, 'in': 235, '^': 235, '&': 235, '>>': 235, '<<=': 235, '-': 235, 'and': 235, '|=': 235, 'is': 235, 'or': 235, '/=': 235, '=': 235, '//': 234, '}': 235, '<': 235, '^=': 235, ',': 235, 'if': 235, '>=': 235, '<<': 235, 'not': 235, 'as': 235, '+': 235, '==': 235, '%': 234, 'from': 235, '%=': 235, '!=': 235, 'NEWLINE': 235, '&=': 235, ']': 235, '*': 234, '+=': 235, '*=': 235, '//=': 235, ':': 235, 'for': 235, '/': 234, '>': 235, '-=': 235, ')': 235, ';': 235, '**=': 235, '>>=': 235, 'else': 235, '<=': 235}, 'with_item': {'-': 236, 'True': 236, '+': 236, '~': 236, '{': 236, 'EMBEDEXPR': 236, 'lambda': 236, '[': 236, 'NAME': 236, '(': 236, 'False': 236, 'None': 236, 'NUMBER': 236, 'not': 236, '...': 236, 'STRING': 236}, 'import_as_name': {'NAME': 232}, 'while_stmt': {'while': 239}, 'funcdef': {'def': 240}, 'arglist_item': {'-': 241, 'True': 241, '+': 241, '~': 241, '{': 241, '*': 242, 'EMBEDEXPR': 241, '**': 243, 'lambda': 241, '[': 241, 'NAME': 241, '(': 241, 'False': 241, 'None': 241, 'NUMBER': 241, 'not': 241, '...': 241, 'STRING': 241}, 'dictorsetmaker__6': {',': 244, '}': 245}, 'lambdef__1': {'NAME': 360, ':': 361, '*': 360, '**': 360}, 'subscriptlist__3': {'-': 301, 'True': 301, '+': 301, ':': 301, '{': 301, 'EMBEDEXPR': 301, '~': 301, 'lambda': 301, '[': 301, 'NAME': 301, ']': 302, '(': 301, 'False': 301, 'None': 301, 'NUMBER': 301, 'not': 301, '...': 301, 'STRING': 301}, 'subscriptlist__2': {',': 5, ']': 6}, 'import_as_names__2': {')': 250, ',': 249, 'NEWLINE': 250, ';': 250}, 'decorated__1': {'class': 251, 'def': 252}, 'return_stmt__1': {'-': 253, 'True': 253, '+': 253, '~': 253, ';': 254, '{': 253, 'EMBEDEXPR': 253, 'lambda': 253, '[': 253, 'NAME': 253, 'NEWLINE': 254, '(': 253, 'False': 253, 'None': 253, 'NUMBER': 253, 'not': 253, '...': 253, 'STRING': 253}, 'testlist_comp__4': {'-': 255, 'True': 255, '+': 255, '(': 255, '{': 255, 'EMBEDEXPR': 255, '~': 255, 'lambda': 255, '[': 255, ')': 256, 'NAME': 255, ']': 256, '*': 255, 'False': 255, 'None': 255, 'NUMBER': 255, 'not': 255, '...': 255, 'STRING': 255}, 'suite': {'True': 133, '+': 133, 'break': 133, '{': 133, 'False': 133, 'from': 133, 'assert': 133, 'lambda': 133, 'EMBEDSTAT': 133, 'global': 133, 'NEWLINE': 134, '*': 133, 'pass': 133, 'None': 133, '~': 133, 'STRING': 133, '-': 133, 'nonlocal': 133, 'yield': 133, 'continue': 133, 'return': 133, '[': 133, 'del': 133, 'NAME': 133, '...': 133, '(': 133, 'EMBEDEXPR': 133, 'not': 133, 'raise': 133, 'import': 133, 'NUMBER': 133}, 'dictorsetmaker__1': {'for': 259, ',': 260, ':': 258, '}': 260}, 'flow_stmt__1': {'raise': 264, 'break': 261, 'continue': 262, 'return': 263, 'yield': 265}, 'and_test__1': {',': 267, ';': 267, 'if': 267, 'from': 267, '%=': 267, 'NEWLINE': 267, '+=': 267, '|=': 267, ']': 267, '-=': 267, '*=': 267, 'for': 267, 'and': 266, '&=': 267, ':': 267, 'or': 267, '/=': 267, '=': 267, '^=': 267, '<<=': 267, ')': 267, '}': 267, '//=': 267, 'else': 267, '**=': 267, '>>=': 267, 'as': 267}, 'if_stmt__2': {'True': 269, '+': 269, 'break': 269, '?>': 269, 'DEDENT': 269, '{': 269, 'False': 269, 'from': 269, 'assert': 269, 'lambda': 269, 'EMBEDSTAT': 269, 'global': 269, 'yield': 269, 'NEWLINE': 269, '*': 269, 'while': 269, 'pass': 269, 'class': 269, 'if': 269, '~': 269, '@': 269, '-': 269, 'nonlocal': 269, 'def': 269, 'for': 269, 'try': 269, 'with': 269, 'continue': 269, 'return': 269, '[': 269, 'del': 269, '}': 269, 'NAME': 269, 'import': 269, '...': 269, '(': 269, 'EMBEDEXPR': 269, 'None': 269, 'not': 269, 'STRING': 269, 'raise': 269, 'else': 268, 'NUMBER': 269}, 'comp_op__1': {'-': 359, 'True': 359, '+': 359, '~': 359, '{': 359, 'False': 359, '[': 359, 'NAME': 359, '(': 359, 'EMBEDEXPR': 359, 'not': 358, 'STRING': 359, 'None': 359, '...': 359, 'NUMBER': 359}, 'lambdef_nocond__1': {'NAME': 270, ':': 271, '*': 270, '**': 270}, 'testlist_comp': {'-': 274, 'True': 274, '+': 274, '(': 274, '{': 274, 'EMBEDEXPR': 274, '~': 274, 'lambda': 274, '[': 274, 'NAME': 274, '*': 274, 'False': 274, 'None': 274, 'NUMBER': 274, 'not': 274, '...': 274, 'STRING': 274}, 'arglist__2': {')': 305, ',': 304}, 'atom__2': {'-': 277, 'True': 277, '+': 277, '(': 277, '{': 277, 'EMBEDEXPR': 277, '~': 277, 'lambda': 277, '[': 277, 'NAME': 277, ']': 278, '*': 277, 'False': 277, 'None': 277, 'NUMBER': 277, 'not': 277, '...': 277, 'STRING': 277}, 'subscript__1': {'-': 279, 'True': 279, '+': 279, '~': 279, '{': 279, 'EMBEDEXPR': 279, 'lambda': 279, '[': 279, 'NAME': 279, ':': 280, '(': 279, 'False': 279, 'None': 279, 'NUMBER': 279, 'not': 279, '...': 279, 'STRING': 279}, 'with_stmt__1': {',': 112, ':': 113}, 'and_test': {'-': 282, 'True': 282, '+': 282, '~': 282, '{': 282, 'False': 282, '[': 282, 'NAME': 282, '(': 282, 'EMBEDEXPR': 282, 'None': 282, 'STRING': 282, 'not': 282, '...': 282, 'NUMBER': 282}, 'exprlist__3': {'in': 34, ',': 33, 'NEWLINE': 34, ';': 34}, 'parameters__1': {')': 285, 'NAME': 284, '*': 284, '**': 284}, 'arglist__1': {'-': 286, 'True': 286, '+': 286, '(': 286, '{': 286, 'EMBEDEXPR': 286, '~': 286, '**': 286, 'lambda': 286, '[': 286, 'NAME': 286, '*': 286, 'False': 286, 'None': 286, 'NUMBER': 286, 'not': 286, '...': 286, 'STRING': 286}, 'testlist': {'-': 139, 'True': 139, '+': 139, '~': 139, '{': 139, 'EMBEDEXPR': 139, 'lambda': 139, '[': 139, 'NAME': 139, '(': 139, 'False': 139, 'None': 139, 'NUMBER': 139, 'not': 139, '...': 139, 'STRING': 139}, 'decorator__2': {'-': 288, 'True': 288, '+': 288, '(': 288, '{': 288, 'EMBEDEXPR': 288, '~': 288, '**': 288, 'lambda': 288, '[': 288, ')': 289, 'NAME': 288, '*': 288, 'False': 288, 'None': 288, 'NUMBER': 288, 'not': 288, '...': 288, 'STRING': 288}, 'varargslist_item': {'NAME': 290, '*': 291, '**': 292}, 'funcdef__1': {':': 294, '->': 293}, 'break_stmt': {'break': 295}, 'import_as_names': {'NAME': 296}, 'comparison__1': {',': 298, '==': 297, 'if': 298, 'from': 298, '%=': 298, '!=': 297, 'NEWLINE': 298, '<<=': 298, '|=': 298, ']': 298, '+=': 298, '*=': 298, '=': 298, 'for': 298, 'and': 298, '&=': 298, ':': 298, ';': 298, ')': 298, 'is': 297, 'or': 298, '/=': 298, '>': 297, '>>=': 298, '}': 298, '<': 297, '^=': 298, '//=': 298, 'else': 298, 'in': 297, '>=': 297, '**=': 298, 'not': 297, '-=': 298, 'as': 298, '<=': 297}, 'except_clause__1': {'-': 371, 'True': 371, '+': 371, '~': 371, '{': 371, 'EMBEDEXPR': 371, 'lambda': 371, '[': 371, 'NAME': 371, ':': 372, '(': 371, 'False': 371, 'None': 371, 'NUMBER': 371, 'not': 371, '...': 371, 'STRING': 371}, 'try_stmt__2': {'True': 300, 'break': 300, 'if': 300, 'assert': 300, 'global': 300, 'pass': 300, 'lambda': 300, 'while': 300, 'None': 300, '-': 300, 'def': 300, 'not': 300, 'with': 300, 'continue': 300, 'return': 300, '[': 300, 'del': 300, '(': 300, 'EMBEDEXPR': 300, '~': 300, 'STRING': 300, '...': 300, 'NUMBER': 300, 'except': 299, '+': 300, '{': 300, 'from': 300, 'DEDENT': 300, 'NEWLINE': 300, 'EMBEDSTAT': 300, '*': 300, 'class': 300, '}': 300, '@': 300, 'nonlocal': 300, 'yield': 300, 'try': 300, 'False': 300, 'NAME': 300, 'else': 300, '?>': 300, 'for': 300, 'finally': 300, 'raise': 300, 'import': 300}, 'comp_if': {'if': 246}, 'expr': {'-': 303, 'True': 303, '+': 303, '{': 303, 'False': 303, '[': 303, 'NAME': 303, '(': 303, 'EMBEDEXPR': 303, 'None': 303, 'STRING': 303, '~': 303, '...': 303, 'NUMBER': 303}, 'raise_stmt': {'raise': 306}, 'arith_expr': {'-': 307, 'True': 307, '+': 307, '{': 307, 'False': 307, '[': 307, 'NAME': 307, '(': 307, 'EMBEDEXPR': 307, 'None': 307, 'STRING': 307, '~': 307, '...': 307, 'NUMBER': 307}, 'subscript__4': {',': 155, ']': 155, ':': 154}, 'subscript': {'-': 281, 'True': 281, '+': 281, ':': 281, '{': 281, 'EMBEDEXPR': 281, '~': 281, 'lambda': 281, '[': 281, 'NAME': 281, '(': 281, 'False': 281, 'None': 281, 'NUMBER': 281, 'not': 281, '...': 281, 'STRING': 281}, 'test__1': {';': 311, 'if': 310, 'from': 311, '%=': 311, 'NEWLINE': 311, '+=': 311, '<<=': 311, '|=': 311, ']': 311, '-=': 311, '*=': 311, '&=': 311, ':': 311, '**=': 311, '/=': 311, '=': 311, '}': 311, ')': 311, '^=': 311, ',': 311, 'for': 311, '//=': 311, '>>=': 311, 'as': 311}, 'yield_expr': {'yield': 386}, 'lambdef': {'lambda': 312}, 'yield_expr__1': {'-': 314, 'True': 314, '+': 314, '~': 314, ';': 315, '{': 314, 'False': 314, 'from': 313, '=': 315, 'lambda': 314, '[': 314, ')': 315, 'NAME': 314, 'NEWLINE': 315, '(': 314, 'EMBEDEXPR': 314, 'None': 314, 'NUMBER': 314, 'not': 314, '...': 314, 'STRING': 314}, 'testlist_star_expr__3': {'True': 127, '+': 127, ';': 128, '{': 127, 'EMBEDEXPR': 127, '%=': 128, 'lambda': 127, 'NEWLINE': 128, '<<=': 128, '&=': 128, '*': 127, '//=': 128, 'None': 127, '+=': 128, 'not': 127, 'STRING': 127, '-': 127, '|=': 128, '*=': 128, '**=': 128, '/=': 128, '=': 128, '>>=': 128, '[': 127, '^=': 128, 'NAME': 127, '(': 127, 'False': 127, '~': 127, '-=': 128, '...': 127, 'NUMBER': 127}, 'assert_stmt': {'assert': 316}, 'tfpdef': {'NAME': 317}, 'import_as_names__1': {'NAME': 318}, 'testlist_star_expr': {'-': 319, 'True': 319, '+': 319, '(': 319, '{': 319, 'EMBEDEXPR': 319, '~': 319, 'lambda': 319, '[': 319, 'NAME': 319, '*': 319, 'False': 319, 'None': 319, 'NUMBER': 319, 'not': 319, '...': 319, 'STRING': 319}, 'eval_input': {'-': 387, 'True': 387, '+': 387, '~': 387, '{': 387, 'EMBEDEXPR': 387, 'lambda': 387, '[': 387, 'NAME': 387, '(': 387, 'False': 387, 'None': 387, 'NUMBER': 387, 'not': 387, '...': 387, 'STRING': 387}, 'parameters': {'(': 321}, 'lambdef_nocond': {'lambda': 322}, 'small_stmt__1': {'True': 324, '+': 324, 'break': 327, '{': 324, 'False': 324, 'from': 328, 'lambda': 324, 'assert': 331, 'EMBEDSTAT': 323, 'global': 329, '*': 324, 'pass': 326, 'None': 324, 'not': 324, 'STRING': 324, '-': 324, 'nonlocal': 330, 'yield': 327, 'continue': 327, 'return': 327, '[': 324, 'del': 325, 'NAME': 324, 'import': 328, '(': 324, 'EMBEDEXPR': 324, '~': 324, 'raise': 327, '...': 324, 'NUMBER': 324}, 'try_stmt__3': {'True': 248, 'break': 248, 'False': 248, 'assert': 248, 'global': 248, 'pass': 248, 'while': 248, 'None': 248, '-': 248, 'def': 248, 'not': 248, 'with': 248, 'continue': 248, 'return': 248, '[': 248, 'del': 248, '}': 248, '(': 248, 'if': 248, 'NEWLINE': 248, 'STRING': 248, '...': 248, 'NUMBER': 248, '+': 248, '{': 248, 'from': 248, 'DEDENT': 248, 'lambda': 248, 'EMBEDSTAT': 248, '*': 248, 'class': 248, '~': 248, '@': 248, 'EMBEDEXPR': 248, 'nonlocal': 248, 'yield': 248, 'try': 248, 'NAME': 248, 'import': 248, '?>': 248, 'for': 248, 'finally': 248, 'raise': 248, 'else': 247}, 'try_stmt__4': {'True': 336, '+': 336, 'break': 336, '?>': 336, 'DEDENT': 336, '{': 336, 'False': 336, 'from': 336, 'lambda': 336, 'assert': 336, 'EMBEDSTAT': 336, 'global': 336, 'yield': 336, 'NEWLINE': 336, '*': 336, 'while': 336, 'pass': 336, 'class': 336, 'if': 336, '~': 336, '@': 336, '-': 336, 'nonlocal': 336, 'def': 336, 'for': 336, 'try': 336, 'with': 336, 'continue': 336, 'return': 336, '[': 336, 'del': 336, '}': 336, 'NAME': 336, 'import': 336, '(': 336, 'EMBEDEXPR': 336, 'None': 336, 'finally': 335, 'not': 336, 'STRING': 336, 'raise': 336, '...': 336, 'NUMBER': 336}, 'decorator__1': {'NEWLINE': 37, '(': 36}, 'subscript__6': {',': 41, ']': 41, ':': 40}, 'dotted_name__1': {'.': 340, ',': 341, 'import': 341, '(': 341, ';': 341, 'NEWLINE': 341, 'as': 341}, 'simple_stmt__3': {'True': 342, '+': 342, 'break': 342, '{': 342, 'False': 342, 'from': 342, 'assert': 342, 'lambda': 342, 'EMBEDSTAT': 342, 'global': 342, 'NEWLINE': 343, '*': 342, 'pass': 342, 'None': 342, '~': 342, 'STRING': 342, '-': 342, 'nonlocal': 342, 'yield': 342, 'continue': 342, 'return': 342, '[': 342, 'del': 342, 'NAME': 342, '...': 342, '(': 342, 'EMBEDEXPR': 342, 'not': 342, 'raise': 342, 'import': 342, 'NUMBER': 342}, 'simple_stmt': {'True': 344, '+': 344, 'break': 344, '{': 344, 'False': 344, 'from': 344, 'assert': 344, 'lambda': 344, 'EMBEDSTAT': 344, 'global': 344, '*': 344, 'pass': 344, 'None': 344, '~': 344, 'STRING': 344, '-': 344, 'nonlocal': 344, 'yield': 344, 'continue': 344, 'return': 344, '[': 344, 'del': 344, 'NAME': 344, '...': 344, '(': 344, 'EMBEDEXPR': 344, 'not': 344, 'raise': 344, 'import': 344, 'NUMBER': 344}, 'typedargslist__2': {')': 346, ',': 345}, 'arith_expr__2': {'-': 348, '+': 347}, 'import_as_name__1': {')': 390, ',': 390, 'NEWLINE': 390, 'as': 389, ';': 390}, 'with_stmt': {'with': 350}, 'expr_stmt__3': {';': 352, '=': 351, 'NEWLINE': 352}, 'import_from': {'from': 353}, 'dotted_as_names': {'NAME': 354}, 'stmt__1': {'class': 144, 'True': 143, '+': 143, 'break': 143, '{': 143, 'False': 143, 'from': 143, 'assert': 143, 'lambda': 143, 'EMBEDSTAT': 143, 'global': 143, '*': 143, 'while': 144, 'pass': 143, 'None': 143, 'if': 144, '~': 143, 'STRING': 143, '-': 143, 'nonlocal': 143, 'def': 144, 'yield': 143, 'try': 144, 'with': 144, 'continue': 143, 'return': 143, '[': 143, 'del': 143, 'for': 144, 'NAME': 143, '...': 143, '(': 143, 'EMBEDEXPR': 143, 'not': 143, '@': 144, 'raise': 143, 'import': 143, 'NUMBER': 143}, 'power': {'True': 339, 'NAME': 339, 'STRING': 339, '(': 339, '{': 339, 'False': 339, 'None': 339, '[': 339, 'EMBEDEXPR': 339, '...': 339, 'NUMBER': 339}, 'star_expr': {'*': 257}, 'varargslist__1': {'NAME': 362, '*': 362, '**': 362}, 'global_stmt': {'global': 363}, 'embed_stmt': {'EMBEDSTAT': 364}, 'comp_if__1': {')': 366, ',': 366, ']': 366, 'if': 365, 'for': 365, '}': 366}, 'subscript__3': {'-': 367, ',': 368, 'True': 367, '+': 367, '~': 367, '{': 367, 'EMBEDEXPR': 367, 'lambda': 367, '[': 367, 'NAME': 367, ']': 368, '(': 367, 'False': 367, ':': 368, 'None': 367, 'NUMBER': 367, 'not': 367, '...': 367, 'STRING': 367}, 'import_as_names__3': {')': 370, 'NAME': 369, 'NEWLINE': 370, ';': 370}, 'expr_stmt': {'-': 226, 'True': 226, '+': 226, '(': 226, '{': 226, 'EMBEDEXPR': 226, '~': 226, 'lambda': 226, '[': 226, 'NAME': 226, '*': 226, 'False': 226, 'None': 226, 'NUMBER': 226, 'not': 226, '...': 226, 'STRING': 226}, 'dictorsetmaker__2': {':': 373}, 'testlist__1': {'-': 233, 'True': 233, '+': 233, '~': 233, '{': 233, 'EMBEDEXPR': 233, 'lambda': 233, '[': 233, 'NAME': 233, '(': 233, 'False': 233, 'None': 233, 'NUMBER': 233, 'not': 233, '...': 233, 'STRING': 233}, 'argument': {'-': 382, 'True': 382, '+': 382, '~': 382, '{': 382, 'EMBEDEXPR': 382, 'lambda': 382, '[': 382, 'NAME': 382, '(': 382, 'False': 382, 'None': 382, 'NUMBER': 382, 'not': 382, '...': 382, 'STRING': 382}, 'sliceop__1': {'-': 333, ',': 334, 'True': 333, '+': 333, '~': 333, '{': 333, 'EMBEDEXPR': 333, 'lambda': 333, '[': 333, 'NAME': 333, ']': 334, '(': 333, 'False': 333, 'None': 333, 'NUMBER': 333, 'not': 333, '...': 333, 'STRING': 333}, 'atom__4': {'|': 309, 'in': 309, '^': 309, '&': 309, '>>': 309, '<<=': 309, '-': 309, 'and': 309, '|=': 309, 'is': 309, 'or': 309, '/=': 309, '=': 309, '//': 309, '[': 309, '<': 309, '^=': 309, ',': 309, '(': 309, 'if': 309, '>=': 309, '<<': 309, '%': 309, 'as': 309, '.': 309, '+': 309, '==': 309, 'from': 309, '%=': 309, '!=': 309, 'not': 309, 'NEWLINE': 309, '&=': 309, ']': 309, '*': 309, '**': 309, '+=': 309, '*=': 309, 'STRING': 308, '//=': 309, ':': 309, 'for': 309, '/': 309, '>': 309, '-=': 309, ')': 309, ';': 309, '**=': 309, '}': 309, '>>=': 309, 'else': 309, '<=': 309}, 'classdef': {'class': 320}, 'varargslist': {'NAME': 388, '*': 388, '**': 388}, 'vfpdef': {'NAME': 349}, 'nonlocal_stmt__1': {',': 391, 'NEWLINE': 392, ';': 392}, 'suite__1': {'True': 393, '+': 393, 'break': 393, 'DEDENT': 394, '{': 393, 'EMBEDEXPR': 393, 'from': 393, 'assert': 393, 'lambda': 393, 'EMBEDSTAT': 393, 'global': 393, 'yield': 393, '*': 393, 'while': 393, 'pass': 393, 'class': 393, 'if': 393, '~': 393, '@': 393, '-': 393, 'nonlocal': 393, 'def': 393, 'for': 393, 'try': 393, 'with': 393, 'continue': 393, 'return': 393, '[': 393, 'del': 393, 'NAME': 393, 'import': 393, '(': 393, 'False': 393, 'None': 393, 'not': 393, 'STRING': 393, 'raise': 393, '...': 393, 'NUMBER': 393}, 'arith_expr__1': {'+': 395, '==': 396, '|': 396, 'in': 396, 'from': 396, '%=': 396, '!=': 396, 'NEWLINE': 396, '^': 396, ']': 396, '|=': 396, '>>': 396, '&': 396, '//=': 396, 'or': 396, '+=': 396, '<<=': 396, 'for': 396, '-': 395, 'and': 396, '&=': 396, ':': 396, ';': 396, ')': 396, 'is': 396, '**=': 396, '/=': 396, '=': 396, '>': 396, '>>=': 396, '*=': 396, '<': 396, '^=': 396, ',': 396, 'else': 396, 'if': 396, '>=': 396, '<<': 396, 'not': 396, '<=': 396, '-=': 396, 'as': 396, '}': 396}}
