
from context import flatten
from toypython.lex import tokenizer
from toypython.ast import *

def set_context(expr, context):
    if isinstance(expr, (Attribute, Subscript, Name)):
        return expr._replace(ctx=context)

    if isinstance(expr, Starred):
        return Starred(expr.location, set_context(expr.value, context), context)

    if isinstance(expr, List) or (isinstance(expr, Tuple) and expr.elts):
        new_elts = [set_context(e, context) for e in expr.elts]
        return expr.__class__(expr.location, new_elts, context)

    print(expr)
    name = expr.__class__.__name__
    if context == Del():
        raise ParseError(expr.location, "can't delete %s" % name)
    else:
        raise ParseError(expr.location, "can't assign to %s" % name)

def ast_for_testlist(tests, commas):
    return tests[0] if not commas else Tuple(tests[0].location, tests, Load())

def ast_for_arguments(items):
    posargs = []
    vararg = None
    after_vararg = False
    kwonlyargs = []
    kwdefaults = []
    kwarg = None
    posdefaults = []

    if items[-1][0] and items[-1][0].type == '**':
        kwarg = items[-1][1]
        items = items[:-1]

    if items[-1][0] and items[-1][0].type == '*' and not items[-1][1]:
        raise ParseError(items[-1][0].location, "named arguments must follow bare *")

    for (star_token, fpdef, default) in items:
        if star_token and star_token.type == '**':
            raise ParseError(star_token.location, "**kwargs must be last in arguments list")

        if star_token and after_vararg:
            raise ParseError(star_token.location, "*args can only be given once")

        if star_token:
            after_vararg = True
            vararg = fpdef
        elif after_vararg:
            kwonlyargs.append(fpdef)
            kwdefaults.append(default)
        else:
            if default:
                posdefaults.append(default)
            elif posdefaults:
                raise ParseError(fpdef.location, "non-default argument follows default argument")
            posargs.append(fpdef)

    return arguments(posargs, vararg, kwonlyargs, kwdefaults, kwarg, posdefaults)

def ast_for_dotted_name(name_tokens):
    rv = None
    for name_token in name_tokens:
        if rv:
            rv = Attribute(rv.location, rv, name_token.value, Load())
        else:
            rv = Name(name_token.location, name_token.value, Load())
    return rv

def ast_for_decorator(loc, name_expr, have_parens, arglist):
    if not have_parens: return name_expr
    return ast_for_call(loc, name_expr, arglist)

def ast_for_comprehension(root_comp_for):
    comps = []

    for item in flatten(root_comp_for):
        if item[0].type == 'for':
            token, (exprlist, exprlist_has_comma), iter = item
            if exprlist_has_comma:
                target = Tuple(exprlist[0].location, exprlist, Store())
            else:
                target = exprlist[0]
            target = set_context(target, Store())
            comps.append(comprehension(target, iter, []))
        else:
            token, cond = item
            comps[-1].ifs.append(cond)

    return comps

token_type_to_operator_class = {
    '|': BitOr,
    '^': BitXor,
    '&': BitAnd,
    '<<': LShift,
    '>>': RShift,
    '+': Add,
    '-': Sub,
    '*': Mult,
    '/': Div,
    '//': FloorDiv,
    '%': Mod,
}

def ast_for_binop(symbols):
    symbols = iter(symbols)
    left = next(symbols)
    while True:
        try:
            op_token = next(symbols)
            op_class = token_type_to_operator_class[op_token.type]
            right = next(symbols)
            left = BinOp(op_token.location, left, op_class(), right)
        except StopIteration:
            return left

def ast_for_power(atom, trailers, factor):
    for trailer in trailers:
        atom = trailer(atom)._replace(location=atom.location)
    if factor:
        atom = BinOp(atom.location, atom, Pow(), factor)
    return atom

def ast_for_call(loc, func, arglist):
    if not arglist: return Call(loc, func, None, None, None, None)

    args = []
    keywords = []
    vararg = None
    kwarg = None
    keyword_names = set()

    star_token, test, root_comp_for, kwvalue = arglist[0]
    if len(arglist) == 1 and root_comp_for:
        value = GeneratorExp(test.location, test, ast_for_comprehension(root_comp_for))
        arglist = [(None, value, None, None)]

    for (star_token, test, root_comp_for, kwvalue) in arglist:
        if root_comp_for:
            raise ParseError(test, "Generator expression must be parenthesized "
                             "if not sole argument")

    if arglist[-1][0] and arglist[-1][0].type == '**':
        kwarg = arglist[-1][1]
        arglist = arglist[:-1]

    for (star_token, test, root_comp_for, kwvalue) in arglist:
        if star_token and star_token.type == '*':
            if vararg:
                raise ParseError(star_token.location, "*expression can only be given once")
            vararg = test
        elif star_token and star_token.type == '**':
            raise ParseError(star_token.location, "**expression must be last in arguments list")
        elif not kwvalue:
            if keywords:
                raise ParseError(test.location, "non-keyword arg after keyword arg")
            if vararg:
                raise ParseError(test.location, "only named arguments may follow *expression")
            args.append(test)
        elif isinstance(test, Lambda):
            raise ParseError(test.location, "lambda cannot contain assignment")
        elif not isinstance(test, Name):
            raise ParseError(test.location, "keyword can't be an expression")
        elif test.id in keyword_names:
            raise ParseError(test.location, "keyword argument repeated")
        else:
            keyword_names.add(test.id)
            keywords.append(keyword(test.id, kwvalue))

    return Call(loc, func, args, keywords, vararg, kwarg)

def ast_for_expr_stmt(base, augassign_op, augassign_value, assignments):
    if augassign_op:
        base = set_context(base, Store())
        if not isinstance(base, (Name, Attribute, Subscript)):
            raise ParseError(base.location, "illegal expression for augmented assignment")
        return AugAssign(base.location, base, augassign_op, augassign_value)

    if assignments:
        value = assignments[-1]
        targets = []
        for target in [base] + assignments[:-1]:
            if isinstance(target, (Yield, YieldFrom)):
                raise ParseError(target.location, "assignment to yield expression not possible")
            targets.append(set_context(target, Store()))
        return Assign(base.location, targets, value)

    return Expr(base.location, base)

def ast_for_if_stmt(tokens, conds, suites, else_suite):
    for token, cond, suite in reversed(list(zip(tokens, conds, suites))):
        else_suite = [If(token.location, cond, suite, else_suite or [])]
    return else_suite

def ast_for_classdef(loc, name, arglist, body):
    dummy_call = ast_for_call(loc, None, arglist)
    return ClassDef(loc, name, dummy_call.args, dummy_call.keywords,
                    dummy_call.starargs, dummy_call.kwargs, body, [])



from context import combine_action

from context import ParseError

def parse(ctx, start_nt=None, close_with=None):
    stack = [close_with or 'EOF', start_nt or 'file_input']
    results = []
    tok_iter = tokenizer(ctx, close_with)
    token = next(tok_iter)

    while True:
        stack_top = stack[-1]

        if isinstance(stack_top, int):
            rule, action = RULES[stack_top]
            stack.pop()
            n = len(rule)
            args = results[len(results) - n:]
            del results[len(results) - n:]
            results.append(action(ctx, *args))

        elif stack_top not in TABLE:
            if token.type != stack_top:
                raise ParseError(token.location, "Expected " + stack_top)
            stack.pop()
            if not stack: break
            results.append(token)
            token = next(tok_iter)

        else:
            row = TABLE[stack_top]
            if token.type not in row:
                raise ParseError(token.location, "Unexpected {}, expected {}".format(
                    repr(token.type), ', '.join(sorted(repr(k) for k in row))))
            rule_num = row[token.type]
            rule, action = RULES[rule_num]
            stack.pop()
            stack.append(rule_num)
            stack.extend(reversed(rule))

    return results[0]

RULES = [(('expr', 'comparison__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda _ctx, r, _all, op, _loc, l: (
l if not op else Compare(l.location, l, op, r)
), normal_vars=['l'], list_vars=['op', 'r']))
), (('flow_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _ctx, _all, _loc: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('testlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda t, _ctx, _all, _loc, c: (
ast_for_testlist(t, c)
), normal_vars=[], list_vars=['c', 't']))
), (('yield', 'yield_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda f, t, _ctx, _all, _loc: (
YieldFrom(_loc, f) if f else Yield(_loc, t)
), normal_vars=['f', 't'], list_vars=[]))
), (('testlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('test', 'subscript__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), ((':', 'subscript__5', 'subscript__6'), 
(lambda ctx, v1, v2, v3: [('d', v1), v2, v3])
), (('import_name',), 
(combine_action(root=lambda ctx, v1: [('i', v1)], user=lambda _ctx, _all, _loc, i: (
i
), normal_vars=['i'], list_vars=[]))
), (('import_from',), 
(combine_action(root=lambda ctx, v1: [('i', v1)], user=lambda _ctx, _all, _loc, i: (
i
), normal_vars=['i'], list_vars=[]))
), (('from', 'import_from__2', 'import', 'import_from__6'), 
(lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), v4])
), (('vfpdef', 'varargslist_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda _ctx, a, _all, _loc, d: (
(None, a, d)
), normal_vars=['a', 'd'], list_vars=[]))
), (('*', 'varargslist_item__2'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda _ctx, a, s, _all, _loc: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'vfpdef'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _ctx, a, s, _all, _loc: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('and', 'not_test', 'and_test__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('l', v2), v3])
), ((), 
(lambda ctx: [])
), (('tfpdef', 'typedargslist_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda _ctx, a, _all, _loc, d: (
(None, a, d)
), normal_vars=['a', 'd'], list_vars=[]))
), (('*', 'typedargslist_item__2'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda _ctx, a, s, _all, _loc: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'tfpdef'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _ctx, a, s, _all, _loc: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('test',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), (('varargslist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('with', 'with_item', 'with_stmt__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('w', v2), v3, (None, v4), ('b', v5)], user=lambda w, _ctx, b, _all, _loc: (
With(_loc, w, b)
), normal_vars=['b'], list_vars=['w']))
), ((',', 'dotted_as_name', 'dotted_as_names__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('comp_for',), 
(lambda ctx, v1: [('dc', v1)])
), (('dictorsetmaker__4',), 
(lambda ctx, v1: [v1])
), (('comp_for',), 
(lambda ctx, v1: [('c', v1)])
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('v', v2)])
), ((), 
(lambda ctx: [])
), ((';', 'simple_stmt__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('return', 'return_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda t, _ctx, _all, _loc: (
Return(_loc, t)
), normal_vars=['t'], list_vars=[]))
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('for', 'exprlist', 'in', 'testlist', ':', 'suite', 'for_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6, v7: [(None, v1), ('t', v2), (None, v3), ('c', v4), (None, v5), ('s', v6), v7], user=lambda t, _ctx, s, _all, e, _loc, c: (
For(_loc, set_context(Tuple(t[0][0].location, t[0], Store()) if t[1] else t[0][0], Store()), c, s, e)
), normal_vars=['c', 'e', 's', 't'], list_vars=[]))
), (('exprlist__2', 'exprlist__3'), 
(lambda ctx, v1, v2: [v1, v2])
), (('lambda', 'lambdef_nocond__1', ':', 'test_nocond'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), ('t', v4)], user=lambda t, a, _ctx, _all, _loc: (
Lambda(_loc, a or arguments(None, None, None, None, None, None), t)
), normal_vars=['a', 't'], list_vars=[]))
), (('testlist_comp__1', 'testlist_comp__2'), 
(combine_action(root=lambda ctx, v1, v2: [v1, v2], user=lambda f, t, _ctx, _all, _loc, c: (
(t, c, f)
), normal_vars=['f'], list_vars=['c', 't']))
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('def', 'NAME', 'parameters', 'funcdef__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6: [(None, v1), ('n', v2), ('p', v3), v4, (None, v5), ('b', v6)], user=lambda _ctx, b, r, _all, p, n, _loc: (
FunctionDef(_loc, n.value, p, b, [], r)
), normal_vars=['b', 'n', 'p', 'r'], list_vars=[]))
), (('subscriptlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), ((',', 'subscriptlist__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), ((':', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('t', v2)])
), ((), 
(lambda ctx: [])
), (('if', 'test', ':', 'suite', 'if_stmt__1', 'if_stmt__2'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6: [('t', v1), ('c', v2), (None, v3), ('s', v4), v5, v6], user=lambda t, _ctx, s, _all, e, _loc, c: (
ast_for_if_stmt(t, c, s, e)
), normal_vars=['e'], list_vars=['c', 's', 't']))
), (('dotted_as_name', 'dotted_as_names__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _ctx, n, _all, _loc: (
n
), normal_vars=[], list_vars=['n']))
), (('break',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Break(_loc)
), normal_vars=[], list_vars=[]))
), (('(', 'parameters__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda t, _ctx, _all, _loc: (
t or arguments(None, None, None, None, None, None)
), normal_vars=['t'], list_vars=[]))
), (('not', 'not_test'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _ctx, e, _all, _loc: (
UnaryOp(_loc, Not(), e)
), normal_vars=['e'], list_vars=[]))
), (('comparison',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _ctx, e, _all, _loc: (
e
), normal_vars=['e'], list_vars=[]))
), ((',', 'import_as_names__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('xor_expr', 'expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _ctx, _all, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('yield_expr',), 
(lambda ctx, v1: [('av', v1)])
), (('testlist',), 
(lambda ctx, v1: [('av', v1)])
), ((',', 'testlist_comp__4'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('test',), 
(lambda ctx, v1: [('u', v1)])
), ((), 
(lambda ctx: [])
), (('@', 'dotted_name', 'decorator__1', 'NEWLINE'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), ('d', v2), v3, (None, v4)], user=lambda _ctx, b, a, _all, _loc, d: (
ast_for_decorator(_loc, ast_for_dotted_name(d), b, a)
), normal_vars=['a', 'b', 'd'], list_vars=[]))
), (('and_test', 'or_test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda _ctx, _all, _loc, l: (
l[0] if len(l) == 1 else BoolOp(l[0].location, And(), l)
), normal_vars=[], list_vars=['l']))
), (('STRING', 'atom__4'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('subscriptlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _ctx, s, _all, _loc, c: (

    s[0] if not c else
    ExtSlice(s) if any(isinstance(k, Slice) for k in s) else
    Index(Tuple(s[0].value.location, [k.value for k in s], Load()))
), normal_vars=[], list_vars=['c', 's']))
), (('and_expr', 'xor_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _ctx, _all, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('shift_expr__2', 'arith_expr', 'shift_expr__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('simple_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _ctx, s, _all, _loc: (
flatten(s)
), normal_vars=['s'], list_vars=[]))
), (('NEWLINE', 'INDENT', 'stmt', 'suite__1', 'DEDENT'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), (None, v2), ('s', v3), v4, (None, v5)], user=lambda _ctx, s, _all, _loc: (
flatten(s)
), normal_vars=[], list_vars=['s']))
), (('assert', 'test', 'assert_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('a', v2), v3], user=lambda _ctx, a, b, _all, _loc: (
Assert(_loc, a, b)
), normal_vars=['a', 'b'], list_vars=[]))
), (('arglist_item', 'arglist__2'), 
(lambda ctx, v1, v2: [('a', v1), v2])
), (('classdef',), 
(lambda ctx, v1: [('cf', v1)])
), (('funcdef',), 
(lambda ctx, v1: [('cf', v1)])
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('n', v2)])
), ((), 
(lambda ctx: [])
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('d', v2)])
), ((), 
(lambda ctx: [])
), (('testlist_star_expr__1', 'testlist_star_expr__2'), 
(combine_action(root=lambda ctx, v1, v2: [v1, v2], user=lambda t, _ctx, _all, _loc, c: (
ast_for_testlist(t, c)
), normal_vars=[], list_vars=['c', 't']))
), (('test', ':', 'test', 'dictorsetmaker__4'), 
(lambda ctx, v1, v2, v3, v4: [('k', v1), (None, v2), ('v', v3), v4])
), ((), 
(lambda ctx: [])
), (('+=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Add()
), normal_vars=[], list_vars=[]))
), (('-=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Sub()
), normal_vars=[], list_vars=[]))
), (('*=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Mul()
), normal_vars=[], list_vars=[]))
), (('/=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Div()
), normal_vars=[], list_vars=[]))
), (('%=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Mod()
), normal_vars=[], list_vars=[]))
), (('&=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
BitAnd()
), normal_vars=[], list_vars=[]))
), (('|=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
BitOr()
), normal_vars=[], list_vars=[]))
), (('^=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
BitXor()
), normal_vars=[], list_vars=[]))
), (('<<=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
LShift()
), normal_vars=[], list_vars=[]))
), (('>>=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
RShift()
), normal_vars=[], list_vars=[]))
), (('**=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Pow()
), normal_vars=[], list_vars=[]))
), (('//=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
FloorDiv()
), normal_vars=[], list_vars=[]))
), (('factor', 'term__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _ctx, _all, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('augassign', 'expr_stmt__2'), 
(lambda ctx, v1, v2: [('aa', v1), v2])
), (('expr_stmt__3',), 
(lambda ctx, v1: [v1])
), (('(', 'atom__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda y, _ctx, _all, _loc, l: (

      y if y else
      Tuple(_loc, None, Load()) if not l else
      GeneratorExp(_loc, l[0][0], ast_for_comprehension(l[2])) if l[2] else
      ast_for_testlist(l[0], l[1])
), normal_vars=['l', 'y'], list_vars=[]))
), (('[', 'atom__2', ']'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _ctx, _all, _loc, l: (

      List(_loc, None, Load()) if not l else
      ListComp(_loc, l[0][0], ast_for_comprehension(l[2])) if l[2] else
      List(_loc, l[0], Load())
), normal_vars=['l'], list_vars=[]))
), (('{', 'atom__3', '}'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _ctx, _all, _loc, d: (
d._replace(location=_loc) if d else Dict(_loc, None, None)
), normal_vars=['d'], list_vars=[]))
), (('NAME',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _ctx, n, _all, _loc: (
Name(_loc, n.value, Load())
), normal_vars=['n'], list_vars=[]))
), (('NUMBER',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _ctx, n, _all, _loc: (
Num(_loc, n.value)
), normal_vars=['n'], list_vars=[]))
), (('STRING', 'atom__4'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda _ctx, s, _all, _loc: (
Str(_loc, ''.join(t.value for t in s))
), normal_vars=[], list_vars=['s']))
), (('...',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Ellipsis(_loc)
), normal_vars=[], list_vars=[]))
), (('None',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
NameConstant(_loc, None)
), normal_vars=[], list_vars=[]))
), (('True',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
NameConstant(_loc, True)
), normal_vars=[], list_vars=[]))
), (('False',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
NameConstant(_loc, False)
), normal_vars=[], list_vars=[]))
), (('EMBEDEXPR',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _ctx, e, _all, _loc: (
EmbedExp(_loc, e.value)
), normal_vars=['e'], list_vars=[]))
), ((',', 'NAME', 'nonlocal_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('not_test', 'and_test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda _ctx, _all, _loc, l: (
l[0] if len(l) == 1 else BoolOp(l[0].location, Or(), l)
), normal_vars=[], list_vars=['l']))
), (('NEWLINE',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Interactive([])
), normal_vars=[], list_vars=[]))
), (('simple_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _ctx, s, _all, _loc: (
Interactive(flatten(s))
), normal_vars=['s'], list_vars=[]))
), (('compound_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _ctx, s, _all, _loc: (
Interactive(flatten(s))
), normal_vars=['s'], list_vars=[]))
), (('sliceop',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), (('small_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _ctx, _all, _loc: (
_all[0]
), normal_vars=[], list_vars=[]))
), ((',', 'with_item', 'with_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('w', v2), v3])
), ((), 
(lambda ctx: [])
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('a', v2)])
), ((), 
(lambda ctx: [])
), (('or_test',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _ctx, e, _all, _loc: (
e
), normal_vars=['e'], list_vars=[]))
), (('lambdef_nocond',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _ctx, e, _all, _loc: (
e
), normal_vars=['e'], list_vars=[]))
), (('pass',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Pass(_loc)
), normal_vars=[], list_vars=[]))
), (('except_clause', ':', 'suite', 'try_stmt__2', 'try_stmt__3', 'try_stmt__4'), 
(lambda ctx, v1, v2, v3, v4, v5, v6: [('ec', v1), (None, v2), ('es', v3), v4, v5, v6])
), (('finally', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('f', v3)])
), (('NAME', 'dotted_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda t, _ctx, _all, _loc: (
t
), normal_vars=[], list_vars=['t']))
), (('arith_expr__2', 'term', 'arith_expr__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('trailer', 'power__1'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), ((), 
(lambda ctx: [])
), (('as', 'expr'), 
(lambda ctx, v1, v2: [(None, v1), ('e', v2)])
), ((), 
(lambda ctx: [])
), (('typedargslist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _ctx, _all, _loc, l: (
ast_for_arguments(l)
), normal_vars=[], list_vars=['l']))
), (('test', 'argument__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda t, _ctx, _all, v, c, _loc: (
(None, t, c, v)
), normal_vars=['c', 't', 'v'], list_vars=[]))
), (('typedargslist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('argument',), 
(combine_action(root=lambda ctx, v1: [('a', v1)], user=lambda _ctx, a, _all, _loc: (
a
), normal_vars=['a'], list_vars=[]))
), (('*', 'test'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _ctx, a, s, _all, _loc: (
(s, a, None, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'test'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _ctx, a, s, _all, _loc: (
(s, a, None, None)
), normal_vars=['a', 's'], list_vars=[]))
), ((',', 'varargslist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('elif', 'test', ':', 'suite', 'if_stmt__1'), 
(lambda ctx, v1, v2, v3, v4, v5: [('t', v1), ('c', v2), (None, v3), ('s', v4), v5])
), ((), 
(lambda ctx: [])
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('l', v3)])
), ((), 
(lambda ctx: [])
), (('global', 'NAME', 'global_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3], user=lambda _ctx, n, _all, _loc: (
Global(_loc, [t.value for t in n])
), normal_vars=[], list_vars=['n']))
), (('expr',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), (('atom', 'power__1', 'power__2'), 
(combine_action(root=lambda ctx, v1, v2, v3: [('a', v1), v2, v3], user=lambda f, t, a, _ctx, _all, _loc: (
ast_for_power(a, t, f)
), normal_vars=['a', 'f'], list_vars=['t']))
), (('*',), 
(lambda ctx, v1: [('s', v1)])
), (('(', 'import_as_names', ')'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), (None, v3)])
), (('import_as_names',), 
(lambda ctx, v1: [('n', v1)])
), (('lambda', 'lambdef__1', ':', 'test'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), ('t', v4)], user=lambda t, a, _ctx, _all, _loc: (
Lambda(_loc, a or arguments(None, None, None, None, None, None), t)
), normal_vars=['a', 't'], list_vars=[]))
), (('comp_iter',), 
(lambda ctx, v1: [('r', v1)])
), ((), 
(lambda ctx: [])
), (('comp_iter',), 
(lambda ctx, v1: [('r', v1)])
), ((), 
(lambda ctx: [])
), (('class', 'NAME', 'classdef__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('n', v2), v3, (None, v4), ('b', v5)], user=lambda _ctx, a, b, _all, n, _loc: (
ast_for_classdef(_loc, n.value, a, b)
), normal_vars=['a', 'b', 'n'], list_vars=[]))
), (('exprlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), ((':', 'test', 'dictorsetmaker__3'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('v1', v2), v3])
), (('test',), 
(lambda ctx, v1: [('u', v1)])
), ((), 
(lambda ctx: [])
), ((':', 'subscript__3', 'subscript__4'), 
(lambda ctx, v1, v2, v3: [('d', v1), v2, v3])
), ((), 
(lambda ctx: [])
), ((':', 'sliceop__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _ctx, s, _all, _loc: (
s
), normal_vars=['s'], list_vars=[]))
), (('or', 'and_test', 'or_test__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('l', v2), v3])
), ((), 
(lambda ctx: [])
), (('or_test', 'test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda _ctx, a, b, _all, _loc, c: (
IfExp(a.location, b, a, c) if b else a
), normal_vars=['a', 'b', 'c'], list_vars=[]))
), (('lambdef',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _ctx, e, _all, _loc: (
e
), normal_vars=['e'], list_vars=[]))
), (('except_clause', ':', 'suite', 'try_stmt__2'), 
(lambda ctx, v1, v2, v3, v4: [('ec', v1), (None, v2), ('es', v3), v4])
), ((), 
(lambda ctx: [])
), (('varargslist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _ctx, _all, _loc, l: (
ast_for_arguments(l)
), normal_vars=[], list_vars=['l']))
), (('varargslist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('from', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('f', v2)])
), (('testlist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('.', 'NAME', 'dotted_name__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('t', v2), v3])
), ((), 
(lambda ctx: [])
), (('subscript', 'subscriptlist__2'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), (('subscript__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda l, _ctx, s, _all, u, _loc, d: (
Slice(l, u, s) if d else Index(l)
), normal_vars=['d', 'l', 's', 'u'], list_vars=[]))
), (('NAME', 'tfpdef__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda t, _ctx, n, _all, _loc: (
arg(_loc, n.value, t)
), normal_vars=['n', 't'], list_vars=[]))
), (('test', 'with_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda t, e, _ctx, _all, _loc: (
withitem(t, set_context(e, Store()) if e else None)
), normal_vars=['e', 't'], list_vars=[]))
), (('simple_stmt__1', 'NEWLINE'), 
(combine_action(root=lambda ctx, v1, v2: [v1, (None, v2)], user=lambda _ctx, s, _all, _loc: (
s
), normal_vars=[], list_vars=['s']))
), (('testlist_comp__1', 'testlist_comp__3'), 
(lambda ctx, v1, v2: [v1, v2])
), ((), 
(lambda ctx: [])
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('|', 'xor_expr', 'expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('(', 'trailer__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _ctx, a, _all, _loc: (
lambda left_expr: ast_for_call(_loc, left_expr, a)
), normal_vars=['a'], list_vars=[]))
), (('[', 'subscriptlist', ']'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('s', v2), (None, v3)], user=lambda _ctx, s, _all, _loc: (
lambda left_expr: Subscript(_loc, left_expr, s, Load())
), normal_vars=['s'], list_vars=[]))
), (('.', 'NAME'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('n', v2)], user=lambda _ctx, n, _all, _loc: (
lambda left_expr: Attribute(_loc, left_expr, n.value, Load())
), normal_vars=['n'], list_vars=[]))
), (('yield_expr',), 
(lambda ctx, v1: [('v', v1)])
), (('testlist_star_expr',), 
(lambda ctx, v1: [('v', v1)])
), ((',', 'dictorsetmaker__5'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('NAME', 'import_as_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _ctx, a, n, _all, _loc: (
alias(n.value, a.value if a else None)
), normal_vars=['a', 'n'], list_vars=[]))
), (('raise', 'raise_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _ctx, a, b, _all, _loc: (
Raise(_loc, a, b)
), normal_vars=['a', 'b'], list_vars=[]))
), ((',', 'testlist_star_expr__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('if_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('while_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('for_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('try_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('with_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('funcdef',), 
(lambda ctx, v1: [(None, v1)])
), (('classdef',), 
(lambda ctx, v1: [(None, v1)])
), (('decorated',), 
(lambda ctx, v1: [(None, v1)])
), ((',', 'dictorsetmaker__7'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('<',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Lt()
), normal_vars=[], list_vars=[]))
), (('>',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Gt()
), normal_vars=[], list_vars=[]))
), (('==',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Eq()
), normal_vars=[], list_vars=[]))
), (('>=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
GtE()
), normal_vars=[], list_vars=[]))
), (('<=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
LtE()
), normal_vars=[], list_vars=[]))
), (('!=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
NotEq()
), normal_vars=[], list_vars=[]))
), (('in',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
In()
), normal_vars=[], list_vars=[]))
), (('not', 'in'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), (None, v2)], user=lambda _ctx, _all, _loc: (
NotIn()
), normal_vars=[], list_vars=[]))
), (('is', 'comp_op__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _ctx, n, _all, _loc: (
IsNot() if n else Is()
), normal_vars=['n'], list_vars=[]))
), (('dictorsetmaker__2',), 
(lambda ctx, v1: [v1])
), (('comp_for',), 
(lambda ctx, v1: [('sc', v1)])
), (('dictorsetmaker__6',), 
(lambda ctx, v1: [v1])
), (('simple_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('compound_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('decorators', 'decorated__1'), 
(combine_action(root=lambda ctx, v1, v2: [('d', v1), v2], user=lambda _ctx, _all, _loc, cf, d: (
cf._replace(decorator_list=d)
), normal_vars=['cf', 'd'], list_vars=[]))
), (('NAME',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _ctx, n, _all, _loc: (
arg(_loc, n.value, None)
), normal_vars=['n'], list_vars=[]))
), (('import_as_name', 'import_as_names__2'), 
(lambda ctx, v1, v2: [('n', v1), v2])
), (('continue',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _ctx, _all, _loc: (
Continue(_loc)
), normal_vars=[], list_vars=[]))
), (('NEWLINE', 'file_input__1'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), (('stmt', 'file_input__1'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('term__2', 'factor', 'term__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
), ((',', 'NAME', 'global_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('exprlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda t, _ctx, _all, _loc, c: (
(t, bool(c))
), normal_vars=[], list_vars=['c', 't']))
), (('NEWLINE', 'eval_input__1'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('.',), 
(lambda ctx, v1: [('d', v1)])
), (('...',), 
(lambda ctx, v1: [('d', v1)])
), (('finally', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('f', v3)])
), ((), 
(lambda ctx: [])
), (('**', 'factor'), 
(lambda ctx, v1, v2: [(None, v1), ('f', v2)])
), ((), 
(lambda ctx: [])
), (('simple_stmt__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), ((',', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('b', v2)])
), ((), 
(lambda ctx: [])
), (('stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _ctx, _all, _loc: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('typedargslist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('yield_expr',), 
(lambda ctx, v1: [('y', v1)])
), (('testlist_comp',), 
(lambda ctx, v1: [('l', v1)])
), ((), 
(lambda ctx: [])
), (('while', 'test', ':', 'suite', 'while_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('c', v2), (None, v3), ('s', v4), v5], user=lambda _ctx, s, _all, e, _loc, c: (
While(_loc, c, s, e)
), normal_vars=['c', 'e', 's'], list_vars=[]))
), (('import_as_names__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('arglist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('testlist_star_expr', 'expr_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda av, t, _ctx, aa, _all, v, _loc: (
ast_for_expr_stmt(t, aa, av, v)
), normal_vars=['aa', 'av', 't'], list_vars=['v']))
), (('comp_for',), 
(combine_action(root=lambda ctx, v1: [('c', v1)], user=lambda _ctx, _all, _loc, c: (
c
), normal_vars=['c'], list_vars=[]))
), (('comp_if',), 
(combine_action(root=lambda ctx, v1: [('c', v1)], user=lambda _ctx, _all, _loc, c: (
c
), normal_vars=['c'], list_vars=[]))
), (('file_input__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _ctx, s, _all, _loc: (
Module(flatten(s))
), normal_vars=[], list_vars=['s']))
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('if', 'test_nocond', 'comp_if__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [('i', v1), ('c', v2), v3], user=lambda _ctx, r, _all, _loc, c, i: (
[(i, c), r or []]
), normal_vars=['c', 'i', 'r'], list_vars=[]))
), (('for', 'exprlist', 'in', 'or_test', 'comp_for__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [('f', v1), ('t', v2), (None, v3), ('i', v4), v5], user=lambda f, t, _ctx, r, _all, _loc, i: (
[(f, t, i), r or []]
), normal_vars=['f', 'i', 'r', 't'], list_vars=[]))
), (('->', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('r', v2)])
), ((), 
(lambda ctx: [])
), ((',', 'arglist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('stmt', 'suite__1'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('test', 'except_clause__2'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), ((), 
(lambda ctx: [])
), (('.', 'import_from__4'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), (('...', 'import_from__4'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), ((), 
(lambda ctx: [])
), (('comp_for',), 
(lambda ctx, v1: [('f', v1)])
), (('testlist_comp__3',), 
(lambda ctx, v1: [v1])
), (('test',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), (('test', 'dictorsetmaker__1'), 
(combine_action(root=lambda ctx, v1, v2: [('k1', v1), v2], user=lambda k1, _ctx, s, sc, _all, v1, dc, v, _loc, k: (

    DictComp(k1.location, k1, v1, ast_for_comprehension(dc)) if dc else
    SetComp(k1.location, k1, ast_for_comprehension(sc)) if sc else
    Dict(k1.location, [k1]+k, [v1]+v) if v1 else
    Set(k1.location, [k1]+s)
), normal_vars=['dc', 'k1', 'sc', 'v1'], list_vars=['k', 's', 'v']))
), (('from', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('b', v2)])
), ((), 
(lambda ctx: [])
), (('test', 'dictorsetmaker__6'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('dotted_name', 'dotted_as_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _ctx, a, n, _all, _loc: (
alias('.'.join(t.value for t in n), a.value if a else None)
), normal_vars=['a', 'n'], list_vars=[]))
), (('varargslist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('*', 'expr'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _ctx, e, _all, _loc: (
Starred(_loc, e, Load())
), normal_vars=['e'], list_vars=[]))
), ((',', 'testlist__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('vfpdef',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('testlist_comp',), 
(lambda ctx, v1: [('l', v1)])
), ((), 
(lambda ctx: [])
), (('EMBEDSTAT',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _ctx, e, _all, _loc: (
EmbedStat(_loc, e.value)
), normal_vars=['e'], list_vars=[]))
), (('arglist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _ctx, a, _all, _loc: (
a
), normal_vars=[], list_vars=['a']))
), (('arith_expr', 'shift_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _ctx, _all, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('*',), 
(lambda ctx, v1: [(None, v1)])
), (('/',), 
(lambda ctx, v1: [(None, v1)])
), (('%',), 
(lambda ctx, v1: [(None, v1)])
), (('//',), 
(lambda ctx, v1: [(None, v1)])
), (('test', 'testlist__2'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), (('import_from__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _ctx, s, _all, m, n, _loc, d: (
ImportFrom(_loc, '.'.join(t.value for t in m) if m else None, [alias('*', None)] if s else n, sum(len(t.type) for t in d))
), normal_vars=['m', 'n', 's'], list_vars=['d']))
), (('tfpdef',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('import_as_names__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _ctx, n, _all, _loc: (
n
), normal_vars=[], list_vars=['n']))
), (('del', 'exprlist'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('l', v2)], user=lambda _ctx, _all, _loc, l: (
Delete(_loc, [set_context(e, Del()) for e in l[0]])
), normal_vars=['l'], list_vars=[]))
), (('test',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), (('decorator', 'decorators__1'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), ((), 
(lambda ctx: [])
), (('sliceop',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), (('testlist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('compound_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _ctx, _all, _loc: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('not',), 
(lambda ctx, v1: [('n', v1)])
), ((), 
(lambda ctx: [])
), (('shift_expr', 'and_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _ctx, _all, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('import', 'dotted_as_names'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('n', v2)], user=lambda _ctx, n, _all, _loc: (
Import(_loc, n)
), normal_vars=['n'], list_vars=[]))
), (('decorator', 'decorators__1'), 
(combine_action(root=lambda ctx, v1, v2: [('d', v1), v2], user=lambda _ctx, _all, _loc, d: (
d
), normal_vars=[], list_vars=['d']))
), (('typedargslist_item', 'typedargslist__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), (('(', 'classdef__2', ')'), 
(lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)])
), ((), 
(lambda ctx: [])
), (('=', 'expr_stmt__4', 'expr_stmt__3'), 
(lambda ctx, v1, v2, v3: [(None, v1), v2, v3])
), ((), 
(lambda ctx: [])
), (('comp_op', 'expr', 'comparison__1'), 
(lambda ctx, v1, v2, v3: [('op', v1), ('r', v2), v3])
), ((), 
(lambda ctx: [])
), (('except', 'except_clause__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda t, _ctx, n, _all, _loc: (
lambda body: ExceptHandler(_loc, t, n.value if n else None, body)
), normal_vars=['n', 't'], list_vars=[]))
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('d', v2)])
), ((), 
(lambda ctx: [])
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('a', v2)])
), ((), 
(lambda ctx: [])
), (('yield_expr',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _ctx, e, _all, _loc: (
Expr(e.location, e)
), normal_vars=['e'], list_vars=[]))
), (('dotted_name',), 
(lambda ctx, v1: [('m', v1)])
), (('import_from__3', 'import_from__4', 'import_from__5'), 
(lambda ctx, v1, v2, v3: [v1, v2, v3])
), (('testlist_star_expr__1', 'testlist_star_expr__2'), 
(lambda ctx, v1, v2: [v1, v2])
), ((), 
(lambda ctx: [])
), (('term', 'arith_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _ctx, _all, _loc: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), ((',', 'typedargslist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('test', 'raise_stmt__2'), 
(lambda ctx, v1, v2: [('a', v1), v2])
), ((), 
(lambda ctx: [])
), (('+', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _ctx, e, _all, _loc: (
UnaryOp(_loc, UAdd(), e)
), normal_vars=['e'], list_vars=[]))
), (('-', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _ctx, e, _all, _loc: (
UnaryOp(_loc, USub(), e)
), normal_vars=['e'], list_vars=[]))
), (('~', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _ctx, e, _all, _loc: (
UnaryOp(_loc, Invert(), e)
), normal_vars=['e'], list_vars=[]))
), (('power',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _ctx, e, _all, _loc: (
e
), normal_vars=['e'], list_vars=[]))
), (('nonlocal', 'NAME', 'nonlocal_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3], user=lambda _ctx, n, _all, _loc: (
Nonlocal(_loc, [t.value for t in n])
), normal_vars=[], list_vars=['n']))
), (('<<',), 
(lambda ctx, v1: [(None, v1)])
), (('>>',), 
(lambda ctx, v1: [(None, v1)])
), (('^', 'and_expr', 'xor_expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('try', ':', 'suite', 'try_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), (None, v2), ('b', v3), v4], user=lambda es, _loc, f, _ctx, b, _all, ec, l: (
Try(_loc, b, [kc(ks) for kc, ks in zip(ec, es)], l, f)
), normal_vars=['b', 'f', 'l'], list_vars=['ec', 'es']))
), (('small_stmt', 'simple_stmt__2'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), (('+',), 
(lambda ctx, v1: [(None, v1)])
), (('-',), 
(lambda ctx, v1: [(None, v1)])
), (('dotted_name',), 
(lambda ctx, v1: [('m', v1)])
), ((), 
(lambda ctx: [])
), (('if', 'or_test', 'else', 'test'), 
(lambda ctx, v1, v2, v3, v4: [(None, v1), ('b', v2), (None, v3), ('c', v4)])
), ((), 
(lambda ctx: [])
), ((',', 'exprlist__4'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('(', 'decorator__2', ')'), 
(lambda ctx, v1, v2, v3: [('b', v1), v2, (None, v3)])
), ((), 
(lambda ctx: [])
), (('&', 'shift_expr', 'and_expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('dictorsetmaker',), 
(lambda ctx, v1: [('d', v1)])
), ((), 
(lambda ctx: [])
), (('varargslist_item', 'varargslist__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), (('testlist', 'eval_input__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda t, _ctx, _all, _loc: (
Expression(t)
), normal_vars=['t'], list_vars=[]))
), (('embed_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('expr_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('del_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('pass_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('flow_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('import_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('global_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('nonlocal_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('assert_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('break_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('continue_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('return_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('raise_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('yield_stmt',), 
(lambda ctx, v1: [(None, v1)])
)]
TABLE = {'comparison': {'EMBEDEXPR': 0, '+': 0, 'None': 0, 'NUMBER': 0, '-': 0, '{': 0, '(': 0, 'True': 0, 'STRING': 0, 'False': 0, '[': 0, '~': 0, 'NAME': 0, '...': 0}, 'flow_stmt': {'yield': 1, 'break': 1, 'return': 1, 'continue': 1, 'raise': 1}, 'testlist': {'EMBEDEXPR': 2, '+': 2, 'NAME': 2, 'NUMBER': 2, '-': 2, '{': 2, '(': 2, 'True': 2, 'STRING': 2, 'False': 2, 'lambda': 2, '[': 2, '~': 2, 'not': 2, 'None': 2, '...': 2}, 'yield_expr': {'yield': 3}, 'and_expr': {'EMBEDEXPR': 331, '+': 331, 'None': 331, 'NUMBER': 331, '-': 331, '{': 331, '(': 331, 'True': 331, 'STRING': 331, 'False': 331, '[': 331, '~': 331, 'NAME': 331, '...': 331}, 'subscript__1': {'EMBEDEXPR': 6, '+': 6, 'NAME': 6, 'NUMBER': 6, '-': 6, '{': 6, '(': 6, 'True': 6, 'STRING': 6, 'False': 6, 'lambda': 6, ':': 7, '[': 6, '~': 6, 'not': 6, 'None': 6, '...': 6}, 'dotted_as_names': {'NAME': 52}, 'import_from__1': {'from': 10}, 'varargslist_item': {'*': 12, 'NAME': 11, '**': 13}, 'and_test__1': {'as': 15, ';': 15, '**=': 15, 'for': 15, 'else': 15, 'or': 15, '//=': 15, '/=': 15, '<<=': 15, ',': 15, ':)': 15, '|=': 15, 'if': 15, '&=': 15, 'NEWLINE': 15, '*=': 15, '-=': 15, ':': 15, '>>=': 15, '%=': 15, 'and': 14, '}': 15, ')': 15, '^=': 15, '=': 15, '+=': 15, 'from': 15, ']': 15}, 'varargslist_item__2': {',': 304, 'NAME': 303, ':': 304}, 'for_stmt__1': {'yield': 20, 'EOF': 20, 'EMBEDSTAT': 20, 'with': 20, '{': 20, '(': 20, 'break': 20, 'from': 20, 'nonlocal': 20, 'for': 20, 'EMBEDEXPR': 20, 'DEDENT': 20, 'while': 20, 'continue': 20, '~': 20, 'return': 20, 'lambda': 20, 'else': 19, 'try': 20, 'None': 20, 'pass': 20, '...': 20, '+': 20, '@': 20, 'NAME': 20, 'NUMBER': 20, 'def': 20, 'class': 20, 'if': 20, 'global': 20, 'False': 20, 'STRING': 20, 'assert': 20, '[': 20, 'import': 20, 'del': 20, 'not': 20, 'raise': 20, '-': 20, '*': 20, 'NEWLINE': 20, 'True': 20}, 'testlist_star_expr__1': {'EMBEDEXPR': 21, '+': 21, 'NAME': 21, 'NUMBER': 21, '-': 21, '{': 21, '(': 21, '*': 22, 'True': 21, 'STRING': 21, 'False': 21, 'lambda': 21, '[': 21, '~': 21, 'not': 21, 'None': 21, '...': 21}, 'lambdef_nocond__1': {'*': 23, 'NAME': 23, '**': 23, ':': 24}, 'with_stmt': {'with': 25}, 'dotted_as_names__1': {';': 27, ',': 26, 'NEWLINE': 27}, 'varargslist': {'*': 180, 'NAME': 180, '**': 180}, 'argument__1': {')': 32, '=': 31, 'for': 30, ',': 32}, 'simple_stmt__2': {';': 33, 'NEWLINE': 34}, 'return_stmt': {'return': 35}, 'while_stmt__1': {'yield': 37, 'EOF': 37, 'EMBEDSTAT': 37, 'with': 37, '{': 37, '(': 37, 'break': 37, 'from': 37, 'nonlocal': 37, 'for': 37, 'EMBEDEXPR': 37, 'DEDENT': 37, 'while': 37, 'continue': 37, '~': 37, 'return': 37, 'lambda': 37, 'else': 36, 'try': 37, 'None': 37, 'pass': 37, '...': 37, '+': 37, '@': 37, 'NAME': 37, 'NUMBER': 37, 'def': 37, 'class': 37, 'if': 37, 'global': 37, 'False': 37, 'STRING': 37, 'assert': 37, '[': 37, 'import': 37, 'del': 37, 'not': 37, 'raise': 37, '-': 37, '*': 37, 'NEWLINE': 37, 'True': 37}, 'sliceop__1': {'EMBEDEXPR': 290, '+': 290, 'NAME': 290, 'NUMBER': 290, '-': 290, '{': 290, '(': 290, 'True': 290, ']': 291, 'STRING': 290, 'False': 290, 'lambda': 290, '[': 290, '~': 290, ',': 291, 'not': 290, 'None': 290, '...': 290}, 'lambdef_nocond': {'lambda': 40}, 'while_stmt': {'while': 264}, 'if_stmt__2': {'yield': 43, 'EOF': 43, 'EMBEDSTAT': 43, 'with': 43, '{': 43, '(': 43, 'break': 43, 'from': 43, 'nonlocal': 43, 'for': 43, 'EMBEDEXPR': 43, 'DEDENT': 43, 'while': 43, 'continue': 43, '~': 43, 'return': 43, 'lambda': 43, 'else': 42, 'try': 43, 'None': 43, 'pass': 43, '...': 43, '+': 43, '@': 43, 'NAME': 43, 'NUMBER': 43, 'def': 43, 'class': 43, 'if': 43, 'global': 43, 'False': 43, 'STRING': 43, 'assert': 43, '[': 43, 'import': 43, 'del': 43, 'not': 43, 'raise': 43, '-': 43, '*': 43, 'NEWLINE': 43, 'True': 43}, 'funcdef': {'def': 44}, 'subscriptlist__3': {'EMBEDEXPR': 45, '+': 45, 'NAME': 45, 'NUMBER': 45, '-': 45, '{': 45, '(': 45, 'True': 45, ']': 46, 'STRING': 45, 'False': 45, 'lambda': 45, ':': 45, '[': 45, '~': 45, 'not': 45, 'None': 45, '...': 45}, 'subscriptlist__2': {',': 47, ']': 48}, 'subscript__2': {',': 172, ':': 171, ']': 172}, 'if_stmt': {'if': 51}, 'import_stmt': {'import': 8, 'from': 9}, 'break_stmt': {'break': 53}, 'parameters': {'(': 54}, 'not_test': {'EMBEDEXPR': 56, '+': 56, 'None': 56, 'NUMBER': 56, '-': 56, '{': 56, '(': 56, 'True': 56, 'STRING': 56, 'False': 56, '[': 56, '~': 56, 'not': 55, 'NAME': 56, '...': 56}, 'import_as_names__2': {')': 58, ';': 58, ',': 57, 'NEWLINE': 58}, 'expr': {'EMBEDEXPR': 59, '+': 59, 'None': 59, 'NUMBER': 59, '-': 59, '{': 59, '(': 59, 'True': 59, 'STRING': 59, 'False': 59, '[': 59, '~': 59, 'NAME': 59, '...': 59}, 'expr_stmt__2': {'yield': 60, 'EMBEDEXPR': 61, 'NAME': 61, 'NUMBER': 61, '-': 61, '{': 61, '(': 61, 'True': 61, 'STRING': 61, 'False': 61, 'lambda': 61, '[': 61, '+': 61, '~': 61, 'not': 61, 'None': 61, '...': 61}, 'testlist_comp__3': {')': 63, ',': 62, ']': 63}, 'subscript__5': {'EMBEDEXPR': 64, '+': 64, 'NAME': 64, 'NUMBER': 64, '-': 64, '{': 64, '(': 64, 'True': 64, ']': 65, 'STRING': 64, 'False': 64, 'lambda': 64, ':': 65, '[': 64, '~': 64, ',': 65, 'not': 64, 'None': 64, '...': 64}, 'decorator': {'@': 66}, 'or_test': {'EMBEDEXPR': 67, '+': 67, 'None': 67, 'NUMBER': 67, '-': 67, '{': 67, '(': 67, 'True': 67, 'STRING': 67, 'False': 67, '[': 67, '~': 67, 'not': 67, 'NAME': 67, '...': 67}, 'atom__4': {'as': 69, '>=': 69, '<=': 69, '<': 69, '*': 69, 'for': 69, 'in': 69, ':': 69, '<<=': 69, '//=': 69, '>>': 69, '>': 69, ':)': 69, '!=': 69, '&=': 69, '&': 69, '<<': 69, '*=': 69, '==': 69, ',': 69, '%=': 69, 'and': 69, '}': 69, ')': 69, '/=': 69, 'NEWLINE': 69, ']': 69, '+': 69, '**=': 69, '^': 69, ';': 69, '%': 69, 'is': 69, '[': 69, '^=': 69, 'else': 69, '//': 69, '(': 69, '|': 69, '-': 69, 'if': 69, '/': 69, 'STRING': 68, '-=': 69, '=': 69, '>>=': 69, '**': 69, 'or': 69, 'not': 69, '|=': 69, '+=': 69, '.': 69, 'from': 69}, 'with_item__1': {'as': 136, ',': 137, ':': 137}, 'xor_expr': {'EMBEDEXPR': 71, '+': 71, 'None': 71, 'NUMBER': 71, '-': 71, '{': 71, '(': 71, 'True': 71, 'STRING': 71, 'False': 71, '[': 71, '~': 71, 'NAME': 71, '...': 71}, 'shift_expr__1': {'as': 73, ';': 73, '>=': 73, '<=': 73, '^': 73, '<': 73, 'from': 73, 'for': 73, 'in': 73, ':': 73, 'is': 73, 'else': 73, 'or': 73, '//=': 73, '<<=': 73, '>>': 72, '**=': 73, '>': 73, ':)': 73, '|': 73, '^=': 73, 'if': 73, '!=': 73, '&=': 73, '&': 73, '<<': 72, '|=': 73, '*=': 73, '==': 73, '-=': 73, ',': 73, '>>=': 73, '%=': 73, 'and': 73, '}': 73, ')': 73, '/=': 73, 'not': 73, '=': 73, '+=': 73, 'NEWLINE': 73, ']': 73}, 'suite': {'yield': 74, 'EMBEDEXPR': 74, 'NEWLINE': 75, 'EMBEDSTAT': 74, '-': 74, 'break': 74, 'nonlocal': 74, 'del': 74, 'continue': 74, 'return': 74, 'lambda': 74, '~': 74, 'NAME': 74, 'pass': 74, '...': 74, '+': 74, '(': 74, 'None': 74, 'NUMBER': 74, '{': 74, 'global': 74, 'False': 74, 'STRING': 74, 'assert': 74, '[': 74, 'import': 74, 'not': 74, 'True': 74, '*': 74, 'from': 74, 'raise': 74}, 'assert_stmt': {'assert': 76}, 'arglist__1': {'EMBEDEXPR': 77, '(': 77, 'None': 77, 'NUMBER': 77, '-': 77, '{': 77, '*': 77, 'True': 77, 'STRING': 77, 'False': 77, 'lambda': 77, '[': 77, '+': 77, '~': 77, 'not': 77, 'NAME': 77, '**': 77, '...': 77}, 'decorated__1': {'def': 79, 'class': 78}, 'except_clause__2': {'as': 80, ':': 81}, 'varargslist_item__1': {',': 83, '=': 82, ':': 83}, 'testlist_star_expr': {'EMBEDEXPR': 84, '(': 84, 'None': 84, 'NUMBER': 84, '-': 84, '{': 84, '*': 84, 'True': 84, 'STRING': 84, 'False': 84, 'lambda': 84, '[': 84, '+': 84, '~': 84, 'not': 84, 'NAME': 84, '...': 84}, 'dictorsetmaker__6': {',': 218, '}': 219}, 'expr_stmt': {'EMBEDEXPR': 269, '(': 269, 'None': 269, 'NUMBER': 269, '-': 269, '{': 269, '*': 269, 'True': 269, 'STRING': 269, 'False': 269, 'lambda': 269, '[': 269, '+': 269, '~': 269, 'not': 269, 'NAME': 269, '...': 269}, 'import_from__5': {'NAME': 369, 'import': 370}, 'term': {'EMBEDEXPR': 99, '+': 99, 'None': 99, 'NUMBER': 99, '-': 99, '{': 99, '(': 99, 'True': 99, 'STRING': 99, 'False': 99, '[': 99, '~': 99, 'NAME': 99, '...': 99}, 'expr_stmt__1': {'**=': 100, '&=': 100, ';': 101, '*=': 100, '-=': 100, '=': 101, '^=': 100, '//=': 100, '/=': 100, '<<=': 100, '>>=': 100, '|=': 100, '+=': 100, 'NEWLINE': 101, '%=': 100}, 'atom': {'None': 109, '(': 102, 'False': 111, '[': 103, 'NUMBER': 106, '...': 108, '{': 104, 'NAME': 105, 'EMBEDEXPR': 112, 'True': 110, 'STRING': 107}, 'decorators__1': {'@': 322, 'def': 323, 'class': 323}, 'compound_stmt': {'@': 328, 'for': 328, 'if': 328, 'class': 328, 'try': 328, 'with': 328, 'while': 328, 'def': 328}, 'small_stmt__1': {'EMBEDEXPR': 384, 'yield': 387, '{': 384, '-': 384, 'break': 387, '*': 384, 'del': 385, 'continue': 387, 'False': 384, 'lambda': 384, '~': 384, 'None': 384, 'pass': 386, '...': 384, '+': 384, '(': 384, 'NAME': 384, 'NUMBER': 384, 'EMBEDSTAT': 383, 'global': 389, 'return': 387, 'STRING': 384, 'assert': 391, '[': 384, 'import': 388, 'not': 384, 'raise': 387, 'nonlocal': 390, 'from': 388, 'True': 384}, 'and_test': {'EMBEDEXPR': 115, '+': 115, 'None': 115, 'NUMBER': 115, '-': 115, '{': 115, '(': 115, 'True': 115, 'STRING': 115, 'False': 115, '[': 115, '~': 115, 'not': 115, 'NAME': 115, '...': 115}, 'single_input': {'yield': 117, 'EMBEDEXPR': 117, 'with': 118, '{': 117, '-': 117, 'break': 117, 'from': 117, 'nonlocal': 117, 'for': 118, 'del': 117, 'while': 118, 'continue': 117, 'return': 117, 'lambda': 117, '~': 117, '@': 118, 'try': 118, 'NAME': 117, 'pass': 117, '...': 117, '+': 117, '(': 117, 'None': 117, 'NUMBER': 117, 'EMBEDSTAT': 117, 'class': 118, 'def': 118, 'if': 118, 'global': 117, 'False': 117, 'STRING': 117, 'assert': 117, '[': 117, 'import': 117, 'not': 117, 'True': 117, '*': 117, 'NEWLINE': 116, 'raise': 117}, 'import_as_names__1': {'NAME': 236}, 'subscript__4': {',': 120, ':': 119, ']': 120}, 'small_stmt': {'yield': 121, 'EMBEDEXPR': 121, 'EMBEDSTAT': 121, '-': 121, 'break': 121, 'nonlocal': 121, 'del': 121, 'continue': 121, 'return': 121, 'lambda': 121, '~': 121, 'NAME': 121, 'pass': 121, '...': 121, '+': 121, '(': 121, 'None': 121, 'NUMBER': 121, '{': 121, 'global': 121, 'False': 121, 'STRING': 121, 'assert': 121, '[': 121, 'import': 121, 'not': 121, 'True': 121, '*': 121, 'from': 121, 'raise': 121}, 'dotted_as_name__1': {'as': 124, ';': 125, ',': 125, 'NEWLINE': 125}, 'test_nocond': {'EMBEDEXPR': 126, '+': 126, 'None': 126, 'NUMBER': 126, '-': 126, '{': 126, '(': 126, 'True': 126, 'STRING': 126, 'lambda': 127, 'False': 126, '[': 126, '~': 126, 'not': 126, 'NAME': 126, '...': 126}, 'for_stmt': {'for': 38}, 'pass_stmt': {'pass': 128}, 'try_stmt__1': {'finally': 130, 'except': 129}, 'dotted_name': {'NAME': 131}, 'arith_expr__1': {'as': 133, '>=': 133, '<=': 133, '<': 133, 'for': 133, 'in': 133, '<<=': 133, '//=': 133, '>>': 133, '>': 133, ':)': 133, '!=': 133, '&=': 133, '&': 133, '<<': 133, '*=': 133, '==': 133, ',': 133, '%=': 133, 'and': 133, '}': 133, ')': 133, '/=': 133, '=': 133, 'NEWLINE': 133, ']': 133, '+': 132, '**=': 133, '^': 133, ';': 133, 'is': 133, '^=': 133, 'else': 133, 'if': 133, '|': 133, '-': 132, '-=': 133, ':': 133, '>>=': 133, 'or': 133, 'not': 133, '|=': 133, '+=': 133, 'from': 133}, 'typedargslist_item': {'*': 17, 'NAME': 16, '**': 18}, 'funcdef__1': {':': 278, '->': 277}, 'typedargslist': {'*': 138, 'NAME': 138, '**': 138}, 'argument': {'EMBEDEXPR': 139, '+': 139, 'NAME': 139, 'NUMBER': 139, '-': 139, '{': 139, '(': 139, 'True': 139, 'STRING': 139, 'False': 139, 'lambda': 139, '[': 139, '~': 139, 'not': 139, 'None': 139, '...': 139}, 'parameters__1': {')': 141, '*': 140, 'NAME': 140, '**': 140}, 'exprlist__1': {'EMBEDEXPR': 39, '+': 39, 'None': 39, 'NUMBER': 39, '-': 39, '{': 39, '(': 39, '*': 39, 'True': 39, 'STRING': 39, 'False': 39, '[': 39, '~': 39, 'NAME': 39, '...': 39}, 'stmt': {'yield': 258, 'EMBEDEXPR': 258, 'for': 258, '{': 258, '(': 258, 'break': 258, 'nonlocal': 258, 'with': 258, 'continue': 258, 'while': 258, 'del': 258, '~': 258, 'return': 258, 'lambda': 258, 'EMBEDSTAT': 258, 'try': 258, 'None': 258, 'pass': 258, '...': 258, '+': 258, '@': 258, 'NAME': 258, 'NUMBER': 258, 'def': 258, 'class': 258, 'if': 258, 'global': 258, 'False': 258, 'STRING': 258, 'assert': 258, '[': 258, 'import': 258, 'not': 258, 'raise': 258, '-': 258, '*': 258, 'from': 258, 'True': 258}, 'arglist_item': {'EMBEDEXPR': 144, '+': 144, 'NAME': 144, 'NUMBER': 144, '-': 144, '{': 144, '(': 144, '*': 145, 'True': 144, 'STRING': 144, 'False': 144, 'lambda': 144, '[': 144, '~': 144, 'not': 144, 'None': 144, '**': 146, '...': 144}, 'arglist': {'EMBEDEXPR': 308, '(': 308, 'None': 308, 'NUMBER': 308, '-': 308, '{': 308, '*': 308, 'True': 308, 'STRING': 308, 'False': 308, 'lambda': 308, '[': 308, '+': 308, '~': 308, 'not': 308, 'NAME': 308, '**': 308, '...': 308}, 'varargslist__2': {',': 147, ':': 148}, 'if_stmt__1': {'yield': 150, 'EOF': 150, 'with': 150, '{': 150, '(': 150, 'break': 150, 'from': 150, 'nonlocal': 150, 'for': 150, 'EMBEDEXPR': 150, 'DEDENT': 150, 'while': 150, 'continue': 150, 'else': 150, 'return': 150, 'lambda': 150, 'EMBEDSTAT': 150, '~': 150, 'try': 150, 'None': 150, 'pass': 150, '...': 150, '+': 150, '@': 150, 'NAME': 150, 'NUMBER': 150, 'def': 150, 'class': 150, 'if': 150, 'global': 150, 'False': 150, 'STRING': 150, 'assert': 150, '[': 150, 'import': 150, 'del': 150, 'elif': 149, 'not': 150, 'raise': 150, '-': 150, '*': 150, 'NEWLINE': 150, 'True': 150}, 'global_stmt': {'global': 153}, 'exprlist__2': {'EMBEDEXPR': 154, '+': 154, 'None': 154, 'NUMBER': 154, '-': 154, '{': 154, '(': 154, '*': 155, 'True': 154, 'STRING': 154, 'False': 154, '[': 154, '~': 154, 'NAME': 154, '...': 154}, 'power': {'EMBEDEXPR': 156, '(': 156, 'False': 156, '[': 156, 'NUMBER': 156, '...': 156, '{': 156, 'None': 156, 'NAME': 156, 'True': 156, 'STRING': 156}, 'testlist_star_expr__2': {'|=': 209, '**=': 209, '&=': 209, ';': 209, '*=': 209, '-=': 209, ',': 208, '^=': 209, '%=': 209, '/=': 209, '<<=': 209, '>>=': 209, '=': 209, '+=': 209, 'NEWLINE': 209, '//=': 209}, 'lambdef': {'lambda': 160}, 'comp_if__1': {',': 164, 'if': 163, '}': 164, ')': 164, 'for': 163, ']': 164}, 'classdef': {'class': 165}, 'exprlist__4': {'EMBEDEXPR': 166, '+': 166, 'None': 166, 'NUMBER': 166, '-': 166, '{': 166, '(': 166, '*': 166, ';': 167, 'in': 167, 'True': 166, 'STRING': 166, 'False': 166, '[': 166, '~': 166, 'NAME': 166, 'NEWLINE': 167, '...': 166}, 'subscript__3': {'EMBEDEXPR': 169, '+': 169, 'NAME': 169, 'NUMBER': 169, '-': 169, '{': 169, '(': 169, 'True': 169, ']': 170, 'STRING': 169, 'False': 169, 'lambda': 169, ':': 170, '[': 169, '~': 169, ',': 170, 'not': 169, 'None': 169, '...': 169}, 'sliceop': {':': 173}, 'or_test__1': {'as': 175, ';': 175, '**=': 175, 'for': 175, 'else': 175, 'or': 174, '//=': 175, '/=': 175, '<<=': 175, ',': 175, ':)': 175, '|=': 175, 'if': 175, '&=': 175, 'NEWLINE': 175, '*=': 175, '-=': 175, ':': 175, '>>=': 175, '%=': 175, '}': 175, ')': 175, '^=': 175, '=': 175, '+=': 175, 'from': 175, ']': 175}, 'test': {'EMBEDEXPR': 176, '+': 176, 'None': 176, 'NUMBER': 176, '-': 176, '{': 176, '(': 176, 'True': 176, 'STRING': 176, 'lambda': 177, 'False': 176, '[': 176, '~': 176, 'not': 176, 'NAME': 176, '...': 176}, 'try_stmt__2': {'EOF': 179, '{': 179, '-': 179, 'nonlocal': 179, 'with': 179, 'DEDENT': 179, 'while': 179, '~': 179, 'finally': 179, 'None': 179, 'pass': 179, '+': 179, '...': 179, 'class': 179, 'global': 179, 'assert': 179, 'except': 178, 'import': 179, '*': 179, 'NEWLINE': 179, 'True': 179, 'yield': 179, 'for': 179, 'EMBEDSTAT': 179, 'break': 179, '@': 179, 'EMBEDEXPR': 179, 'del': 179, 'continue': 179, 'False': 179, 'lambda': 179, 'else': 179, 'try': 179, 'NAME': 179, 'def': 179, '(': 179, 'NUMBER': 179, 'if': 179, 'return': 179, 'STRING': 179, '[': 179, 'not': 179, 'from': 179, 'raise': 179}, 'dictorsetmaker__3': {',': 29, 'for': 28, '}': 29}, 'lambdef__1': {'*': 181, 'NAME': 181, '**': 181, ':': 182}, 'yield_expr__1': {'EMBEDEXPR': 184, '+': 184, 'NAME': 184, 'NUMBER': 184, '-': 184, '{': 184, '(': 184, '=': 185, ';': 185, 'True': 184, 'STRING': 184, ')': 185, 'False': 184, 'lambda': 184, 'NEWLINE': 185, '[': 184, '~': 184, 'not': 184, 'None': 184, 'from': 183, '...': 184}, 'dotted_name__1': {'(': 187, ',': 187, 'import': 187, 'as': 187, '.': 186, ';': 187, 'NEWLINE': 187}, 'subscriptlist__1': {'EMBEDEXPR': 188, '+': 188, 'NAME': 188, 'NUMBER': 188, '-': 188, '{': 188, '(': 188, 'True': 188, 'STRING': 188, 'False': 188, 'lambda': 188, ':': 188, '[': 188, '~': 188, 'not': 188, 'None': 188, '...': 188}, 'subscript': {'EMBEDEXPR': 189, '+': 189, 'NAME': 189, 'NUMBER': 189, '-': 189, '{': 189, '(': 189, 'True': 189, 'STRING': 189, 'False': 189, 'lambda': 189, ':': 189, '[': 189, '~': 189, 'not': 189, 'None': 189, '...': 189}, 'tfpdef': {'NAME': 190}, 'with_item': {'EMBEDEXPR': 191, '+': 191, 'NAME': 191, 'NUMBER': 191, '-': 191, '{': 191, '(': 191, 'True': 191, 'STRING': 191, 'False': 191, 'lambda': 191, '[': 191, '~': 191, 'not': 191, 'None': 191, '...': 191}, 'simple_stmt': {'yield': 192, 'EMBEDEXPR': 192, 'EMBEDSTAT': 192, '-': 192, 'break': 192, 'nonlocal': 192, 'del': 192, 'continue': 192, 'return': 192, 'lambda': 192, '~': 192, 'NAME': 192, 'pass': 192, '...': 192, '+': 192, '(': 192, 'None': 192, 'NUMBER': 192, '{': 192, 'global': 192, 'False': 192, 'STRING': 192, 'assert': 192, '[': 192, 'import': 192, 'not': 192, 'True': 192, '*': 192, 'from': 192, 'raise': 192}, 'testlist_comp__4': {'EMBEDEXPR': 193, '(': 193, 'None': 193, 'NUMBER': 193, '-': 193, '{': 193, '*': 193, 'True': 193, ']': 194, 'STRING': 193, ')': 194, 'False': 193, 'lambda': 193, '[': 193, '+': 193, '~': 193, 'not': 193, 'NAME': 193, '...': 193}, 'classdef__2': {'EMBEDEXPR': 195, '(': 195, 'None': 195, 'NUMBER': 195, '-': 195, '{': 195, '*': 195, 'True': 195, 'STRING': 195, ')': 196, 'False': 195, 'lambda': 195, '[': 195, '+': 195, '~': 195, 'not': 195, 'NAME': 195, '**': 195, '...': 195}, 'expr__1': {'as': 198, ';': 198, '>=': 198, '<=': 198, '<': 198, 'from': 198, 'for': 198, 'in': 198, ':': 198, 'is': 198, 'else': 198, 'or': 198, '//=': 198, '<<=': 198, '**=': 198, '>': 198, ':)': 198, '|': 197, '^=': 198, 'if': 198, '!=': 198, '&=': 198, '|=': 198, '*=': 198, '==': 198, '-=': 198, ',': 198, '>>=': 198, '%=': 198, 'and': 198, '}': 198, ')': 198, '/=': 198, 'not': 198, '=': 198, '+=': 198, 'NEWLINE': 198, ']': 198}, 'trailer': {'(': 199, '.': 201, '[': 200}, 'expr_stmt__4': {'yield': 202, 'EMBEDEXPR': 203, 'NAME': 203, 'NUMBER': 203, '-': 203, '{': 203, '(': 203, '*': 203, 'True': 203, 'STRING': 203, 'False': 203, 'lambda': 203, '[': 203, '+': 203, '~': 203, 'not': 203, 'None': 203, '...': 203}, 'dictorsetmaker__4': {',': 204, '}': 205}, 'import_as_name': {'NAME': 206}, 'raise_stmt': {'raise': 207}, 'testlist_comp__1': {'EMBEDEXPR': 320, '+': 320, 'NAME': 320, 'NUMBER': 320, '-': 320, '{': 320, '(': 320, '*': 321, 'True': 320, 'STRING': 320, 'False': 320, 'lambda': 320, '[': 320, '~': 320, 'not': 320, 'None': 320, '...': 320}, 'compound_stmt__1': {'@': 217, 'for': 212, 'if': 210, 'class': 216, 'try': 213, 'with': 214, 'while': 211, 'def': 215}, 'comp_op': {'>': 221, '==': 222, 'is': 228, '>=': 223, '<=': 224, '!=': 225, '<': 220, 'not': 227, 'in': 226}, 'dictorsetmaker__1': {',': 231, ':': 229, 'for': 230, '}': 231}, 'stmt__1': {'yield': 232, 'EMBEDEXPR': 232, 'with': 233, 'EMBEDSTAT': 232, '-': 232, 'break': 232, 'nonlocal': 232, 'for': 233, 'del': 232, 'while': 233, 'continue': 232, 'return': 232, 'lambda': 232, '~': 232, '@': 233, 'try': 233, 'NAME': 232, 'pass': 232, '...': 232, '+': 232, '(': 232, 'None': 232, 'NUMBER': 232, '{': 232, 'class': 233, 'def': 233, 'if': 233, 'global': 232, 'False': 232, 'STRING': 232, 'assert': 232, '[': 232, 'import': 232, 'not': 232, 'True': 232, '*': 232, 'from': 232, 'raise': 232}, 'typedargslist__3': {')': 260, '*': 259, 'NAME': 259, '**': 259}, 'vfpdef': {'NAME': 235}, 'dictorsetmaker__5': {'EMBEDEXPR': 85, '+': 85, 'NAME': 85, 'NUMBER': 85, '-': 85, '{': 85, '(': 85, 'True': 85, 'STRING': 85, 'False': 85, 'lambda': 85, '[': 85, '}': 86, '~': 85, 'not': 85, 'None': 85, '...': 85}, 'import_from__6': {'(': 158, '*': 157, 'NAME': 159}, 'continue_stmt': {'continue': 237}, 'file_input__1': {'EMBEDEXPR': 239, 'yield': 239, 'for': 239, '{': 239, '(': 239, 'break': 239, 'from': 239, 'nonlocal': 239, 'with': 239, 'continue': 239, 'while': 239, 'del': 239, '~': 239, 'return': 239, 'lambda': 239, 'EMBEDSTAT': 239, 'try': 239, 'None': 239, 'pass': 239, '...': 239, '+': 239, '@': 239, 'NAME': 239, 'NUMBER': 239, 'def': 239, 'class': 239, 'if': 239, 'global': 239, 'EOF': 240, 'False': 239, 'STRING': 239, 'assert': 239, '[': 239, 'import': 239, 'not': 239, 'raise': 239, '-': 239, '*': 239, 'NEWLINE': 238, 'True': 239}, 'term__1': {'as': 242, '>=': 242, '<=': 242, '<': 242, '*': 241, 'for': 242, 'in': 242, ':': 242, '<<=': 242, '//=': 242, '>>': 242, '>': 242, ':)': 242, '!=': 242, '/': 241, '&': 242, '<<': 242, '*=': 242, '==': 242, 'if': 242, '%=': 242, 'and': 242, '}': 242, ')': 242, '/=': 242, 'NEWLINE': 242, ']': 242, '+': 242, '**=': 242, '^': 242, ';': 242, '%': 241, 'is': 242, '^=': 242, 'else': 242, ',': 242, '//': 241, '|': 242, '-': 242, '&=': 242, '-=': 242, '=': 242, '>>=': 242, 'or': 242, 'not': 242, '|=': 242, '+=': 242, 'from': 242}, 'global_stmt__1': {';': 244, ',': 243, 'NEWLINE': 244}, 'exprlist': {'EMBEDEXPR': 245, '+': 245, 'None': 245, 'NUMBER': 245, '-': 245, '{': 245, '(': 245, '*': 245, 'True': 245, 'STRING': 245, 'False': 245, '[': 245, '~': 245, 'NAME': 245, '...': 245}, 'eval_input__1': {':)': 247, 'NEWLINE': 246}, 'import_from__3': {'.': 248, '...': 249}, 'try_stmt__4': {'yield': 251, 'EOF': 251, 'with': 251, '{': 251, '(': 251, 'break': 251, 'from': 251, 'nonlocal': 251, 'for': 251, 'EMBEDEXPR': 251, 'DEDENT': 251, 'while': 251, 'continue': 251, '~': 251, 'finally': 250, 'lambda': 251, 'EMBEDSTAT': 251, 'try': 251, 'None': 251, 'False': 251, 'pass': 251, '...': 251, '+': 251, '@': 251, 'NAME': 251, 'NUMBER': 251, 'def': 251, 'class': 251, 'if': 251, 'global': 251, 'return': 251, 'STRING': 251, 'assert': 251, '[': 251, 'import': 251, 'del': 251, 'not': 251, 'raise': 251, '-': 251, '*': 251, 'NEWLINE': 251, 'True': 251}, 'power__2': {'as': 253, '>=': 253, '<=': 253, '<': 253, '*': 253, 'for': 253, 'in': 253, ':': 253, '<<=': 253, '//=': 253, '>>': 253, '>': 253, ':)': 253, '!=': 253, '&=': 253, '&': 253, '<<': 253, '*=': 253, '==': 253, ',': 253, '%=': 253, 'and': 253, '}': 253, ')': 253, '/=': 253, '//': 253, 'NEWLINE': 253, ']': 253, '+': 253, '**=': 253, '^': 253, ';': 253, '%': 253, 'is': 253, '^=': 253, 'else': 253, '-': 253, '|': 253, 'if': 253, '/': 253, '-=': 253, '=': 253, '>>=': 253, '**': 252, 'or': 253, 'not': 253, '|=': 253, '+=': 253, 'from': 253}, 'simple_stmt__3': {'yield': 254, 'EMBEDEXPR': 254, 'NEWLINE': 255, 'EMBEDSTAT': 254, '-': 254, 'break': 254, 'nonlocal': 254, 'del': 254, 'continue': 254, 'return': 254, 'lambda': 254, '~': 254, 'NAME': 254, 'pass': 254, '...': 254, '+': 254, '(': 254, 'None': 254, 'NUMBER': 254, '{': 254, 'global': 254, 'False': 254, 'STRING': 254, 'assert': 254, '[': 254, 'import': 254, 'not': 254, 'True': 254, '*': 254, 'from': 254, 'raise': 254}, 'assert_stmt__1': {';': 257, ',': 256, 'NEWLINE': 257}, 'augassign': {'**=': 97, '&=': 92, '>>=': 96, '*=': 89, '-=': 88, '^=': 94, '%=': 91, '<<=': 95, '/=': 90, '|=': 93, '+=': 87, '//=': 98}, 'decorator__2': {'EMBEDEXPR': 142, '(': 142, 'None': 142, 'NUMBER': 142, '-': 142, '{': 142, '*': 142, 'True': 142, 'STRING': 142, ')': 143, 'False': 142, 'lambda': 142, '[': 142, '+': 142, '~': 142, 'not': 142, 'NAME': 142, '**': 142, '...': 142}, 'atom__1': {'yield': 261, 'EMBEDEXPR': 262, 'NAME': 262, 'NUMBER': 262, '-': 262, '{': 262, '(': 262, '*': 262, 'True': 262, 'STRING': 262, ')': 263, 'False': 262, 'lambda': 262, '[': 262, '+': 262, '~': 262, 'not': 262, 'None': 262, '...': 262}, 'testlist_comp': {'EMBEDEXPR': 41, '(': 41, 'None': 41, 'NUMBER': 41, '-': 41, '{': 41, '*': 41, 'True': 41, 'STRING': 41, 'False': 41, 'lambda': 41, '[': 41, '+': 41, '~': 41, 'not': 41, 'NAME': 41, '...': 41}, 'import_as_names__3': {')': 266, ';': 266, 'NAME': 265, 'NEWLINE': 266}, 'arglist__3': {'EMBEDEXPR': 267, '(': 267, 'None': 267, 'NUMBER': 267, '-': 267, '{': 267, '*': 267, 'True': 267, 'STRING': 267, ')': 268, 'False': 267, 'lambda': 267, '[': 267, '+': 267, '~': 267, 'not': 267, 'NAME': 267, '**': 267, '...': 267}, 'with_stmt__1': {',': 122, ':': 123}, 'nonlocal_stmt__1': {';': 114, ',': 113, 'NEWLINE': 114}, 'file_input': {'EMBEDEXPR': 272, 'EOF': 272, 'for': 272, '{': 272, '(': 272, 'break': 272, 'from': 272, 'nonlocal': 272, 'with': 272, 'continue': 272, 'while': 272, 'del': 272, '~': 272, 'return': 272, 'lambda': 272, 'yield': 272, 'EMBEDSTAT': 272, 'try': 272, 'None': 272, 'pass': 272, '...': 272, '+': 272, '@': 272, 'NAME': 272, 'NUMBER': 272, 'def': 272, 'class': 272, 'if': 272, 'global': 272, 'False': 272, 'STRING': 272, 'assert': 272, '[': 272, 'import': 272, 'not': 272, 'raise': 272, '-': 272, '*': 272, 'NEWLINE': 272, 'True': 272}, 'trailer__1': {'EMBEDEXPR': 273, '(': 273, 'None': 273, 'NUMBER': 273, '-': 273, '{': 273, '*': 273, 'True': 273, 'STRING': 273, ')': 274, 'False': 273, 'lambda': 273, '[': 273, '+': 273, '~': 273, 'not': 273, 'NAME': 273, '**': 273, '...': 273}, 'comp_if': {'if': 275}, 'comp_for': {'for': 276}, 'except_clause__1': {'EMBEDEXPR': 283, '+': 283, 'NAME': 283, 'NUMBER': 283, '-': 283, '{': 283, '(': 283, 'True': 283, 'STRING': 283, 'False': 283, 'lambda': 283, ':': 284, '[': 283, '~': 283, 'not': 283, 'None': 283, '...': 283}, 'arglist__2': {')': 280, ',': 279}, 'suite__1': {'yield': 281, 'EMBEDEXPR': 281, 'for': 281, '{': 281, '(': 281, 'break': 281, 'nonlocal': 281, 'with': 281, 'continue': 281, 'while': 281, 'del': 281, '~': 281, 'return': 281, 'lambda': 281, 'EMBEDSTAT': 281, 'try': 281, 'None': 281, 'pass': 281, '...': 281, '+': 281, '@': 281, 'NAME': 281, 'NUMBER': 281, 'def': 281, 'class': 281, 'if': 281, 'global': 281, 'False': 281, 'STRING': 281, 'assert': 281, '[': 281, 'DEDENT': 282, 'import': 281, 'not': 281, 'raise': 281, '-': 281, '*': 281, 'from': 281, 'True': 281}, 'import_from__4': {'.': 285, 'NAME': 287, '...': 286, 'import': 287}, 'testlist_comp__2': {')': 289, ',': 289, 'for': 288, ']': 289}, 'comp_for__1': {',': 162, 'if': 161, '}': 162, ')': 162, 'for': 161, ']': 162}, 'dictorsetmaker': {'EMBEDEXPR': 292, '+': 292, 'NAME': 292, 'NUMBER': 292, '-': 292, '{': 292, '(': 292, 'True': 292, 'STRING': 292, 'False': 292, 'lambda': 292, '[': 292, '~': 292, 'not': 292, 'None': 292, '...': 292}, 'raise_stmt__2': {';': 294, 'NEWLINE': 294, 'from': 293}, 'dictorsetmaker__7': {'EMBEDEXPR': 295, '+': 295, 'NAME': 295, 'NUMBER': 295, '-': 295, '{': 295, '(': 295, 'True': 295, 'STRING': 295, 'False': 295, 'lambda': 295, '[': 295, '}': 296, '~': 295, 'not': 295, 'None': 295, '...': 295}, 'varargslist__3': {'*': 298, 'NAME': 298, '**': 298, ':': 299}, 'and_expr__1': {'as': 378, ';': 378, '>=': 378, '<=': 378, '^': 378, '<': 378, 'from': 378, 'for': 378, 'in': 378, ':': 378, 'is': 378, 'else': 378, 'or': 378, '//=': 378, '<<=': 378, '**=': 378, '>': 378, ':)': 378, '|': 378, '^=': 378, 'if': 378, '!=': 378, '&=': 378, '&': 377, '|=': 378, '*=': 378, '==': 378, '-=': 378, ',': 378, '>>=': 378, '%=': 378, 'and': 378, '}': 378, ')': 378, '/=': 378, 'not': 378, '=': 378, '+=': 378, 'NEWLINE': 378, ']': 378}, 'testlist__2': {':)': 302, '=': 302, ')': 302, ',': 301, ';': 302, 'NEWLINE': 302, ':': 302}, 'atom__2': {'EMBEDEXPR': 305, '(': 305, 'None': 305, 'NUMBER': 305, '-': 305, '{': 305, '*': 305, 'True': 305, ']': 306, 'STRING': 305, 'False': 305, 'lambda': 305, '[': 305, '+': 305, '~': 305, 'not': 305, 'NAME': 305, '...': 305}, 'varargslist__1': {'*': 381, 'NAME': 381, '**': 381}, 'embed_stmt': {'EMBEDSTAT': 307}, 'try_stmt__3': {'yield': 152, 'EOF': 152, 'EMBEDSTAT': 152, 'with': 152, '{': 152, '(': 152, 'break': 152, 'from': 152, 'nonlocal': 152, 'for': 152, 'EMBEDEXPR': 152, 'DEDENT': 152, 'while': 152, 'continue': 152, '~': 152, 'finally': 152, 'lambda': 152, 'else': 151, 'try': 152, 'None': 152, 'False': 152, 'pass': 152, '...': 152, '+': 152, '@': 152, 'NAME': 152, 'NUMBER': 152, 'def': 152, 'class': 152, 'if': 152, 'global': 152, 'return': 152, 'STRING': 152, 'assert': 152, '[': 152, 'import': 152, 'del': 152, 'not': 152, 'raise': 152, '-': 152, '*': 152, 'NEWLINE': 152, 'True': 152}, 'shift_expr': {'EMBEDEXPR': 309, '+': 309, 'None': 309, 'NUMBER': 309, '-': 309, '{': 309, '(': 309, 'True': 309, 'STRING': 309, 'False': 309, '[': 309, '~': 309, 'NAME': 309, '...': 309}, 'term__2': {'*': 310, '//': 313, '%': 312, '/': 311}, 'testlist__1': {'EMBEDEXPR': 314, '+': 314, 'NAME': 314, 'NUMBER': 314, '-': 314, '{': 314, '(': 314, 'True': 314, 'STRING': 314, 'False': 314, 'lambda': 314, '[': 314, '~': 314, 'not': 314, 'None': 314, '...': 314}, 'import_from': {'from': 315}, 'typedargslist_item__2': {')': 317, ',': 317, 'NAME': 316}, 'import_as_names': {'NAME': 318}, 'del_stmt': {'del': 319}, 'subscript__6': {',': 325, ':': 324, ']': 325}, 'return_stmt__1': {'EMBEDEXPR': 326, '+': 326, 'NAME': 326, 'NUMBER': 326, '-': 326, '{': 326, '(': 326, ';': 327, 'True': 326, 'STRING': 326, 'False': 326, 'lambda': 326, '[': 326, '~': 326, 'not': 326, 'None': 326, 'NEWLINE': 327, '...': 326}, 'dictorsetmaker__2': {':': 168}, 'comp_op__1': {'EMBEDEXPR': 330, '+': 330, 'None': 330, 'NUMBER': 330, '-': 330, '{': 330, '(': 330, 'True': 330, 'STRING': 330, 'False': 330, '[': 330, '~': 330, 'not': 329, 'NAME': 330, '...': 330}, 'testlist__3': {'EMBEDEXPR': 4, '{': 4, ':)': 5, ';': 5, ':': 5, 'False': 4, 'lambda': 4, '[': 4, '~': 4, 'NAME': 4, '...': 4, '+': 4, '(': 4, 'None': 4, 'NUMBER': 4, '-': 4, 'STRING': 4, '=': 5, ')': 5, 'not': 4, 'NEWLINE': 5, 'True': 4}, 'import_name': {'import': 332}, 'decorators': {'@': 333}, 'typedargslist__1': {'*': 334, 'NAME': 334, '**': 334}, 'classdef__1': {'(': 335, ':': 336}, 'expr_stmt__3': {';': 338, '=': 337, 'NEWLINE': 338}, 'comparison__1': {'as': 340, 'for': 340, '>=': 339, '<=': 339, '<': 339, ';': 340, 'in': 339, ':': 340, 'is': 339, 'else': 340, 'or': 340, '//=': 340, '/=': 340, '<<=': 340, '**=': 340, '>': 339, ':)': 340, '|=': 340, 'if': 340, '!=': 339, '&=': 340, 'NEWLINE': 340, '*=': 340, '==': 339, '-=': 340, ',': 340, '>>=': 340, '%=': 340, 'and': 340, '}': 340, ')': 340, '^=': 340, 'not': 339, '=': 340, '+=': 340, 'from': 340, ']': 340}, 'typedargslist_item__1': {')': 343, ',': 343, '=': 342}, 'import_as_name__1': {')': 345, 'as': 344, ';': 345, ',': 345, 'NEWLINE': 345}, 'yield_stmt': {'yield': 346}, 'import_from__2': {'.': 348, 'NAME': 347, '...': 348}, 'testlist_star_expr__3': {'EMBEDEXPR': 349, '**=': 350, '{': 349, '*': 349, ';': 350, '<<=': 350, 'False': 349, 'lambda': 349, '^=': 350, '//=': 350, '~': 349, 'None': 349, '...': 349, '+': 349, '(': 349, 'NAME': 349, 'NUMBER': 349, '-': 349, '&=': 350, '|=': 350, '*=': 350, 'STRING': 349, '-=': 350, '[': 349, '>>=': 350, '%=': 350, '/=': 350, 'not': 349, '=': 350, '+=': 350, 'NEWLINE': 350, 'True': 349}, 'arith_expr': {'EMBEDEXPR': 351, '+': 351, 'None': 351, 'NUMBER': 351, '-': 351, '{': 351, '(': 351, 'True': 351, 'STRING': 351, 'False': 351, '[': 351, '~': 351, 'NAME': 351, '...': 351}, 'typedargslist__2': {')': 353, ',': 352}, 'raise_stmt__1': {'EMBEDEXPR': 354, '+': 354, 'NAME': 354, 'NUMBER': 354, '-': 354, '{': 354, '(': 354, ';': 355, 'True': 354, 'STRING': 354, 'False': 354, 'lambda': 354, '[': 354, '~': 354, 'not': 354, 'None': 354, 'NEWLINE': 355, '...': 354}, 'comp_iter': {'for': 270, 'if': 271}, 'factor': {'+': 356, 'EMBEDEXPR': 359, 'None': 359, 'NUMBER': 359, '-': 357, '{': 359, '(': 359, 'True': 359, 'STRING': 359, 'False': 359, '[': 359, '~': 358, 'NAME': 359, '...': 359}, 'nonlocal_stmt': {'nonlocal': 360}, 'shift_expr__2': {'>>': 362, '<<': 361}, 'xor_expr__1': {'as': 364, ';': 364, '>=': 364, '<=': 364, '^': 363, '<': 364, 'from': 364, 'for': 364, 'in': 364, ':': 364, 'is': 364, 'else': 364, 'or': 364, '//=': 364, '<<=': 364, '**=': 364, '>': 364, ':)': 364, '|': 364, '^=': 364, 'if': 364, '!=': 364, '&=': 364, '|=': 364, '*=': 364, '==': 364, '-=': 364, ',': 364, '>>=': 364, '%=': 364, 'and': 364, '}': 364, ')': 364, '/=': 364, 'not': 364, '=': 364, '+=': 364, 'NEWLINE': 364, ']': 364}, 'try_stmt': {'try': 365}, 'simple_stmt__1': {'yield': 366, 'EMBEDEXPR': 366, 'EMBEDSTAT': 366, '-': 366, 'break': 366, 'nonlocal': 366, 'del': 366, 'continue': 366, 'return': 366, 'lambda': 366, '~': 366, 'NAME': 366, 'pass': 366, '...': 366, '+': 366, '(': 366, 'None': 366, 'NUMBER': 366, '{': 366, 'global': 366, 'False': 366, 'STRING': 366, 'assert': 366, '[': 366, 'import': 366, 'not': 366, 'True': 366, '*': 366, 'from': 366, 'raise': 366}, 'arith_expr__2': {'+': 367, '-': 368}, 'tfpdef__1': {')': 50, '=': 50, ':': 49, ',': 50}, 'test__1': {'as': 372, ';': 372, '**=': 372, 'for': 372, '^=': 372, '//=': 372, '<<=': 372, ',': 372, ':)': 372, '|=': 372, 'if': 371, '&=': 372, 'NEWLINE': 372, '*=': 372, '-=': 372, ':': 372, '>>=': 372, '%=': 372, '}': 372, ')': 372, '/=': 372, '=': 372, '+=': 372, 'from': 372, ']': 372}, 'exprlist__3': {';': 374, ',': 373, 'in': 374, 'NEWLINE': 374}, 'power__1': {'as': 135, '>=': 135, '<=': 135, '<': 135, '*': 135, 'for': 135, 'in': 135, ':': 135, '<<=': 135, '//=': 135, '>>': 135, '>': 135, ':)': 135, '!=': 135, '&=': 135, '&': 135, '<<': 135, '*=': 135, '==': 135, ',': 135, '%=': 135, 'and': 135, '}': 135, ')': 135, '/=': 135, 'NEWLINE': 135, ']': 135, '+': 135, '**=': 135, '^': 135, ';': 135, '%': 135, 'is': 135, '[': 134, '^=': 135, 'else': 135, '//': 135, '(': 134, '|': 135, '-': 135, '.': 134, '/': 135, '-=': 135, '=': 135, '>>=': 135, '**': 135, 'if': 135, 'or': 135, 'not': 135, '|=': 135, '+=': 135, 'from': 135}, 'decorator__1': {'(': 375, 'NEWLINE': 376}, 'star_expr': {'*': 300}, 'atom__3': {'EMBEDEXPR': 379, '+': 379, 'NAME': 379, 'NUMBER': 379, '-': 379, '{': 379, '(': 379, 'True': 379, 'STRING': 379, 'False': 379, 'lambda': 379, '[': 379, '}': 380, '~': 379, 'not': 379, 'None': 379, '...': 379}, 'except_clause': {'except': 341}, 'eval_input': {'EMBEDEXPR': 382, '+': 382, 'NAME': 382, 'NUMBER': 382, '-': 382, '{': 382, '(': 382, 'True': 382, 'STRING': 382, 'False': 382, 'lambda': 382, '[': 382, '~': 382, 'not': 382, 'None': 382, '...': 382}, 'dotted_as_name': {'NAME': 297}, 'decorated': {'@': 234}, 'flow_stmt__1': {'break': 392, 'yield': 396, 'return': 394, 'continue': 393, 'raise': 395}, 'subscriptlist': {'EMBEDEXPR': 70, '+': 70, 'NAME': 70, 'NUMBER': 70, '-': 70, '{': 70, '(': 70, 'True': 70, 'STRING': 70, 'False': 70, 'lambda': 70, ':': 70, '[': 70, '~': 70, 'not': 70, 'None': 70, '...': 70}}
