
from context import flatten
from toypython.lex import tokenizer
from toypython.ast import *

def set_context(expr, context):
    if isinstance(expr, (Attribute, Subscript, Name)):
        return expr._replace(ctx=context)

    if isinstance(expr, Starred):
        return Starred(expr.location, set_context(expr.value, context), context)

    if isinstance(expr, List) or (isinstance(expr, Tuple) and expr.elts):
        new_elts = [set_context(e, context) for e in expr.elts]
        return expr.__class__(expr.location, new_elts, context)

    print(expr)
    name = expr.__class__.__name__
    if context == Del():
        raise ParseError(expr.location, "can't delete %s" % name)
    else:
        raise ParseError(expr.location, "can't assign to %s" % name)

def ast_for_testlist(tests, commas):
    return tests[0] if not commas else Tuple(tests[0].location, tests, Load())

def ast_for_arguments(items):
    posargs = []
    vararg = None
    after_vararg = False
    kwonlyargs = []
    kwdefaults = []
    kwarg = None
    posdefaults = []

    if items[-1][0] and items[-1][0].type == '**':
        kwarg = items[-1][1]
        items = items[:-1]

    if items[-1][0] and items[-1][0].type == '*' and not items[-1][1]:
        raise ParseError(items[-1][0].location, "named arguments must follow bare *")

    for (star_token, fpdef, default) in items:
        if star_token and star_token.type == '**':
            raise ParseError(star_token.location, "**kwargs must be last in arguments list")

        if star_token and after_vararg:
            raise ParseError(star_token.location, "*args can only be given once")

        if star_token:
            after_vararg = True
            vararg = fpdef
        elif after_vararg:
            kwonlyargs.append(fpdef)
            kwdefaults.append(default)
        else:
            if default:
                posdefaults.append(default)
            elif posdefaults:
                raise ParseError(fpdef.location, "non-default argument follows default argument")
            posargs.append(fpdef)

    return arguments(posargs, vararg, kwonlyargs, kwdefaults, kwarg, posdefaults)

def ast_for_dotted_name(name_tokens):
    rv = None
    for name_token in name_tokens:
        if rv:
            rv = Attribute(rv.location, rv, name_token.value, Load())
        else:
            rv = Name(name_token.location, name_token.value, Load())
    return rv

def ast_for_decorator(loc, name_expr, have_parens, arglist):
    if not have_parens: return name_expr
    return ast_for_call(loc, name_expr, arglist)

def ast_for_comprehension(root_comp_for):
    comps = []

    for item in flatten(root_comp_for):
        if item[0].type == 'for':
            token, (exprlist, exprlist_has_comma), iter = item
            if exprlist_has_comma:
                target = Tuple(exprlist[0].location, exprlist, Store())
            else:
                target = exprlist[0]
            target = set_context(target, Store())
            comps.append(comprehension(target, iter, []))
        else:
            token, cond = item
            comps[-1].ifs.append(cond)

    return comps

token_type_to_operator_class = {
    '|': BitOr,
    '^': BitXor,
    '&': BitAnd,
    '<<': LShift,
    '>>': RShift,
    '+': Add,
    '-': Sub,
    '*': Mult,
    '/': Div,
    '//': FloorDiv,
    '%': Mod,
}

def ast_for_binop(symbols):
    symbols = iter(symbols)
    left = next(symbols)
    while True:
        try:
            op_token = next(symbols)
            op_class = token_type_to_operator_class[op_token.type]
            right = next(symbols)
            left = BinOp(op_token.location, left, op_class(), right)
        except StopIteration:
            return left

def ast_for_power(atom, trailers, factor):
    for trailer in trailers:
        atom = trailer(atom)._replace(location=atom.location)
    if factor:
        atom = BinOp(atom.location, atom, Pow(), factor)
    return atom

def ast_for_call(loc, func, arglist):
    if not arglist: return Call(loc, func, None, None, None, None)

    args = []
    keywords = []
    vararg = None
    kwarg = None
    keyword_names = set()

    star_token, test, root_comp_for, kwvalue = arglist[0]
    if len(arglist) == 1 and root_comp_for:
        value = GeneratorExp(test.location, test, ast_for_comprehension(root_comp_for))
        arglist = [(None, value, None, None)]

    for (star_token, test, root_comp_for, kwvalue) in arglist:
        if root_comp_for:
            raise ParseError(test, "Generator expression must be parenthesized "
                             "if not sole argument")

    if arglist[-1][0] and arglist[-1][0].type == '**':
        kwarg = arglist[-1][1]
        arglist = arglist[:-1]

    for (star_token, test, root_comp_for, kwvalue) in arglist:
        if star_token and star_token.type == '*':
            if vararg:
                raise ParseError(star_token.location, "*expression can only be given once")
            vararg = test
        elif star_token and star_token.type == '**':
            raise ParseError(star_token.location, "**expression must be last in arguments list")
        elif not kwvalue:
            if keywords:
                raise ParseError(test.location, "non-keyword arg after keyword arg")
            if vararg:
                raise ParseError(test.location, "only named arguments may follow *expression")
            args.append(test)
        elif isinstance(test, Lambda):
            raise ParseError(test.location, "lambda cannot contain assignment")
        elif not isinstance(test, Name):
            raise ParseError(test.location, "keyword can't be an expression")
        elif test.id in keyword_names:
            raise ParseError(test.location, "keyword argument repeated")
        else:
            keyword_names.add(test.id)
            keywords.append(keyword(test.id, kwvalue))

    return Call(loc, func, args, keywords, vararg, kwarg)

def ast_for_expr_stmt(base, augassign_op, augassign_value, assignments):
    if augassign_op:
        base = set_context(base, Store())
        if not isinstance(base, (Name, Attribute, Subscript)):
            raise ParseError(base.location, "illegal expression for augmented assignment")
        return AugAssign(base.location, base, augassign_op, augassign_value)

    if assignments:
        value = assignments[-1]
        targets = []
        for target in [base] + assignments[:-1]:
            if isinstance(target, (Yield, YieldFrom)):
                raise ParseError(target.location, "assignment to yield expression not possible")
            targets.append(set_context(target, Store()))
        return Assign(base.location, targets, value)

    return Expr(base.location, base)

def ast_for_if_stmt(tokens, conds, suites, else_suite):
    for token, cond, suite in reversed(list(zip(tokens, conds, suites))):
        else_suite = [If(token.location, cond, suite, else_suite or [])]
    return else_suite

def ast_for_classdef(loc, name, arglist, body):
    dummy_call = ast_for_call(loc, None, arglist)
    return ClassDef(loc, name, dummy_call.args, dummy_call.keywords,
                    dummy_call.starargs, dummy_call.kwargs, body, [])



from context import combine_action

from context import ParseError

def parse(ctx, start_nt=None, close_with=None):
    stack = [close_with or 'EOF', start_nt or 'file_input']
    results = []
    tok_iter = tokenizer(ctx, close_with)
    token = next(tok_iter)

    while True:
        stack_top = stack[-1]

        if isinstance(stack_top, int):
            rule, action = RULES[stack_top]
            stack.pop()
            n = len(rule)
            args = results[len(results) - n:]
            del results[len(results) - n:]
            results.append(action(ctx, *args))

        elif stack_top not in TABLE:
            if token.type != stack_top:
                raise ParseError(token.location, "Expected " + stack_top)
            stack.pop()
            if not stack: break
            results.append(token)
            token = next(tok_iter)

        else:
            row = TABLE[stack_top]
            if token.type not in row:
                raise ParseError(token.location, "Unexpected {}, expected {}".format(
                    repr(token.type), ', '.join(sorted(repr(k) for k in row))))
            rule_num = row[token.type]
            rule, action = RULES[rule_num]
            stack.pop()
            stack.append(rule_num)
            stack.extend(reversed(rule))

    return results[0]

RULES = [(('yield_expr',), 
(lambda ctx, v1: [('y', v1)])
), (('testlist_comp',), 
(lambda ctx, v1: [('l', v1)])
), ((), 
(lambda ctx: [])
), ((':', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('t', v2)])
), ((), 
(lambda ctx: [])
), (('exprlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda t, c, _loc, _all, _ctx: (
(t, bool(c))
), normal_vars=[], list_vars=['c', 't']))
), (('+=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Add()
), normal_vars=[], list_vars=[]))
), (('-=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Sub()
), normal_vars=[], list_vars=[]))
), (('*=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Mul()
), normal_vars=[], list_vars=[]))
), (('/=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Div()
), normal_vars=[], list_vars=[]))
), (('%=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Mod()
), normal_vars=[], list_vars=[]))
), (('&=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
BitAnd()
), normal_vars=[], list_vars=[]))
), (('|=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
BitOr()
), normal_vars=[], list_vars=[]))
), (('^=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
BitXor()
), normal_vars=[], list_vars=[]))
), (('<<=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
LShift()
), normal_vars=[], list_vars=[]))
), (('>>=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
RShift()
), normal_vars=[], list_vars=[]))
), (('**=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Pow()
), normal_vars=[], list_vars=[]))
), (('//=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
FloorDiv()
), normal_vars=[], list_vars=[]))
), (('comp_iter',), 
(lambda ctx, v1: [('r', v1)])
), ((), 
(lambda ctx: [])
), (('embed_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('expr_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('del_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('pass_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('flow_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('import_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('global_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('nonlocal_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('assert_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('and', 'not_test', 'and_test__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('l', v2), v3])
), ((), 
(lambda ctx: [])
), (('not_test', 'and_test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda _loc, _all, l, _ctx: (
l[0] if len(l) == 1 else BoolOp(l[0].location, Or(), l)
), normal_vars=[], list_vars=['l']))
), (('+',), 
(lambda ctx, v1: [(None, v1)])
), (('-',), 
(lambda ctx, v1: [(None, v1)])
), (('or', 'and_test', 'or_test__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('l', v2), v3])
), ((), 
(lambda ctx: [])
), (('subscript', 'subscriptlist__2'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), (('sliceop',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), (('test', 'with_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda t, _loc, _all, e, _ctx: (
withitem(t, set_context(e, Store()) if e else None)
), normal_vars=['e', 't'], list_vars=[]))
), (('typedargslist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, _all, l, _ctx: (
ast_for_arguments(l)
), normal_vars=[], list_vars=['l']))
), (('(', 'parameters__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda t, _loc, _all, _ctx: (
t or arguments(None, None, None, None, None, None)
), normal_vars=['t'], list_vars=[]))
), (('comp_iter',), 
(lambda ctx, v1: [('r', v1)])
), ((), 
(lambda ctx: [])
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('l', v3)])
), ((), 
(lambda ctx: [])
), (('*', 'expr'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _loc, _all, e, _ctx: (
Starred(_loc, e, Load())
), normal_vars=['e'], list_vars=[]))
), (('NEWLINE', 'file_input__1'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), (('stmt', 'file_input__1'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('testlist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('d', v2)])
), ((), 
(lambda ctx: [])
), (('if', 'test', ':', 'suite', 'if_stmt__1', 'if_stmt__2'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6: [('t', v1), ('c', v2), (None, v3), ('s', v4), v5, v6], user=lambda t, c, _loc, e, s, _all, _ctx: (
ast_for_if_stmt(t, c, s, e)
), normal_vars=['e'], list_vars=['c', 's', 't']))
), (('and_expr', 'xor_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _loc, _all, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), ((',', 'subscriptlist__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), ((',', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('b', v2)])
), ((), 
(lambda ctx: [])
), (('arith_expr', 'shift_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _loc, _all, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('import', 'dotted_as_names'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('n', v2)], user=lambda _loc, _all, _ctx, n: (
Import(_loc, n)
), normal_vars=['n'], list_vars=[]))
), (('try', ':', 'suite', 'try_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), (None, v2), ('b', v3), v4], user=lambda f, _loc, _all, es, ec, _ctx, b, l: (
Try(_loc, b, [kc(ks) for kc, ks in zip(ec, es)], l, f)
), normal_vars=['b', 'f', 'l'], list_vars=['ec', 'es']))
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('d', v2)])
), ((), 
(lambda ctx: [])
), (('+', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _loc, _all, e, _ctx: (
UnaryOp(_loc, UAdd(), e)
), normal_vars=['e'], list_vars=[]))
), (('-', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _loc, _all, e, _ctx: (
UnaryOp(_loc, USub(), e)
), normal_vars=['e'], list_vars=[]))
), (('~', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _loc, _all, e, _ctx: (
UnaryOp(_loc, Invert(), e)
), normal_vars=['e'], list_vars=[]))
), (('power',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _loc, _all, e, _ctx: (
e
), normal_vars=['e'], list_vars=[]))
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('(', 'decorator__2', ')'), 
(lambda ctx, v1, v2, v3: [('b', v1), v2, (None, v3)])
), ((), 
(lambda ctx: [])
), (('del', 'exprlist'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('l', v2)], user=lambda _loc, _all, l, _ctx: (
Delete(_loc, [set_context(e, Del()) for e in l[0]])
), normal_vars=['l'], list_vars=[]))
), (('class', 'NAME', 'classdef__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('n', v2), v3, (None, v4), ('b', v5)], user=lambda _loc, _all, a, _ctx, b, n: (
ast_for_classdef(_loc, n.value, a, b)
), normal_vars=['a', 'b', 'n'], list_vars=[]))
), (('typedargslist_item', 'typedargslist__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), (('testlist', 'eval_input__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda t, _loc, _all, _ctx: (
Expression(t)
), normal_vars=['t'], list_vars=[]))
), (('^', 'and_expr', 'xor_expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('testlist_star_expr__1', 'testlist_star_expr__2'), 
(lambda ctx, v1, v2: [v1, v2])
), ((), 
(lambda ctx: [])
), (('arglist_item', 'arglist__2'), 
(lambda ctx, v1, v2: [('a', v1), v2])
), (('import_name',), 
(combine_action(root=lambda ctx, v1: [('i', v1)], user=lambda i, _loc, _all, _ctx: (
i
), normal_vars=['i'], list_vars=[]))
), (('import_from',), 
(combine_action(root=lambda ctx, v1: [('i', v1)], user=lambda i, _loc, _all, _ctx: (
i
), normal_vars=['i'], list_vars=[]))
), (('subscriptlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, c, _all, s, _ctx: (

    s[0] if not c else
    ExtSlice(s) if any(isinstance(k, Slice) for k in s) else
    Index(Tuple(s[0].value.location, [k.value for k in s], Load()))
), normal_vars=[], list_vars=['c', 's']))
), (('nonlocal', 'NAME', 'nonlocal_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3], user=lambda _loc, _all, _ctx, n: (
Nonlocal(_loc, [t.value for t in n])
), normal_vars=[], list_vars=['n']))
), (('comp_for',), 
(lambda ctx, v1: [('f', v1)])
), (('testlist_comp__3',), 
(lambda ctx, v1: [v1])
), (('except_clause', ':', 'suite', 'try_stmt__2'), 
(lambda ctx, v1, v2, v3, v4: [('ec', v1), (None, v2), ('es', v3), v4])
), ((), 
(lambda ctx: [])
), (('(', 'classdef__2', ')'), 
(lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)])
), ((), 
(lambda ctx: [])
), (('typedargslist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('xor_expr', 'expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _loc, _all, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('raise', 'raise_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _loc, b, _all, a, _ctx: (
Raise(_loc, a, b)
), normal_vars=['a', 'b'], list_vars=[]))
), (('comp_op', 'expr', 'comparison__1'), 
(lambda ctx, v1, v2, v3: [('op', v1), ('r', v2), v3])
), ((), 
(lambda ctx: [])
), (('comp_for',), 
(lambda ctx, v1: [('c', v1)])
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('v', v2)])
), ((), 
(lambda ctx: [])
), (('varargslist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('a', v2)])
), ((), 
(lambda ctx: [])
), (('test', 'raise_stmt__2'), 
(lambda ctx, v1, v2: [('a', v1), v2])
), ((), 
(lambda ctx: [])
), (('from', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('f', v2)])
), (('testlist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('import_from__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, _all, d, s, _ctx, m, n: (
ImportFrom(_loc, '.'.join(t.value for t in m) if m else None, [alias('*', None)] if s else n, sum(len(t.type) for t in d))
), normal_vars=['m', 'n', 's'], list_vars=['d']))
), (('import_as_names__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('shift_expr', 'and_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _loc, _all, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('except_clause', ':', 'suite', 'try_stmt__2', 'try_stmt__3', 'try_stmt__4'), 
(lambda ctx, v1, v2, v3, v4, v5, v6: [('ec', v1), (None, v2), ('es', v3), v4, v5, v6])
), (('finally', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('f', v3)])
), (('test',), 
(lambda ctx, v1: [('u', v1)])
), ((), 
(lambda ctx: [])
), (('tfpdef', 'typedargslist_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda _loc, _all, d, a, _ctx: (
(None, a, d)
), normal_vars=['a', 'd'], list_vars=[]))
), (('*', 'typedargslist_item__2'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda _loc, _all, s, a, _ctx: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'tfpdef'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _loc, _all, s, a, _ctx: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('NAME', 'tfpdef__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda t, _loc, _all, _ctx, n: (
arg(_loc, n.value, t)
), normal_vars=['n', 't'], list_vars=[]))
), (('simple_stmt__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('global', 'NAME', 'global_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3], user=lambda _loc, _all, _ctx, n: (
Global(_loc, [t.value for t in n])
), normal_vars=[], list_vars=['n']))
), (('test',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), (('if_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('while_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('for_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('try_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('with_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('funcdef',), 
(lambda ctx, v1: [(None, v1)])
), (('classdef',), 
(lambda ctx, v1: [(None, v1)])
), (('decorated',), 
(lambda ctx, v1: [(None, v1)])
), (('break_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('continue_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('return_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('raise_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('yield_stmt',), 
(lambda ctx, v1: [(None, v1)])
), ((',', 'NAME', 'global_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('EMBEDSTAT',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _loc, _all, e, _ctx: (
EmbedStat(_loc, e.value)
), normal_vars=['e'], list_vars=[]))
), (('test', 'dictorsetmaker__6'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('decorators', 'decorated__1'), 
(combine_action(root=lambda ctx, v1, v2: [('d', v1), v2], user=lambda _loc, _all, d, cf, _ctx: (
cf._replace(decorator_list=d)
), normal_vars=['cf', 'd'], list_vars=[]))
), ((',', 'varargslist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), ((';', 'simple_stmt__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), ((',', 'dictorsetmaker__5'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('test', ':', 'test', 'dictorsetmaker__4'), 
(lambda ctx, v1, v2, v3, v4: [('k', v1), (None, v2), ('v', v3), v4])
), ((), 
(lambda ctx: [])
), (('STRING', 'atom__4'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('or_test',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _loc, _all, e, _ctx: (
e
), normal_vars=['e'], list_vars=[]))
), (('lambdef_nocond',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _loc, _all, e, _ctx: (
e
), normal_vars=['e'], list_vars=[]))
), (('decorator', 'decorators__1'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), ((), 
(lambda ctx: [])
), (('flow_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, _all, _ctx: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('arglist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, _all, a, _ctx: (
a
), normal_vars=[], list_vars=['a']))
), (('|', 'xor_expr', 'expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('dotted_name',), 
(lambda ctx, v1: [('m', v1)])
), (('import_from__3', 'import_from__4', 'import_from__5'), 
(lambda ctx, v1, v2, v3: [v1, v2, v3])
), (('or_test', 'test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda _loc, c, _all, a, _ctx, b: (
IfExp(a.location, b, a, c) if b else a
), normal_vars=['a', 'b', 'c'], list_vars=[]))
), (('lambdef',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _loc, _all, e, _ctx: (
e
), normal_vars=['e'], list_vars=[]))
), ((',', 'testlist__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('stmt', 'suite__1'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('exprlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('a', v2)])
), ((), 
(lambda ctx: [])
), (('test',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), (('dotted_name',), 
(lambda ctx, v1: [('m', v1)])
), ((), 
(lambda ctx: [])
), (('lambda', 'lambdef_nocond__1', ':', 'test_nocond'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), ('t', v4)], user=lambda t, _loc, _all, a, _ctx: (
Lambda(_loc, a or arguments(None, None, None, None, None, None), t)
), normal_vars=['a', 't'], list_vars=[]))
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('testlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), ((',', 'dictorsetmaker__7'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('test', 'subscript__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), ((':', 'subscript__5', 'subscript__6'), 
(lambda ctx, v1, v2, v3: [('d', v1), v2, v3])
), (('classdef',), 
(lambda ctx, v1: [('cf', v1)])
), (('funcdef',), 
(lambda ctx, v1: [('cf', v1)])
), (('factor', 'term__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _loc, _all, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('shift_expr__2', 'arith_expr', 'shift_expr__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('file_input__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, _all, s, _ctx: (
Module(flatten(s))
), normal_vars=[], list_vars=['s']))
), (('if', 'or_test', 'else', 'test'), 
(lambda ctx, v1, v2, v3, v4: [(None, v1), ('b', v2), (None, v3), ('c', v4)])
), ((), 
(lambda ctx: [])
), (('.', 'import_from__4'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), (('...', 'import_from__4'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), ((), 
(lambda ctx: [])
), (('varargslist_item', 'varargslist__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), (('except', 'except_clause__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda t, _loc, _all, _ctx, n: (
lambda body: ExceptHandler(_loc, t, n.value if n else None, body)
), normal_vars=['n', 't'], list_vars=[]))
), (('finally', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('f', v3)])
), ((), 
(lambda ctx: [])
), (('(', 'atom__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _loc, y, _all, l, _ctx: (

      y if y else
      Tuple(_loc, None, Load()) if not l else
      GeneratorExp(_loc, l[0][0], ast_for_comprehension(l[2])) if l[2] else
      ast_for_testlist(l[0], l[1])
), normal_vars=['l', 'y'], list_vars=[]))
), (('[', 'atom__2', ']'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _loc, _all, l, _ctx: (

      List(_loc, None, Load()) if not l else
      ListComp(_loc, l[0][0], ast_for_comprehension(l[2])) if l[2] else
      List(_loc, l[0], Load())
), normal_vars=['l'], list_vars=[]))
), (('{', 'atom__3', '}'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _loc, _all, d, _ctx: (
d._replace(location=_loc) if d else Dict(_loc, None, None)
), normal_vars=['d'], list_vars=[]))
), (('NAME',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _loc, _all, _ctx, n: (
Name(_loc, n.value, Load())
), normal_vars=['n'], list_vars=[]))
), (('NUMBER',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _loc, _all, _ctx, n: (
Num(_loc, n.value)
), normal_vars=['n'], list_vars=[]))
), (('STRING', 'atom__4'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda _loc, _all, s, _ctx: (
Str(_loc, ''.join(t.value for t in s))
), normal_vars=[], list_vars=['s']))
), (('...',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Ellipsis(_loc)
), normal_vars=[], list_vars=[]))
), (('None',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
NameConstant(_loc, None)
), normal_vars=[], list_vars=[]))
), (('True',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
NameConstant(_loc, True)
), normal_vars=[], list_vars=[]))
), (('False',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
NameConstant(_loc, False)
), normal_vars=[], list_vars=[]))
), (('EMBEDEXPR',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _loc, _all, e, _ctx: (
EmbedExp(_loc, e.value)
), normal_vars=['e'], list_vars=[]))
), (('expr',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), (('expr', 'comparison__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda _loc, _all, r, _ctx, l, op: (
l if not op else Compare(l.location, l, op, r)
), normal_vars=['l'], list_vars=['op', 'r']))
), (('decorator', 'decorators__1'), 
(combine_action(root=lambda ctx, v1, v2: [('d', v1), v2], user=lambda _loc, _all, d, _ctx: (
d
), normal_vars=[], list_vars=['d']))
), (('testlist_star_expr__1', 'testlist_star_expr__2'), 
(combine_action(root=lambda ctx, v1, v2: [v1, v2], user=lambda t, c, _loc, _all, _ctx: (
ast_for_testlist(t, c)
), normal_vars=[], list_vars=['c', 't']))
), (('sliceop',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), (('exprlist__2', 'exprlist__3'), 
(lambda ctx, v1, v2: [v1, v2])
), (('<<',), 
(lambda ctx, v1: [(None, v1)])
), (('>>',), 
(lambda ctx, v1: [(None, v1)])
), (('vfpdef',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('varargslist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, _all, l, _ctx: (
ast_for_arguments(l)
), normal_vars=[], list_vars=['l']))
), (('NAME', 'import_as_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _loc, _all, _ctx, a, n: (
alias(n.value, a.value if a else None)
), normal_vars=['a', 'n'], list_vars=[]))
), (('testlist_comp__1', 'testlist_comp__3'), 
(lambda ctx, v1, v2: [v1, v2])
), ((), 
(lambda ctx: [])
), (('dictorsetmaker',), 
(lambda ctx, v1: [('d', v1)])
), ((), 
(lambda ctx: [])
), (('comp_for',), 
(lambda ctx, v1: [('dc', v1)])
), (('dictorsetmaker__4',), 
(lambda ctx, v1: [v1])
), (('simple_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _loc, _all, s, _ctx: (
flatten(s)
), normal_vars=['s'], list_vars=[]))
), (('NEWLINE', 'INDENT', 'stmt', 'suite__1', 'DEDENT'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), (None, v2), ('s', v3), v4, (None, v5)], user=lambda _loc, _all, s, _ctx: (
flatten(s)
), normal_vars=[], list_vars=['s']))
), (('test',), 
(lambda ctx, v1: [('u', v1)])
), ((), 
(lambda ctx: [])
), (('tfpdef',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('from', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('b', v2)])
), ((), 
(lambda ctx: [])
), (('->', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('r', v2)])
), ((), 
(lambda ctx: [])
), (('argument',), 
(combine_action(root=lambda ctx, v1: [('a', v1)], user=lambda _loc, _all, a, _ctx: (
a
), normal_vars=['a'], list_vars=[]))
), (('*', 'test'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _loc, _all, s, a, _ctx: (
(s, a, None, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'test'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _loc, _all, s, a, _ctx: (
(s, a, None, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('simple_stmt__1', 'NEWLINE'), 
(combine_action(root=lambda ctx, v1, v2: [v1, (None, v2)], user=lambda _loc, _all, s, _ctx: (
s
), normal_vars=[], list_vars=['s']))
), (('break',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Break(_loc)
), normal_vars=[], list_vars=[]))
), (('for', 'exprlist', 'in', 'or_test', 'comp_for__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [('f', v1), ('t', v2), (None, v3), ('i', v4), v5], user=lambda t, _loc, _all, r, f, i, _ctx: (
[(f, t, i), r or []]
), normal_vars=['f', 'i', 'r', 't'], list_vars=[]))
), (('varargslist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('yield_expr',), 
(lambda ctx, v1: [('v', v1)])
), (('testlist_star_expr',), 
(lambda ctx, v1: [('v', v1)])
), (('stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, _all, _ctx: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('**', 'factor'), 
(lambda ctx, v1, v2: [(None, v1), ('f', v2)])
), ((), 
(lambda ctx: [])
), (('<',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Lt()
), normal_vars=[], list_vars=[]))
), (('>',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Gt()
), normal_vars=[], list_vars=[]))
), (('==',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Eq()
), normal_vars=[], list_vars=[]))
), (('>=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
GtE()
), normal_vars=[], list_vars=[]))
), (('<=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
LtE()
), normal_vars=[], list_vars=[]))
), (('!=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
NotEq()
), normal_vars=[], list_vars=[]))
), (('in',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
In()
), normal_vars=[], list_vars=[]))
), (('not', 'in'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), (None, v2)], user=lambda _loc, _all, _ctx: (
NotIn()
), normal_vars=[], list_vars=[]))
), (('is', 'comp_op__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _loc, _all, _ctx, n: (
IsNot() if n else Is()
), normal_vars=['n'], list_vars=[]))
), (('dotted_name', 'dotted_as_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _loc, _all, _ctx, a, n: (
alias('.'.join(t.value for t in n), a.value if a else None)
), normal_vars=['a', 'n'], list_vars=[]))
), (('test', 'testlist__2'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('testlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda t, c, _loc, _all, _ctx: (
ast_for_testlist(t, c)
), normal_vars=[], list_vars=['c', 't']))
), (('assert', 'test', 'assert_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('a', v2), v3], user=lambda _loc, b, _all, a, _ctx: (
Assert(_loc, a, b)
), normal_vars=['a', 'b'], list_vars=[]))
), ((',', 'arglist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('continue',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Continue(_loc)
), normal_vars=[], list_vars=[]))
), ((',', 'testlist_star_expr__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('typedargslist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('from', 'import_from__2', 'import', 'import_from__6'), 
(lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), v4])
), (('varargslist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('yield', 'yield_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda t, _loc, _all, _ctx, f: (
YieldFrom(_loc, f) if f else Yield(_loc, t)
), normal_vars=['f', 't'], list_vars=[]))
), (('if', 'test_nocond', 'comp_if__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [('i', v1), ('c', v2), v3], user=lambda _loc, c, _all, r, _ctx, i: (
[(i, c), r or []]
), normal_vars=['c', 'i', 'r'], list_vars=[]))
), ((',', 'NAME', 'nonlocal_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('return', 'return_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda t, _loc, _all, _ctx: (
Return(_loc, t)
), normal_vars=['t'], list_vars=[]))
), (('NAME', 'dotted_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda t, _loc, _all, _ctx: (
t
), normal_vars=[], list_vars=['t']))
), (('subscriptlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), ((',', 'typedargslist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('small_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, _all, _ctx: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('subscript__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, _all, d, s, u, _ctx, l: (
Slice(l, u, s) if d else Index(l)
), normal_vars=['d', 'l', 's', 'u'], list_vars=[]))
), (('yield_expr',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _loc, _all, e, _ctx: (
Expr(e.location, e)
), normal_vars=['e'], list_vars=[]))
), (('pass',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Pass(_loc)
), normal_vars=[], list_vars=[]))
), (('small_stmt', 'simple_stmt__2'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), (('compound_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, _all, _ctx: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('while', 'test', ':', 'suite', 'while_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('c', v2), (None, v3), ('s', v4), v5], user=lambda _loc, c, _all, e, s, _ctx: (
While(_loc, c, s, e)
), normal_vars=['c', 'e', 's'], list_vars=[]))
), (('term__2', 'factor', 'term__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('dotted_as_name', 'dotted_as_names__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _loc, _all, _ctx, n: (
n
), normal_vars=[], list_vars=['n']))
), (('and_test', 'or_test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda _loc, _all, l, _ctx: (
l[0] if len(l) == 1 else BoolOp(l[0].location, And(), l)
), normal_vars=[], list_vars=['l']))
), (('*',), 
(lambda ctx, v1: [('s', v1)])
), (('(', 'import_as_names', ')'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), (None, v3)])
), (('import_as_names',), 
(lambda ctx, v1: [('n', v1)])
), (('*',), 
(lambda ctx, v1: [(None, v1)])
), (('/',), 
(lambda ctx, v1: [(None, v1)])
), (('%',), 
(lambda ctx, v1: [(None, v1)])
), (('//',), 
(lambda ctx, v1: [(None, v1)])
), ((',', 'dotted_as_name', 'dotted_as_names__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('vfpdef', 'varargslist_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda _loc, _all, d, a, _ctx: (
(None, a, d)
), normal_vars=['a', 'd'], list_vars=[]))
), (('*', 'varargslist_item__2'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda _loc, _all, s, a, _ctx: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'vfpdef'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda _loc, _all, s, a, _ctx: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('=', 'expr_stmt__4', 'expr_stmt__3'), 
(lambda ctx, v1, v2, v3: [(None, v1), v2, v3])
), ((), 
(lambda ctx: [])
), (('.',), 
(lambda ctx, v1: [('d', v1)])
), (('...',), 
(lambda ctx, v1: [('d', v1)])
), ((',', 'import_as_names__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('term', 'arith_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _loc, _all, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('simple_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('compound_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('comp_for',), 
(combine_action(root=lambda ctx, v1: [('c', v1)], user=lambda _loc, c, _all, _ctx: (
c
), normal_vars=['c'], list_vars=[]))
), (('comp_if',), 
(combine_action(root=lambda ctx, v1: [('c', v1)], user=lambda _loc, c, _all, _ctx: (
c
), normal_vars=['c'], list_vars=[]))
), ((':', 'test', 'dictorsetmaker__3'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('v1', v2), v3])
), (('for', 'exprlist', 'in', 'testlist', ':', 'suite', 'for_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6, v7: [(None, v1), ('t', v2), (None, v3), ('c', v4), (None, v5), ('s', v6), v7], user=lambda t, c, _loc, e, s, _all, _ctx: (
For(_loc, set_context(Tuple(t[0][0].location, t[0], Store()) if t[1] else t[0][0], Store()), c, s, e)
), normal_vars=['c', 'e', 's', 't'], list_vars=[]))
), (('NAME',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _loc, _all, _ctx, n: (
arg(_loc, n.value, None)
), normal_vars=['n'], list_vars=[]))
), (('import_as_names__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _loc, _all, _ctx, n: (
n
), normal_vars=[], list_vars=['n']))
), (('def', 'NAME', 'parameters', 'funcdef__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6: [(None, v1), ('n', v2), ('p', v3), v4, (None, v5), ('b', v6)], user=lambda _loc, p, _all, r, _ctx, b, n: (
FunctionDef(_loc, n.value, p, b, [], r)
), normal_vars=['b', 'n', 'p', 'r'], list_vars=[]))
), (('test', 'dictorsetmaker__1'), 
(combine_action(root=lambda ctx, v1, v2: [('k1', v1), v2], user=lambda v1, _loc, _all, k1, v, dc, sc, s, _ctx, k: (

    DictComp(k1.location, k1, v1, ast_for_comprehension(dc)) if dc else
    SetComp(k1.location, k1, ast_for_comprehension(sc)) if sc else
    Dict(k1.location, [k1]+k, [v1]+v) if v1 else
    Set(k1.location, [k1]+s)
), normal_vars=['dc', 'k1', 'sc', 'v1'], list_vars=['k', 's', 'v']))
), (('trailer', 'power__1'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), ((), 
(lambda ctx: [])
), (('lambda', 'lambdef__1', ':', 'test'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), ('t', v4)], user=lambda t, _loc, _all, a, _ctx: (
Lambda(_loc, a or arguments(None, None, None, None, None, None), t)
), normal_vars=['a', 't'], list_vars=[]))
), (('test',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), (('dictorsetmaker__2',), 
(lambda ctx, v1: [v1])
), (('comp_for',), 
(lambda ctx, v1: [('sc', v1)])
), (('dictorsetmaker__6',), 
(lambda ctx, v1: [v1])
), (('&', 'shift_expr', 'and_expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('NEWLINE', 'eval_input__1'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('not',), 
(lambda ctx, v1: [('n', v1)])
), ((), 
(lambda ctx: [])
), (('.', 'NAME', 'dotted_name__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('t', v2), v3])
), ((), 
(lambda ctx: [])
), (('test', 'except_clause__2'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), ((), 
(lambda ctx: [])
), (('test', 'argument__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda t, c, _loc, _all, v, _ctx: (
(None, t, c, v)
), normal_vars=['c', 't', 'v'], list_vars=[]))
), (('@', 'dotted_name', 'decorator__1', 'NEWLINE'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), ('d', v2), v3, (None, v4)], user=lambda _loc, _all, d, a, _ctx, b: (
ast_for_decorator(_loc, ast_for_dotted_name(d), b, a)
), normal_vars=['a', 'b', 'd'], list_vars=[]))
), (('elif', 'test', ':', 'suite', 'if_stmt__1'), 
(lambda ctx, v1, v2, v3, v4, v5: [('t', v1), ('c', v2), (None, v3), ('s', v4), v5])
), ((), 
(lambda ctx: [])
), ((':', 'sliceop__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _loc, _all, s, _ctx: (
s
), normal_vars=['s'], list_vars=[]))
), (('(', 'trailer__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _loc, _all, a, _ctx: (
lambda left_expr: ast_for_call(_loc, left_expr, a)
), normal_vars=['a'], list_vars=[]))
), (('[', 'subscriptlist', ']'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('s', v2), (None, v3)], user=lambda _loc, _all, s, _ctx: (
lambda left_expr: Subscript(_loc, left_expr, s, Load())
), normal_vars=['s'], list_vars=[]))
), (('.', 'NAME'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('n', v2)], user=lambda _loc, _all, _ctx, n: (
lambda left_expr: Attribute(_loc, left_expr, n.value, Load())
), normal_vars=['n'], list_vars=[]))
), (('augassign', 'expr_stmt__2'), 
(lambda ctx, v1, v2: [('aa', v1), v2])
), (('expr_stmt__3',), 
(lambda ctx, v1: [v1])
), (('not', 'not_test'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _loc, _all, e, _ctx: (
UnaryOp(_loc, Not(), e)
), normal_vars=['e'], list_vars=[]))
), (('comparison',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _loc, _all, e, _ctx: (
e
), normal_vars=['e'], list_vars=[]))
), (('NEWLINE',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _loc, _all, _ctx: (
Interactive([])
), normal_vars=[], list_vars=[]))
), (('simple_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _loc, _all, s, _ctx: (
Interactive(flatten(s))
), normal_vars=['s'], list_vars=[]))
), (('compound_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _loc, _all, s, _ctx: (
Interactive(flatten(s))
), normal_vars=['s'], list_vars=[]))
), (('testlist_comp',), 
(lambda ctx, v1: [('l', v1)])
), ((), 
(lambda ctx: [])
), (('with', 'with_item', 'with_stmt__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('w', v2), v3, (None, v4), ('b', v5)], user=lambda _loc, b, _all, w, _ctx: (
With(_loc, w, b)
), normal_vars=['b'], list_vars=['w']))
), (('arith_expr__2', 'term', 'arith_expr__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
), ((',', 'testlist_comp__4'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('as', 'expr'), 
(lambda ctx, v1, v2: [(None, v1), ('e', v2)])
), ((), 
(lambda ctx: [])
), (('testlist_star_expr', 'expr_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda t, aa, _loc, _all, v, _ctx, av: (
ast_for_expr_stmt(t, aa, av, v)
), normal_vars=['aa', 'av', 't'], list_vars=['v']))
), (('atom', 'power__1', 'power__2'), 
(combine_action(root=lambda ctx, v1, v2, v3: [('a', v1), v2, v3], user=lambda t, _loc, _all, a, f, _ctx: (
ast_for_power(a, t, f)
), normal_vars=['a', 'f'], list_vars=['t']))
), ((',', 'with_item', 'with_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('w', v2), v3])
), ((), 
(lambda ctx: [])
), (('import_as_name', 'import_as_names__2'), 
(lambda ctx, v1, v2: [('n', v1), v2])
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('n', v2)])
), ((), 
(lambda ctx: [])
), (('arglist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), ((':', 'subscript__3', 'subscript__4'), 
(lambda ctx, v1, v2, v3: [('d', v1), v2, v3])
), ((), 
(lambda ctx: [])
), (('yield_expr',), 
(lambda ctx, v1: [('av', v1)])
), (('testlist',), 
(lambda ctx, v1: [('av', v1)])
), ((',', 'exprlist__4'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('testlist_comp__1', 'testlist_comp__2'), 
(combine_action(root=lambda ctx, v1, v2: [v1, v2], user=lambda t, c, _loc, _all, f, _ctx: (
(t, c, f)
), normal_vars=['f'], list_vars=['c', 't']))
)]
TABLE = {'not_test': {'[': 368, 'NAME': 368, 'None': 368, '-': 368, '(': 368, '...': 368, '{': 368, 'STRING': 368, '+': 368, 'False': 368, '~': 368, 'True': 368, 'NUMBER': 368, 'not': 367, 'EMBEDEXPR': 368}, 'atom__1': {'[': 1, '*': 1, 'NAME': 1, 'False': 1, 'None': 1, '(': 1, 'yield': 0, '-': 1, ')': 2, '...': 1, '{': 1, 'EMBEDEXPR': 1, '+': 1, 'lambda': 1, '~': 1, 'True': 1, 'NUMBER': 1, 'not': 1, 'STRING': 1}, 'tfpdef__1': {',': 4, ':': 3, ')': 4, '=': 4}, 'expr_stmt__1': {'|=': 365, '*=': 365, '>>=': 365, '^=': 365, '%=': 365, '&=': 365, '/=': 365, '-=': 365, 'NEWLINE': 366, '**=': 365, '<<=': 365, ';': 366, '=': 366, '//=': 365, '+=': 365}, 'augassign': {'*=': 8, '>>=': 15, '^=': 13, '%=': 10, '&=': 11, '/=': 9, '|=': 12, '-=': 7, '**=': 16, '<<=': 14, '//=': 17, '+=': 6}, 'dictorsetmaker__5': {'[': 154, 'NAME': 154, 'False': 154, 'None': 154, '(': 154, '-': 154, '...': 154, '{': 154, 'STRING': 154, '+': 154, 'lambda': 154, '~': 154, 'True': 154, 'NUMBER': 154, '}': 155, 'not': 154, 'EMBEDEXPR': 154}, 'small_stmt__1': {'[': 21, 'NAME': 21, 'None': 21, '-': 21, 'raise': 24, 'return': 24, 'EMBEDSTAT': 20, 'EMBEDEXPR': 21, 'lambda': 21, '~': 21, 'pass': 23, 'True': 21, 'global': 26, 'assert': 28, '*': 21, 'import': 25, 'yield': 24, '(': 21, 'continue': 24, '...': 21, 'break': 24, '{': 21, 'STRING': 21, '+': 21, 'nonlocal': 27, 'False': 21, 'NUMBER': 21, 'del': 22, 'not': 21, 'from': 25}, 'with_stmt': {'with': 374}, 'and_test__1': {'>>=': 30, ':': 30, '|=': 30, '^=': 30, 'and': 29, 'as': 30, ')': 30, '*=': 30, 'from': 30, '-=': 30, ',': 30, 'else': 30, 'or': 30, '&=': 30, '+=': 30, 'for': 30, '}': 30, '%=': 30, ']': 30, 'NEWLINE': 30, '/=': 30, 'if': 30, '**=': 30, '<<=': 30, ';': 30, '@@@': 30, '//=': 30, '=': 30}, 'and_test': {'[': 31, 'NAME': 31, 'None': 31, '(': 31, '-': 31, '...': 31, '{': 31, 'STRING': 31, '+': 31, 'False': 31, '~': 31, 'True': 31, 'NUMBER': 31, 'not': 31, 'EMBEDEXPR': 31}, 'arith_expr__2': {'-': 33, '+': 32}, 'or_test__1': {'|=': 35, ':': 35, '>>=': 35, '^=': 35, 'as': 35, '//=': 35, ')': 35, '*=': 35, 'from': 35, '-=': 35, ',': 35, 'else': 35, '}': 35, '&=': 35, '+=': 35, 'for': 35, 'or': 34, '%=': 35, ']': 35, 'NEWLINE': 35, '/=': 35, 'if': 35, '**=': 35, '<<=': 35, ';': 35, '@@@': 35, '=': 35}, 'subscript__6': {',': 38, ':': 37, ']': 38}, 'with_item': {'[': 39, 'NAME': 39, 'False': 39, 'None': 39, '(': 39, '-': 39, '...': 39, '{': 39, 'STRING': 39, '+': 39, 'lambda': 39, '~': 39, 'True': 39, 'NUMBER': 39, 'not': 39, 'EMBEDEXPR': 39}, 'typedargslist': {'*': 40, 'NAME': 40, '**': 40}, 'parameters': {'(': 41}, 'comp_for__1': {'for': 42, ',': 43, ']': 43, 'if': 42, '}': 43, ')': 43}, 'try_stmt__3': {'[': 45, 'NAME': 45, 'None': 45, 'if': 45, 'try': 45, 'EMBEDSTAT': 45, 'def': 45, 'EOF': 45, 'EMBEDEXPR': 45, 'raise': 45, '~': 45, 'True': 45, 'assert': 45, '*': 45, 'class': 45, 'yield': 45, '(': 45, 'continue': 45, '...': 45, 'STRING': 45, 'while': 45, 'False': 45, 'global': 45, 'not': 45, 'with': 45, '</python>': 45, 'for': 45, 'finally': 45, 'from': 45, 'pass': 45, 'else': 44, '@': 45, 'NEWLINE': 45, 'import': 45, 'nonlocal': 45, '-': 45, 'return': 45, 'break': 45, '{': 45, '+': 45, 'DEDENT': 45, 'lambda': 45, 'NUMBER': 45, 'del': 45}, 'star_expr': {'*': 46}, 'simple_stmt__1': {'[': 301, 'NAME': 301, '+': 301, 'None': 301, '(': 301, 'yield': 301, 'return': 301, 'EMBEDSTAT': 301, 'EMBEDEXPR': 301, 'raise': 301, '~': 301, 'True': 301, 'NUMBER': 301, 'assert': 301, '*': 301, 'import': 301, 'nonlocal': 301, '-': 301, 'continue': 301, '...': 301, 'break': 301, '{': 301, 'STRING': 301, 'pass': 301, 'lambda': 301, 'False': 301, 'global': 301, 'del': 301, 'not': 301, 'from': 301}, 'return_stmt__1': {'[': 50, 'NAME': 50, 'False': 50, 'None': 50, 'NEWLINE': 51, '(': 50, '-': 50, '...': 50, '{': 50, 'STRING': 50, '+': 50, 'lambda': 50, '~': 50, 'True': 50, 'NUMBER': 50, ';': 51, 'not': 50, 'EMBEDEXPR': 50}, 'power__1': {'[': 339, '**': 340, '>>=': 340, '^=': 340, '-': 340, '//=': 340, 'is': 340, ':': 340, '<=': 340, '^': 340, '}': 340, '*': 340, '<<': 340, '%=': 340, '&': 340, ']': 340, '(': 339, '/=': 340, 'if': 340, '>>': 340, '**=': 340, 'and': 340, '//': 340, '>=': 340, 'not': 340, '=': 340, '|=': 340, '/': 340, 'as': 340, '<<=': 340, '&=': 340, ')': 340, '-=': 340, 'in': 340, '%': 340, 'from': 340, '.': 339, 'else': 340, 'or': 340, '+=': 340, 'for': 340, '*=': 340, 'NEWLINE': 340, '>': 340, '|': 340, '<': 340, '==': 340, '!=': 340, '+': 340, ';': 340, '@@@': 340, ',': 340}, 'typedargslist_item__1': {',': 53, ')': 53, '=': 52}, 'typedargslist__2': {',': 295, ')': 296}, 'if_stmt': {'if': 54}, 'xor_expr': {'[': 55, 'NAME': 55, 'None': 55, '(': 55, '-': 55, '...': 55, '{': 55, 'STRING': 55, '+': 55, 'False': 55, '~': 55, 'True': 55, 'NUMBER': 55, 'EMBEDEXPR': 55}, 'subscriptlist__2': {',': 56, ']': 57}, 'assert_stmt__1': {'NEWLINE': 59, ',': 58, ';': 59}, 'shift_expr': {'[': 60, 'NAME': 60, 'None': 60, '(': 60, '-': 60, '...': 60, '{': 60, 'STRING': 60, '+': 60, 'False': 60, '~': 60, 'True': 60, 'NUMBER': 60, 'EMBEDEXPR': 60}, 'import_name': {'import': 61}, 'try_stmt': {'try': 62}, 'varargslist_item__1': {',': 64, ':': 64, '=': 63}, 'factor': {'[': 68, 'NAME': 68, 'None': 68, '-': 66, '(': 68, '...': 68, '{': 68, 'STRING': 68, '+': 65, 'False': 68, '~': 67, 'True': 68, 'NUMBER': 68, 'EMBEDEXPR': 68}, 'if_stmt__2': {'[': 70, 'with': 70, 'NAME': 70, '</python>': 70, 'None': 70, 'if': 70, '-': 70, 'DEDENT': 70, 'try': 70, 'return': 70, 'for': 70, 'EMBEDSTAT': 70, 'def': 70, 'EOF': 70, '@': 70, 'EMBEDEXPR': 70, 'raise': 70, '~': 70, 'True': 70, 'else': 69, 'NUMBER': 70, 'assert': 70, '*': 70, 'class': 70, 'NEWLINE': 70, 'import': 70, '{': 70, 'yield': 70, '(': 70, 'continue': 70, '...': 70, 'pass': 70, 'while': 70, 'STRING': 70, '+': 70, 'nonlocal': 70, 'lambda': 70, 'False': 70, 'global': 70, 'del': 70, 'break': 70, 'not': 70, 'from': 70}, 'decorator__1': {'NEWLINE': 72, '(': 71}, 'del_stmt': {'del': 73}, 'classdef': {'class': 74}, 'typedargslist__1': {'*': 75, 'NAME': 75, '**': 75}, 'eval_input': {'[': 76, 'NAME': 76, 'False': 76, 'None': 76, '(': 76, '-': 76, '...': 76, '{': 76, 'STRING': 76, '+': 76, 'lambda': 76, '~': 76, 'True': 76, 'NUMBER': 76, 'not': 76, 'EMBEDEXPR': 76}, 'xor_expr__1': {'|=': 78, ':': 78, '>>=': 78, 'if': 78, '//=': 78, 'as': 78, '<<=': 78, '%=': 78, ')': 78, 'for': 78, 'in': 78, '^=': 78, 'from': 78, '==': 78, '<=': 78, '^': 77, 'else': 78, '}': 78, '&=': 78, '@@@': 78, '+=': 78, '-=': 78, 'or': 78, '*=': 78, ']': 78, 'NEWLINE': 78, '>': 78, '|': 78, '/=': 78, '<': 78, ',': 78, 'is': 78, '!=': 78, '**=': 78, 'and': 78, ';': 78, '>=': 78, 'not': 78, '=': 78}, 'testlist_star_expr__3': {'[': 79, '|=': 80, 'NAME': 79, '**=': 80, '>>=': 80, '^=': 80, '-': 79, '%=': 80, '-=': 80, 'EMBEDEXPR': 79, 'lambda': 79, '~': 79, 'True': 79, '//=': 80, ';': 80, '&=': 80, '+=': 80, '*=': 80, '*': 79, 'NEWLINE': 80, 'None': 79, '(': 79, '/=': 80, '...': 79, '{': 79, 'STRING': 79, '+': 79, '<<=': 80, 'False': 79, 'NUMBER': 79, 'not': 79, '=': 80}, 'arglist__1': {'[': 81, '*': 81, 'NAME': 81, 'False': 81, '**': 81, 'None': 81, '(': 81, '-': 81, '...': 81, '{': 81, 'EMBEDEXPR': 81, '+': 81, 'lambda': 81, '~': 81, 'True': 81, 'NUMBER': 81, 'not': 81, 'STRING': 81}, 'dictorsetmaker__1': {'for': 345, ',': 346, ':': 344, '}': 346}, 'subscriptlist__1': {'[': 36, 'NAME': 36, 'False': 36, 'None': 36, '(': 36, 'lambda': 36, '-': 36, '...': 36, ':': 36, 'EMBEDEXPR': 36, '+': 36, '{': 36, '~': 36, 'True': 36, 'NUMBER': 36, 'not': 36, 'STRING': 36}, 'nonlocal_stmt': {'nonlocal': 85}, 'try_stmt__2': {'[': 89, 'NAME': 89, 'None': 89, 'if': 89, 'try': 89, 'EMBEDSTAT': 89, 'def': 89, 'EOF': 89, 'EMBEDEXPR': 89, 'raise': 89, '~': 89, 'True': 89, 'assert': 89, '*': 89, 'class': 89, 'yield': 89, '(': 89, 'continue': 89, '...': 89, 'STRING': 89, 'while': 89, 'except': 88, 'global': 89, 'not': 89, 'with': 89, '</python>': 89, 'for': 89, 'finally': 89, 'from': 89, 'False': 89, 'pass': 89, 'else': 89, '@': 89, 'NEWLINE': 89, 'import': 89, 'nonlocal': 89, '-': 89, 'return': 89, 'break': 89, '{': 89, '+': 89, 'DEDENT': 89, 'lambda': 89, 'NUMBER': 89, 'del': 89}, 'classdef__1': {':': 91, '(': 90}, 'typedargslist__3': {'*': 92, 'NAME': 92, ')': 93, '**': 92}, 'and_expr__1': {'|=': 348, ':': 348, '>>=': 348, 'if': 348, 'as': 348, '<<=': 348, '%=': 348, ')': 348, 'for': 348, 'in': 348, '^=': 348, 'from': 348, '==': 348, '<=': 348, '^': 348, 'else': 348, '}': 348, '&=': 348, '@@@': 348, '+=': 348, '-=': 348, 'or': 348, '*=': 348, '&': 347, ']': 348, 'NEWLINE': 348, '>': 348, '|': 348, '/=': 348, '<': 348, ',': 348, 'is': 348, '!=': 348, 'not': 348, '**=': 348, 'and': 348, ';': 348, '>=': 348, '//=': 348, '=': 348}, 'expr': {'[': 94, 'NAME': 94, 'None': 94, '(': 94, '-': 94, '...': 94, '{': 94, 'STRING': 94, '+': 94, 'False': 94, '~': 94, 'True': 94, 'NUMBER': 94, 'EMBEDEXPR': 94}, 'expr__1': {'|=': 165, ':': 165, '>>=': 165, 'if': 165, 'as': 165, '%=': 165, ')': 165, 'from': 165, 'for': 165, 'in': 165, '^=': 165, '//=': 165, 'and': 165, '==': 165, '<=': 165, 'is': 165, 'else': 165, '}': 165, '&=': 165, '@@@': 165, '+=': 165, '-=': 165, 'or': 165, '*=': 165, ']': 165, 'NEWLINE': 165, '>': 165, '|': 164, '/=': 165, '<': 165, ',': 165, '!=': 165, '**=': 165, '<<=': 165, ';': 165, '>=': 165, 'not': 165, '=': 165}, 'funcdef__1': {':': 248, '->': 247}, 'argument__1': {'for': 98, ',': 100, ')': 100, '=': 99}, 'lambdef__1': {':': 102, '*': 101, 'NAME': 101, '**': 101}, 'dotted_as_name__1': {'NEWLINE': 104, ',': 104, 'as': 103, ';': 104}, 'raise_stmt__1': {'[': 105, 'NAME': 105, 'False': 105, 'None': 105, 'NEWLINE': 106, '(': 105, '-': 105, '...': 105, '{': 105, 'STRING': 105, '+': 105, 'lambda': 105, '~': 105, 'True': 105, 'NUMBER': 105, ';': 106, 'not': 105, 'EMBEDEXPR': 105}, 'yield_expr__1': {'[': 108, 'NAME': 108, 'False': 108, 'None': 108, 'NEWLINE': 109, '-': 108, '(': 108, ')': 109, '=': 109, '...': 108, 'STRING': 108, '{': 108, 'EMBEDEXPR': 108, '+': 108, 'lambda': 108, '~': 108, 'True': 108, 'NUMBER': 108, ';': 109, 'not': 108, 'from': 107}, 'import_from': {'from': 110}, 'break_stmt': {'break': 253}, 'arglist_item': {'[': 249, '*': 250, 'NAME': 249, 'False': 249, '**': 251, 'None': 249, '(': 249, '-': 249, '...': 249, '{': 249, 'STRING': 249, '+': 249, 'lambda': 249, '~': 249, 'True': 249, 'NUMBER': 249, 'not': 249, 'EMBEDEXPR': 249}, 'testlist__1': {'[': 272, 'NAME': 272, 'False': 272, 'None': 272, '(': 272, '-': 272, '...': 272, '{': 272, 'STRING': 272, '+': 272, 'lambda': 272, '~': 272, 'True': 272, 'NUMBER': 272, 'not': 272, 'EMBEDEXPR': 272}, 'typedargslist_item': {'*': 119, 'NAME': 118, '**': 120}, 'tfpdef': {'NAME': 121}, 'simple_stmt__3': {'[': 122, 'NAME': 122, '+': 122, 'None': 122, '(': 122, 'yield': 122, 'return': 122, 'EMBEDSTAT': 122, 'EMBEDEXPR': 122, 'raise': 122, '~': 122, 'True': 122, 'NUMBER': 122, 'assert': 122, '*': 122, 'NEWLINE': 123, 'import': 122, 'nonlocal': 122, '-': 122, 'continue': 122, '...': 122, 'break': 122, '{': 122, 'STRING': 122, 'pass': 122, 'lambda': 122, 'False': 122, 'global': 122, 'del': 122, 'not': 122, 'from': 122}, 'dotted_as_names': {'NAME': 306}, 'arglist__2': {',': 277, ')': 278}, 'compound_stmt__1': {'for': 129, 'with': 131, 'def': 132, 'while': 128, '@': 134, 'class': 133, 'if': 127, 'try': 130}, 'flow_stmt__1': {'yield': 139, 'break': 135, 'raise': 138, 'continue': 136, 'return': 137}, 'argument': {'[': 357, 'NAME': 357, 'False': 357, 'None': 357, '(': 357, '-': 357, '...': 357, '{': 357, 'STRING': 357, '+': 357, 'lambda': 357, '~': 357, 'True': 357, 'NUMBER': 357, 'not': 357, 'EMBEDEXPR': 357}, 'global_stmt__1': {'NEWLINE': 141, ',': 140, ';': 141}, 'while_stmt__1': {'[': 143, 'with': 143, 'NAME': 143, '</python>': 143, 'None': 143, 'if': 143, '-': 143, 'DEDENT': 143, 'try': 143, 'return': 143, 'for': 143, 'EMBEDSTAT': 143, 'def': 143, 'EOF': 143, '@': 143, 'EMBEDEXPR': 143, 'raise': 143, '~': 143, 'True': 143, 'else': 142, 'NUMBER': 143, 'assert': 143, '*': 143, 'class': 143, 'NEWLINE': 143, 'import': 143, '{': 143, 'yield': 143, '(': 143, 'continue': 143, '...': 143, 'pass': 143, 'while': 143, 'STRING': 143, '+': 143, 'nonlocal': 143, 'lambda': 143, 'False': 143, 'global': 143, 'del': 143, 'break': 143, 'not': 143, 'from': 143}, 'embed_stmt': {'EMBEDSTAT': 144}, 'decorated': {'@': 147}, 'varargslist__2': {',': 148, ':': 149}, 'simple_stmt__2': {'NEWLINE': 151, ';': 150}, 'comp_if__1': {'for': 18, ',': 19, ']': 19, 'if': 18, '}': 19, ')': 19}, 'if_stmt__1': {'[': 360, 'NAME': 360, 'None': 360, 'if': 360, 'try': 360, 'EMBEDSTAT': 360, 'def': 360, 'EOF': 360, 'EMBEDEXPR': 360, 'raise': 360, '~': 360, 'True': 360, 'assert': 360, '*': 360, 'class': 360, 'yield': 360, '(': 360, 'continue': 360, '...': 360, 'STRING': 360, 'while': 360, 'False': 360, 'global': 360, 'not': 360, 'with': 360, '</python>': 360, 'for': 360, 'from': 360, 'elif': 359, 'pass': 360, 'else': 360, '@': 360, 'NEWLINE': 360, 'import': 360, 'nonlocal': 360, '-': 360, 'return': 360, 'break': 360, '{': 360, '+': 360, 'DEDENT': 360, 'lambda': 360, 'NUMBER': 360, 'del': 360}, 'decorators__1': {'@': 160, 'def': 161, 'class': 161}, 'or_test': {'[': 307, 'NAME': 307, 'None': 307, '(': 307, '-': 307, '...': 307, '{': 307, 'STRING': 307, '+': 307, 'False': 307, '~': 307, 'True': 307, 'NUMBER': 307, 'not': 307, 'EMBEDEXPR': 307}, 'arglist': {'[': 163, '*': 163, 'NAME': 163, 'False': 163, '**': 163, 'None': 163, '(': 163, '-': 163, '...': 163, '{': 163, 'EMBEDEXPR': 163, '+': 163, 'lambda': 163, '~': 163, 'True': 163, 'NUMBER': 163, 'not': 163, 'STRING': 163}, 'test__1': {'|=': 200, ':': 200, '>>=': 200, '^=': 200, 'as': 200, '//=': 200, ')': 200, '*=': 200, 'from': 200, '-=': 200, ',': 200, '}': 200, '&=': 200, '+=': 200, 'for': 200, '%=': 200, ']': 200, 'NEWLINE': 200, '/=': 200, 'if': 199, '**=': 200, '<<=': 200, ';': 200, '@@@': 200, '=': 200}, 'import_from__2': {'.': 167, 'NAME': 166, '...': 167}, 'test': {'[': 168, 'NAME': 168, 'None': 168, '(': 168, 'lambda': 169, '-': 168, '...': 168, '{': 168, 'STRING': 168, '+': 168, 'False': 168, '~': 168, 'True': 168, 'NUMBER': 168, 'not': 168, 'EMBEDEXPR': 168}, 'trailer': {'[': 363, '.': 364, '(': 362}, 'suite__1': {'[': 172, 'with': 172, 'DEDENT': 173, 'NAME': 172, 'None': 172, 'if': 172, '-': 172, 'lambda': 172, 'try': 172, 'return': 172, 'for': 172, 'EMBEDSTAT': 172, 'def': 172, 'while': 172, '@': 172, 'EMBEDEXPR': 172, 'raise': 172, '~': 172, 'True': 172, 'NUMBER': 172, 'assert': 172, '*': 172, 'class': 172, 'import': 172, 'yield': 172, '(': 172, 'continue': 172, '...': 172, 'pass': 172, '{': 172, 'STRING': 172, '+': 172, 'nonlocal': 172, 'False': 172, 'global': 172, 'del': 172, 'break': 172, 'not': 172, 'from': 172}, 'import_as_name__1': {'NEWLINE': 177, ',': 177, 'as': 176, ')': 177, ';': 177}, 'testlist_star_expr__1': {'[': 178, '*': 179, 'NAME': 178, 'False': 178, 'None': 178, '(': 178, '-': 178, '...': 178, '{': 178, 'STRING': 178, '+': 178, 'lambda': 178, '~': 178, 'True': 178, 'NUMBER': 178, 'not': 178, 'EMBEDEXPR': 178}, 'import_from__5': {'import': 181, 'NAME': 180}, 'lambdef_nocond': {'lambda': 182}, 'classdef__2': {'[': 183, '*': 183, 'NAME': 183, 'False': 183, '**': 183, 'None': 183, '(': 183, '-': 183, ')': 184, '...': 183, '{': 183, 'EMBEDEXPR': 183, '+': 183, 'lambda': 183, '~': 183, 'True': 183, 'NUMBER': 183, 'not': 183, 'STRING': 183}, 'for_stmt__1': {'[': 186, 'with': 186, 'NAME': 186, '</python>': 186, 'None': 186, 'if': 186, '-': 186, 'DEDENT': 186, 'try': 186, 'return': 186, 'for': 186, 'EMBEDSTAT': 186, 'def': 186, 'EOF': 186, '@': 186, 'EMBEDEXPR': 186, 'raise': 186, '~': 186, 'True': 186, 'else': 185, 'NUMBER': 186, 'assert': 186, '*': 186, 'class': 186, 'NEWLINE': 186, 'import': 186, '{': 186, 'yield': 186, '(': 186, 'continue': 186, '...': 186, 'pass': 186, 'while': 186, 'STRING': 186, '+': 186, 'nonlocal': 186, 'lambda': 186, 'False': 186, 'global': 186, 'del': 186, 'break': 186, 'not': 186, 'from': 186}, 'testlist__3': {'[': 187, 'NAME': 187, 'None': 187, '-': 187, ')': 188, ':': 188, 'EMBEDEXPR': 187, 'lambda': 187, '~': 187, 'True': 187, ';': 188, '@@@': 188, 'NEWLINE': 188, '(': 187, '...': 187, '{': 187, 'STRING': 187, '+': 187, 'False': 187, 'NUMBER': 187, 'not': 187, '=': 188}, 'dictorsetmaker__6': {',': 189, '}': 190}, 'subscript__1': {'[': 191, 'NAME': 191, 'False': 191, 'None': 191, '(': 191, '-': 191, '...': 191, ':': 192, '{': 191, 'STRING': 191, '+': 191, 'lambda': 191, '~': 191, 'True': 191, 'NUMBER': 191, 'not': 191, 'EMBEDEXPR': 191}, 'decorated__1': {'def': 194, 'class': 193}, 'term': {'[': 195, 'NAME': 195, 'None': 195, '(': 195, '-': 195, '...': 195, '{': 195, 'STRING': 195, '+': 195, 'False': 195, '~': 195, 'True': 195, 'NUMBER': 195, 'EMBEDEXPR': 195}, 'shift_expr__1': {'|=': 197, ':': 197, '>>=': 197, 'if': 197, 'as': 197, '//=': 197, ')': 197, ';': 197, 'for': 197, 'in': 197, 'from': 197, 'and': 197, '==': 197, '<=': 197, '^': 197, 'else': 197, '}': 197, '&=': 197, '@@@': 197, '+=': 197, '-=': 197, '<<': 196, '%=': 197, '&': 197, ']': 197, '*=': 197, 'NEWLINE': 197, '>': 197, '|': 197, '/=': 197, '<': 197, '^=': 197, 'or': 197, 'is': 197, '>>': 196, '**=': 197, '<<=': 197, '!=': 197, '>=': 197, ',': 197, 'not': 197, '=': 197}, 'file_input': {'[': 198, 'with': 198, 'NAME': 198, '</python>': 198, 'None': 198, 'if': 198, '-': 198, 'lambda': 198, 'try': 198, 'return': 198, 'for': 198, 'EMBEDSTAT': 198, 'def': 198, 'EOF': 198, '@': 198, 'EMBEDEXPR': 198, 'raise': 198, '~': 198, 'True': 198, 'NUMBER': 198, 'assert': 198, '*': 198, 'class': 198, 'NEWLINE': 198, 'import': 198, '{': 198, 'yield': 198, '(': 198, 'continue': 198, '...': 198, 'pass': 198, 'while': 198, 'STRING': 198, '+': 198, 'nonlocal': 198, 'False': 198, 'global': 198, 'del': 198, 'break': 198, 'not': 198, 'from': 198}, 'import_from__4': {'NAME': 203, '.': 201, 'import': 203, '...': 202}, 'varargslist__1': {'*': 204, 'NAME': 204, '**': 204}, 'except_clause': {'except': 205}, 'atom': {'[': 209, 'NAME': 211, 'STRING': 213, 'EMBEDEXPR': 218, '{': 210, 'True': 216, 'NUMBER': 212, 'None': 215, '(': 208, 'False': 217, '...': 214}, 'exprlist__2': {'[': 219, '*': 220, 'NAME': 219, 'None': 219, '(': 219, '-': 219, '...': 219, '{': 219, 'STRING': 219, '+': 219, 'False': 219, '~': 219, 'True': 219, 'NUMBER': 219, 'EMBEDEXPR': 219}, 'comparison': {'[': 221, 'NAME': 221, 'None': 221, '(': 221, '-': 221, '...': 221, '{': 221, 'STRING': 221, '+': 221, 'False': 221, '~': 221, 'True': 221, 'NUMBER': 221, 'EMBEDEXPR': 221}, 'decorators': {'@': 222}, 'testlist_star_expr': {'[': 223, '*': 223, 'NAME': 223, 'False': 223, 'None': 223, '(': 223, '-': 223, '...': 223, '{': 223, 'EMBEDEXPR': 223, '+': 223, 'lambda': 223, '~': 223, 'True': 223, 'NUMBER': 223, 'not': 223, 'STRING': 223}, 'subscript__4': {',': 225, ':': 224, ']': 225}, 'exprlist__1': {'[': 226, '*': 226, 'NAME': 226, 'None': 226, '-': 226, '(': 226, '...': 226, '{': 226, 'STRING': 226, '+': 226, 'False': 226, '~': 226, 'True': 226, 'NUMBER': 226, 'EMBEDEXPR': 226}, 'file_input__1': {'[': 48, 'with': 48, 'NAME': 48, '</python>': 49, 'None': 48, 'if': 48, '-': 48, 'lambda': 48, 'try': 48, 'return': 48, 'for': 48, 'EMBEDSTAT': 48, 'def': 48, 'while': 48, '@': 48, 'EMBEDEXPR': 48, 'raise': 48, '~': 48, 'True': 48, 'NUMBER': 48, 'assert': 48, '*': 48, 'EOF': 49, 'class': 48, 'NEWLINE': 47, 'import': 48, 'yield': 48, '(': 48, 'continue': 48, '...': 48, 'pass': 48, '{': 48, 'STRING': 48, '+': 48, 'nonlocal': 48, 'False': 48, 'global': 48, 'del': 48, 'break': 48, 'not': 48, 'from': 48}, 'varargslist_item__2': {',': 230, 'NAME': 229, ':': 230}, 'varargslist': {'*': 231, 'NAME': 231, '**': 231}, 'import_as_name': {'NAME': 232}, 'testlist_comp__4': {'[': 233, '*': 233, 'NAME': 233, 'False': 233, 'None': 233, '(': 233, ']': 234, '-': 233, ')': 234, '...': 233, '{': 233, 'EMBEDEXPR': 233, '+': 233, 'lambda': 233, '~': 233, 'True': 233, 'NUMBER': 233, 'not': 233, 'STRING': 233}, 'atom__3': {'[': 235, 'NAME': 235, 'False': 235, 'None': 235, '(': 235, '-': 235, '...': 235, '{': 235, 'STRING': 235, '+': 235, 'lambda': 235, '~': 235, 'True': 235, 'NUMBER': 235, '}': 236, 'not': 235, 'EMBEDEXPR': 235}, 'dictorsetmaker__3': {'for': 237, ',': 238, '}': 238}, 'suite': {'[': 239, 'NAME': 239, '+': 239, 'None': 239, '(': 239, 'yield': 239, 'return': 239, 'EMBEDSTAT': 239, 'EMBEDEXPR': 239, 'raise': 239, '~': 239, 'True': 239, 'NUMBER': 239, 'assert': 239, '*': 239, 'NEWLINE': 240, 'import': 239, 'nonlocal': 239, '-': 239, 'continue': 239, '...': 239, 'break': 239, '{': 239, 'STRING': 239, 'pass': 239, 'lambda': 239, 'False': 239, 'global': 239, 'del': 239, 'not': 239, 'from': 239}, 'subscript__5': {'[': 241, 'NAME': 241, 'False': 241, 'None': 241, '(': 241, ':': 242, ']': 242, '-': 241, '...': 241, ',': 242, '{': 241, 'STRING': 241, '+': 241, 'lambda': 241, '~': 241, 'True': 241, 'NUMBER': 241, 'not': 241, 'EMBEDEXPR': 241}, 'typedargslist_item__2': {',': 244, 'NAME': 243, ')': 244}, 'raise_stmt__2': {'NEWLINE': 246, ';': 246, 'from': 245}, 'comparison__1': {'|=': 97, ':': 97, '>>=': 97, 'if': 97, 'and': 97, 'as': 97, '%=': 97, ')': 97, 'for': 97, 'in': 96, '^=': 97, 'from': 97, '<=': 96, ',': 97, 'else': 97, '}': 97, '&=': 97, '@@@': 97, '+=': 97, '-=': 97, 'or': 97, 'is': 96, ']': 97, '*=': 97, 'NEWLINE': 97, '>': 96, '/=': 97, '<': 96, '==': 96, '!=': 96, 'not': 96, '**=': 97, '<<=': 97, ';': 97, '>=': 96, '//=': 97, '=': 97}, 'try_stmt__1': {'finally': 115, 'except': 114}, 'simple_stmt': {'[': 252, 'NAME': 252, '+': 252, 'None': 252, '(': 252, 'yield': 252, 'return': 252, 'EMBEDSTAT': 252, 'EMBEDEXPR': 252, 'raise': 252, '~': 252, 'True': 252, 'NUMBER': 252, 'assert': 252, '*': 252, 'import': 252, 'nonlocal': 252, '-': 252, 'continue': 252, '...': 252, 'break': 252, '{': 252, 'STRING': 252, 'pass': 252, 'lambda': 252, 'False': 252, 'global': 252, 'del': 252, 'not': 252, 'from': 252}, 'testlist_comp__1': {'[': 125, '*': 126, 'NAME': 125, 'False': 125, 'None': 125, '(': 125, '-': 125, '...': 125, '{': 125, 'STRING': 125, '+': 125, 'lambda': 125, '~': 125, 'True': 125, 'NUMBER': 125, 'not': 125, 'EMBEDEXPR': 125}, 'raise_stmt': {'raise': 95}, 'comp_for': {'for': 254}, 'lambdef_nocond__1': {':': 256, '*': 255, 'NAME': 255, '**': 255}, 'expr_stmt__4': {'[': 258, '*': 258, 'NAME': 258, 'False': 258, 'None': 258, '(': 258, 'yield': 257, '-': 258, '...': 258, '{': 258, 'EMBEDEXPR': 258, '+': 258, 'lambda': 258, '~': 258, 'True': 258, 'NUMBER': 258, 'not': 258, 'STRING': 258}, 'stmt': {'[': 259, 'with': 259, 'NAME': 259, 'None': 259, 'if': 259, '-': 259, 'lambda': 259, 'try': 259, 'return': 259, 'for': 259, 'EMBEDSTAT': 259, 'def': 259, 'while': 259, '@': 259, 'EMBEDEXPR': 259, 'raise': 259, '~': 259, 'True': 259, 'NUMBER': 259, 'assert': 259, '*': 259, 'class': 259, 'import': 259, 'yield': 259, '(': 259, 'continue': 259, '...': 259, 'pass': 259, '{': 259, 'STRING': 259, '+': 259, 'nonlocal': 259, 'False': 259, 'global': 259, 'del': 259, 'break': 259, 'not': 259, 'from': 259}, 'arith_expr__1': {'>>=': 376, 'if': 376, '-': 375, '//=': 376, '<=': 376, '^': 376, '}': 376, '<<': 376, '%=': 376, '&': 376, ']': 376, '/=': 376, '^=': 376, '>>': 376, '**=': 376, 'and': 376, '>=': 376, 'not': 376, '=': 376, '|=': 376, ':': 376, 'as': 376, '<<=': 376, '&=': 376, ')': 376, 'for': 376, 'in': 376, 'from': 376, 'is': 376, 'else': 376, 'or': 376, '+=': 376, '-=': 376, '*=': 376, 'NEWLINE': 376, '>': 376, '|': 376, '<': 376, '==': 376, '!=': 376, '+': 375, ';': 376, '@@@': 376, ',': 376}, 'comp_op': {'==': 264, '!=': 267, 'in': 268, '<=': 266, 'is': 270, '>=': 265, '>': 263, 'not': 269, '<': 262}, 'dotted_as_name': {'NAME': 271}, 'subscript__3': {'[': 116, 'NAME': 116, 'False': 116, 'None': 116, '(': 116, ':': 117, ']': 117, '-': 116, '...': 116, ',': 117, '{': 116, 'STRING': 116, '+': 116, 'lambda': 116, '~': 116, 'True': 116, 'NUMBER': 116, 'not': 116, 'EMBEDEXPR': 116}, 'trailer__1': {'[': 273, '*': 273, 'NAME': 273, 'False': 273, '**': 273, 'None': 273, '(': 273, '-': 273, ')': 274, '...': 273, '{': 273, 'EMBEDEXPR': 273, '+': 273, 'lambda': 273, '~': 273, 'True': 273, 'NUMBER': 273, 'not': 273, 'STRING': 273}, 'testlist': {'[': 275, 'NAME': 275, 'False': 275, 'None': 275, '(': 275, '-': 275, '...': 275, '{': 275, 'STRING': 275, '+': 275, 'lambda': 275, '~': 275, 'True': 275, 'NUMBER': 275, 'not': 275, 'EMBEDEXPR': 275}, 'assert_stmt': {'assert': 276}, 'continue_stmt': {'continue': 279}, 'dictorsetmaker__4': {',': 152, '}': 153}, 'parameters__1': {'*': 282, 'NAME': 282, ')': 283, '**': 282}, 'import_from__1': {'from': 284}, 'varargslist__3': {':': 286, '*': 285, 'NAME': 285, '**': 285}, 'yield_expr': {'yield': 287}, 'comp_if': {'if': 288}, 'nonlocal_stmt__1': {'NEWLINE': 290, ',': 289, ';': 290}, 'return_stmt': {'return': 291}, 'dotted_name': {'NAME': 292}, 'import_as_names__3': {'NEWLINE': 112, ';': 112, 'NAME': 111, ')': 112}, 'small_stmt': {'[': 297, 'NAME': 297, '+': 297, 'None': 297, '(': 297, 'yield': 297, 'return': 297, 'EMBEDSTAT': 297, 'EMBEDEXPR': 297, 'raise': 297, '~': 297, 'True': 297, 'NUMBER': 297, 'assert': 297, '*': 297, 'import': 297, 'nonlocal': 297, '-': 297, 'continue': 297, '...': 297, 'break': 297, '{': 297, 'STRING': 297, 'pass': 297, 'lambda': 297, 'False': 297, 'global': 297, 'del': 297, 'not': 297, 'from': 297}, 'subscript': {'[': 298, 'NAME': 298, 'False': 298, 'None': 298, '(': 298, 'lambda': 298, '-': 298, '...': 298, ':': 298, 'EMBEDEXPR': 298, '+': 298, '{': 298, '~': 298, 'True': 298, 'NUMBER': 298, 'not': 298, 'STRING': 298}, 'yield_stmt': {'yield': 299}, 'subscriptlist': {'[': 84, 'NAME': 84, 'False': 84, 'None': 84, '(': 84, 'lambda': 84, '-': 84, '...': 84, ':': 84, 'EMBEDEXPR': 84, '+': 84, '{': 84, '~': 84, 'True': 84, 'NUMBER': 84, 'not': 84, 'STRING': 84}, 'testlist_comp__2': {'for': 86, ',': 87, ')': 87, ']': 87}, 'compound_stmt': {'with': 302, '@': 302, 'def': 302, 'while': 302, 'for': 302, 'class': 302, 'if': 302, 'try': 302}, 'while_stmt': {'while': 303}, 'term__1': {'>>=': 305, '^=': 305, '|': 305, 'as': 305, ':': 305, '<=': 305, '^': 305, '}': 305, '*': 304, '<<': 305, '%=': 305, '&': 305, ']': 305, '/=': 305, 'if': 305, '>>': 305, '**=': 305, 'and': 305, '//': 304, '>=': 305, 'not': 305, '=': 305, '|=': 305, '/': 304, '//=': 305, '<<=': 305, '&=': 305, ')': 305, '-=': 305, 'in': 305, '%': 304, 'from': 305, 'is': 305, 'else': 305, 'or': 305, '+=': 305, 'for': 305, '*=': 305, 'NEWLINE': 305, '>': 305, '-': 305, '<': 305, '==': 305, '!=': 305, '+': 305, ';': 305, '@@@': 305, ',': 305}, 'global_stmt': {'global': 124}, 'flow_stmt': {'raise': 162, 'break': 162, 'yield': 162, 'continue': 162, 'return': 162}, 'import_from__6': {'*': 308, 'NAME': 310, '(': 309}, 'exprlist__4': {'[': 174, '*': 174, 'NAME': 174, 'None': 174, 'NEWLINE': 175, '-': 174, '(': 174, '...': 174, 'in': 175, '{': 174, 'STRING': 174, '+': 174, 'False': 174, '~': 174, 'True': 174, 'NUMBER': 174, ';': 175, 'EMBEDEXPR': 174}, 'dotted_as_names__1': {'NEWLINE': 316, ',': 315, ';': 316}, 'varargslist_item': {'*': 318, 'NAME': 317, '**': 319}, 'decorator__2': {'[': 320, '*': 320, 'NAME': 320, 'False': 320, '**': 320, 'None': 320, '(': 320, '-': 320, ')': 321, '...': 320, '{': 320, 'EMBEDEXPR': 320, '+': 320, 'lambda': 320, '~': 320, 'True': 320, 'NUMBER': 320, 'not': 320, 'STRING': 320}, 'expr_stmt__3': {'NEWLINE': 323, ';': 323, '=': 322}, 'import_from__3': {'.': 324, '...': 325}, 'import_as_names__2': {'NEWLINE': 327, ',': 326, ')': 327, ';': 327}, 'testlist_star_expr__2': {'>>=': 281, '%=': 281, '|=': 281, '^=': 281, 'NEWLINE': 281, '*=': 281, '//=': 281, '/=': 281, '-=': 281, ',': 280, '**=': 281, '<<=': 281, ';': 281, '+=': 281, '&=': 281, '=': 281}, 'arith_expr': {'[': 328, 'NAME': 328, 'None': 328, '(': 328, '-': 328, '...': 328, '{': 328, 'STRING': 328, '+': 328, 'False': 328, '~': 328, 'True': 328, 'NUMBER': 328, 'EMBEDEXPR': 328}, 'stmt__1': {'[': 329, 'with': 330, 'NAME': 329, '+': 329, 'None': 329, 'if': 330, '(': 329, 'yield': 329, '@': 330, 'return': 329, 'for': 330, 'EMBEDSTAT': 329, 'def': 330, 'while': 330, 'try': 330, 'EMBEDEXPR': 329, 'raise': 329, '~': 329, 'True': 329, 'NUMBER': 329, 'assert': 329, '*': 329, 'class': 330, 'import': 329, 'nonlocal': 329, '-': 329, 'continue': 329, '...': 329, 'break': 329, '{': 329, 'STRING': 329, 'pass': 329, 'lambda': 329, 'False': 329, 'global': 329, 'del': 329, 'not': 329, 'from': 329}, 'comp_iter': {'for': 331, 'if': 332}, 'lambdef': {'lambda': 341}, 'dictorsetmaker__2': {':': 333}, 'import_stmt': {'import': 82, 'from': 83}, 'dotted_name__1': {'.': 353, ';': 354, 'NEWLINE': 354, 'import': 354, 'as': 354, ',': 354, '(': 354}, 'vfpdef': {'NAME': 335}, 'import_as_names': {'NAME': 336}, 'funcdef': {'def': 337}, 'dictorsetmaker': {'[': 338, 'NAME': 338, 'False': 338, 'None': 338, '(': 338, '-': 338, '...': 338, '{': 338, 'STRING': 338, '+': 338, 'lambda': 338, '~': 338, 'True': 338, 'NUMBER': 338, 'not': 338, 'EMBEDEXPR': 338}, 'and_expr': {'[': 113, 'NAME': 113, 'None': 113, '(': 113, '-': 113, '...': 113, '{': 113, 'STRING': 113, '+': 113, 'False': 113, '~': 113, 'True': 113, 'NUMBER': 113, 'EMBEDEXPR': 113}, 'for_stmt': {'for': 334}, 'sliceop__1': {'[': 342, 'NAME': 342, 'False': 342, 'None': 342, '(': 342, ']': 343, '-': 342, '...': 342, ',': 343, '{': 342, 'STRING': 342, '+': 342, 'lambda': 342, '~': 342, 'True': 342, 'NUMBER': 342, 'not': 342, 'EMBEDEXPR': 342}, 'sliceop': {':': 361}, 'exprlist': {'[': 5, '*': 5, 'NAME': 5, 'None': 5, '-': 5, '(': 5, '...': 5, '{': 5, 'STRING': 5, '+': 5, 'False': 5, '~': 5, 'True': 5, 'NUMBER': 5, 'EMBEDEXPR': 5}, 'eval_input__1': {'NEWLINE': 349, '@@@': 350}, 'comp_op__1': {'[': 352, 'NAME': 352, 'None': 352, '-': 352, '(': 352, '...': 352, '{': 352, 'STRING': 352, '+': 352, 'False': 352, '~': 352, 'True': 352, 'NUMBER': 352, 'not': 351, 'EMBEDEXPR': 352}, 'subscriptlist__3': {'[': 293, 'NAME': 293, 'False': 293, ']': 294, 'None': 293, '(': 293, 'lambda': 293, '-': 293, '...': 293, ':': 293, 'EMBEDEXPR': 293, '+': 293, '{': 293, '~': 293, 'True': 293, 'NUMBER': 293, 'not': 293, 'STRING': 293}, 'except_clause__1': {'[': 355, 'NAME': 355, 'False': 355, 'None': 355, '(': 355, '-': 355, '...': 355, ':': 356, '{': 355, 'STRING': 355, '+': 355, 'lambda': 355, '~': 355, 'True': 355, 'NUMBER': 355, 'not': 355, 'EMBEDEXPR': 355}, 'shift_expr__2': {'<<': 227, '>>': 228}, 'decorator': {'@': 358}, 'atom__4': {'[': 157, '**': 157, '>>=': 157, '^=': 157, '-': 157, 'as': 157, ':': 157, '<=': 157, '^': 157, '}': 157, '*': 157, '<<': 157, '%=': 157, '&': 157, ']': 157, '(': 157, '/=': 157, '.': 157, '>>': 157, 'STRING': 156, '**=': 157, 'and': 157, '//': 157, '>=': 157, 'not': 157, '=': 157, '|=': 157, '/': 157, '//=': 157, '<<=': 157, '&=': 157, ')': 157, '-=': 157, 'in': 157, '%': 157, 'from': 157, 'is': 157, 'else': 157, 'or': 157, '+=': 157, 'for': 157, '*=': 157, 'if': 157, 'NEWLINE': 157, '>': 157, '|': 157, '<': 157, '==': 157, '!=': 157, '+': 157, ';': 157, '@@@': 157, ',': 157}, 'test_nocond': {'[': 158, 'NAME': 158, 'None': 158, '(': 158, 'lambda': 159, '-': 158, '...': 158, '{': 158, 'STRING': 158, '+': 158, 'False': 158, '~': 158, 'True': 158, 'NUMBER': 158, 'not': 158, 'EMBEDEXPR': 158}, 'testlist__2': {',': 170, ':': 171, 'NEWLINE': 171, ';': 171, '@@@': 171, ')': 171, '=': 171}, 'term__2': {'%': 313, '*': 311, '/': 312, '//': 314}, 'try_stmt__4': {'[': 207, 'with': 207, 'NAME': 207, '</python>': 207, 'None': 207, 'if': 207, '-': 207, 'DEDENT': 207, 'try': 207, 'return': 207, 'for': 207, 'finally': 206, 'def': 207, 'EOF': 207, '@': 207, 'EMBEDEXPR': 207, 'raise': 207, '~': 207, 'True': 207, 'NUMBER': 207, 'assert': 207, '*': 207, 'class': 207, 'EMBEDSTAT': 207, 'NEWLINE': 207, 'import': 207, '{': 207, 'yield': 207, '(': 207, 'continue': 207, '...': 207, 'pass': 207, 'while': 207, 'STRING': 207, '+': 207, 'nonlocal': 207, 'lambda': 207, 'False': 207, 'global': 207, 'del': 207, 'break': 207, 'not': 207, 'from': 207}, 'single_input': {'[': 370, 'with': 371, 'NAME': 370, 'None': 370, 'if': 371, '(': 370, 'yield': 370, '@': 371, 'return': 370, 'for': 371, 'EMBEDSTAT': 370, 'def': 371, 'while': 371, 'try': 371, 'EMBEDEXPR': 370, 'raise': 370, '~': 370, 'True': 370, 'NUMBER': 370, 'assert': 370, '*': 370, 'class': 371, 'NEWLINE': 369, 'import': 370, 'nonlocal': 370, '-': 370, 'continue': 370, '...': 370, 'pass': 370, '{': 370, 'STRING': 370, '+': 370, 'lambda': 370, 'False': 370, 'global': 370, 'del': 370, 'break': 370, 'not': 370, 'from': 370}, 'atom__2': {'[': 372, '*': 372, 'NAME': 372, 'False': 372, 'None': 372, '(': 372, ']': 373, '-': 372, '...': 372, '{': 372, 'EMBEDEXPR': 372, '+': 372, 'lambda': 372, '~': 372, 'True': 372, 'NUMBER': 372, 'not': 372, 'STRING': 372}, 'dictorsetmaker__7': {'[': 145, 'NAME': 145, 'False': 145, 'None': 145, '(': 145, '-': 145, '...': 145, '{': 145, 'STRING': 145, '+': 145, 'lambda': 145, '~': 145, 'True': 145, 'NUMBER': 145, '}': 146, 'not': 145, 'EMBEDEXPR': 145}, 'power__2': {'**': 260, '>>=': 261, '^=': 261, '|': 261, '//=': 261, ':': 261, '<=': 261, '^': 261, '}': 261, '*': 261, '%': 261, '%=': 261, '&': 261, ']': 261, '/=': 261, 'if': 261, '>>': 261, '**=': 261, 'and': 261, '//': 261, '>=': 261, 'not': 261, '=': 261, '|=': 261, '/': 261, 'as': 261, '<<=': 261, '&=': 261, ')': 261, '-=': 261, 'in': 261, '<<': 261, 'from': 261, 'is': 261, 'else': 261, 'or': 261, '+=': 261, 'for': 261, '*=': 261, 'NEWLINE': 261, '>': 261, '-': 261, '<': 261, '==': 261, '!=': 261, '+': 261, ';': 261, '@@@': 261, ',': 261}, 'testlist_comp__3': {',': 377, ')': 378, ']': 378}, 'with_item__1': {',': 380, ':': 380, 'as': 379}, 'expr_stmt': {'[': 381, '*': 381, 'NAME': 381, 'False': 381, 'None': 381, '(': 381, '-': 381, '...': 381, '{': 381, 'EMBEDEXPR': 381, '+': 381, 'lambda': 381, '~': 381, 'True': 381, 'NUMBER': 381, 'not': 381, 'STRING': 381}, 'power': {'[': 382, 'STRING': 382, 'NAME': 382, '{': 382, 'None': 382, 'True': 382, 'NUMBER': 382, 'EMBEDEXPR': 382, '(': 382, 'False': 382, '...': 382}, 'with_stmt__1': {',': 383, ':': 384}, 'pass_stmt': {'pass': 300}, 'import_as_names__1': {'NAME': 385}, 'except_clause__2': {':': 387, 'as': 386}, 'arglist__3': {'[': 388, '*': 388, 'NAME': 388, 'False': 388, '**': 388, 'None': 388, '(': 388, '-': 388, ')': 389, '...': 388, '{': 388, 'EMBEDEXPR': 388, '+': 388, 'lambda': 388, '~': 388, 'True': 388, 'NUMBER': 388, 'not': 388, 'STRING': 388}, 'subscript__2': {',': 391, ':': 390, ']': 391}, 'expr_stmt__2': {'[': 393, 'NAME': 393, 'False': 393, 'None': 393, '(': 393, 'yield': 392, '-': 393, '...': 393, '{': 393, 'EMBEDEXPR': 393, '+': 393, 'lambda': 393, '~': 393, 'True': 393, 'NUMBER': 393, 'not': 393, 'STRING': 393}, 'exprlist__3': {'NEWLINE': 395, ',': 394, ';': 395, 'in': 395}, 'testlist_comp': {'[': 396, '*': 396, 'NAME': 396, 'False': 396, 'None': 396, '(': 396, '-': 396, '...': 396, '{': 396, 'EMBEDEXPR': 396, '+': 396, 'lambda': 396, '~': 396, 'True': 396, 'NUMBER': 396, 'not': 396, 'STRING': 396}}
