
from context import flatten
from toypython.lex import tokenizer
from toypython.ast import *

def set_context(expr, context):
    if isinstance(expr, (Attribute, Subscript, Name)):
        return expr._replace(ctx=context)

    if isinstance(expr, Starred):
        return Starred(expr.location, set_context(expr.value, context), context)

    if isinstance(expr, List) or (isinstance(expr, Tuple) and expr.elts):
        new_elts = [set_context(e, context) for e in expr.elts]
        return expr.__class__(expr.location, new_elts, context)

    print(expr)
    name = expr.__class__.__name__
    if context == Del():
        raise ParseError(expr.location, "can't delete %s" % name)
    else:
        raise ParseError(expr.location, "can't assign to %s" % name)

def ast_for_testlist(tests, commas):
    return tests[0] if not commas else Tuple(tests[0].location, tests, Load())

def ast_for_arguments(items):
    posargs = []
    vararg = None
    after_vararg = False
    kwonlyargs = []
    kwdefaults = []
    kwarg = None
    posdefaults = []

    if items[-1][0] and items[-1][0].type == '**':
        kwarg = items[-1][1]
        items = items[:-1]

    if items[-1][0] and items[-1][0].type == '*' and not items[-1][1]:
        raise ParseError(items[-1][0].location, "named arguments must follow bare *")

    for (star_token, fpdef, default) in items:
        if star_token and star_token.type == '**':
            raise ParseError(star_token.location, "**kwargs must be last in arguments list")

        if star_token and after_vararg:
            raise ParseError(star_token.location, "*args can only be given once")

        if star_token:
            after_vararg = True
            vararg = fpdef
        elif after_vararg:
            kwonlyargs.append(fpdef)
            kwdefaults.append(default)
        else:
            if default:
                posdefaults.append(default)
            elif posdefaults:
                raise ParseError(fpdef.location, "non-default argument follows default argument")
            posargs.append(fpdef)

    return arguments(posargs, vararg, kwonlyargs, kwdefaults, kwarg, posdefaults)

def ast_for_dotted_name(name_tokens):
    rv = None
    for name_token in name_tokens:
        if rv:
            rv = Attribute(rv.location, rv, name_token.value, Load())
        else:
            rv = Name(name_token.location, name_token.value, Load())
    return rv

def ast_for_decorator(loc, name_expr, have_parens, arglist):
    if not have_parens: return name_expr
    return ast_for_call(loc, name_expr, arglist)

def ast_for_comprehension(root_comp_for):
    comps = []

    for item in flatten(root_comp_for):
        if item[0].type == 'for':
            token, (exprlist, exprlist_has_comma), iter = item
            if exprlist_has_comma:
                target = Tuple(exprlist[0].location, exprlist, Store())
            else:
                target = exprlist[0]
            target = set_context(target, Store())
            comps.append(comprehension(target, iter, []))
        else:
            token, cond = item
            comps[-1].ifs.append(cond)

    return comps

token_type_to_operator_class = {
    '|': BitOr,
    '^': BitXor,
    '&': BitAnd,
    '<<': LShift,
    '>>': RShift,
    '+': Add,
    '-': Sub,
    '*': Mult,
    '/': Div,
    '//': FloorDiv,
    '%': Mod,
}

def ast_for_binop(symbols):
    symbols = iter(symbols)
    left = next(symbols)
    while True:
        try:
            op_token = next(symbols)
            op_class = token_type_to_operator_class[op_token.type]
            right = next(symbols)
            left = BinOp(op_token.location, left, op_class(), right)
        except StopIteration:
            return left

def ast_for_power(atom, trailers, factor):
    for trailer in trailers:
        atom = trailer(atom)._replace(location=atom.location)
    if factor:
        atom = BinOp(atom.location, atom, Pow(), factor)
    return atom

def ast_for_call(loc, func, arglist):
    if not arglist: return Call(loc, func, None, None, None, None)

    args = []
    keywords = []
    vararg = None
    kwarg = None
    keyword_names = set()

    star_token, test, root_comp_for, kwvalue = arglist[0]
    if len(arglist) == 1 and root_comp_for:
        value = GeneratorExp(test.location, test, ast_for_comprehension(root_comp_for))
        arglist = [(None, value, None, None)]

    for (star_token, test, root_comp_for, kwvalue) in arglist:
        if root_comp_for:
            raise ParseError(test, "Generator expression must be parenthesized "
                             "if not sole argument")

    if arglist[-1][0] and arglist[-1][0].type == '**':
        kwarg = arglist[-1][1]
        arglist = arglist[:-1]

    for (star_token, test, root_comp_for, kwvalue) in arglist:
        if star_token and star_token.type == '*':
            if vararg:
                raise ParseError(star_token.location, "*expression can only be given once")
            vararg = test
        elif star_token and star_token.type == '**':
            raise ParseError(star_token.location, "**expression must be last in arguments list")
        elif not kwvalue:
            if keywords:
                raise ParseError(test.location, "non-keyword arg after keyword arg")
            if vararg:
                raise ParseError(test.location, "only named arguments may follow *expression")
            args.append(test)
        elif isinstance(test, Lambda):
            raise ParseError(test.location, "lambda cannot contain assignment")
        elif not isinstance(test, Name):
            raise ParseError(test.location, "keyword can't be an expression")
        elif test.id in keyword_names:
            raise ParseError(test.location, "keyword argument repeated")
        else:
            keyword_names.add(test.id)
            keywords.append(keyword(test.id, kwvalue))

    return Call(loc, func, args, keywords, vararg, kwarg)

def ast_for_expr_stmt(base, augassign_op, augassign_value, assignments):
    if augassign_op:
        base = set_context(base, Store())
        if not isinstance(base, (Name, Attribute, Subscript)):
            raise ParseError(base.location, "illegal expression for augmented assignment")
        return AugAssign(base.location, base, augassign_op, augassign_value)

    if assignments:
        value = assignments[-1]
        targets = []
        for target in [base] + assignments[:-1]:
            if isinstance(target, (Yield, YieldFrom)):
                raise ParseError(target.location, "assignment to yield expression not possible")
            targets.append(set_context(target, Store()))
        return Assign(base.location, targets, value)

    return Expr(base.location, base)

def ast_for_if_stmt(tokens, conds, suites, else_suite):
    for token, cond, suite in reversed(list(zip(tokens, conds, suites))):
        else_suite = [If(token.location, cond, suite, else_suite or [])]
    return else_suite

def ast_for_classdef(loc, name, arglist, body):
    dummy_call = ast_for_call(loc, None, arglist)
    return ClassDef(loc, name, dummy_call.args, dummy_call.keywords,
                    dummy_call.starargs, dummy_call.kwargs, body, [])



from context import combine_action

from context import ParseError

def parse(ctx, start_nt=None, close_with=None):
    stack = [close_with or 'EOF', start_nt or 'file_input']
    results = []
    tok_iter = tokenizer(ctx, close_with)
    token = next(tok_iter)

    while True:
        stack_top = stack[-1]

        if isinstance(stack_top, int):
            rule, action = RULES[stack_top]
            stack.pop()
            n = len(rule)
            args = results[len(results) - n:]
            del results[len(results) - n:]
            results.append(action(ctx, *args))

        elif stack_top not in TABLE:
            if token.type != stack_top:
                raise ParseError(token.location, "Expected " + stack_top)
            stack.pop()
            if not stack: break
            results.append(token)
            token = next(tok_iter)

        else:
            row = TABLE[stack_top]
            if token.type not in row:
                raise ParseError(token.location, "Unexpected {}, expected {}".format(
                    repr(token.type), ', '.join(sorted(repr(k) for k in row))))
            rule_num = row[token.type]
            rule, action = RULES[rule_num]
            stack.pop()
            stack.append(rule_num)
            stack.extend(reversed(rule))

    return results[0]

RULES = [(('simple_stmt__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('test',), 
(lambda ctx, v1: [('u', v1)])
), ((), 
(lambda ctx: [])
), (('argument',), 
(combine_action(root=lambda ctx, v1: [('a', v1)], user=lambda a, _all, _loc, _ctx: (
a
), normal_vars=['a'], list_vars=[]))
), (('*', 'test'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda a, _all, s, _loc, _ctx: (
(s, a, None, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'test'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda a, _all, s, _loc, _ctx: (
(s, a, None, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('augassign', 'expr_stmt__2'), 
(lambda ctx, v1, v2: [('aa', v1), v2])
), (('expr_stmt__3',), 
(lambda ctx, v1: [v1])
), (('expr', 'comparison__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda op, _loc, l, r, _all, _ctx: (
l if not op else Compare(l.location, l, op, r)
), normal_vars=['l'], list_vars=['op', 'r']))
), (('NEWLINE', 'eval_input__1'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), ((',', 'NAME', 'global_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('testlist_star_expr__1', 'testlist_star_expr__2'), 
(lambda ctx, v1, v2: [v1, v2])
), ((), 
(lambda ctx: [])
), ((',', 'dotted_as_name', 'dotted_as_names__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('testlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, c, t, _loc, _ctx: (
ast_for_testlist(t, c)
), normal_vars=[], list_vars=['c', 't']))
), (('subscript', 'subscriptlist__2'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), (('shift_expr', 'and_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _loc, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('dictorsetmaker__2',), 
(lambda ctx, v1: [v1])
), (('comp_for',), 
(lambda ctx, v1: [('sc', v1)])
), (('dictorsetmaker__6',), 
(lambda ctx, v1: [v1])
), (('testlist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('d', v2)])
), ((), 
(lambda ctx: [])
), (('**', 'factor'), 
(lambda ctx, v1, v2: [(None, v1), ('f', v2)])
), ((), 
(lambda ctx: [])
), (('or', 'and_test', 'or_test__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('l', v2), v3])
), ((), 
(lambda ctx: [])
), (('import_name',), 
(combine_action(root=lambda ctx, v1: [('i', v1)], user=lambda i, _all, _loc, _ctx: (
i
), normal_vars=['i'], list_vars=[]))
), (('import_from',), 
(combine_action(root=lambda ctx, v1: [('i', v1)], user=lambda i, _all, _loc, _ctx: (
i
), normal_vars=['i'], list_vars=[]))
), (('yield_expr',), 
(lambda ctx, v1: [('y', v1)])
), (('testlist_comp',), 
(lambda ctx, v1: [('l', v1)])
), ((), 
(lambda ctx: [])
), ((':', 'sliceop__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, s, _loc, _ctx: (
s
), normal_vars=['s'], list_vars=[]))
), (('testlist_comp__1', 'testlist_comp__3'), 
(lambda ctx, v1, v2: [v1, v2])
), ((), 
(lambda ctx: [])
), (('dotted_as_name', 'dotted_as_names__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _all, n, _loc, _ctx: (
n
), normal_vars=[], list_vars=['n']))
), (('dictorsetmaker',), 
(lambda ctx, v1: [('d', v1)])
), ((), 
(lambda ctx: [])
), (('import_as_names__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('arith_expr__2', 'term', 'arith_expr__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('or_test', 'test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda _ctx, c, _loc, a, _all, b: (
IfExp(a.location, b, a, c) if b else a
), normal_vars=['a', 'b', 'c'], list_vars=[]))
), (('lambdef',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _loc, _ctx: (
e
), normal_vars=['e'], list_vars=[]))
), (('subscriptlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, s, c, _loc, _ctx: (

    s[0] if not c else
    ExtSlice(s) if any(isinstance(k, Slice) for k in s) else
    Index(Tuple(s[0].value.location, [k.value for k in s], Load()))
), normal_vars=[], list_vars=['c', 's']))
), (('NEWLINE', 'file_input__1'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), (('stmt', 'file_input__1'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('test', 'dictorsetmaker__1'), 
(combine_action(root=lambda ctx, v1, v2: [('k1', v1), v2], user=lambda v1, s, k1, sc, k, _loc, dc, _all, v, _ctx: (

    DictComp(k1.location, k1, v1, ast_for_comprehension(dc)) if dc else
    SetComp(k1.location, k1, ast_for_comprehension(sc)) if sc else
    Dict(k1.location, [k1]+k, [v1]+v) if v1 else
    Set(k1.location, [k1]+s)
), normal_vars=['dc', 'k1', 'sc', 'v1'], list_vars=['k', 's', 'v']))
), (('typedargslist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('global', 'NAME', 'global_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3], user=lambda _all, n, _loc, _ctx: (
Global(_loc, [t.value for t in n])
), normal_vars=[], list_vars=['n']))
), (('NAME', 'import_as_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda a, _all, n, _loc, _ctx: (
alias(n.value, a.value if a else None)
), normal_vars=['a', 'n'], list_vars=[]))
), (('except_clause', ':', 'suite', 'try_stmt__2', 'try_stmt__3', 'try_stmt__4'), 
(lambda ctx, v1, v2, v3, v4, v5, v6: [('ec', v1), (None, v2), ('es', v3), v4, v5, v6])
), (('finally', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('f', v3)])
), ((',', 'typedargslist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('test',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), ((',', 'subscriptlist__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('arglist_item', 'arglist__2'), 
(lambda ctx, v1, v2: [('a', v1), v2])
), (('varargslist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _ctx, _loc, l: (
ast_for_arguments(l)
), normal_vars=[], list_vars=['l']))
), (('NAME', 'dotted_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda _all, t, _loc, _ctx: (
t
), normal_vars=[], list_vars=['t']))
), (('testlist', 'eval_input__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda _all, t, _loc, _ctx: (
Expression(t)
), normal_vars=['t'], list_vars=[]))
), (('return', 'return_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, t, _loc, _ctx: (
Return(_loc, t)
), normal_vars=['t'], list_vars=[]))
), (('<',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Lt()
), normal_vars=[], list_vars=[]))
), (('>',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Gt()
), normal_vars=[], list_vars=[]))
), (('==',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Eq()
), normal_vars=[], list_vars=[]))
), (('>=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
GtE()
), normal_vars=[], list_vars=[]))
), (('<=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
LtE()
), normal_vars=[], list_vars=[]))
), (('!=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
NotEq()
), normal_vars=[], list_vars=[]))
), (('in',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
In()
), normal_vars=[], list_vars=[]))
), (('not', 'in'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), (None, v2)], user=lambda _all, _loc, _ctx: (
NotIn()
), normal_vars=[], list_vars=[]))
), (('is', 'comp_op__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, n, _loc, _ctx: (
IsNot() if n else Is()
), normal_vars=['n'], list_vars=[]))
), (('NAME', 'tfpdef__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda _all, n, t, _loc, _ctx: (
arg(_loc, n.value, t)
), normal_vars=['n', 't'], list_vars=[]))
), (('if', 'test_nocond', 'comp_if__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [('i', v1), ('c', v2), v3], user=lambda i, c, _loc, r, _all, _ctx: (
[(i, c), r or []]
), normal_vars=['c', 'i', 'r'], list_vars=[]))
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('n', v2)])
), ((), 
(lambda ctx: [])
), (('import_as_name', 'import_as_names__2'), 
(lambda ctx, v1, v2: [('n', v1), v2])
), (('test', 'dictorsetmaker__6'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('dotted_name',), 
(lambda ctx, v1: [('m', v1)])
), ((), 
(lambda ctx: [])
), (('small_stmt', 'simple_stmt__2'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('testlist_star_expr', 'expr_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda av, _loc, aa, _all, t, v, _ctx: (
ast_for_expr_stmt(t, aa, av, v)
), normal_vars=['aa', 'av', 't'], list_vars=['v']))
), (('^', 'and_expr', 'xor_expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('not',), 
(lambda ctx, v1: [('n', v1)])
), ((), 
(lambda ctx: [])
), (('sliceop',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), ((':', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('t', v2)])
), ((), 
(lambda ctx: [])
), (('exprlist__2', 'exprlist__3'), 
(lambda ctx, v1, v2: [v1, v2])
), (('(', 'classdef__2', ')'), 
(lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)])
), ((), 
(lambda ctx: [])
), (('@', 'dotted_name', 'decorator__1', 'NEWLINE'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), ('d', v2), v3, (None, v4)], user=lambda d, _ctx, _loc, a, _all, b: (
ast_for_decorator(_loc, ast_for_dotted_name(d), b, a)
), normal_vars=['a', 'b', 'd'], list_vars=[]))
), (('testlist_star_expr__1', 'testlist_star_expr__2'), 
(combine_action(root=lambda ctx, v1, v2: [v1, v2], user=lambda _all, c, t, _loc, _ctx: (
ast_for_testlist(t, c)
), normal_vars=[], list_vars=['c', 't']))
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('a', v2)])
), ((), 
(lambda ctx: [])
), (('except', 'except_clause__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, n, t, _loc, _ctx: (
lambda body: ExceptHandler(_loc, t, n.value if n else None, body)
), normal_vars=['n', 't'], list_vars=[]))
), (('flow_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _loc, _ctx: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('comp_iter',), 
(lambda ctx, v1: [('r', v1)])
), ((), 
(lambda ctx: [])
), (('simple_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('compound_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('+', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _all, e, _loc, _ctx: (
UnaryOp(_loc, UAdd(), e)
), normal_vars=['e'], list_vars=[]))
), (('-', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _all, e, _loc, _ctx: (
UnaryOp(_loc, USub(), e)
), normal_vars=['e'], list_vars=[]))
), (('~', 'factor'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _all, e, _loc, _ctx: (
UnaryOp(_loc, Invert(), e)
), normal_vars=['e'], list_vars=[]))
), (('power',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _loc, _ctx: (
e
), normal_vars=['e'], list_vars=[]))
), (('dotted_name',), 
(lambda ctx, v1: [('m', v1)])
), (('import_from__3', 'import_from__4', 'import_from__5'), 
(lambda ctx, v1, v2, v3: [v1, v2, v3])
), ((',', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('b', v2)])
), ((), 
(lambda ctx: [])
), (('+=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Add()
), normal_vars=[], list_vars=[]))
), (('-=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Sub()
), normal_vars=[], list_vars=[]))
), (('*=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Mul()
), normal_vars=[], list_vars=[]))
), (('/=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Div()
), normal_vars=[], list_vars=[]))
), (('%=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Mod()
), normal_vars=[], list_vars=[]))
), (('&=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
BitAnd()
), normal_vars=[], list_vars=[]))
), (('|=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
BitOr()
), normal_vars=[], list_vars=[]))
), (('^=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
BitXor()
), normal_vars=[], list_vars=[]))
), (('<<=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
LShift()
), normal_vars=[], list_vars=[]))
), (('>>=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
RShift()
), normal_vars=[], list_vars=[]))
), (('**=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Pow()
), normal_vars=[], list_vars=[]))
), (('//=',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
FloorDiv()
), normal_vars=[], list_vars=[]))
), (('typedargslist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _ctx, _loc, l: (
ast_for_arguments(l)
), normal_vars=[], list_vars=['l']))
), (('test', 'argument__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda c, _loc, _all, t, v, _ctx: (
(None, t, c, v)
), normal_vars=['c', 't', 'v'], list_vars=[]))
), (('exprlist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, c, t, _loc, _ctx: (
(t, bool(c))
), normal_vars=[], list_vars=['c', 't']))
), (('decorators', 'decorated__1'), 
(combine_action(root=lambda ctx, v1, v2: [('d', v1), v2], user=lambda d, cf, _all, _loc, _ctx: (
cf._replace(decorator_list=d)
), normal_vars=['cf', 'd'], list_vars=[]))
), ((':', 'subscript__3', 'subscript__4'), 
(lambda ctx, v1, v2, v3: [('d', v1), v2, v3])
), ((), 
(lambda ctx: [])
), (('from', 'import_from__2', 'import', 'import_from__6'), 
(lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), v4])
), (('yield', 'yield_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _ctx, t, _loc, f: (
YieldFrom(_loc, f) if f else Yield(_loc, t)
), normal_vars=['f', 't'], list_vars=[]))
), (('except_clause', ':', 'suite', 'try_stmt__2'), 
(lambda ctx, v1, v2, v3, v4: [('ec', v1), (None, v2), ('es', v3), v4])
), ((), 
(lambda ctx: [])
), (('test', ':', 'test', 'dictorsetmaker__4'), 
(lambda ctx, v1, v2, v3, v4: [('k', v1), (None, v2), ('v', v3), v4])
), ((), 
(lambda ctx: [])
), ((',', 'NAME', 'nonlocal_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3])
), ((), 
(lambda ctx: [])
), (('stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _loc, _ctx: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('lambda', 'lambdef__1', ':', 'test'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), ('t', v4)], user=lambda a, _all, t, _loc, _ctx: (
Lambda(_loc, a or arguments(None, None, None, None, None, None), t)
), normal_vars=['a', 't'], list_vars=[]))
), (('import_from__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda d, s, n, _loc, _all, m, _ctx: (
ImportFrom(_loc, '.'.join(t.value for t in m) if m else None, [alias('*', None)] if s else n, sum(len(t.type) for t in d))
), normal_vars=['m', 'n', 's'], list_vars=['d']))
), (('(', 'atom__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _all, _loc, _ctx, y, l: (

      y if y else
      Tuple(_loc, None, Load()) if not l else
      GeneratorExp(_loc, l[0][0], ast_for_comprehension(l[2])) if l[2] else
      ast_for_testlist(l[0], l[1])
), normal_vars=['l', 'y'], list_vars=[]))
), (('[', 'atom__2', ']'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _all, _ctx, _loc, l: (

      List(_loc, None, Load()) if not l else
      ListComp(_loc, l[0][0], ast_for_comprehension(l[2])) if l[2] else
      List(_loc, l[0], Load())
), normal_vars=['l'], list_vars=[]))
), (('{', 'atom__3', '}'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda d, _all, _loc, _ctx: (
d._replace(location=_loc) if d else Dict(_loc, None, None)
), normal_vars=['d'], list_vars=[]))
), (('NAME',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _all, n, _loc, _ctx: (
Name(_loc, n.value, Load())
), normal_vars=['n'], list_vars=[]))
), (('NUMBER',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _all, n, _loc, _ctx: (
Num(_loc, n.value)
), normal_vars=['n'], list_vars=[]))
), (('STRING', 'atom__4'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda _all, s, _loc, _ctx: (
Str(_loc, ''.join(t.value for t in s))
), normal_vars=[], list_vars=['s']))
), (('...',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Ellipsis(_loc)
), normal_vars=[], list_vars=[]))
), (('None',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
NameConstant(_loc, None)
), normal_vars=[], list_vars=[]))
), (('True',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
NameConstant(_loc, True)
), normal_vars=[], list_vars=[]))
), (('False',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
NameConstant(_loc, False)
), normal_vars=[], list_vars=[]))
), (('EMBEDEXPR',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _loc, _ctx: (
EmbedExp(_loc, e.value)
), normal_vars=['e'], list_vars=[]))
), (('def', 'NAME', 'parameters', 'funcdef__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6: [(None, v1), ('n', v2), ('p', v3), v4, (None, v5), ('b', v6)], user=lambda _ctx, n, _loc, r, _all, p, b: (
FunctionDef(_loc, n.value, p, b, [], r)
), normal_vars=['b', 'n', 'p', 'r'], list_vars=[]))
), (('(', 'decorator__2', ')'), 
(lambda ctx, v1, v2, v3: [('b', v1), v2, (None, v3)])
), ((), 
(lambda ctx: [])
), (('arith_expr', 'shift_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _loc, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('compound_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _loc, _ctx: (
_all[0]
), normal_vars=[], list_vars=[]))
), (('class', 'NAME', 'classdef__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('n', v2), v3, (None, v4), ('b', v5)], user=lambda _ctx, n, _loc, a, _all, b: (
ast_for_classdef(_loc, n.value, a, b)
), normal_vars=['a', 'b', 'n'], list_vars=[]))
), (('while', 'test', ':', 'suite', 'while_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('c', v2), (None, v3), ('s', v4), v5], user=lambda s, c, _loc, _all, e, _ctx: (
While(_loc, c, s, e)
), normal_vars=['c', 'e', 's'], list_vars=[]))
), (('test', 'testlist__2'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), (('if_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('while_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('for_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('try_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('with_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('funcdef',), 
(lambda ctx, v1: [(None, v1)])
), (('classdef',), 
(lambda ctx, v1: [(None, v1)])
), (('decorated',), 
(lambda ctx, v1: [(None, v1)])
), (('break_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('continue_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('return_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('raise_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('yield_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('tfpdef',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('import_as_names__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, n, _loc, _ctx: (
n
), normal_vars=[], list_vars=['n']))
), (('*',), 
(lambda ctx, v1: [(None, v1)])
), (('/',), 
(lambda ctx, v1: [(None, v1)])
), (('%',), 
(lambda ctx, v1: [(None, v1)])
), (('//',), 
(lambda ctx, v1: [(None, v1)])
), (('if', 'or_test', 'else', 'test'), 
(lambda ctx, v1, v2, v3, v4: [(None, v1), ('b', v2), (None, v3), ('c', v4)])
), ((), 
(lambda ctx: [])
), (('testlist_comp',), 
(lambda ctx, v1: [('l', v1)])
), ((), 
(lambda ctx: [])
), (('testlist_comp__1', 'testlist_comp__2'), 
(combine_action(root=lambda ctx, v1, v2: [v1, v2], user=lambda c, _loc, f, _all, t, _ctx: (
(t, c, f)
), normal_vars=['f'], list_vars=['c', 't']))
), (('assert', 'test', 'assert_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('a', v2), v3], user=lambda a, _all, _ctx, _loc, b: (
Assert(_loc, a, b)
), normal_vars=['a', 'b'], list_vars=[]))
), (('term', 'arith_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _loc, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('STRING', 'atom__4'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('comp_for',), 
(lambda ctx, v1: [('f', v1)])
), (('testlist_comp__3',), 
(lambda ctx, v1: [v1])
), ((',', 'arglist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('break',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Break(_loc)
), normal_vars=[], list_vars=[]))
), (('<<',), 
(lambda ctx, v1: [(None, v1)])
), (('>>',), 
(lambda ctx, v1: [(None, v1)])
), (('varargslist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('typedargslist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('varargslist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('decorator', 'decorators__1'), 
(combine_action(root=lambda ctx, v1, v2: [('d', v1), v2], user=lambda d, _all, _loc, _ctx: (
d
), normal_vars=[], list_vars=['d']))
), (('NEWLINE',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Interactive([])
), normal_vars=[], list_vars=[]))
), (('simple_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _all, s, _loc, _ctx: (
Interactive(flatten(s))
), normal_vars=['s'], list_vars=[]))
), (('compound_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _all, s, _loc, _ctx: (
Interactive(flatten(s))
), normal_vars=['s'], list_vars=[]))
), (('expr',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), (('pass',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Pass(_loc)
), normal_vars=[], list_vars=[]))
), (('from', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('f', v2)])
), (('testlist',), 
(lambda ctx, v1: [('t', v1)])
), ((), 
(lambda ctx: [])
), (('embed_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('expr_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('del_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('pass_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('flow_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('import_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('global_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('nonlocal_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('assert_stmt',), 
(lambda ctx, v1: [(None, v1)])
), (('simple_stmt',), 
(combine_action(root=lambda ctx, v1: [('s', v1)], user=lambda _all, s, _loc, _ctx: (
flatten(s)
), normal_vars=['s'], list_vars=[]))
), (('NEWLINE', 'INDENT', 'stmt', 'suite__1', 'DEDENT'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), (None, v2), ('s', v3), v4, (None, v5)], user=lambda _all, s, _loc, _ctx: (
flatten(s)
), normal_vars=[], list_vars=['s']))
), (('for', 'exprlist', 'in', 'or_test', 'comp_for__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [('f', v1), ('t', v2), (None, v3), ('i', v4), v5], user=lambda i, _loc, r, f, _all, t, _ctx: (
[(f, t, i), r or []]
), normal_vars=['f', 'i', 'r', 't'], list_vars=[]))
), (('factor', 'term__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _loc, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('(', 'trailer__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda a, _all, _loc, _ctx: (
lambda left_expr: ast_for_call(_loc, left_expr, a)
), normal_vars=['a'], list_vars=[]))
), (('[', 'subscriptlist', ']'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('s', v2), (None, v3)], user=lambda _all, s, _loc, _ctx: (
lambda left_expr: Subscript(_loc, left_expr, s, Load())
), normal_vars=['s'], list_vars=[]))
), (('.', 'NAME'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('n', v2)], user=lambda _all, n, _loc, _ctx: (
lambda left_expr: Attribute(_loc, left_expr, n.value, Load())
), normal_vars=['n'], list_vars=[]))
), (('shift_expr__2', 'arith_expr', 'shift_expr__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('for', 'exprlist', 'in', 'testlist', ':', 'suite', 'for_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6, v7: [(None, v1), ('t', v2), (None, v3), ('c', v4), (None, v5), ('s', v6), v7], user=lambda s, c, _loc, _all, e, t, _ctx: (
For(_loc, set_context(Tuple(t[0][0].location, t[0], Store()) if t[1] else t[0][0], Store()), c, s, e)
), normal_vars=['c', 'e', 's', 't'], list_vars=[]))
), (('xor_expr', 'expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _loc, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), (('elif', 'test', ':', 'suite', 'if_stmt__1'), 
(lambda ctx, v1, v2, v3, v4, v5: [('t', v1), ('c', v2), (None, v3), ('s', v4), v5])
), ((), 
(lambda ctx: [])
), ((',', 'testlist_comp__4'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('raise', 'raise_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda a, _all, _ctx, _loc, b: (
Raise(_loc, a, b)
), normal_vars=['a', 'b'], list_vars=[]))
), (('subscript__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda d, s, _loc, l, _all, u, _ctx: (
Slice(l, u, s) if d else Index(l)
), normal_vars=['d', 'l', 's', 'u'], list_vars=[]))
), (('arglist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('d', v2)])
), ((), 
(lambda ctx: [])
), (('testlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('(', 'parameters__1', ')'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), v2, (None, v3)], user=lambda _all, t, _loc, _ctx: (
t or arguments(None, None, None, None, None, None)
), normal_vars=['t'], list_vars=[]))
), (('&', 'shift_expr', 'and_expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('simple_stmt__1', 'NEWLINE'), 
(combine_action(root=lambda ctx, v1, v2: [v1, (None, v2)], user=lambda _all, s, _loc, _ctx: (
s
), normal_vars=[], list_vars=['s']))
), (('dotted_name', 'dotted_as_name__1'), 
(combine_action(root=lambda ctx, v1, v2: [('n', v1), v2], user=lambda a, _all, n, _loc, _ctx: (
alias('.'.join(t.value for t in n), a.value if a else None)
), normal_vars=['a', 'n'], list_vars=[]))
), (('atom', 'power__1', 'power__2'), 
(combine_action(root=lambda ctx, v1, v2, v3: [('a', v1), v2, v3], user=lambda _loc, f, a, _all, t, _ctx: (
ast_for_power(a, t, f)
), normal_vars=['a', 'f'], list_vars=['t']))
), (('NAME',), 
(combine_action(root=lambda ctx, v1: [('n', v1)], user=lambda _all, n, _loc, _ctx: (
arg(_loc, n.value, None)
), normal_vars=['n'], list_vars=[]))
), (('.', 'NAME', 'dotted_name__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('t', v2), v3])
), ((), 
(lambda ctx: [])
), (('decorator', 'decorators__1'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), ((), 
(lambda ctx: [])
), (('classdef',), 
(lambda ctx, v1: [('cf', v1)])
), (('funcdef',), 
(lambda ctx, v1: [('cf', v1)])
), (('*', 'expr'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _all, e, _loc, _ctx: (
Starred(_loc, e, Load())
), normal_vars=['e'], list_vars=[]))
), (('tfpdef', 'typedargslist_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda a, d, _all, _loc, _ctx: (
(None, a, d)
), normal_vars=['a', 'd'], list_vars=[]))
), (('*', 'typedargslist_item__2'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda a, _all, s, _loc, _ctx: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'tfpdef'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda a, _all, s, _loc, _ctx: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('and_test', 'or_test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda _all, _ctx, _loc, l: (
l[0] if len(l) == 1 else BoolOp(l[0].location, And(), l)
), normal_vars=[], list_vars=['l']))
), (('yield_expr',), 
(lambda ctx, v1: [('av', v1)])
), (('testlist',), 
(lambda ctx, v1: [('av', v1)])
), (('comp_iter',), 
(lambda ctx, v1: [('r', v1)])
), ((), 
(lambda ctx: [])
), (('+',), 
(lambda ctx, v1: [(None, v1)])
), (('-',), 
(lambda ctx, v1: [(None, v1)])
), (('=', 'expr_stmt__4', 'expr_stmt__3'), 
(lambda ctx, v1, v2, v3: [(None, v1), v2, v3])
), ((), 
(lambda ctx: [])
), ((',', 'with_item', 'with_stmt__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('w', v2), v3])
), ((), 
(lambda ctx: [])
), (('test', 'with_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('t', v1), v2], user=lambda _all, e, t, _loc, _ctx: (
withitem(t, set_context(e, Store()) if e else None)
), normal_vars=['e', 't'], list_vars=[]))
), (('sliceop',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), (('import', 'dotted_as_names'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('n', v2)], user=lambda _all, n, _loc, _ctx: (
Import(_loc, n)
), normal_vars=['n'], list_vars=[]))
), ((',', 'testlist__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), (('or_test',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _loc, _ctx: (
e
), normal_vars=['e'], list_vars=[]))
), (('lambdef_nocond',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _loc, _ctx: (
e
), normal_vars=['e'], list_vars=[]))
), (('not_test', 'and_test__1'), 
(combine_action(root=lambda ctx, v1, v2: [('l', v1), v2], user=lambda _all, _ctx, _loc, l: (
l[0] if len(l) == 1 else BoolOp(l[0].location, Or(), l)
), normal_vars=[], list_vars=['l']))
), (('trailer', 'power__1'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), ((), 
(lambda ctx: [])
), (('test', 'raise_stmt__2'), 
(lambda ctx, v1, v2: [('a', v1), v2])
), ((), 
(lambda ctx: [])
), (('nonlocal', 'NAME', 'nonlocal_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), v3], user=lambda _all, n, _loc, _ctx: (
Nonlocal(_loc, [t.value for t in n])
), normal_vars=[], list_vars=['n']))
), (('as', 'expr'), 
(lambda ctx, v1, v2: [(None, v1), ('e', v2)])
), ((), 
(lambda ctx: [])
), ((',', 'testlist_star_expr__3'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
), ((',', 'dictorsetmaker__7'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), ((',', 'import_as_names__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('test',), 
(lambda ctx, v1: [('u', v1)])
), ((), 
(lambda ctx: [])
), (('stmt', 'suite__1'), 
(lambda ctx, v1, v2: [('s', v1), v2])
), ((), 
(lambda ctx: [])
), (('test', 'subscript__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), ((':', 'subscript__5', 'subscript__6'), 
(lambda ctx, v1, v2, v3: [('d', v1), v2, v3])
), (('and_expr', 'xor_expr__1'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), v2], user=lambda _all, _loc, _ctx: (
ast_for_binop(_all)
), normal_vars=[], list_vars=[]))
), ((',', 'varargslist__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('try', ':', 'suite', 'try_stmt__1'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), (None, v2), ('b', v3), v4], user=lambda l, f, _all, b, es, _loc, ec, _ctx: (
Try(_loc, b, [kc(ks) for kc, ks in zip(ec, es)], l, f)
), normal_vars=['b', 'f', 'l'], list_vars=['ec', 'es']))
), (('arglist__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda a, _all, _loc, _ctx: (
a
), normal_vars=[], list_vars=['a']))
), (('lambda', 'lambdef_nocond__1', ':', 'test_nocond'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4: [(None, v1), v2, (None, v3), ('t', v4)], user=lambda a, _all, t, _loc, _ctx: (
Lambda(_loc, a or arguments(None, None, None, None, None, None), t)
), normal_vars=['a', 't'], list_vars=[]))
), (('continue',), 
(combine_action(root=lambda ctx, v1: [(None, v1)], user=lambda _all, _loc, _ctx: (
Continue(_loc)
), normal_vars=[], list_vars=[]))
), (('test', 'except_clause__2'), 
(lambda ctx, v1, v2: [('t', v1), v2])
), ((), 
(lambda ctx: [])
), (('comp_for',), 
(lambda ctx, v1: [('dc', v1)])
), (('dictorsetmaker__4',), 
(lambda ctx, v1: [v1])
), (('term__2', 'factor', 'term__1'), 
(lambda ctx, v1, v2, v3: [v1, (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('comp_for',), 
(combine_action(root=lambda ctx, v1: [('c', v1)], user=lambda _all, c, _loc, _ctx: (
c
), normal_vars=['c'], list_vars=[]))
), (('comp_if',), 
(combine_action(root=lambda ctx, v1: [('c', v1)], user=lambda _all, c, _loc, _ctx: (
c
), normal_vars=['c'], list_vars=[]))
), (('typedargslist_item', 'typedargslist__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), (('if', 'test', ':', 'suite', 'if_stmt__1', 'if_stmt__2'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5, v6: [('t', v1), ('c', v2), (None, v3), ('s', v4), v5, v6], user=lambda s, c, _loc, _all, e, t, _ctx: (
ast_for_if_stmt(t, c, s, e)
), normal_vars=['e'], list_vars=['c', 's', 't']))
), (('yield_expr',), 
(lambda ctx, v1: [('v', v1)])
), (('testlist_star_expr',), 
(lambda ctx, v1: [('v', v1)])
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('l', v3)])
), ((), 
(lambda ctx: [])
), (('subscriptlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('comp_for',), 
(lambda ctx, v1: [('c', v1)])
), (('=', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('v', v2)])
), ((), 
(lambda ctx: [])
), (('->', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('r', v2)])
), ((), 
(lambda ctx: [])
), (('vfpdef',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('comp_op', 'expr', 'comparison__1'), 
(lambda ctx, v1, v2, v3: [('op', v1), ('r', v2), v3])
), ((), 
(lambda ctx: [])
), (('else', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('e', v3)])
), ((), 
(lambda ctx: [])
), (('varargslist_item', 'varargslist__2'), 
(lambda ctx, v1, v2: [('l', v1), v2])
), (('not', 'not_test'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('e', v2)], user=lambda _all, e, _loc, _ctx: (
UnaryOp(_loc, Not(), e)
), normal_vars=['e'], list_vars=[]))
), (('comparison',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _loc, _ctx: (
e
), normal_vars=['e'], list_vars=[]))
), (('and', 'not_test', 'and_test__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('l', v2), v3])
), ((), 
(lambda ctx: [])
), (('as', 'NAME'), 
(lambda ctx, v1, v2: [(None, v1), ('a', v2)])
), ((), 
(lambda ctx: [])
), (('*',), 
(lambda ctx, v1: [('s', v1)])
), (('(', 'import_as_names', ')'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('n', v2), (None, v3)])
), (('import_as_names',), 
(lambda ctx, v1: [('n', v1)])
), (('.', 'import_from__4'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), (('...', 'import_from__4'), 
(lambda ctx, v1, v2: [('d', v1), v2])
), ((), 
(lambda ctx: [])
), ((';', 'simple_stmt__3'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('test',), 
(lambda ctx, v1: [('t', v1)])
), (('star_expr',), 
(lambda ctx, v1: [('t', v1)])
), ((':', 'test', 'dictorsetmaker__3'), 
(lambda ctx, v1, v2, v3: [(None, v1), ('v1', v2), v3])
), (('yield_expr',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _loc, _ctx: (
Expr(e.location, e)
), normal_vars=['e'], list_vars=[]))
), (('del', 'exprlist'), 
(combine_action(root=lambda ctx, v1, v2: [(None, v1), ('l', v2)], user=lambda _all, _ctx, _loc, l: (
Delete(_loc, [set_context(e, Del()) for e in l[0]])
), normal_vars=['l'], list_vars=[]))
), (('exprlist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('small_stmt__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, _loc, _ctx: (
_all[0]
), normal_vars=[], list_vars=[]))
), ((',', 'dictorsetmaker__5'), 
(lambda ctx, v1, v2: [(None, v1), v2])
), ((), 
(lambda ctx: [])
), (('from', 'test'), 
(lambda ctx, v1, v2: [(None, v1), ('b', v2)])
), ((), 
(lambda ctx: [])
), (('varargslist',), 
(lambda ctx, v1: [('a', v1)])
), ((), 
(lambda ctx: [])
), (('file_input__1',), 
(combine_action(root=lambda ctx, v1: [v1], user=lambda _all, s, _loc, _ctx: (
Module(flatten(s))
), normal_vars=[], list_vars=['s']))
), (('EMBEDSTAT',), 
(combine_action(root=lambda ctx, v1: [('e', v1)], user=lambda _all, e, _loc, _ctx: (
EmbedStat(_loc, e.value)
), normal_vars=['e'], list_vars=[]))
), (('test',), 
(lambda ctx, v1: [('s', v1)])
), ((), 
(lambda ctx: [])
), (('finally', ':', 'suite'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), ('f', v3)])
), ((), 
(lambda ctx: [])
), (('with', 'with_item', 'with_stmt__1', ':', 'suite'), 
(combine_action(root=lambda ctx, v1, v2, v3, v4, v5: [(None, v1), ('w', v2), v3, (None, v4), ('b', v5)], user=lambda _all, w, _ctx, _loc, b: (
With(_loc, w, b)
), normal_vars=['b'], list_vars=['w']))
), (('vfpdef', 'varargslist_item__1'), 
(combine_action(root=lambda ctx, v1, v2: [('a', v1), v2], user=lambda a, d, _all, _loc, _ctx: (
(None, a, d)
), normal_vars=['a', 'd'], list_vars=[]))
), (('*', 'varargslist_item__2'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), v2], user=lambda a, _all, s, _loc, _ctx: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('**', 'vfpdef'), 
(combine_action(root=lambda ctx, v1, v2: [('s', v1), ('a', v2)], user=lambda a, _all, s, _loc, _ctx: (
(s, a, None)
), normal_vars=['a', 's'], list_vars=[]))
), (('|', 'xor_expr', 'expr__1'), 
(lambda ctx, v1, v2, v3: [(None, v1), (None, v2), v3])
), ((), 
(lambda ctx: [])
), (('arglist__1',), 
(lambda ctx, v1: [v1])
), ((), 
(lambda ctx: [])
), (('.',), 
(lambda ctx, v1: [('d', v1)])
), (('...',), 
(lambda ctx, v1: [('d', v1)])
), ((',', 'exprlist__4'), 
(lambda ctx, v1, v2: [('c', v1), v2])
), ((), 
(lambda ctx: [])
)]
TABLE = {'simple_stmt__3': {'*': 0, 'lambda': 0, 'True': 0, 'EMBEDSTAT': 0, 'continue': 0, 'False': 0, '...': 0, 'raise': 0, '[': 0, 'break': 0, 'from': 0, '+': 0, 'nonlocal': 0, '(': 0, 'return': 0, 'EMBEDEXPR': 0, 'NAME': 0, 'yield': 0, 'NEWLINE': 1, 'global': 0, '~': 0, 'pass': 0, 'None': 0, 'NUMBER': 0, '{': 0, 'import': 0, 'STRING': 0, '-': 0, 'del': 0, 'not': 0, 'assert': 0}, 'subscript__3': {':': 3, 'EMBEDEXPR': 2, 'NAME': 2, 'lambda': 2, '(': 2, 'True': 2, 'NUMBER': 2, '~': 2, 'None': 2, 'False': 2, '{': 2, '...': 2, 'STRING': 2, '-': 2, 'not': 2, '[': 2, ']': 3, ',': 3, '+': 2}, 'arglist_item': {'EMBEDEXPR': 4, 'NAME': 4, 'lambda': 4, '**': 6, '(': 4, 'True': 4, 'NUMBER': 4, '~': 4, 'None': 4, 'False': 4, '{': 4, '...': 4, 'STRING': 4, '-': 4, 'not': 4, '[': 4, '+': 4, '*': 5}, 'expr_stmt__1': {'%=': 7, '|=': 7, '**=': 7, ';': 8, '>>=': 7, '-=': 7, '<<=': 7, '+=': 7, '/=': 7, 'NEWLINE': 8, '&=': 7, '^=': 7, '=': 8, '*=': 7, '//=': 7}, 'comparison': {'EMBEDEXPR': 9, 'NAME': 9, '(': 9, 'True': 9, 'NUMBER': 9, '~': 9, 'None': 9, 'False': 9, '{': 9, '+': 9, 'STRING': 9, '-': 9, '[': 9, '...': 9}, 'eval_input__1': {'NEWLINE': 10}, 'global_stmt__1': {'NEWLINE': 13, ',': 12, ';': 13}, 'testlist_star_expr__3': {'|=': 15, '*': 14, 'lambda': 14, ';': 15, 'True': 14, 'NUMBER': 14, '=': 15, 'False': 14, '<<=': 15, '...': 14, '[': 14, 'NEWLINE': 15, '&=': 15, '*=': 15, '//=': 15, '%=': 15, 'EMBEDEXPR': 14, 'NAME': 14, '**=': 15, '(': 14, '>>=': 15, '~': 14, 'None': 14, '{': 14, '+': 14, '+=': 15, 'STRING': 14, '-': 14, 'not': 14, '^=': 15, '-=': 15, '/=': 15}, 'dotted_as_names__1': {'NEWLINE': 17, ',': 16, ';': 17}, 'testlist': {'EMBEDEXPR': 18, 'NAME': 18, 'lambda': 18, '(': 18, 'True': 18, 'NUMBER': 18, '~': 18, 'None': 18, 'False': 18, '{': 18, '...': 18, 'STRING': 18, '-': 18, 'not': 18, '[': 18, '+': 18}, 'subscriptlist__1': {':': 19, 'EMBEDEXPR': 19, 'NAME': 19, 'lambda': 19, '(': 19, 'True': 19, 'None': 19, '~': 19, 'NUMBER': 19, 'False': 19, '{': 19, '...': 19, 'STRING': 19, '-': 19, 'not': 19, '[': 19, '+': 19}, 'and_expr': {'EMBEDEXPR': 20, 'NAME': 20, '(': 20, 'True': 20, 'NUMBER': 20, '~': 20, 'None': 20, 'False': 20, '{': 20, '+': 20, 'STRING': 20, '-': 20, '[': 20, '...': 20}, 'dictorsetmaker__1': {':': 21, ',': 23, 'for': 22, '}': 23}, 'return_stmt__1': {'EMBEDEXPR': 24, 'NAME': 24, 'lambda': 24, '(': 24, 'True': 24, 'NUMBER': 24, '~': 24, ';': 25, 'None': 24, 'False': 24, '{': 24, '...': 24, 'STRING': 24, '-': 24, 'not': 24, '[': 24, 'NEWLINE': 25, '+': 24}, 'typedargslist_item__1': {')': 27, ',': 27, '=': 26}, 'power__2': {'|=': 29, '*': 29, ')': 29, 'as': 29, '&': 29, 'if': 29, '}': 29, '=': 29, '/=': 29, '**': 28, ']': 29, 'else': 29, '<': 29, '//=': 29, 'in': 29, '%': 29, 'NEWLINE': 29, '-=': 29, '^': 29, 'is': 29, '-': 29, 'not': 29, '==': 29, '!=': 29, '^=': 29, ':': 29, '|': 29, ';': 29, 'or': 29, '>>': 29, '+=': 29, '&=': 29, 'from': 29, 'and': 29, '//': 29, '*=': 29, '%=': 29, '<<': 29, '**=': 29, '>>=': 29, '>=': 29, '<<=': 29, '+': 29, '>': 29, '<=': 29, 'for': 29, '/': 29, ',': 29}, 'or_test__1': {')': 31, '|=': 31, 'as': 31, 'if': 31, '}': 31, 'or': 30, '/=': 31, '+=': 31, 'from': 31, ']': 31, '&=': 31, '*=': 31, '//=': 31, '%=': 31, '**=': 31, 'NEWLINE': 31, '-=': 31, ';': 31, '<<=': 31, 'else': 31, '>>=': 31, 'for': 31, ':': 31, ',': 31, '^=': 31, '=': 31}, 'import_stmt': {'from': 33, 'import': 32}, 'atom__1': {')': 36, '*': 35, 'NAME': 35, 'lambda': 35, 'yield': 34, 'True': 35, 'None': 35, '~': 35, 'NUMBER': 35, 'False': 35, '{': 35, '...': 35, 'STRING': 35, '+': 35, 'not': 35, '[': 35, 'EMBEDEXPR': 35, '-': 35, '(': 35}, 'sliceop': {':': 37}, 'testlist_comp__4': {')': 39, '*': 38, 'NAME': 38, 'lambda': 38, 'EMBEDEXPR': 38, 'True': 38, 'None': 38, '~': 38, 'NUMBER': 38, 'False': 38, '{': 38, '...': 38, 'STRING': 38, '-': 38, 'not': 38, '[': 38, ']': 39, '+': 38, '(': 38}, 'dotted_as_names': {'NAME': 40}, 'atom__3': {'EMBEDEXPR': 41, 'NAME': 41, 'lambda': 41, '(': 41, 'True': 41, 'NUMBER': 41, '~': 41, 'None': 41, 'False': 41, '{': 41, '...': 41, 'STRING': 41, '-': 41, 'not': 41, '[': 41, '}': 42, '+': 41}, 'import_as_names__3': {')': 44, 'NEWLINE': 44, 'NAME': 43, ';': 44}, 'arith_expr__1': {'|=': 46, ')': 46, '|': 46, 'as': 46, '&': 46, '%=': 46, 'if': 46, '}': 46, '=': 46, '/=': 46, '+=': 46, 'for': 46, '&=': 46, ']': 46, 'from': 46, 'and': 46, '*=': 46, '<': 46, '>>': 46, '//=': 46, ':': 46, 'in': 46, ';': 46, '<<': 46, '**=': 46, 'NEWLINE': 46, '^=': 46, '-=': 46, '^': 46, '<<=': 46, 'is': 46, '+': 45, '>': 46, '>>=': 46, '<=': 46, '-': 45, 'not': 46, 'else': 46, ',': 46, '==': 46, '!=': 46, '>=': 46, 'or': 46}, 'test': {'EMBEDEXPR': 47, 'NAME': 47, 'lambda': 48, '(': 47, 'True': 47, 'NUMBER': 47, '~': 47, 'None': 47, 'False': 47, '{': 47, '+': 47, 'STRING': 47, '-': 47, 'not': 47, '[': 47, '...': 47}, 'subscriptlist': {':': 49, 'EMBEDEXPR': 49, 'NAME': 49, 'lambda': 49, '(': 49, 'True': 49, 'None': 49, '~': 49, 'NUMBER': 49, 'False': 49, '{': 49, '...': 49, 'STRING': 49, '-': 49, 'not': 49, '[': 49, '+': 49}, 'file_input__1': {'for': 51, '*': 51, 'lambda': 51, 'if': 51, 'True': 51, 'EMBEDSTAT': 51, 'def': 51, 'False': 51, '...': 51, 'raise': 51, '[': 51, 'from': 51, 'break': 51, 'import': 51, 'while': 51, '(': 51, 'continue': 51, 'EMBEDEXPR': 51, 'NAME': 51, 'try': 51, '@': 51, 'yield': 51, 'NEWLINE': 50, 'return': 51, '~': 51, 'pass': 51, 'None': 51, 'NUMBER': 51, '{': 51, 'nonlocal': 51, 'STRING': 51, '-': 51, 'del': 51, 'not': 51, '+': 51, 'global': 51, 'with': 51, 'class': 51, 'assert': 51, 'EOF': 52}, 'dictorsetmaker': {'EMBEDEXPR': 53, 'NAME': 53, 'lambda': 53, '(': 53, 'True': 53, 'NUMBER': 53, '~': 53, 'None': 53, 'False': 53, '{': 53, '...': 53, 'STRING': 53, '-': 53, 'not': 53, '[': 53, '+': 53}, 'typedargslist__3': {')': 55, '*': 54, 'NAME': 54, '**': 54}, 'global_stmt': {'global': 56}, 'import_as_name': {'NAME': 57}, 'typedargslist__2': {')': 61, ',': 60}, 'testlist_comp__1': {'EMBEDEXPR': 62, 'NAME': 62, 'lambda': 62, '(': 62, 'True': 62, 'NUMBER': 62, '~': 62, 'None': 62, 'False': 62, '{': 62, '...': 62, 'STRING': 62, '-': 62, 'not': 62, '[': 62, '+': 62, '*': 63}, 'dictorsetmaker__2': {':': 367}, 'arglist__1': {'*': 66, 'NAME': 66, 'lambda': 66, '**': 66, 'EMBEDEXPR': 66, 'True': 66, 'None': 66, '~': 66, 'NUMBER': 66, 'False': 66, '{': 66, '...': 66, 'STRING': 66, '-': 66, 'not': 66, '[': 66, '+': 66, '(': 66}, 'varargslist': {'*': 67, 'NAME': 67, '**': 67}, 'dotted_name': {'NAME': 68}, 'eval_input': {'EMBEDEXPR': 69, 'NAME': 69, 'lambda': 69, '(': 69, 'True': 69, 'NUMBER': 69, '~': 69, 'None': 69, 'False': 69, '{': 69, '...': 69, 'STRING': 69, '-': 69, 'not': 69, '[': 69, '+': 69}, 'return_stmt': {'return': 70}, 'comp_op': {'>': 72, 'in': 77, '<=': 75, 'not': 78, 'is': 79, '==': 73, '>=': 74, '<': 71, '!=': 76}, 'tfpdef': {'NAME': 80}, 'comp_if': {'if': 81}, 'except_clause__2': {':': 83, 'as': 82}, 'import_as_names__1': {'NAME': 84}, 'dictorsetmaker__7': {'EMBEDEXPR': 85, 'NAME': 85, 'lambda': 85, '(': 85, 'True': 85, 'NUMBER': 85, '~': 85, 'None': 85, 'False': 85, '{': 85, '...': 85, 'STRING': 85, '-': 85, 'not': 85, '[': 85, '}': 86, '+': 85}, 'import_from__5': {'NAME': 87, 'import': 88}, 'simple_stmt__1': {'*': 89, 'lambda': 89, 'True': 89, 'EMBEDSTAT': 89, 'continue': 89, 'False': 89, '...': 89, 'raise': 89, '[': 89, 'break': 89, 'from': 89, '+': 89, 'nonlocal': 89, '(': 89, 'return': 89, 'EMBEDEXPR': 89, 'NAME': 89, 'yield': 89, 'global': 89, '~': 89, 'pass': 89, 'None': 89, 'NUMBER': 89, '{': 89, 'import': 89, 'STRING': 89, '-': 89, 'del': 89, 'not': 89, 'assert': 89}, 'classdef__2': {')': 91, '*': 90, 'NAME': 90, 'lambda': 90, '**': 90, 'EMBEDEXPR': 90, 'True': 90, 'None': 90, '~': 90, 'NUMBER': 90, 'False': 90, '{': 90, '...': 90, 'STRING': 90, '-': 90, 'not': 90, '[': 90, '+': 90, '(': 90}, 'expr_stmt': {'*': 92, 'NAME': 92, 'lambda': 92, 'EMBEDEXPR': 92, 'True': 92, 'None': 92, '~': 92, 'NUMBER': 92, 'False': 92, '{': 92, '...': 92, 'STRING': 92, '-': 92, 'not': 92, '[': 92, '+': 92, '(': 92}, 'xor_expr__1': {')': 94, '|=': 94, '|': 94, 'as': 94, '%=': 94, 'if': 94, '}': 94, '=': 94, '+=': 94, '/=': 94, '&=': 94, ']': 94, 'from': 94, 'and': 94, '<': 94, '*=': 94, '//=': 94, ':': 94, 'in': 94, ';': 94, '**=': 94, 'NEWLINE': 94, '>=': 94, '-=': 94, '^': 93, '<<=': 94, 'is': 94, 'else': 94, '>': 94, '>>=': 94, '<=': 94, 'for': 94, 'not': 94, ',': 94, '==': 94, '!=': 94, '^=': 94, 'or': 94}, 'comp_op__1': {'EMBEDEXPR': 96, 'NAME': 96, '(': 96, 'True': 96, 'NUMBER': 96, '~': 96, 'None': 96, 'False': 96, '{': 96, '...': 96, 'STRING': 96, '-': 96, 'not': 95, '[': 96, '+': 96}, 'subscript__4': {':': 97, ',': 98, ']': 98}, 'tfpdef__1': {':': 99, ')': 100, '=': 100, ',': 100}, 'classdef__1': {':': 103, '(': 102}, 'comparison__1': {')': 347, '|=': 347, 'as': 347, '%=': 347, 'if': 347, '}': 347, '=': 347, '/=': 347, '+=': 347, '&=': 347, ']': 347, 'from': 347, 'and': 347, '<': 346, '*=': 347, '!=': 346, ':': 347, 'in': 346, '**=': 347, 'NEWLINE': 347, '^=': 347, '-=': 347, ';': 347, '<<=': 347, '<=': 346, 'else': 347, '>': 346, '>>=': 347, 'is': 346, 'for': 347, 'not': 346, ',': 347, '==': 346, '//=': 347, '>=': 346, 'or': 347}, 'testlist_star_expr': {'*': 105, 'NAME': 105, 'lambda': 105, 'EMBEDEXPR': 105, 'True': 105, 'None': 105, '~': 105, 'NUMBER': 105, 'False': 105, '{': 105, '...': 105, 'STRING': 105, '-': 105, 'not': 105, '[': 105, '+': 105, '(': 105}, 'for_stmt__1': {'for': 107, '*': 107, 'NUMBER': 107, 'lambda': 107, 'DEDENT': 107, 'if': 107, 'True': 107, 'EMBEDSTAT': 107, 'def': 107, 'False': 107, '...': 107, 'nonlocal': 107, '[': 107, 'from': 107, 'break': 107, 'import': 107, 'while': 107, '(': 107, 'continue': 107, 'EMBEDEXPR': 107, 'NAME': 107, 'try': 107, '@': 107, 'yield': 107, 'NEWLINE': 107, 'return': 107, '~': 107, 'pass': 107, 'None': 107, 'raise': 107, '{': 107, 'else': 106, 'STRING': 107, '-': 107, 'del': 107, 'EOF': 107, '+': 107, 'global': 107, 'with': 107, 'class': 107, 'assert': 107, 'not': 107}, 'dotted_as_name__1': {'NEWLINE': 109, ',': 109, 'as': 108, ';': 109}, 'except_clause': {'except': 110}, 'flow_stmt': {'return': 111, 'break': 111, 'raise': 111, 'continue': 111, 'yield': 111}, 'comp_for__1': {')': 113, 'for': 112, ']': 113, '}': 113, ',': 113, 'if': 112}, 'stmt__1': {'*': 114, 'lambda': 114, 'if': 115, 'True': 114, 'EMBEDSTAT': 114, 'continue': 114, 'False': 114, 'while': 115, '...': 114, 'raise': 114, '[': 114, 'break': 114, 'from': 114, '+': 114, 'nonlocal': 114, '(': 114, 'return': 114, 'EMBEDEXPR': 114, 'NAME': 114, 'try': 115, '@': 115, 'yield': 114, 'global': 114, '~': 114, 'pass': 114, 'None': 114, 'NUMBER': 114, '{': 114, 'import': 114, 'STRING': 114, '-': 114, 'del': 114, 'not': 114, 'for': 115, 'def': 115, 'with': 115, 'class': 115, 'assert': 114}, 'factor': {'EMBEDEXPR': 119, 'NAME': 119, '(': 119, 'True': 119, 'NUMBER': 119, '~': 118, 'None': 119, 'False': 119, '{': 119, '...': 119, 'STRING': 119, '-': 117, '[': 119, '+': 116}, 'import_from__2': {'.': 121, 'NAME': 120, '...': 121}, 'assert_stmt__1': {'NEWLINE': 123, ',': 122, ';': 123}, 'augassign': {'%=': 128, '|=': 130, '**=': 134, '>>=': 133, '-=': 125, '<<=': 132, '+=': 124, '/=': 127, '&=': 129, '^=': 131, '*=': 126, '//=': 135}, 'typedargslist': {'*': 136, 'NAME': 136, '**': 136}, 'argument': {'EMBEDEXPR': 137, 'NAME': 137, 'lambda': 137, '(': 137, 'True': 137, 'NUMBER': 137, '~': 137, 'None': 137, 'False': 137, '{': 137, '...': 137, 'STRING': 137, '-': 137, 'not': 137, '[': 137, '+': 137}, 'exprlist': {'*': 138, 'NAME': 138, 'EMBEDEXPR': 138, 'True': 138, 'NUMBER': 138, '~': 138, 'None': 138, 'False': 138, '{': 138, '...': 138, 'STRING': 138, '-': 138, '[': 138, '+': 138, '(': 138}, 'decorated': {'@': 139}, 'subscript__2': {':': 140, ',': 141, ']': 141}, 'test_nocond': {'EMBEDEXPR': 292, 'NAME': 292, 'lambda': 293, '(': 292, 'True': 292, 'NUMBER': 292, '~': 292, 'None': 292, 'False': 292, '{': 292, '+': 292, 'STRING': 292, '-': 292, 'not': 292, '[': 292, '...': 292}, 'and_test': {'EMBEDEXPR': 294, 'NAME': 294, '(': 294, 'True': 294, 'NUMBER': 294, '~': 294, 'None': 294, 'False': 294, '{': 294, '+': 294, 'STRING': 294, '-': 294, 'not': 294, '[': 294, '...': 294}, 'embed_stmt': {'EMBEDSTAT': 380}, 'try_stmt__2': {'*': 145, 'pass': 145, 'if': 145, 'STRING': 145, '...': 145, '[': 145, 'break': 145, '+': 145, 'while': 145, 'EMBEDEXPR': 145, 'NAME': 145, 'NEWLINE': 145, 'None': 145, 'nonlocal': 145, '-': 145, 'del': 145, 'EOF': 145, 'with': 145, 'class': 145, 'not': 145, 'return': 145, 'lambda': 145, 'True': 145, 'EMBEDSTAT': 145, 'def': 145, 'finally': 145, 'False': 145, 'raise': 145, 'from': 145, 'except': 144, 'continue': 145, 'try': 145, '@': 145, 'yield': 145, 'global': 145, '~': 145, 'import': 145, 'NUMBER': 145, '{': 145, 'else': 145, '(': 145, 'for': 145, 'DEDENT': 145, 'assert': 145}, 'dictorsetmaker__5': {'EMBEDEXPR': 146, 'NAME': 146, 'lambda': 146, '(': 146, 'True': 146, 'NUMBER': 146, '~': 146, 'None': 146, 'False': 146, '{': 146, '...': 146, 'STRING': 146, '-': 146, 'not': 146, '[': 146, '}': 147, '+': 146}, 'stmt': {'*': 150, 'lambda': 150, 'if': 150, 'True': 150, 'EMBEDSTAT': 150, 'continue': 150, 'False': 150, '...': 150, 'raise': 150, '[': 150, 'from': 150, 'break': 150, 'import': 150, 'while': 150, '(': 150, 'return': 150, 'EMBEDEXPR': 150, 'NAME': 150, 'try': 150, '@': 150, 'yield': 150, 'global': 150, '~': 150, 'pass': 150, 'None': 150, 'NUMBER': 150, '{': 150, 'nonlocal': 150, 'STRING': 150, '-': 150, 'del': 150, 'not': 150, '+': 150, 'for': 150, 'def': 150, 'with': 150, 'class': 150, 'assert': 150}, 'lambdef': {'lambda': 151}, 'import_from': {'from': 152}, 'exprlist__3': {'NEWLINE': 396, ',': 395, ';': 396, 'in': 396}, 'del_stmt': {'del': 369}, 'funcdef': {'def': 164}, 'decorator__1': {'NEWLINE': 166, '(': 165}, 'shift_expr': {'EMBEDEXPR': 167, 'NAME': 167, '(': 167, 'True': 167, 'NUMBER': 167, '~': 167, 'None': 167, 'False': 167, '{': 167, '+': 167, 'STRING': 167, '-': 167, '[': 167, '...': 167}, 'decorators': {'@': 216}, 'while_stmt': {'while': 170}, 'testlist__1': {'EMBEDEXPR': 171, 'NAME': 171, 'lambda': 171, '(': 171, 'True': 171, 'NUMBER': 171, '~': 171, 'None': 171, 'False': 171, '{': 171, '...': 171, 'STRING': 171, '-': 171, 'not': 171, '[': 171, '+': 171}, 'compound_stmt__1': {'for': 174, 'with': 176, '@': 179, 'if': 172, 'try': 175, 'class': 178, 'def': 177, 'while': 173}, 'for_stmt': {'for': 244}, 'typedargslist_item__2': {')': 186, ',': 186, 'NAME': 185}, 'simple_stmt__2': {'NEWLINE': 364, ';': 363}, 'trailer__1': {')': 188, '*': 187, 'NAME': 187, 'lambda': 187, '**': 187, 'EMBEDEXPR': 187, 'True': 187, 'None': 187, '~': 187, 'NUMBER': 187, 'False': 187, '{': 187, '...': 187, 'STRING': 187, '-': 187, 'not': 187, '[': 187, '+': 187, '(': 187}, 'import_as_names': {'NAME': 189}, 'term__2': {'/': 191, '*': 190, '%': 192, '//': 193}, 'test__1': {')': 195, '|=': 195, 'as': 195, ';': 195, '}': 195, '=': 195, '+=': 195, '/=': 195, 'from': 195, ']': 195, '&=': 195, '*=': 195, '//=': 195, '%=': 195, '**=': 195, 'NEWLINE': 195, '-=': 195, 'if': 194, '<<=': 195, '>>=': 195, 'for': 195, ':': 195, ',': 195, '^=': 195}, 'with_item': {'EMBEDEXPR': 286, 'NAME': 286, 'lambda': 286, '(': 286, 'True': 286, 'NUMBER': 286, '~': 286, 'None': 286, 'False': 286, '{': 286, '...': 286, 'STRING': 286, '-': 286, 'not': 286, '[': 286, '+': 286}, 'atom__2': {'*': 196, 'NAME': 196, 'lambda': 196, 'EMBEDEXPR': 196, 'True': 196, 'None': 196, '~': 196, 'NUMBER': 196, 'False': 196, '{': 196, '...': 196, 'STRING': 196, '-': 196, 'not': 196, '[': 196, ']': 197, '+': 196, '(': 196}, 'testlist_comp': {'*': 198, 'NAME': 198, 'lambda': 198, 'EMBEDEXPR': 198, 'True': 198, 'None': 198, '~': 198, 'NUMBER': 198, 'False': 198, '{': 198, '...': 198, 'STRING': 198, '-': 198, 'not': 198, '[': 198, '+': 198, '(': 198}, 'assert_stmt': {'assert': 199}, 'testlist_star_expr__1': {'EMBEDEXPR': 365, 'NAME': 365, 'lambda': 365, '(': 365, 'True': 365, 'NUMBER': 365, '~': 365, 'None': 365, 'False': 365, '{': 365, '...': 365, 'STRING': 365, '-': 365, 'not': 365, '[': 365, '+': 365, '*': 366}, 'atom__4': {'|=': 202, ')': 202, 'as': 202, '&': 202, '**': 202, '*': 202, 'STRING': 201, '}': 202, '=': 202, '/=': 202, '|': 202, '[': 202, ']': 202, 'else': 202, '<': 202, '//=': 202, 'in': 202, '%': 202, 'NEWLINE': 202, '-=': 202, 'if': 202, 'is': 202, '-': 202, 'not': 202, '==': 202, '!=': 202, '^=': 202, '^': 202, ':': 202, '.': 202, ';': 202, 'or': 202, '>>': 202, '+=': 202, '&=': 202, 'from': 202, 'and': 202, '//': 202, '*=': 202, '%=': 202, '<<': 202, '**=': 202, '>>=': 202, '>=': 202, '<<=': 202, '+': 202, '>': 202, '(': 202, '<=': 202, 'for': 202, '/': 202, ',': 202}, 'testlist_comp__2': {')': 204, ',': 204, 'for': 203, ']': 204}, 'break_stmt': {'break': 207}, 'shift_expr__2': {'<<': 208, '>>': 209}, 'lambdef__1': {':': 211, '*': 210, 'NAME': 210, '**': 210}, 'parameters__1': {')': 213, '*': 212, 'NAME': 212, '**': 212}, 'varargslist__3': {':': 215, '*': 214, 'NAME': 214, '**': 214}, 'single_input': {'*': 218, 'lambda': 218, 'if': 219, 'True': 218, 'EMBEDSTAT': 218, 'continue': 218, 'False': 218, 'while': 219, '...': 218, 'raise': 218, '[': 218, 'break': 218, 'from': 218, '+': 218, 'nonlocal': 218, '(': 218, 'return': 218, 'EMBEDEXPR': 218, 'NAME': 218, 'try': 219, '@': 219, 'yield': 218, 'NEWLINE': 217, 'global': 218, '~': 218, 'pass': 218, 'None': 218, 'NUMBER': 218, '{': 218, 'import': 218, 'STRING': 218, '-': 218, 'del': 218, 'not': 218, 'for': 219, 'def': 219, 'with': 219, 'class': 219, 'assert': 218}, 'exprlist__2': {'EMBEDEXPR': 220, 'NAME': 220, '(': 220, 'True': 220, 'NUMBER': 220, '~': 220, 'None': 220, 'False': 220, '{': 220, '+': 220, 'STRING': 220, '-': 220, '[': 220, '...': 220, '*': 221}, 'pass_stmt': {'pass': 222}, 'yield_expr__1': {')': 225, 'EMBEDEXPR': 224, 'NAME': 224, 'lambda': 224, '(': 224, 'True': 224, 'None': 224, '~': 224, ';': 225, 'NUMBER': 224, 'False': 224, '{': 224, '...': 224, 'STRING': 224, '-': 224, 'not': 224, '[': 224, 'NEWLINE': 225, 'from': 223, '+': 224, '=': 225}, 'small_stmt__1': {'*': 227, 'lambda': 227, 'True': 227, 'EMBEDSTAT': 226, 'continue': 230, 'False': 227, '...': 227, 'raise': 230, 'None': 227, '[': 227, 'from': 231, 'break': 230, 'import': 231, 'nonlocal': 233, 'return': 230, 'EMBEDEXPR': 227, 'NAME': 227, 'yield': 230, '(': 227, 'global': 232, '~': 227, 'pass': 229, 'NUMBER': 227, '{': 227, '+': 227, 'STRING': 227, '-': 227, 'del': 228, 'not': 227, 'assert': 234}, 'suite': {'*': 235, 'lambda': 235, 'True': 235, 'EMBEDSTAT': 235, 'continue': 235, 'False': 235, '...': 235, 'raise': 235, '[': 235, 'break': 235, 'from': 235, '+': 235, 'nonlocal': 235, '(': 235, 'return': 235, 'EMBEDEXPR': 235, 'NAME': 235, 'yield': 235, 'NEWLINE': 236, 'global': 235, '~': 235, 'pass': 235, 'None': 235, 'NUMBER': 235, '{': 235, 'import': 235, 'STRING': 235, '-': 235, 'del': 235, 'not': 235, 'assert': 235}, 'comp_for': {'for': 237}, 'varargslist_item__1': {':': 255, ',': 255, '=': 254}, 'trailer': {'(': 239, '.': 241, '[': 240}, 'shift_expr__1': {')': 243, '|=': 243, '|': 243, 'as': 243, '&': 243, '%=': 243, '^': 243, '}': 243, '=': 243, '+=': 243, '>>': 242, '**=': 243, '&=': 243, ']': 243, 'from': 243, 'and': 243, '<': 243, '*=': 243, '!=': 243, ':': 243, 'in': 243, ';': 243, '<<': 242, '/=': 243, 'NEWLINE': 243, '^=': 243, '-=': 243, 'if': 243, '<<=': 243, 'is': 243, 'else': 243, '>': 243, '>>=': 243, '<=': 243, 'for': 243, 'not': 243, ',': 243, '==': 243, '//=': 243, '>=': 243, 'or': 243}, 'flow_stmt__1': {'return': 182, 'break': 180, 'raise': 183, 'continue': 181, 'yield': 184}, 'expr': {'EMBEDEXPR': 245, 'NAME': 245, '(': 245, 'True': 245, 'NUMBER': 245, '~': 245, 'None': 245, 'False': 245, '{': 245, '+': 245, 'STRING': 245, '-': 245, '[': 245, '...': 245}, 'if_stmt__1': {'for': 247, '*': 247, 'lambda': 247, 'DEDENT': 247, 'if': 247, 'True': 247, 'EMBEDSTAT': 247, 'def': 247, 'False': 247, '...': 247, 'raise': 247, 'elif': 246, '[': 247, 'from': 247, 'break': 247, 'import': 247, 'else': 247, 'while': 247, '(': 247, 'continue': 247, 'EMBEDEXPR': 247, 'NAME': 247, 'try': 247, '@': 247, 'yield': 247, 'NEWLINE': 247, 'return': 247, '~': 247, 'pass': 247, 'None': 247, 'NUMBER': 247, '{': 247, 'nonlocal': 247, 'STRING': 247, '-': 247, 'del': 247, 'EOF': 247, '+': 247, 'global': 247, 'with': 247, 'class': 247, 'assert': 247, 'not': 247}, 'raise_stmt': {'raise': 250}, 'subscript': {':': 251, 'EMBEDEXPR': 251, 'NAME': 251, 'lambda': 251, '(': 251, 'True': 251, 'None': 251, '~': 251, 'NUMBER': 251, 'False': 251, '{': 251, '...': 251, 'STRING': 251, '-': 251, 'not': 251, '[': 251, '+': 251}, 'decorator__2': {')': 253, '*': 252, 'NAME': 252, 'lambda': 252, '**': 252, 'EMBEDEXPR': 252, 'True': 252, 'None': 252, '~': 252, 'NUMBER': 252, 'False': 252, '{': 252, '...': 252, 'STRING': 252, '-': 252, 'not': 252, '[': 252, '+': 252, '(': 252}, 'nonlocal_stmt__1': {'NEWLINE': 149, ',': 148, ';': 149}, 'testlist__3': {':': 257, 'EMBEDEXPR': 256, ')': 257, 'NAME': 256, 'lambda': 256, '(': 256, 'True': 256, 'NUMBER': 256, '~': 256, ';': 257, 'None': 256, 'False': 256, '{': 256, '...': 256, 'STRING': 256, '-': 256, 'not': 256, '[': 256, 'NEWLINE': 257, '+': 256, '=': 257}, 'raise_stmt__2': {'NEWLINE': 376, 'from': 375, ';': 376}, 'compound_stmt': {'for': 168, 'try': 168, '@': 168, 'if': 168, 'with': 168, 'class': 168, 'def': 168, 'while': 168}, 'try_stmt__1': {'except': 58, 'finally': 59}, 'and_expr__1': {')': 260, '|=': 260, '|': 260, 'as': 260, '&': 259, '%=': 260, '^': 260, '}': 260, '=': 260, '+=': 260, '/=': 260, '&=': 260, ']': 260, 'from': 260, 'and': 260, '<': 260, '*=': 260, '//=': 260, ':': 260, 'in': 260, ';': 260, '**=': 260, 'NEWLINE': 260, '>=': 260, '-=': 260, 'if': 260, '<<=': 260, 'is': 260, 'else': 260, '>': 260, '>>=': 260, '<=': 260, 'for': 260, 'not': 260, ',': 260, '==': 260, '!=': 260, '^=': 260, 'or': 260}, 'exprlist__1': {'*': 101, 'NAME': 101, 'EMBEDEXPR': 101, 'True': 101, 'NUMBER': 101, '~': 101, 'None': 101, 'False': 101, '{': 101, '...': 101, 'STRING': 101, '-': 101, '[': 101, '+': 101, '(': 101}, 'lambdef_nocond__1': {':': 378, '*': 377, 'NAME': 377, '**': 377}, 'power': {'True': 263, 'EMBEDEXPR': 263, 'NAME': 263, '[': 263, '(': 263, 'STRING': 263, 'None': 263, 'NUMBER': 263, 'False': 263, '{': 263, '...': 263}, 'vfpdef': {'NAME': 264}, 'dotted_name__1': {'(': 266, '.': 265, 'as': 266, ';': 266, 'NEWLINE': 266, ',': 266, 'import': 266}, 'decorators__1': {'def': 268, 'class': 268, '@': 267}, 'decorated__1': {'def': 270, 'class': 269}, 'star_expr': {'*': 271}, 'or_test': {'EMBEDEXPR': 275, 'NAME': 275, '(': 275, 'True': 275, 'NUMBER': 275, '~': 275, 'None': 275, 'False': 275, '{': 275, '+': 275, 'STRING': 275, '-': 275, 'not': 275, '[': 275, '...': 275}, 'parameters': {'(': 258}, 'expr_stmt__2': {'EMBEDEXPR': 277, 'NAME': 277, 'lambda': 277, 'yield': 276, 'True': 277, 'None': 277, '~': 277, 'NUMBER': 277, 'False': 277, '{': 277, '...': 277, 'STRING': 277, '-': 277, 'not': 277, '[': 277, '+': 277, '(': 277}, 'comp_if__1': {')': 279, 'for': 278, ']': 279, '}': 279, ',': 279, 'if': 278}, 'import_from__3': {'.': 393, '...': 394}, 'arith_expr__2': {'-': 281, '+': 280}, 'expr_stmt__3': {'NEWLINE': 283, '=': 282, ';': 283}, 'with_stmt__1': {':': 285, ',': 284}, 'subscriptlist__2': {',': 64, ']': 65}, 'subscript__6': {':': 287, ',': 288, ']': 288}, 'simple_stmt': {'*': 261, 'lambda': 261, 'True': 261, 'EMBEDSTAT': 261, 'continue': 261, 'False': 261, '...': 261, 'raise': 261, '[': 261, 'break': 261, 'from': 261, '+': 261, 'nonlocal': 261, '(': 261, 'return': 261, 'EMBEDEXPR': 261, 'NAME': 261, 'yield': 261, 'global': 261, '~': 261, 'pass': 261, 'None': 261, 'NUMBER': 261, '{': 261, 'import': 261, 'STRING': 261, '-': 261, 'del': 261, 'not': 261, 'assert': 261}, 'import_name': {'import': 289}, 'testlist__2': {':': 291, ')': 291, ';': 291, 'NEWLINE': 291, ',': 290, '=': 291}, 'import_from__1': {'from': 142}, 'yield_expr': {'yield': 143}, 'power__1': {'|=': 296, '*': 296, ')': 296, 'as': 296, '&': 296, '**': 296, '^': 296, '}': 296, '=': 296, '/=': 296, '|': 296, '[': 295, ']': 296, 'else': 296, '<': 296, '//=': 296, 'in': 296, '%': 296, 'NEWLINE': 296, '-=': 296, 'if': 296, 'is': 296, '-': 296, 'not': 296, '==': 296, '!=': 296, '^=': 296, ':': 296, '.': 295, ';': 296, 'or': 296, '>>': 296, '+=': 296, '&=': 296, 'from': 296, 'and': 296, '//': 296, '*=': 296, '%=': 296, '<<': 296, '**=': 296, '>>=': 296, '>=': 296, '<<=': 296, '+': 296, '>': 296, '(': 295, '<=': 296, 'for': 296, '/': 296, ',': 296}, 'raise_stmt__1': {'EMBEDEXPR': 297, 'NAME': 297, 'lambda': 297, '(': 297, 'True': 297, 'NUMBER': 297, '~': 297, ';': 298, 'None': 297, 'False': 297, '{': 297, '...': 297, 'STRING': 297, '-': 297, 'not': 297, '[': 297, 'NEWLINE': 298, '+': 297}, 'nonlocal_stmt': {'nonlocal': 299}, 'with_item__1': {':': 301, ',': 301, 'as': 300}, 'testlist_star_expr__2': {'|=': 303, '%=': 303, '**=': 303, ';': 303, '>>=': 303, '=': 303, '<<=': 303, '+=': 303, '/=': 303, 'NEWLINE': 303, '&=': 303, ',': 302, '^=': 303, '-=': 303, '*=': 303, '//=': 303}, 'dictorsetmaker__6': {',': 304, '}': 305}, 'import_as_names__2': {')': 307, ',': 306, ';': 307, 'NEWLINE': 307}, 'suite__1': {'*': 310, 'lambda': 310, 'DEDENT': 311, 'if': 310, 'True': 310, 'EMBEDSTAT': 310, 'continue': 310, 'False': 310, '...': 310, 'raise': 310, '[': 310, 'from': 310, 'break': 310, 'import': 310, 'while': 310, '(': 310, 'return': 310, 'EMBEDEXPR': 310, 'NAME': 310, 'try': 310, '@': 310, 'yield': 310, 'global': 310, '~': 310, 'pass': 310, 'None': 310, 'NUMBER': 310, '{': 310, 'nonlocal': 310, 'STRING': 310, '-': 310, 'del': 310, 'not': 310, '+': 310, 'for': 310, 'def': 310, 'with': 310, 'class': 310, 'assert': 310}, 'try_stmt': {'try': 317}, 'xor_expr': {'EMBEDEXPR': 314, 'NAME': 314, '(': 314, 'True': 314, 'NUMBER': 314, '~': 314, 'None': 314, 'False': 314, '{': 314, '+': 314, 'STRING': 314, '-': 314, '[': 314, '...': 314}, 'varargslist__2': {':': 316, ',': 315}, 'subscript__1': {':': 313, 'EMBEDEXPR': 312, 'NAME': 312, 'lambda': 312, '(': 312, 'True': 312, 'NUMBER': 312, '~': 312, 'None': 312, 'False': 312, '{': 312, '...': 312, 'STRING': 312, '-': 312, 'not': 312, '[': 312, '+': 312}, 'lambdef_nocond': {'lambda': 319}, 'continue_stmt': {'continue': 320}, 'except_clause__1': {':': 322, 'EMBEDEXPR': 321, 'NAME': 321, 'lambda': 321, '(': 321, 'True': 321, 'NUMBER': 321, '~': 321, 'None': 321, 'False': 321, '{': 321, '...': 321, 'STRING': 321, '-': 321, 'not': 321, '[': 321, '+': 321}, 'dictorsetmaker__3': {',': 324, 'for': 323, '}': 324}, 'term__1': {'|=': 326, '*': 325, ')': 326, 'as': 326, '&': 326, 'if': 326, '}': 326, '=': 326, '/=': 326, ']': 326, 'else': 326, '<': 326, '//=': 326, 'in': 326, '%': 325, 'NEWLINE': 326, '-=': 326, '^': 326, 'is': 326, '-': 326, 'not': 326, '==': 326, '!=': 326, '^=': 326, ':': 326, '|': 326, ';': 326, 'or': 326, '>>': 326, '+=': 326, 'from': 326, '&=': 326, 'and': 326, '//': 325, '*=': 326, '%=': 326, '<<': 326, '**=': 326, '>>=': 326, '>=': 326, '<<=': 326, '+': 326, '>': 326, '<=': 326, 'for': 326, '/': 325, ',': 326}, 'comp_iter': {'for': 327, 'if': 328}, 'subscript__5': {':': 309, 'EMBEDEXPR': 308, 'NAME': 308, 'lambda': 308, '(': 308, 'True': 308, 'NUMBER': 308, '~': 308, 'None': 308, 'False': 308, '{': 308, '...': 308, 'STRING': 308, '-': 308, 'not': 308, '[': 308, ']': 309, ',': 309, '+': 308}, 'if_stmt': {'if': 330}, 'classdef': {'class': 169}, 'while_stmt__1': {'for': 334, '*': 334, 'NUMBER': 334, 'lambda': 334, 'DEDENT': 334, 'if': 334, 'True': 334, 'EMBEDSTAT': 334, 'def': 334, 'False': 334, '...': 334, 'nonlocal': 334, '[': 334, 'from': 334, 'break': 334, 'import': 334, 'while': 334, '(': 334, 'continue': 334, 'EMBEDEXPR': 334, 'NAME': 334, 'try': 334, '@': 334, 'yield': 334, 'NEWLINE': 334, 'return': 334, '~': 334, 'pass': 334, 'None': 334, 'raise': 334, '{': 334, 'else': 333, 'STRING': 334, '-': 334, 'del': 334, 'EOF': 334, '+': 334, 'global': 334, 'with': 334, 'class': 334, 'assert': 334, 'not': 334}, 'try_stmt__3': {'for': 336, '*': 336, 'NUMBER': 336, 'lambda': 336, 'DEDENT': 336, 'if': 336, 'True': 336, 'EMBEDSTAT': 336, 'def': 336, 'finally': 336, '...': 336, '+': 336, 'nonlocal': 336, '[': 336, 'from': 336, 'break': 336, 'import': 336, 'while': 336, '(': 336, 'continue': 336, 'EMBEDEXPR': 336, 'NAME': 336, 'try': 336, '@': 336, 'yield': 336, 'NEWLINE': 336, 'return': 336, '~': 336, 'pass': 336, 'None': 336, 'raise': 336, '{': 336, 'else': 335, 'STRING': 336, '-': 336, 'del': 336, 'EOF': 336, 'False': 336, 'global': 336, 'with': 336, 'class': 336, 'assert': 336, 'not': 336}, 'subscriptlist__3': {':': 337, 'EMBEDEXPR': 337, 'NAME': 337, 'lambda': 337, '(': 337, 'True': 337, 'None': 337, '~': 337, 'NUMBER': 337, 'False': 337, '{': 337, '...': 337, 'STRING': 337, '-': 337, 'not': 337, '[': 337, ']': 338, '+': 337}, 'argument__1': {')': 341, ',': 341, '=': 340, 'for': 339}, 'funcdef__1': {':': 343, '->': 342}, 'varargslist_item__2': {':': 345, ',': 345, 'NAME': 344}, 'decorator': {'@': 104}, 'if_stmt__2': {'for': 349, '*': 349, 'NUMBER': 349, 'lambda': 349, 'DEDENT': 349, 'if': 349, 'True': 349, 'EMBEDSTAT': 349, 'def': 349, 'False': 349, '...': 349, 'nonlocal': 349, '[': 349, 'from': 349, 'break': 349, 'import': 349, 'while': 349, '(': 349, 'continue': 349, 'EMBEDEXPR': 349, 'NAME': 349, 'try': 349, '@': 349, 'yield': 349, 'NEWLINE': 349, 'return': 349, '~': 349, 'pass': 349, 'None': 349, 'raise': 349, '{': 349, 'else': 348, 'STRING': 349, '-': 349, 'del': 349, 'EOF': 349, '+': 349, 'global': 349, 'with': 349, 'class': 349, 'assert': 349, 'not': 349}, 'varargslist__1': {'*': 350, 'NAME': 350, '**': 350}, 'not_test': {'EMBEDEXPR': 352, 'NAME': 352, '(': 352, 'True': 352, 'NUMBER': 352, '~': 352, 'None': 352, 'False': 352, '{': 352, '...': 352, 'STRING': 352, '-': 352, 'not': 351, '[': 352, '+': 352}, 'and_test__1': {')': 354, '|=': 354, 'as': 354, 'if': 354, '}': 354, '=': 354, '/=': 354, '+=': 354, 'from': 354, ']': 354, '&=': 354, 'and': 353, '*=': 354, '//=': 354, '%=': 354, '**=': 354, 'NEWLINE': 354, '-=': 354, ';': 354, '<<=': 354, 'else': 354, '>>=': 354, 'for': 354, ':': 354, ',': 354, '^=': 354, 'or': 354}, 'import_as_name__1': {')': 356, 'NEWLINE': 356, ';': 356, 'as': 355, ',': 356}, 'import_from__6': {'*': 357, 'NAME': 359, '(': 358}, 'typedargslist__1': {'*': 329, 'NAME': 329, '**': 329}, 'import_from__4': {'import': 362, '.': 360, 'NAME': 362, '...': 361}, 'file_input': {'for': 379, '*': 379, 'lambda': 379, 'if': 379, 'True': 379, 'EMBEDSTAT': 379, 'def': 379, 'False': 379, '...': 379, 'raise': 379, '[': 379, 'from': 379, 'break': 379, 'import': 379, 'while': 379, '(': 379, 'continue': 379, 'EMBEDEXPR': 379, 'NAME': 379, 'try': 379, '@': 379, 'yield': 379, 'NEWLINE': 379, 'return': 379, '~': 379, 'pass': 379, 'None': 379, 'NUMBER': 379, '{': 379, 'nonlocal': 379, 'STRING': 379, '-': 379, 'del': 379, 'EOF': 379, '+': 379, 'global': 379, 'with': 379, 'class': 379, 'assert': 379, 'not': 379}, 'expr_stmt__4': {'*': 332, 'NAME': 332, 'lambda': 332, 'yield': 331, 'True': 332, 'None': 332, '~': 332, 'NUMBER': 332, 'False': 332, '{': 332, '...': 332, 'STRING': 332, '+': 332, 'not': 332, '[': 332, 'EMBEDEXPR': 332, '-': 332, '(': 332}, 'arith_expr': {'EMBEDEXPR': 200, 'NAME': 200, '(': 200, 'True': 200, 'NUMBER': 200, '~': 200, 'None': 200, 'False': 200, '{': 200, '+': 200, 'STRING': 200, '-': 200, '[': 200, '...': 200}, 'yield_stmt': {'yield': 368}, 'term': {'EMBEDEXPR': 238, 'NAME': 238, '(': 238, 'True': 238, 'NUMBER': 238, '~': 238, 'None': 238, 'False': 238, '{': 238, '+': 238, 'STRING': 238, '-': 238, '[': 238, '...': 238}, 'exprlist__4': {'*': 370, 'NAME': 370, 'in': 371, 'EMBEDEXPR': 370, 'True': 370, 'NUMBER': 370, '~': 370, ';': 371, 'None': 370, 'False': 370, '{': 370, '...': 370, 'STRING': 370, '-': 370, '[': 370, 'NEWLINE': 371, '+': 370, '(': 370}, 'small_stmt': {'*': 372, 'lambda': 372, 'True': 372, 'EMBEDSTAT': 372, 'continue': 372, 'False': 372, '...': 372, 'raise': 372, '[': 372, 'break': 372, 'from': 372, '+': 372, 'nonlocal': 372, '(': 372, 'return': 372, 'EMBEDEXPR': 372, 'NAME': 372, 'yield': 372, 'global': 372, '~': 372, 'pass': 372, 'None': 372, 'NUMBER': 372, '{': 372, 'import': 372, 'STRING': 372, '-': 372, 'del': 372, 'not': 372, 'assert': 372}, 'dictorsetmaker__4': {',': 373, '}': 374}, 'arglist__2': {')': 206, ',': 205}, 'dotted_as_name': {'NAME': 262}, 'try_stmt__4': {'for': 384, '*': 384, 'lambda': 384, 'DEDENT': 384, 'if': 384, 'True': 384, 'EMBEDSTAT': 384, 'def': 384, 'finally': 383, '...': 384, '+': 384, 'raise': 384, '[': 384, 'from': 384, 'break': 384, 'import': 384, 'while': 384, '(': 384, 'continue': 384, 'EMBEDEXPR': 384, 'NAME': 384, 'try': 384, '@': 384, 'yield': 384, 'NEWLINE': 384, 'return': 384, '~': 384, 'pass': 384, 'None': 384, 'NUMBER': 384, '{': 384, 'nonlocal': 384, 'STRING': 384, '-': 384, 'del': 384, 'EOF': 384, 'False': 384, 'global': 384, 'with': 384, 'class': 384, 'assert': 384, 'not': 384}, 'typedargslist_item': {'*': 273, 'NAME': 272, '**': 274}, 'testlist_comp__3': {')': 249, ',': 248, ']': 249}, 'sliceop__1': {'EMBEDEXPR': 381, 'NAME': 381, 'lambda': 381, '(': 381, 'True': 381, 'NUMBER': 381, '~': 381, 'None': 381, 'False': 381, '{': 381, '...': 381, 'STRING': 381, '-': 381, 'not': 381, '[': 381, ']': 382, ',': 382, '+': 381}, 'arglist': {'*': 318, 'NAME': 318, 'lambda': 318, '**': 318, 'EMBEDEXPR': 318, 'True': 318, 'None': 318, '~': 318, 'NUMBER': 318, 'False': 318, '{': 318, '...': 318, 'STRING': 318, '-': 318, 'not': 318, '[': 318, '+': 318, '(': 318}, 'with_stmt': {'with': 385}, 'varargslist_item': {'*': 387, 'NAME': 386, '**': 388}, 'expr__1': {')': 390, '|=': 390, '|': 389, 'as': 390, '%=': 390, 'if': 390, '}': 390, '=': 390, '+=': 390, '/=': 390, '&=': 390, ']': 390, 'from': 390, 'and': 390, '<': 390, '*=': 390, '//=': 390, ':': 390, 'in': 390, '**=': 390, 'NEWLINE': 390, '>=': 390, '-=': 390, ';': 390, '<<=': 390, 'is': 390, 'else': 390, '>': 390, '>>=': 390, '<=': 390, 'for': 390, 'not': 390, ',': 390, '==': 390, '!=': 390, '^=': 390, 'or': 390}, 'arglist__3': {')': 392, '*': 391, 'NAME': 391, 'lambda': 391, '**': 391, 'EMBEDEXPR': 391, 'True': 391, 'None': 391, '~': 391, 'NUMBER': 391, 'False': 391, '{': 391, '...': 391, 'STRING': 391, '-': 391, 'not': 391, '[': 391, '+': 391, '(': 391}, 'atom': {'True': 161, '(': 153, 'NAME': 156, '[': 154, 'EMBEDEXPR': 163, 'STRING': 158, 'None': 160, 'NUMBER': 157, 'False': 162, '{': 155, '...': 159}}
