/*  
 * File:   SelectObject.cpp
 * Author: Marcel Duris
 * 
 * Created on February 19, 2012, 4:01 PM
 */

#include "SelectObject.h"
#include "GrabcutAction.h"
#include <string>
#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <iostream>
#include <cmath>

using namespace cv;
using namespace std;

const string GrabcutAction::FS_FGD = "fgdPxls";
const string GrabcutAction::FS_BGD = "bgdPxls";
const string GrabcutAction::FS_PR_FGD = "prFgdPxls";
const string GrabcutAction::FS_PR_BGD = "prBgdPxls";
const string GrabcutAction::FS_ROI = "roiRect";

GrabcutAction::GrabcutAction(Mat img) {
    if (img.channels() != 3) {
        cvtColor(img, orig, CV_GRAY2RGB);
    } else {
        img.copyTo(orig);
    }
    normalize(orig, orig, 0, 255, CV_MINMAX);
    orig.convertTo(orig, CV_8UC3);

    init();
}

GrabcutAction::GrabcutAction(Mat img, string outfile) {
    if (img.channels() != 3) {
        cvtColor(img, orig, CV_GRAY2RGB);
    } else {
        img.copyTo(orig);
    }
    normalize(orig, orig, 0, 255, CV_MINMAX);
    orig.convertTo(orig, CV_8UC3);

    init();

    this->outfile = outfile;
}

GrabcutAction::GrabcutAction(const GrabcutAction& orig) {
}

GrabcutAction::~GrabcutAction() {
}

void GrabcutAction::init() {
    roiBegin.x = -1;
    roiBegin.y = -1;
    roiEnd.x = -1;
    roiEnd.y = -1;

    //roiRect = Rect(-1, -1, -1, -1);
    roiRect = Rect(0, 0, orig.cols, orig.rows);

    FGC = Scalar(0, 0, 255);
    PFGC = Scalar(230, 130, 255);
    PBGC = Scalar(255, 0, 0);
    BGC = Scalar(255, 255, 160);
    GREEN = Scalar(0, 255, 0);
    WHITE = Scalar(255, 255, 255);
    BLACK = Scalar(0, 0, 0);

    phase = SELECT_FGD;
    isInitialized = false;

    mask.release();

    bgdPxls.clear();
    fgdPxls.clear();
    prBgdPxls.clear();
    prFgdPxls.clear();
}

void GrabcutAction::loadFromMask(Mat mask, Mat &dst) {
    grabCut(orig, mask, roiRect, bgdModel, fgdModel, 1);
    orig.copyTo(dst, mask & 1);
}

void GrabcutAction::loadFromFile(string file, Mat &dst) {
    load(file);
    phase = SELECT_BGD + 1;
    nextIter();
    orig.copyTo(dst, mask & 1);
}

void GrabcutAction::addPoint(Point p) {

    switch (phase) {
        case SELECT_ROI: //Select ROI
            if (roiBegin.x == -1) {
                roiBegin = p;
            } else {
                roiEnd = roiBegin;
                roiBegin = p;

                int x = min(roiBegin.x, roiEnd.x);
                int y = min(roiBegin.y, roiEnd.y);

                int w = max(roiBegin.x, roiEnd.x) - x;
                int h = max(roiBegin.y, roiEnd.y) - y;

                roiRect = Rect(x, y, w, h);
            }
            break;

        case SELECT_FGD: //FGD
            fgdPxls.push_back(p);
            break;
        case SELECT_PR_FGD: //PR FGD
            prFgdPxls.push_back(p);
            break;
        case SELECT_PR_BGD: //PR BGD
            prBgdPxls.push_back(p);
            break;
        case SELECT_BGD: //BGD
            bgdPxls.push_back(p);
            break;
    }

}

void GrabcutAction::processMouseInput(int event, int x, int y, int flags) {
    if (event == CV_EVENT_LBUTTONDOWN) {
        addPoint(Point(x, y));
    }
}

Mat GrabcutAction::getImg() {
    Mat img;
    if (isInitialized) {
        orig.copyTo(img, mask & 1);
    } else {
        orig.copyTo(img);
    }

    if (roiRect.x != -1) {
        rectangle(img, roiRect, BLACK, 2);
    }

    vector< Point >::iterator it;

    for (it = fgdPxls.begin(); it != fgdPxls.end(); ++it) {
        circle(img, *it, RADIUS, FGC, -1);
        circle(img, *it, RADIUS, BLACK);
    }
    for (it = prFgdPxls.begin(); it != prFgdPxls.end(); ++it) {
        circle(img, *it, RADIUS, PFGC, -1);
        circle(img, *it, RADIUS, BLACK);
    }
    for (it = prBgdPxls.begin(); it != prBgdPxls.end(); ++it) {
        circle(img, *it, RADIUS, PBGC, -1);
        circle(img, *it, RADIUS, BLACK);
    }
    for (it = bgdPxls.begin(); it != bgdPxls.end(); ++it) {
        circle(img, *it, RADIUS, BGC, -1);
        circle(img, *it, RADIUS, BLACK);
    }

    Mat result;
    img.copyTo(result);

    return result;
}

Mat GrabcutAction::getMask() {
    Mat result;

    if (mask.empty()) {
        return result;
    }
    
    pushPixelsToMask(true);

    mask.copyTo(result);

    return result;
}

void GrabcutAction::processKeyboardInput(int key) {

    if (key == 'a') {
        phase++;

        switch (phase) {
            case SELECT_ROI:
                cout << "Select Region of Interest\n";
                break;
            case SELECT_FGD:
                cout << "Select Foreground Pixels\n";
                break;
            case SELECT_PR_FGD:
                cout << "Select Probable Foreground Pixels\n";
                break;
            case SELECT_PR_BGD:
                cout << "Select Probable Background Pixels\n";
                break;
            case SELECT_BGD:
                cout << "Select Background Pixels\n";
                break;
        }

        if (phase > SELECT_BGD) {
            nextIter();
            phase = SELECT_FGD;
        }

    }

    if ((key == 'q') || (key == 'n')) {
        phase = FINISHED;
    }

    if (key == 'r') {
        init();
    }

    if (key == 's') {
        //Perzistencia nie je riesena systemovo, tak radsej nech nie je vobec
        //save(outfile);
    }

}
void GrabcutAction::pushPixelsToMask(bool forceReset) {
    
    if (mask.empty() || forceReset) {
        mask.create(orig.size(), CV_8U);
        mask.setTo(GC_BGD);
        (mask(roiRect)).setTo(Scalar(GC_PR_FGD));
    }

    vector< Point >::iterator it;
    for (it = fgdPxls.begin(); it != fgdPxls.end(); ++it) {
        circle(mask, *it, 2, GC_FGD, -1);
    }
    for (it = prFgdPxls.begin(); it != prFgdPxls.end(); ++it) {
        circle(mask, *it, 2, GC_PR_FGD, -1);
    }
    for (it = prBgdPxls.begin(); it != prBgdPxls.end(); ++it) {
        circle(mask, *it, 2, GC_PR_BGD, -1);
    }
    for (it = bgdPxls.begin(); it != bgdPxls.end(); ++it) {
        circle(mask, *it, 2, GC_BGD, -1);
    }
}

void GrabcutAction::nextIter() {
    if (phase <= SELECT_BGD) {
        return;
    }

    pushPixelsToMask();
    
    if (isInitialized) {
        Rect r;
        Point seed = fgdPxls.back();
        //floodFill( orig, seed, Scalar(255), &r, Scalar(10), Scalar(10));
        grabCut(orig, mask, roiRect, bgdModel, fgdModel, 1);
    } else {
        grabCut(orig, mask, roiRect, bgdModel, fgdModel, 1, GC_INIT_WITH_MASK);

        isInitialized = true;
    }

    //    bgdPxls.clear(); fgdPxls.clear();
    //    prBgdPxls.clear(); prFgdPxls.clear();
}

bool GrabcutAction::finished() {
    return phase == FINISHED;
}

void GrabcutAction::save(string output) {
    if (output.length() == 0) {
        cout << "Could not save to file, filename is empty.\n";
        return;
    }

    FileStorage fs(output, FileStorage::WRITE);

    fs << FS_ROI;
    fs << "{:";
    fs << "x" << roiRect.x;
    fs << "y" << roiRect.y;
    fs << "w" << roiRect.width;
    fs << "h" << roiRect.height;
    fs << "}";

    //== Foreground Pixels ==
    fs << FS_FGD;
    fs << "[:";
    if (!fgdPxls.empty()) fs << fgdPxls;
    fs << "]";

    //== Background Pixels ==
    fs << FS_BGD;
    fs << "[:";
    if (!bgdPxls.empty()) fs << bgdPxls;
    fs << "]";

    //== Probably Foreground Pixels ==
    fs << FS_PR_FGD;
    fs << "[:";
    if (!prFgdPxls.empty()) fs << prFgdPxls;
    fs << "]";

    //== Probably Background Pixels ==
    fs << FS_PR_BGD;
    fs << "[:";
    if (!prBgdPxls.empty()) fs << prBgdPxls;
    fs << "]";

    fs.release();
}

void GrabcutAction::load(string input) {
    FileStorage fs(input, FileStorage::READ);

    FileNode fsRoi = fs[FS_ROI];
    fsRoi["x"] >> roiRect.x;
    fsRoi["y"] >> roiRect.y;
    fsRoi["w"] >> roiRect.width;
    fsRoi["h"] >> roiRect.height;

    fs[FS_FGD] >> fgdPxls;
    fs[FS_BGD] >> bgdPxls;
    fs[FS_PR_FGD] >> prFgdPxls;
    fs[FS_PR_BGD] >> prBgdPxls;

    fs.release();
}
