/* 
 * File:   fourier.cpp
 * Author: Marcel Duris
 *
 * Created on January 24, 2012, 10:35 PM
 */

#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>


#include "utils.h"
#include "fourier.h"
#include <opencv2/imgproc/imgproc_c.h>

#include <iostream>
 
using namespace cv;
using namespace std;

void oponentModel(Mat src, Mat &dst) {
    Mat colorPlanes[3];
    split(src, colorPlanes);

    Mat R;
    add(colorPlanes[0], colorPlanes[1], R);
    divide(R, Mat(R.size(), R.type(), Scalar(2)), R);
    subtract(colorPlanes[2], R, R);

    Mat G;
    add(colorPlanes[0], colorPlanes[2], G);
    divide(G, Mat(R.size(), R.type(), Scalar(2)), G);
    subtract(colorPlanes[1], G, G);

    Mat B;
    add(colorPlanes[1], colorPlanes[2], B);
    divide(B, Mat(R.size(), R.type(), Scalar(2)), B);
    subtract(colorPlanes[0], B, B);

    Mat Y;
    Mat Ytmp1;
    Mat Ytmp2;
    add(colorPlanes[1], colorPlanes[2], Ytmp1);
    divide(Ytmp1, Mat(R.size(), R.type(), Scalar(2)), Ytmp1);
    absdiff(colorPlanes[1], colorPlanes[2], Ytmp2);
    divide(Ytmp1, Mat(R.size(), R.type(), Scalar(2)), Ytmp2);
    subtract(Ytmp1, Ytmp2, Y);
    subtract(Y, colorPlanes[2], Y);

    Mat RG;
    subtract(R, G, RG);
    Mat BY;
    subtract(B, Y, BY);
    Mat I;
    add(colorPlanes[0], colorPlanes[1], I);
    add(colorPlanes[2], I, I);
    divide(I, Mat(R.size(), R.type(), Scalar(3)), I);

    colorPlanes[0] = RG;
    colorPlanes[1] = BY;
    colorPlanes[2] = I;

    merge(colorPlanes, 3, dst);
}

/* Fourierova transformacia len pre ucely power spectra*/
void fourier(Mat I, Mat &dst){
    if( I.empty())
        return;
    
    Mat padded;                            //expand input image to optimal size
    int m = getOptimalDFTSize( I.rows );
    int n = getOptimalDFTSize( I.cols ); // on the border add zero values
    copyMakeBorder(I, padded, 0, m - I.rows, 0, n - I.cols, BORDER_CONSTANT, Scalar::all(0));
    Mat planes[] = {padded, Mat::zeros(padded.size(), CV_32F)};
    Mat complexI;
    merge(planes, 2, complexI);         // Add to the expanded another plane with zeros
    
    dft(complexI, complexI);            // this way the result may fit in the source matrix
    
    // compute the magnitude and switch to logarithmic scale
    // => log(1 + sqrt(Re(DFT(I))^2 + Im(DFT(I))^2))
    split(complexI, planes);                   // planes[0] = Re(DFT(I), planes[1] = Im(DFT(I))
    magnitude(planes[0], planes[1], planes[0]);// planes[0] = magnitude  
    Mat magI = planes[0];
    
    //magI += Scalar::all(1);                    // switch to logarithmic scale
    //log(magI, magI);

    magI = magI(Rect(0, 0, magI.cols & -2, magI.rows & -2));

    // rearrange the quadrants of Fourier image  so that the origin is at the image center        
    int cx = magI.cols/2;
    int cy = magI.rows/2;

    Mat q0(magI, Rect(0, 0, cx, cy));   // Top-Left - Create a ROI per quadrant 
    Mat q1(magI, Rect(cx, 0, cx, cy));  // Top-Right
    Mat q2(magI, Rect(0, cy, cx, cy));  // Bottom-Left
    Mat q3(magI, Rect(cx, cy, cx, cy)); // Bottom-Right

    Mat tmp;                           // swap quadrants (Top-Left with Bottom-Right)
    q0.copyTo(tmp);
    q3.copyTo(q0);
    tmp.copyTo(q3);

    q1.copyTo(tmp);                    // swap quadrant (Top-Right with Bottom-Left)
    q2.copyTo(q1);
    tmp.copyTo(q2);
    magI += Scalar::all(1);                    // switch to logarithmic scale
    log(magI, magI);
    
    magI.copyTo(dst);
}

float powerSpectrum(Mat src, float colF){
    Mat magI;
    
    fourier(src, magI);
    
    IplImage stmp = magI;
    IplImage *isrc = &stmp;
    IplImage *idst = cvCloneImage(isrc);
    cvLogPolar(isrc, idst, cvPoint2D32f(isrc->width / 2, isrc->height / 2), 20, CV_INTER_LINEAR+CV_WARP_FILL_OUTLIERS);
    
    Mat result(idst);
    
    vector<Point2f> toPlot1;
    vector<Point2f> toPlot2;
    vector<Point2f> toPlot3;
    
    for (int i = 0; i < result.cols * colF; i ++){
        float value;
        Mat col = result.col(i);
        value = norm(col, NORM_L1);
        //value /= result.rows;
        toPlot1.push_back(Point2f(i, value));
        
        col += Scalar::all(1);//logaritmujem iba raz
        log(col, col);
        value = norm(col, NORM_L1);
        toPlot2.push_back(Point2f(i, value));
        
        col += Scalar::all(1);//ale druhykrat treba, aby som to pekne vykreslil
        log(col, col);
        value = norm(col, NORM_L1);
        toPlot3.push_back(Point2f(i, value));
        
        //cout << value << " ";
    }
    
    Vec4f res;
    Mat line(toPlot3);
    fitLine(line, res, CV_DIST_L2, 0, 0.01, 0.01);// hladam exponent, preto plot3 a nie plot2
    
    return -10 * res[1] / res[0];
    
    magI += Scalar::all(1);                    // switch to logarithmic scale
    log(magI, magI);

    normalize(magI, magI, 0, 1, CV_MINMAX); // Transform the matrix with float values into a 
    
}

void partialPowerSpectrum(Mat src, Mat &dst, int blockSize) {
    
    if (src.channels() != 1){
        cout << "Chcem ciernobiely obrazok!";
        return;
    }
    
    if (src.depth() != CV_32F) {
        src.convertTo(src, CV_32F);
        dst.convertTo(dst, CV_32F);
    }
    
    Mat tmp;
    
    anidiff(src, src, 10, 0.25, 10);
        
    float ispectrum = powerSpectrum(src, 0.25);
    
    int rowCount = src.rows / blockSize;
    int colCount = src.cols / blockSize;
    
    Mat_<float> result(rowCount * blockSize, colCount * blockSize);
    
    for (int row = 0; row < rowCount; row++) {
        for (int col = 0; col < colCount; col++) {
            int w = blockSize;//(col + 1) * blockSize > src.cols ? src.cols - col * blockSize : blockSize;
            int h = blockSize;//(row + 1) * blockSize > src.rows ? src.rows - row * blockSize : blockSize;
            
            Mat roi(src, Rect(col * blockSize, row * blockSize, w, h));
            Mat roiDst(result, Rect(col * blockSize, row * blockSize, w, h));
            
            float lspectrum = powerSpectrum(roi, 1);
            
            float metric = (lspectrum - ispectrum) / ispectrum;
            
            roiDst = Scalar::all(metric);
        }
    }
    
    multiply(result, Mat(result.size(), CV_32F, Scalar(-1)), result);
    result += Scalar::all(1);
    log(result, result);
    
    normalize(result, result, 0, 1, CV_MINMAX);
    
    dst = result;
}

void symPad(Mat src, Mat &dst){
    int M = src.rows;//getOptimalDFTSize(img.rows);
    int N = src.cols;//getOptimalDFTSize(img.cols);
    
    Mat large(M * 2, N * 2, src.type());
    
    Mat q0(large, Rect(0, 0, N, M));
    Mat q1(large, Rect(N, 0, N, M));
    Mat q2(large, Rect(0, M, N, M));
    Mat q3(large, Rect(N, M, N, M));
    
    //TU
    M = 2 * src.rows;//getOptimalDFTSize(2 * src.rows);
    N = 2 * src.rows;//getOptimalDFTSize(2 * src.cols);
    
    src.copyTo(q0);
    flip(src, q3, -1);
    
    src.copyTo(q1);
    flip(src, q2, -1);

    Mat colorPlanes[3];
    split(large, colorPlanes);
    
    int borderRows = 0;//M - 2 * src.rows;
    int borderCols = 0;//N - 2 * src.cols;
    
    copyMakeBorder(large, dst, 0, borderRows, 0, borderCols, BORDER_CONSTANT, Scalar::all(0));
}

void quaternionSpectrum(Mat img, Mat &dst) {

    Mat padded;
    symPad(img, padded);
        
    Mat colorPlanes[3];
    split(padded, colorPlanes);
    Mat paddedI = colorPlanes[0];
    Mat paddedJ = colorPlanes[1];
    Mat paddedK = colorPlanes[2];
    
    Mat planesSimplex[] = {Mat::zeros(paddedI.size(), CV_32F), Mat_<float>(paddedI)};
    Mat complexSimplex;
    merge(planesSimplex, 2, complexSimplex);

    dft(complexSimplex, complexSimplex);

    Mat planesPerplex[] = {Mat_<float>(paddedJ), Mat_<float>(paddedK)};
    Mat complexPerplex;
    merge(planesPerplex, 2, complexPerplex);

    dft(complexPerplex, complexPerplex);

    // Compute quaternion phase, normalize to unit length
    split(complexSimplex, planesSimplex);
    magnitude(planesSimplex[0], planesSimplex[1], planesSimplex[0]);
    split(complexPerplex, planesPerplex);
    magnitude(planesPerplex[0], planesPerplex[1], planesPerplex[0]);

    magnitude(planesSimplex[0], planesPerplex[0], planesSimplex[0]);
    Mat mag = planesSimplex[0];

    split(complexSimplex, planesSimplex);
    divide(planesSimplex[0], mag, planesSimplex[0]);
    divide(planesSimplex[1], mag, planesSimplex[1]);
    merge(planesSimplex, 2, complexSimplex);

    split(complexPerplex, planesPerplex);
    divide(planesPerplex[0], mag, planesPerplex[0]);
    divide(planesPerplex[1], mag, planesPerplex[1]);
    merge(planesPerplex, 2, complexPerplex);

    // Inverse Fourier Transform: Quaternion phase -> Saliency map
    dft(complexSimplex, complexSimplex, DFT_INVERSE + DFT_SCALE);
    dft(complexPerplex, complexPerplex, DFT_INVERSE + DFT_SCALE);

    // Quaternion simplex part magnitude
    split(complexSimplex, planesSimplex);
    magnitude(planesSimplex[0], planesSimplex[1], planesSimplex[0]);
    // Quaternion perplex part magnitude
    split(complexPerplex, planesPerplex);
    magnitude(planesPerplex[0], planesPerplex[1], planesPerplex[0]);
    // Overal magnitude and final saliency map
    magnitude(planesSimplex[0], planesPerplex[0], planesSimplex[0]);
    mag = planesSimplex[0];

    mag = mag(Rect(0, 0, img.cols, img.rows));

    normalize(mag, dst, 0, 1, CV_MINMAX);
}

void simpleSpectrum(Mat img, Mat &dst) { 
    //Simple Spectrum neslobodno ratat z oponent modelu
    
    Mat_<float> f, mag, padded;
    Mat_<float> planes[2];
    cvtColor( img, img, CV_RGB2GRAY );
    
    img.convertTo(img, CV_32FC1);
    normalize(img, img, 0, 1, CV_MINMAX);
    int m = getOptimalDFTSize( img.rows );
    int n = getOptimalDFTSize( img.cols ); // on the border add zero values
    copyMakeBorder(img, padded, 0, m - img.rows, 0, n - img.cols, BORDER_CONSTANT, Scalar::all(0));
     
    Mat tmp[] = {padded, Mat::zeros(padded.size(), CV_32F)};
    merge(tmp, 2, f);
    
    dft(f, f);
    
    split(f, planes);
    magnitude(planes[0], planes[1], mag);
    
    divide(planes[0], mag, planes[0]);//toto odkomentuj
    divide(planes[1], mag, planes[1]);//toto zakomentuj
    magnitude(planes[0], planes[1], mag);
    merge(planes, 2, f);
    
    dft(f, f, DFT_INVERSE + DFT_SCALE);
    split(f, planes);
    magnitude(planes[0], planes[1], mag);
    
    mag = mag(Rect(0, 0, img.cols, img.rows));
    
    dst.create(mag.size(), CV_32F);
    mag.copyTo(dst);    
}

void saliency(Mat src, Mat &dst, bool useSimpleTransform, bool useOponent) {
    if (src.depth() != CV_32F) {
        src.convertTo(src, CV_32F);
        dst.convertTo(dst, CV_32F);
    }
    
    if (src.channels() != 3){
        cout << "Potrebujem farebny obrazok!\n"; 
        return;
    }

    Mat diff;
    if (useOponent) {
        oponentModel(src, diff);
    } else {
        diff = src;
    }
    
    if (useSimpleTransform) {
        simpleSpectrum(diff, dst);
    } else {
        quaternionSpectrum(diff, dst);
    }

    pow(dst, 2, dst);    
}

void getThrustum(Mat &dst, int border) {
    Mat result(dst.size(), CV_32F, Scalar(1));
    
    for (int dist = 0; dist < border; dist++) {
        float val = (float)(dist) / border;
        for (int col = 0; col < result.cols; col++) {
            result.at<float>(col, dist) = val;
            result.at<float>(col, dst.rows - dist - 1) = val;
        }
    }
    
    dst = result;
    Mat colm = result.clone();
    multiply(result, colm.t(), dst);    
}

void globalSaliency(Mat src, Mat &dst) {
    Mat img;
    
    src.convertTo(img, CV_32F);
    normalize(img, img, 0, 1, CV_MINMAX);
    
    saliency(img, dst, false, false);
    normalize(dst, dst, 0, 1, CV_MINMAX);
}

void sectionedSaliency(Mat src, Mat &dst, int blockSize) {
        
    if (src.depth() != CV_32F) {
        src.convertTo(src, CV_32F);
        dst.convertTo(dst, CV_32F);
    }
    
    normalize(src, src, 0, 1, CV_MINMAX);
    
    int overlap = blockSize / 3;
    int offset = overlap / 2;
    int rowCount = src.rows / overlap - 2;
    int colCount = src.cols / overlap - 2;
    
    
    Mat_<float> weight(blockSize, blockSize);
    getThrustum(weight, overlap);
    
    Mat_<float> result((rowCount + 2) * overlap + offset, (colCount + 2) * overlap + offset);
    Mat divisor(result.size(), CV_32F, Scalar(0));
    
    copyMakeBorder(src, src, 0, blockSize, 0, blockSize, BORDER_CONSTANT, Scalar::all(0));
    
    for (int ow = 0; ow < 1; ow++) {
        for (int row = 0; row < rowCount; row++){
            for (int col = 0; col < colCount; col++){
                Rect r(ow * offset + col * overlap, ow * offset + row * overlap, blockSize, blockSize);
                Mat roi(src, r);
                Mat roiDst(result, r);
                Mat roiDiv(divisor, r);
                Mat sal;
                saliency(roi, sal);
                multiply(sal, weight, sal);
                roiDst += sal;
                roiDiv += weight;
            }
        }
    }
    
    divide(result, divisor, result);
    Mat_<float> tmp(result, Rect(0, 0, (colCount + 2) * overlap, (rowCount + 2) * overlap));
    normalize(tmp, dst, 0, 1, CV_MINMAX);    
}

void combinedSaliency(Mat src, Mat &dst) {
    Mat img, sal, sals, salg;
    
    src.convertTo(img, CV_32F);
    
    normalize(img, img, 0, 1, CV_MINMAX);

    sectionedSaliency(img, sals, 150);
    saliency(img, salg);

    multiply(salg, sals, sal);
    sqrt(sal, dst);
}
