/* 
 * File:   fourier.h
 * Author: Marcel Duris
 *
 * Created on January 24, 2012, 10:35 PM
 * 
 * Implementacia zakladnych metod vypoctu saliencie pomocou Fourierovych
 * transformacii.
 */

#ifndef FOURIER_H
#define	FOURIER_H

#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>

#include <iostream>
 
using namespace cv;
using namespace std;

#define SALIENCY_GL_GEOMETRIC 1;

float powerSpectrum(Mat src, float colF);

/**
 * Testovacia funkcia na ziskanie power spectra, tento smer neposkytol dobre vysledky
 * 
 * @param src vstup
 * @param dst vystup
 * @param blockSize po ako velkych polickach ma ratat hodnoty
 */
void partialPowerSpectrum(Mat src, Mat &dst, int blockSize = 50);

/**
 * Vypocet fazoveho spektra Fourierovej transformacie a inverzna transformacia 
 * na mapu saliencie. Funkcia neumrie, ak pustena so zlymi parametrami, ale 
 * neposkytne ziaden vysledok.
 * 
 * Funkcia nie je mienena na priame pouzitie, namiesto nej treba pouzit 
 * funkciu saliency(src, dst, useSimpleTransform, useOponent).
 * 
 * @param img Vstupny farebny obrazok
 * @param dst Vystup
 */
void quaternionSpectrum(Mat img, Mat &dst);

/**
 * Jednoduchsia verzia mapy saliencie pomocou Fourierovej transformacie 
 * intenzity. Lepsie vysledky poskytuje funkcia quaternionoveho spektra.
 * 
 * Funkcia nie je mienena na priame pouzitie, namiesto nej treba pouzit 
 * funkciu globalSaliency(src, dst, useSimpleTransform, useOponent).
 * 
 * @param img Vsupny obrazok
 * @param dst Vystup
 */
void simpleSpectrum(Mat img, Mat &dst);

/**
 * Vypocita mapu saliencie. Umoznuje vyuzit pristup zalozeny na fazovom spektre
 * kvatenrionovej Fourierovej transformacie ale aj jednoduchsi pristup zalozeny
 * na klasickej transformacii. Vychodzie hodnoty by nemalo byt potrebne
 * upravovat. Kombinacia jednoduchej verzie a oponentoveho farebneho modelu dava
 * obzvlast zle vysledky.
 * 
 * Namiesto tejto funkcie je vhodnejsie pouzit funkcie globalSaliency, 
 * localSaliency alebo combinedSaliency, nakolko tie vykonaju este miernu
 * normalizaciu.
 * 
 * @param src Vstup, musi byt farebny pre kvaternionovu mapu
 * @param dst Nenormalizovany vystup
 * @param useSimpleTransform Ak true, pouzije sa jednoduchsia verzia
 * @param useOponent Ak true, pouzije sa oponent farebny bodel.
 */
void saliency(Mat src, Mat &dst, bool useSimpleTransform = false, bool useOponent = false); 

/**
 * Kombinuje vysledky lokalnej a globalnej mapy saliencie.
 * 
 * @param src Vstup
 * @param dst Vystup
 */
void combinedSaliency(Mat src, Mat &dst);

/**
 * Vypocita mapu saliencie z globalneho hladiska zohladnujuceho cely vstup. 
 * Pouziva Fourierovu transformaciu na kvaternionoch.
 * 
 * @param src Farebny vstup
 * @param dst Vystup
 */
void globalSaliency(Mat src, Mat &dst);

/**
 * Vypocita mapu saliencie na mensej skale a odstrani tak miernu falosnu 
 * salienciu na velkych plochach, ktore vznikaju napriklad pocas hmly.
 * 
 * @param src Farebny vstup
 * @param dst Vystup
 * @param blockSize Velkost bloku, v ktorom sa rata lokalna saliencia
 */
void sectionedSaliency(Mat src, Mat &dst, int blockSize = 150);

#endif	/* FOURIER_H */

