/* 
 * File:   main.cpp
 * Author: Marcel Duris
 *
 * Created on October 1, 2011, 3:41 PM
 */

#include <cstdlib>
#include <cstdio>
#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>

#include <iostream>
#include <fstream>
#include <sys/types.h>
#include <dirent.h>
#include <string>

#include "utils.h"
#include "fourier.h"
#include "SelectObject.h"
#include "GrabcutAction.h"
#include "RoiAction.h"
#include "tests.h"
#include "FloodfillAction.h"
#include "ImageTestSet.h"

using namespace cv;
using namespace std;

/*
 * Rozne cinnosti, ktore vieme vykonat:
 *      - Vypocitat odpoved charakteristickych hran
 *      - Vypocitat odpoved mriezky Gaborovych filtrov
 *      - Zostavit sablony pre charakteristicke hrany
 *      - Zostavit sablony pre mriezky Gaborovych filrov
 *      - Vytiahnut sablony zo suboru popisov do samostatnych bitmap
 */
enum Methods {
    EDGE, GRID, EDGE_MASK, GRID_MASK, SPLIT
};

/**
 * Help.
 */
void help() {
    cout << "Pouzitie: visibility <metoda> <subor_sablon> <obrazok> [--showResp]\n";
    cout << "metoda:\n";
    cout << "\t--edge \t\t- detekcia hran v mape saliencie\n";
    cout << "\t--grid \t\t- mriezka odpovedi Gaborovho filtra\n";
    cout << "\t--edgeMask \t- zostavenie databazy pre hrany v mape saliencie\n";
    cout << "\t--gridMask \t- zostavenie databazy pre mriezku Gaborovho filtra\n";

    cout << "\n\t--showResp - zobrazenie ciselnych hodnot odpovedi algoritmu\n";
}

/**
 * Vybuduje subor popisov pre dany algoritmus.
 * 
 * @param imgFile Cesta k stupnemu suboru s obrazkom
 * @param maskFile Cesta k vystupnemu suboru .yml 
 * @param method Pouzity algoritmus
 */
void buildMask(string imgFile, string maskFile, int method) {
    cout << "Vstup: " << imgFile << endl;
    cout << "Subor sablon:" << maskFile << endl;

    Mat img;
    vector<maskRegion> masks;

    img = imread(imgFile);

    if (method == GRID_MASK) {
        gridMask(img, masks);
    } else if (method == EDGE_MASK) {
        edgeMask(img, masks);
    } else {
        cout << "Neznama metoda " << method << endl;
        return;
    }

    saveMaskDesc(masks, maskFile);

    cout << "Sablony boli ulozene do suboru " << maskFile << endl << endl;
}

/**
 * Vypise odpovede algoritmu.
 * 
 * @param imgFile cesta k vstupnemu obrazku
 * @param maskFile cesta k suboru popisu sablon
 * @param method pouzity algoritmus
 * @param noResp ak true vypise y/n, inak vypise ciselne odpovede algoritmu
 */
void printResponses(string imgFile, string maskFile, int method, bool noResp = true) {
    cout << "Vstup: " << imgFile << endl;
    cout << "Subor sablon:" << maskFile << endl;

    Mat img;
    vector<maskRegion> masks;
    vector<float> resps;

    img = imread(imgFile);
    loadMaskDesc(maskFile, masks);

    if (method == GRID) {
        gridResponse(img, masks, resps);
    } else if (method == EDGE) {
        edgeResponse(img, masks, resps);
    } else {
        cout << "Neznama metoda " << method << endl;
        return;
    }

    vector<float>::iterator it = resps.begin(), it_end = resps.end();

    for (int i = 0; it != it_end; ++it, i++) {
        cout << i << " ";

        if (noResp) {
            cout << ((masks[i].threshold <= (*it)) ? "y" : "n") << endl;
        } else {
            cout << *it << endl;
        }
    }

    cout << "===" << endl;

}

/**
 * Rozdeli subor obsahujuci sablony na jednotlive obrazky pre kazdu sablonu.
 * 
 * @param maskFile cesta k suboru obsahujucemu popisy sablon
 * @param outPath printf vyraz popisujuci cestu k vystupnym suborom, mal by 
 *      obsahovat prave raz kluc %0d
 */
void splitMasks(string maskFile, const char *outPath) {
    vector<maskRegion> masks;
    loadMaskDesc(maskFile, masks);

    saveMaskDescSeparate(masks, outPath);
}

/*
 * Hlavna metoda.
 */
int main(int argc, char** argv) {
    
    if (argc > 2) {
        int method = -1;
        if (strcmp(argv[1], "--edge") == 0) {
            method = EDGE;
        } else if (strcmp(argv[1], "--grid") == 0) {
            method = GRID;
        } else if (strcmp(argv[1], "--edgeMask") == 0) {
            method = EDGE_MASK;
        } else if (strcmp(argv[1], "--gridMask") == 0) {
            method = GRID_MASK;
        } else if (strcmp(argv[1], "--split") == 0) {
            splitMasks(argv[2], argv[3]);
        } else {
            cout << "Unknown method " << method << endl;
            return 1;
        }

        bool noResp = true;
        if (argc > 4) {
            if (strcmp(argv[4], "--showResp") == 0) {
                noResp = false;
            }
        }

        if ((method == EDGE) || (method == GRID)) {
            printResponses(argv[3], argv[2], method, noResp);
        }

        if ((method == EDGE_MASK) || (method == GRID_MASK)) {
            buildMask(argv[3], argv[2], method);
        }

        return 0;
    } else {
        help();
    }

    return 0;

}

