/* 
 * File:   DescriptorProcessor.cpp
 * Author: Marcel Duris
 * 
 * Created on December 11, 2011, 3:32 PM
 */

#include <opencv2/imgproc/imgproc_c.h>


#include "utils.h"

#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>

#include <string>
#include <cstdio>
#include <time.h>

#include "SelectAction.h"
#include "FloodfillAction.h"
#include "RoiAction.h"
#include "GrabcutAction.h"
#include "PixelAction.h"
#include "SelectObject.h"
#include "fourier.h"

using namespace cv;

string tbSize = "Size";
string tbLambda = "Lambda";
string tbTheta = "Theta";
string tbPsi = "Psi";
string tbSigma = "Sigma";
string tbGamma = "Gamma";

string iwindow = "image";
string twindow = "trackbars";
string owindow = "original";

void conduction(Mat src, Mat &dst, float kappa, int type = 0) {

    Mat divisor = Mat(src.size(), src.type(), kappa);

    divide(src, divisor, dst);
    pow(dst, 2, dst);

    if (type == 0) {
        // First type exp(-(delta/kappa).^2);
        exp(-1 * dst, dst);
    } else {
        // Second type 1./(1 + (deltaN/kappa).^2);
        add(dst, 1, dst);
        divide(Mat(dst.size(), dst.type(), Scalar(1)), dst, dst);
    }
}

void addConduction(Mat c, Mat delta, float lambda, Mat &dst) {

    multiply(c, delta, c);
    c *= lambda;

    add(c, dst, dst);
}

void anidiff(Mat src, Mat& dst, float kappa, float lambda, int iterCount) {

    // pre parametre k=10, l=0.25, ic=10 sa mi zda, ze je to dost podobne ako
    // izraelske difuzovanie beltramim
    /*float kappa = 10;
    float lambda = 0.25;
    int iterCount = 10;*/
    for (int i = 0; i < iterCount; i++) {

        // Differences in each direction
        Mat deltaN;
        Mat deltaS;
        Mat deltaE;
        Mat deltaW;

        //tato temna magia urobi to, ze posunie vstup o jeden pixel do kazdeho smeru
        // a potom od posunuteho obrazku odcita povodny,
        copyMakeBorder(src, deltaN, 1, 0, 0, 0, BORDER_CONSTANT, Scalar::all(0));
        deltaN = deltaN(Rect(0, 0, src.cols, src.rows));
        deltaN -= src;
        copyMakeBorder(src, deltaS, 0, 1, 0, 0, BORDER_CONSTANT, Scalar::all(0));
        deltaS = deltaS(Rect(0, 1, src.cols, src.rows));
        deltaS -= src;
        copyMakeBorder(src, deltaE, 0, 0, 1, 0, BORDER_CONSTANT, Scalar::all(0));
        deltaE = deltaE(Rect(0, 0, src.cols, src.rows));
        deltaE -= src;
        copyMakeBorder(src, deltaW, 0, 0, 0, 1, BORDER_CONSTANT, Scalar::all(0));
        deltaW = deltaW(Rect(1, 0, src.cols, src.rows));
        deltaW -= src;

        // Vykona tepelnu conduction, mozu byt rozne typy
        Mat cN, cS, cE, cW;
        conduction(deltaN, cN, kappa, 1);
        conduction(deltaS, cS, kappa, 1);
        conduction(deltaE, cE, kappa, 1);
        conduction(deltaW, cW, kappa, 1);

        // Result
        // diff = diff + lambda*(cN.*deltaN + cS.*deltaS + cE.*deltaE + cW.*deltaW);
        addConduction(cN, deltaN, lambda, src);
        addConduction(cS, deltaS, lambda, src);
        addConduction(cE, deltaE, lambda, src);
        addConduction(cW, deltaW, lambda, src);

        //add(src, src, src);
    }

    src.copyTo(dst);

}

void anidiffColor(Mat src, Mat &dst, float kappa, float lambda, int iterCount) {
    // = 10, float lambda = 0.25, int iterCount = 10) {
    Mat colorPlanes[3];
    split(src, colorPlanes);

    for (int i = 0; i < 3; i++) {
        anidiff(colorPlanes[i], colorPlanes[i], kappa, lambda, iterCount);
    }

    merge(colorPlanes, 3, dst);
}

float xprime(float x, float y, float theta) {
    return x * cos(theta) + y * sin(theta);
}

float yprime(float x, float y, float theta) {
    return -1 * x * sin(theta) + y * cos(theta);
}

Mat getGabor(int width, int height, float lambda, float theta, float psi, float sigma, float gamma) {
    int xoffset = width / 2;
    int yoffset = height / 2;
    Mat_<float> kernel = Mat(width, height, CV_32F);
    for (int x = 0; x < width; x++) {
        for (int y = 0; y < height; y++) {
            //cvGet2D(kernel, x, y)[0] = cos(2 * CV_PI * xprime(x / 100.0, y / 100.0, theta) / lambda + psi);
            int xo = x - xoffset;
            int yo = y - yoffset;
            float gauss = exp(
                    -1 * (pow(xprime(xo, yo, theta), 2) + pow(gamma, 2) * pow(yprime(xo, yo, theta), 2)) /
                    (2.0 * pow(sigma, 2)));
            float wave = cos(2 * CV_PI * xprime(xo, yo, theta) / lambda + psi);

            kernel(x, y) = gauss * wave;
        }
    }

    return kernel;
}

void getGradient(Mat src, Mat &dst) {

    //  Mat src, src_gray;
    Mat src_gray;
    Mat grad;

    int scale = 1;
    int delta = 0;
    int ddepth = CV_16S;

    if (!src.data) {
        dst = Mat();
        return;
    }

    GaussianBlur(src, src, Size(3, 3), 0, 0, BORDER_DEFAULT);

    /// Convert it to gray
    if (src.channels() > 1) {
        cvtColor(src, src_gray, CV_RGB2GRAY);
    } else {
        src_gray = src;
    }

    /// Generate grad_x and grad_y
    Mat grad_x, grad_y;
    Mat abs_grad_x, abs_grad_y;

    /// Gradient X
    //Scharr( src_gray, grad_x, ddepth, 1, 0, scale, delta, BORDER_DEFAULT );
    Sobel(src_gray, grad_x, ddepth, 1, 0, 3, scale, delta, BORDER_DEFAULT);
    convertScaleAbs(grad_x, abs_grad_x);

    /// Gradient Y
    //Scharr( src_gray, grad_y, ddepth, 0, 1, scale, delta, BORDER_DEFAULT );
    Sobel(src_gray, grad_y, ddepth, 0, 1, 3, scale, delta, BORDER_DEFAULT);
    convertScaleAbs(grad_y, abs_grad_y);

    /// Total Gradient (approximate)
    addWeighted(abs_grad_x, 0.5, abs_grad_y, 0.5, 0, grad);
    //imshow("Sobel Demo - Simple Edge Detector", grad);
    //waitKey();

    dst = grad;

}

void scaleImage(int width, void* imgPtr) {
    float aspectRatio = 16 / 10;
    float w = width;
    float h = w / aspectRatio;

    Mat img = *((Mat*) imgPtr);

    float ratio = min(w / img.cols, h / img.rows);

    resize(img, img, Size(floor(img.cols * ratio), floor(img.rows * ratio)));
    imshow(owindow, img);
}

void onTrackbarChange(int pos, void* ptr) {
    int size = getTrackbarPos(tbSize, twindow);
    float lambda = 20.0 * getTrackbarPos(tbLambda, twindow) / 1000;
    float theta = 2 * CV_PI * getTrackbarPos(tbTheta, twindow) / 1000;
    float psi = 2 * CV_PI * getTrackbarPos(tbPsi, twindow) / 1000;
    float sigma = 10.0 * getTrackbarPos(tbSigma, twindow) / 1000;
    float gamma = 10.0 * getTrackbarPos(tbGamma, twindow) / 1000;

    Mat kernel = getGabor(size, size, lambda, theta, psi, sigma, gamma);

    cout << "size: " << size;
    cout << " lambda: " << lambda;
    cout << " theta: " << theta;
    cout << " psi: " << psi;
    cout << " sigma: " << sigma;
    cout << " gamma: " << gamma << "\n";

    Mat img = *((Mat*) ptr);
    Mat dst;
    filter2D(img, dst, -1, kernel);

    imshow(iwindow, dst);
}

void setPseudocolor(float intensity, Point pos, Mat planes[]) {
    int x = pos.y;
    int y = pos.x;
    intensity = 1 - intensity;

    if (intensity < 0.2) {
        planes[0].at<float>(x, y) = (0.2 - intensity) * 5;
        planes[1].at<float>(x, y) = 0; //
        //planes[1].at<float>(x, y) = ((1 - intensity) - 0.75) * 4;//
        planes[2].at<float>(x, y) = 1;
        return;
    }

    if (intensity < 0.4) {
        planes[0].at<float>(x, y) = 0;
        planes[1].at<float>(x, y) = (intensity - 0.2) * 5; //
        //planes[1].at<float>(x, y) = ((1 - intensity) - 0.75) * 4;//
        planes[2].at<float>(x, y) = 1;
        return;
    }

    if (intensity < 0.6) {
        planes[0].at<float>(x, y) = 0;
        planes[1].at<float>(x, y) = 1;
        //planes[2].at<float>(x, y) = (intensity - 0.25) * 4;//
        planes[2].at<float>(x, y) = (0.2 - (intensity - 0.4)) * 5; //
        return;
    }

    if (intensity < 0.8) {
        planes[0].at<float>(x, y) = (intensity - 0.6) * 5; //
        //planes[0].at<float>(x, y) = ((0.75 - intensity) - 0) * 4;//
        planes[1].at<float>(x, y) = 1;
        planes[2].at<float>(x, y) = 0;
        return;
    }

    if (intensity <= 1) {
        planes[0].at<float>(x, y) = 1;
        planes[1].at<float>(x, y) = ((1 - intensity)) * 5; //
        //planes[1].at<float>(x, y) = (intensity - 0.75) * 4;;//
        planes[2].at<float>(x, y) = 0;
    }


}

void demonstrateGabor(Mat img) {

    float aspectRatio = 16 / 10;
    float w = 800;
    float h = w / aspectRatio;
    float ratio = min(w / img.cols, h / img.rows);
    resize(img, img, Size(floor(img.cols * ratio), floor(img.rows * ratio)));
    Mat *ptr = &img;


    //=================Original=================================================
    scaleImage(800, ptr);
    imshow(owindow, img);

    int size = 50;
    int lambda = 250; //20/1000 5
    int theta = 250; //2pi/1000 ..
    int psi = 250; //2pi/1000 0
    int sigma = 200; //10/1000 2
    int gamma = 100; //10/1000 1

    //=================Settings=================================================
    namedWindow(twindow, CV_WINDOW_NORMAL);
    createTrackbar(tbSize, twindow, &size, 50, onTrackbarChange, ptr);
    createTrackbar(tbLambda, twindow, &lambda, 1000, onTrackbarChange, ptr);
    createTrackbar(tbTheta, twindow, &theta, 1000, onTrackbarChange, ptr);
    createTrackbar(tbPsi, twindow, &psi, 1000, onTrackbarChange, ptr);
    createTrackbar(tbSigma, twindow, &sigma, 1000, onTrackbarChange, ptr);
    createTrackbar(tbGamma, twindow, &gamma, 1000, onTrackbarChange, ptr);

    //=================Gabor====================================================
    namedWindow(iwindow, CV_WINDOW_KEEPRATIO);
    onTrackbarChange(0, ptr);

    while ('q' != waitKey());

}

float getFloatAverage(Mat img) {
    float result = 0;
    for (int row = 0; row < img.rows; row++) {
        for (int col = 0; col < img.cols; col++) {
            result += img.at<float>(row, col);
        }
    }

    return result / (img.cols * img.rows);
}

float getFloatMax(Mat img) {
    float result = 0;
    for (int row = 0; row < img.rows; row++) {
        for (int col = 0; col < img.cols; col++) {
            if (result < img.at<float>(row, col)) {
                result = img.at<float>(row, col);
            }
        }
    }

    return result;
}

void getAverageResponseGrid(Mat img, Mat &dst, int size, float dir) {
    Mat_<float> filteredF, filteredB;
    Mat kernel;

    kernel = getGabor(20, 20, 6, dir, 1 * CV_PI / 2, 1, 0.6);
    filter2D(img, filteredF, -1, kernel);
    filteredF = abs(filteredF);

    kernel = getGabor(20, 20, 6, dir, 3 * CV_PI / 2, 1, 0.6);
    filter2D(img, filteredB, -1, kernel);
    filteredB = abs(filteredB);

    int rows = img.rows / size;
    int cols = img.cols / size;

    Mat result(rows, cols, CV_32FC1, Scalar(0));

    for (int row = 0; row < rows; row++) {
        for (int col = 0; col < cols; col++) {
            //priemer policka
            Rect roi(col * size, row * size, size, size);

            //minMaxIdx( filtered( roi ), NULL, &max, NULL, NULL);
            float avg = getFloatAverage(filteredF(roi)) + getFloatAverage(filteredB(roi)) / 2;
            result.at<float>(row, col) = avg;
        }
    }

    dst = result;

}

void insertNoise(Mat &img, float intensity, int type) {

    if ((img.type() != CV_8UC1) && (img.type() != CV_8UC3)) {
        cout << "Unable to insert noise, need 1 or 3 channel 8U image\n";
        return;
    }

    int noisedPixels = round(img.cols * img.rows * intensity);
    int x, y;

    srand(time(NULL));

    for (int i = 0; i < noisedPixels; i++) {
        x = rand() % img.cols;
        y = rand() % img.rows;

        switch (type) {
            case WHITE:
                if (img.channels() == 1) {
                    img.at<unsigned char>(y, x) = 255;
                }
                if (img.channels() == 3) {
                    img.at<Vec3b > (y, x) = Vec3b(255, 255, 255);
                }
                break;
            case BLACK:
                if (img.channels() == 1) {
                    img.at<unsigned char>(y, x) = 0;
                }
                if (img.channels() == 3) {
                    img.at<Vec3b > (y, x) = Vec3b(0, 0, 0);
                }
                break;
            case COLOR:
                if (img.channels() == 1) {
                    img.at<unsigned char>(y, x) = rand() % 256;
                }
                if (img.channels() == 3) {
                    img.at<Vec3b > (y, x) = Vec3b(rand() % 256, rand() % 256, rand() % 256);
                }
                break;
            case SALT_AND_PEPPER:
                int val = rand() % 2;
                if (img.channels() == 1) {
                    img.at<unsigned char>(y, x) = val;
                }
                if (img.channels() == 3) {
                    img.at<Vec3b > (y, x) = Vec3b(val, val, val);
                }
                break;
        }
    }
}

void pseudocolor(Mat src, Mat &dst) {
    Mat_<float> f = src;
    normalize(f, f, 0, 1, CV_MINMAX);

    Mat planes[] = {
        Mat::zeros(f.size(), CV_32F),
        Mat::zeros(f.size(), CV_32F),
        Mat::zeros(f.size(), CV_32F)
    };

    Mat_<float>::iterator it = f.begin();
    Mat_<float>::iterator end = f.end();
    for (; it != end; it++) {
        setPseudocolor((*it), it.pos(), planes);
    }

    merge(planes, 3, dst);
}

/**
 * Nominativ singularu nazvu smeru z cisla.
 * 
 * @param step
 * @return 
 */
string getDirForStep(int step) {
    if (step == 0) {
        return "horizontalny";
    }

    if (step == 1) {
        return "vertikalny";
    }
}

void getSignificantEdges(Mat img, Mat &dst, int cellSize) {
    Mat tmp, mask;
    img.convertTo(tmp, CV_32FC1);
    normalize(tmp, tmp, 0, 1, CV_MINMAX);

    mask.create(img.size(), CV_8U);

    Mat_<float> iedges;
    threshold(tmp, tmp, 0.5, 1, CV_THRESH_BINARY);
    integral(tmp, iedges);

    iedges.convertTo(iedges, CV_32FC1);

    int rows = img.rows / cellSize;
    int cols = img.cols / cellSize;

    float l, s, s1, s0, r;
    for (int row = 0; row < rows; row++) {
        for (int col = 0; col < cols; col++) {
            l = iedges.at<float>(cellSize * row + cellSize - 1, cellSize * col + cellSize - 1);

            s1 = iedges.at<float>(cellSize * row + cellSize - 1, cellSize * col);

            s0 = iedges.at<float>(cellSize * row, cellSize * col + cellSize - 1);

            s = iedges.at<float>(cellSize * row, cellSize * col);

            r = l - s1 - s0 + s;

            Rect roiRect(cellSize * col, cellSize * row, cellSize, cellSize);
            Scalar res;
            if (r < cellSize * 1.75) {
                res = Scalar(255);
            } else {
                res = Scalar(0);
            }
            Mat(mask, roiRect).setTo(res);
        }
    }

    img.copyTo(dst, mask);
}

float shapeSimilarity(Mat pattern, Mat img) {
    Mat masked;

    if (pattern.channels() == 3) {
        cvtColor(pattern, pattern, CV_RGB2GRAY);
    }

    if (img.channels() == 3) {
        cvtColor(img, img, CV_RGB2GRAY);
    }

    normalize(pattern, pattern, 0, 255, CV_MINMAX);
    normalize(img, img, 0, 255, CV_MINMAX);

    img.copyTo(masked, pattern); //nie len 0 1 ale gausianovat

    Mat tmp;
    integral(pattern, tmp);
    tmp.convertTo(tmp, CV_32FC1);
    float max = tmp.at<float>(masked.rows - 1, masked.cols - 1);
    if (max == 0) {
        max = 1;
    }

    integral(masked, tmp);
    tmp.convertTo(tmp, CV_32FC1);

    return tmp.at<float>(masked.rows - 1, masked.cols - 1) / max;
}

void detectEdges(Mat img, Mat &dst) {
    if (img.channels() == 3) {
        cvtColor(img, img, CV_RGB2GRAY);
    }

    img.convertTo(img, CV_32FC1);

    normalize(img, img, 0, 1, CV_MINMAX);
    anidiff(img, img, 0.15, 0.2, 10);

    // Format vstupu pre Canny musi byt takyto
    img *= 255;
    img.convertTo(img, CV_8UC1);

    int cellSize = 20;
    Rect roi(0, 0, img.cols - img.cols % cellSize, img.rows - img.rows % cellSize);

    Mat detected, edges;
    Canny(img, detected, 15, 15 * CANNY_EDGE_R, 3);

    getSignificantEdges(Mat(detected, roi), edges, cellSize);

    Canny(img(roi), detected, 80, 80 * CANNY_EDGE_R, 3);

    detected += edges;

    dst.create(detected.size(), detected.type());
    detected.copyTo(dst);

}

bool buildEdgeMask(Mat img, Mat &mask) {
    int steps = 2;
    float step = CV_PI / steps;
    Mat detected, masks[3], roi;
    bool cont = false;
    img.copyTo(roi);
    char wname[256];

    if (roi.channels() != 1) {
        cvtColor(roi, roi, CV_RGB2GRAY);
    }

    detectEdges(roi, detected);

    for (int i = 0; i < steps; i++) {

        Mat tmp, kernel = getGabor(20, 20, 6, step * i, 1.5708, 1, 0.6);
        filter2D(detected, tmp, -1, kernel);
        GaussianBlur(tmp, tmp, Size(3, 3), 1.5);
        normalize(tmp, tmp, 0, 255, CV_MINMAX);
        threshold(tmp, tmp, 20, 255, CV_THRESH_BINARY);

        FloodfillAction edgesAction(img(Rect(0, 0, tmp.cols, tmp.rows)), tmp);

        sprintf(wname, "Smer vyberanych hran: %s", getDirForStep(i).c_str());

        cout << wname << endl << endl;

        SelectObject selectEdges(wname, &edgesAction);
        selectEdges.getUserInput(masks[i]);

        cont = selectEdges.continueSelecting();

    }

    masks[2].create(masks[0].size(), masks[0].type());
    masks[2].setTo(Scalar(0));

    merge(masks, 3, mask);

    return cont;
}

void edgeMask(Mat img, vector<maskRegion> &dst, bool append) {
    Mat roi, mask;
    bool cont = true;

    if (!append) {
        dst.clear();
    }

    while (cont) {

        RoiAction ra(img);
        SelectObject so("Vyber pozorovaneho objektu", &ra);
        so.getUserInput(roi);

        cont = buildEdgeMask(roi, mask);

        maskRegion p;
        mask.copyTo(p.mask);
        p.roi = ra.getRoi();
        p.gridSize = -1;
        dst.push_back(p);
    }
}

void edgeMask(Mat img, vector<maskRegion> &dst, vector<maskRegion> rois, bool append) {
    Mat mask;

    if (!append) {
        dst.clear();
    }

    vector<maskRegion>::iterator it = rois.begin(), it_end = rois.end();

    //globalSaliency(img, img);

    for (; it != it_end; ++it) {
        buildEdgeMask(img((*it).roi), mask);

        maskRegion p;
        mask.copyTo(p.mask);
        p.roi = (*it).roi;
        p.gridSize = -1;
        p.threshold = -1;
        dst.push_back(p);
    }

}

float edgeResponse(Mat img, Mat mask, int method, int gaborDirs) {

    img.convertTo(img, CV_32F);
    normalize(img, img, 0, 1, CV_MINMAX);
    //Difuzia na celociselnych vstupoch ma sice pekne vysledky, 
    //take farebne prechody, ostre a neviem co, ale zahadzuju toho vela

    switch (method) {
        case RESPONSE_SAL:
            globalSaliency(img, img);
            break;
        case RESPONSE_LOC:
            sectionedSaliency(img, img);
            break;
        case RESPONSE_CMB:
            combinedSaliency(img, img);
            break;
    }

    Mat detected, filtered;
    detectEdges(img, detected);

    Mat maskChannels[3];
    split(mask, maskChannels);

    float resp = 1;
    float count = 0;
    for (int dir = 0; dir < gaborDirs; dir++) {
        Mat kernel = getGabor(20, 20, 6, dir * CV_PI / gaborDirs, 1.5708, 1, 0.6);
        filter2D(detected, filtered, -1, kernel);
        GaussianBlur(filtered, filtered, Size(3, 3), 1.5);
        threshold(filtered, filtered, 20, 255, CV_THRESH_BINARY);

        float dirResp = shapeSimilarity(maskChannels[dir], filtered);
        if (dirResp > 0) {
            resp *= dirResp;
            count++;
        }
    }

    if (count > 1) {
        return sqrt(resp);
    } else {
        if (count == 1) {
            return resp;
        }
    }

    return 0;
}

void edgeResponse(Mat img, vector<maskRegion> masks, vector<float> &dst) {
    vector<maskRegion>::iterator it = masks.begin(), it_end = masks.end();

    dst.clear();

    for (int i = 0; it != it_end; ++it, i++) {
        float resp = edgeResponse(img((*it).roi), (*it).mask);
        dst.push_back(resp);
    }
}

void grabcutMask(Mat img, vector<maskRegion> &dst) {
    Mat tmp, mask;
    bool cont = true;

    while (cont) {
        RoiAction ra(img);
        SelectObject so("Select ROI", &ra);
        so.getUserInput(tmp);

        globalSaliency(img(ra.getRoi()), tmp);

        tmp.convertTo(tmp, CV_32F);
        normalize(tmp, tmp, 0, 1, CV_MINMAX);
        anidiff(tmp, tmp, 0.15, 0.2, 10);
        normalize(tmp, tmp, 0, 255, CV_MINMAX);
        tmp.convertTo(tmp, img.type());
        pseudocolor(tmp, tmp);

        GrabcutAction gc(tmp);
        SelectObject gs("Select Background and Foreground", &gc);
        gs.getUserInput(mask);

        cont = gs.continueSelecting();

        maskRegion p;
        mask.copyTo(p.mask);
        p.roi = ra.getRoi();
        p.gridSize = -1;
        dst.push_back(p);
    }
}

void gridMask(Mat img, vector<maskRegion> &dst, bool append) {

    Mat masks[3], mask, tmp, show;
    char wname[256];

    if (!append) {
        dst.clear();
    }

    int gridSize = 5;
    int steps = 2;
    bool cont = true;

    while (cont) {

        RoiAction ra(img);
        SelectObject so("Vyber pozorovaneho objektu", &ra);
        so.getUserInput(tmp);

        if (tmp.channels() != 1) {
            cvtColor(tmp, tmp, CV_RGB2GRAY);
        }

        if (tmp.type() != CV_32F) {
            tmp.convertTo(tmp, CV_32F);
            tmp = tmp / 255;
        }

        for (int i = 0; i < steps; i++) {
            Mat resp;

            getAverageResponseGrid(tmp, resp, gridSize, CV_PI / steps * i);
            resize(tmp, show, resp.size(), 0, 0);
            PixelAction pa(show, resp);

            sprintf(wname, "Smer vyberanych hran: %s", getDirForStep(i).c_str());

            cout << wname << endl << endl;

            SelectObject sp(wname, &pa);

            sp.getUserInput(masks[i]);

            cont = sp.continueSelecting();

        }

        masks[2].create(masks[0].size(), masks[0].type());
        masks[2].setTo(Scalar(0));
        merge(masks, 3, mask);

        maskRegion p;
        mask.copyTo(p.mask);
        p.roi = ra.getRoi();
        p.gridSize = gridSize;
        p.threshold = -1;

        dst.push_back(p);
    }

}

float gridResponse(Mat img, Mat mask, int steps) {

    Mat tmp, grid, maskChannels[3];
    float step = CV_PI / steps, resp = 1, count = 0;

    split(mask, maskChannels);

    img.convertTo(tmp, CV_32F);
    if (tmp.channels() == 3) {
        cvtColor(tmp, tmp, CV_RGB2GRAY);
    }

    normalize(tmp, tmp, 0, 1, CV_MINMAX);
    anidiff(tmp, tmp, 0.25, 0.2, 10);

    for (int i = 0; i < steps; i++) {
        Mat kernel;
        kernel = getGabor(20, 20, 6, step * i, 1 * CV_PI / 2, 1, 0.6);
        getAverageResponseGrid(tmp, grid, 5, step * i);
        threshold(grid, grid, GRID_THRES, 1, CV_THRESH_BINARY);

        float dirResp = shapeSimilarity(maskChannels[i], grid * 255);
        if (dirResp > 0) {
            resp *= dirResp;
            count++;
        }

    }

    if (count > 1) {
        return sqrt(resp);
    } else {
        if (count == 1) {
            return resp;
        }
    }

    return 0;
}

void gridResponse(Mat img, vector<maskRegion> masks, vector<float> &dst) {
    vector<maskRegion>::iterator it = masks.begin(), it_end = masks.end();
    dst.clear();

    for (int i = 0; it != it_end; ++it, i++) {
        float resp = gridResponse(img((*it).roi), (*it).mask);
        dst.push_back(resp);
    }
}

void saveMaskDesc(vector<maskRegion> descs, string outputFile) {

    FileStorage fs(outputFile, FileStorage::WRITE);

    int count = descs.size();

    fs << "pairs" << "[";
    for (int i = 0; i < count; i++) {
        fs << "{";
        fs << "mask" << descs[i].mask;

        fs << "x" << descs[i].roi.x;
        fs << "y" << descs[i].roi.y;
        fs << "h" << descs[i].roi.height;
        fs << "w" << descs[i].roi.width;

        fs << "threshold" << descs[i].threshold;
        fs << "}";
    }
    fs << "]";

    fs.release();
}

void loadMaskDesc(string outputFile, vector<maskRegion> &masks) {
    FileStorage fs(outputFile, FileStorage::READ);

    FileNode pairsNode = fs["pairs"];
    FileNodeIterator it = pairsNode.begin(), it_end = pairsNode.end();

    for (; it != it_end; ++it) {
        maskRegion pair;
        (*it)["mask"] >> pair.mask;

        int x = (int) (*it)["x"];
        int y = (int) (*it)["y"];
        int h = (int) (*it)["h"];
        int w = (int) (*it)["w"];

        pair.roi = Rect(x, y, w, h);

        pair.threshold = (float) (*it)["threshold"];

        masks.push_back(pair);
    }
}

void saveMaskDescSeparate(vector<maskRegion> descs, const char *outputFiles) {
    vector<maskRegion>::iterator it = descs.begin(), it_end = descs.end();

    char output[256];
    for (int i = 0; it != it_end; ++it, i++) {
        sprintf(output, outputFiles, i);
        imwrite(output, (*it).mask);
    }

}
