/* 
 * File:   utils.h
 * Author: Marcel Duris
 *
 * Created on December 11, 2011, 3:46 PM
 * 
 * Zakladne poskytovane nastroje na rozpoznavanie objektov pocas roznych 
 * podmienok.
 */

#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>

#include <iostream>
#include <sys/types.h>
#include <dirent.h>
#include <string>

using namespace cv;

#ifndef UTILS_H
#define	UTILS_H


#define SAL_FG_THLD 50
#define SAL_PFG_THLD 25
#define CANNY_EDGE_T 30
#define CANNY_EDGE_R 3

#define RESPONSE_STD 0
#define RESPONSE_SAL 1
#define RESPONSE_LOC 2
#define RESPONSE_CMB 3

#define GRID_THRES 0.095

enum NoiseType {
    WHITE,
    BLACK,
    COLOR,
    SALT_AND_PEPPER
};

struct maskRegion {
    Rect roi;
    int gridSize;
    float threshold;
    Mat mask;
};

/**
 * Anizotropna difuzia farebneho vstupu.
 * 
 * @param src
 * @param dst
 * @param kappa
 * @param lambda
 * @param iterCount
 */
void anidiffColor(Mat src, Mat &dst, float kappa, float lambda, int iterCount);

/**
 * Anizotropna difuzia ciernobieleho vstupu.
 * 
 * @param src
 * @param dst
 * @param kappa
 * @param lambda
 * @param iterCount
 */
void anidiff(Mat src, Mat &dst, float kappa, float lambda, int iterCount);

/**
 * Spusti jednoduchu demonstraciu Gaborovho filtra na vstupnom obrazku.
 * 
 * @param img Vstup
 */
void demonstrateGabor(Mat img);

/**
 * Vrati Gaborov filter zadanych parametrov centrovany na stred vystupnej 
 * matice.
 * 
 * @param width Sirka vystupnej matice.
 * @param height Vyska vystupnej matice.
 * @param lambda Vlnova dlzka v pixeloch
 * @param theta Otocenie
 * @param psi Faza
 * @param sigma Sigma gaussianu
 * @param gamma Excentrickost gaussianu
 * @return Gaborov filter zodpovedajuci danym parametrom
 */
Mat getGabor(int width, int height, float lambda, float theta, float psi, float sigma, float gamma);

/**
 * Vypocita gradient vstupu.
 * 
 * @param src vstup
 * @param dst vystup
 */
void getGradient(Mat src, Mat &dst);

/**
 * Vlozi Salt & Pepper noise do obrazku, povodny obrazok je modifikovany. 
 * Podporuje cierny a biely sum a sum nahodnej intenzity.
 * @param src
 * @param intensity
 * @param type
 */
void insertNoise(Mat &src, float intensity, int type = 0);

/**
 * Ofarbi vstup nepravymi farbami.
 * 
 * @param src Vstup
 * @param dst Vystup
 */
void pseudocolor(Mat src, Mat &dst);

/**
 * Vyhlada vyznamne hrany medzi hranami najdenymi v mape saliencie pomocou 
 * nizkych prahov Cannyho detektora hran.
 * 
 * @param img Vstupny obrazok s hranami
 * @param dst Vystup s prefiltrovanymi hranami
 * @param cellSize Velkost analyzovanych policok
 */
void getSignificantEdges(Mat img, Mat &dst, int cellSize);

/**
 * Porovna primitivne dva booleanove utvary na vstupe
 * 
 * @param pattern
 * @param img
 * @return 
 */
float shapeSimilarity(Mat pattern, Mat img);

/**
 * Algoritmus rozpoznavania charakteristickych hran. Vybuduje masku.
 * @param img
 * @param outFile
 */
void edgeMask(Mat img, vector<maskRegion> &dst, bool append = false);

/**
 * Algoritmus rozpoznavania charakteristickych hran. Pouzije Roi z existujuceho popisu.
 * @param img
 * @param outFile
 */
void edgeMask(Mat img, vector<maskRegion> &dst, vector<maskRegion> rois, bool append = false);

/**
 * Odpoved algoritmu vyhladavania hran v mape saliencie. Testovacia maska ma byt
 * viackanalova matica jednotlive kanaly maju obsahovat odpovede Gaborovho 
 * filtra v ntom smere na ocakavane hrany Cannyho detektora hran.
 * 
 * @param img Vstupna saliency mapa
 * @param mask Testovacia maska 
 * @return miera podobnosti
 */
float edgeResponse(Mat img, Mat mask, int method = RESPONSE_STD, int gaborDirs = 2);

/**
 * Odpoved algoritmu rozpoznavania charakteristickych hran pre cely obrazok a 
 * vsetky sablony v nom, odpovede vrati vo vektore.
 * 
 * @param img vstupny obrazok
 * @param masks vektor mask
 * @param dst vektor odpovedi
 */
void edgeResponse(Mat img, vector<maskRegion> masks, vector<float> &dst);

/**
 * Zostavi masku pre segmentovanie mapy saliencie.
 * 
 * @param img Vstupna scena z ktorej bude pocitana mapa saliencie
 * @param dst Nejaka kontura
 */
void grabcutMask(Mat img, vector<maskRegion> &dst);

/**
 * Zostavi zoznam mask pre mieru jedna.
 * 
 * @param img obrazok, ku ktoremu sa bude miera konstruovat
 * @param masks vystupny vektor mask
 */
void gridMask(Mat img, vector<maskRegion> &dst, bool append = false);

/**
 * Odpoved miery jedna.
 * 
 * @param img region, ktory analyzujeme
 * @param mask maska prisluchajuca regionu
 * @param steps nepovinny pocet smerov, do ktory podelit gaborov filter
 * @return 
 */
float gridResponse(Mat img, Mat mask, int steps = 2);

/**
 * Odpoved miery jedna pre cely obrazok a vsetky masky v nom, odpovede vrati vo 
 * vektore.
 * 
 * @param img vstupny obrazok
 * @param masks vektor mask
 * @param dst vektor odpovedi
 */
void gridResponse(Mat img, vector<maskRegion> masks, vector<float> &dst);

/**
 * Ulozi zoznam parov (subor s maskou, roi vstupu).
 * 
 * @param pairs vector popisjuci masku
 * @param outputFile cesta k suboru, do ktoreho zapisat
 */
void saveMaskDesc(vector<maskRegion> pairs, string outputFile);

/**
 * Nacita zoznam parov (subor s maskou, roi vstupu).
 * 
 * @param pairs vector popisjuci masku
 * @param outputFile cesta k suboru, do ktoreho zapisat
 */
void loadMaskDesc(string inputFile, vector<maskRegion> &masks);

/**
 * Rozbije zoznam regionov na obrazky s maskami.
 * 
 * @param descs
 * @param outputFiles
 */
void saveMaskDescSeparate(vector<maskRegion> descs, const char *outputFiles);

#endif	/* UTILS_H */
