/*  bitstr.c
 * 
 *  R3lib
 * 
 *  Bit string utils
 * 
 *  Copyright (C) 2006-2007  Michal Linhard <michal@linhard.sk>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  as published by the Free Software Foundation; either version 2.1
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 */
#include <stdio.h> // fprintf
#include <string.h> // memset

#include "bitstr.h"

/* initialise bit string 
   this allocates memory for bit string outside
   mem location has to be allocated properly */
void bs_init(t_bitstr* s, BYTE* mem, int strLength) 
{ 
	s->myStringLength = strLength; 
	s->numChar = strLength >> 3; 
	if ((strLength & 7) != 0)   
	{ 
		s->numChar++;
	} 
	if(!(s->numChar))
		s->numChar = 1;
	s->myBooleanString = mem; // there should be numChar Bytes allocated at mem   
} 

/* get value of bit at index */
inline int bs_getVal(t_bitstr* s, int index) 
{
	int charNum;
	int bitNum;
	int temp;
	
	if( !s ||
	    (index < 0) || 
	    (index >= s->myStringLength) || 
	    (s->myStringLength == -1) )
	{
		fprintf(stderr, "error: index out of range or !s or s->myStringLength == -1\n");
		return -1;
	}	    
	    
	
	charNum = index >> 3;
	bitNum = index & 7;
	temp = (int) s->myBooleanString[charNum];
	
	temp >>= bitNum;
	temp &= 1;
	
	return (temp);
}

/* set value of bit at index */
inline void bs_setVal(t_bitstr* s, int index, int indexVal)
{
	int charNum;
	int bitNum;
	BYTE temp, original, mask;
	
	if( !s ||
	    (index < 0) || 
	    (index >= s->myStringLength) || 
	    (s->myStringLength == -1) )
	{
		fprintf(stderr, "error: index out of range or !s or s->myStringLength == -1\n");
		return;
	}	    
	
	charNum = index >> 3;
	bitNum = index & 7;
	original = s->myBooleanString[charNum];
	temp = (char) indexVal;
	
	temp = temp << bitNum;
	mask = 1    << bitNum;
	temp = temp ^ original;
	temp = temp & mask;
	s->myBooleanString[charNum] = temp ^ original;
}

/* set value of bits in interval <start, end> */
inline void bs_setIntv(t_bitstr* s, int start, int end, int indexVal)
{
	int startCharNum, endCharNum;
	int startBitNum, endBitNum;
	BYTE temp, mask, original;

	startCharNum = start >> 3;
	endCharNum = end >> 3;
	startBitNum = start & 7;
	endBitNum = end & 7;
	endBitNum = endBitNum ^ 7; // count from other side
	
	if(startCharNum == endCharNum)
	{
		original = s->myBooleanString[startCharNum];
		temp = 0xff >> startBitNum; // cut 1's from right
		temp = temp << startBitNum;
		mask = 0xff << endBitNum;   // cut 1's from left
		mask = mask >> endBitNum;
		mask = mask & temp;
		temp = -indexVal; 
		temp = temp ^ original;
		temp = temp & mask;
		s->myBooleanString[startCharNum] = temp ^ original;
	}
	else 
	{
		original = s->myBooleanString[startCharNum];
		mask = 0xff >> startBitNum; 
		mask = mask << startBitNum;
		temp = -indexVal;
		temp = temp ^ original;
		temp = temp & mask;
		s->myBooleanString[startCharNum] = temp ^ original;

		original = s->myBooleanString[endCharNum];
		mask = 0xff << endBitNum;       
		mask = mask >> endBitNum;
		temp = -indexVal;
		temp = temp ^ original;
		temp = temp & mask;
		s->myBooleanString[endCharNum] = temp ^ original;
		
		mask = -indexVal;
		temp = endCharNum-startCharNum-1;
		memset((void*)(s->myBooleanString+startCharNum+1), mask, temp);
	}
}

inline void bs_setAll(t_bitstr* s, int val) 
{ 
	if (val) 
	{ 
		memset((void*)s->myBooleanString, 255, s->numChar); 
	} 
	else 
	{ 
		memset((void*)s->myBooleanString, 0, s->numChar); 
	} 
} 

inline int bs_isAllTrue(t_bitstr* s) 
{ 
	int i; 

	if( !s ||
	    (s->myStringLength <= 0) )
	{
		fprintf(stderr, "integrity error\n");
		return -1;
	}	    

	for (i = 0; i < s->myStringLength / 8; i++) 
		if (s->myBooleanString[i] != (BYTE) 255) 
			return 0; 

	for (i = 0; i < (s->myStringLength % 8); i++) 
		if (!bs_getVal(s, s->myStringLength - i - 1)) 
			return 0;

	return 1; 
} 


