/*  r3.c
 * 
 *  R3lib
 * 
 *  R3 tree definition and routines
 * 
 *  Copyright (C) 2006-2007  Michal Linhard <michal@linhard.sk>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  as published by the Free Software Foundation; either version 2.1
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */ 
#include <stdio.h> // fprintf
#include <string.h> // memset, memmove

#include "conf.h" 
#include "r3.h" 
#include "aarray.h"
#include "suffixArray.h"
#ifdef DEBUG2
#include "test/debug.h"
#include "test/test_r3.h"
#endif

/* create r3 tree for given text 
 * initialise r3t structure */
int r3t_create(t_r3t* r3t, BYTE* text, int textLen)
{
	t_r3t_work r3tw;
	int result;

	if(!r3t || !text || textLen < 0 || textLen > FLAGUINT_MAX) 
		return ERROR_INTEGRITY;
	
	textLen++; // now textLen includes also endmarker(dollar sign)	
	r3t->text = text;
	r3t->textLen = textLen; 

	if(textLen == 1)
		return r3th_createEmpty(r3t);

#ifdef DEBUG_PRINTPHASE
	printf("creating suffix array ...\n");
	fflush(stdout);
#endif
    result = createSuffixArray(text, textLen, &(r3tw.sa)); 
    if(result) { 	
#ifdef DEBUG2
		fprintf(stderr, "createSuffixArray returned %i\n", result);
#endif
    	return result; 
    }

#ifdef DEBUG_PRINTPHASE
	printf("creating lcp-table ...\n");
	fflush(stdout);
#endif
	result = createLCPTable(text, textLen, r3tw.sa, &(r3tw.lcp));
	if(result) {
#ifdef DEBUG2
		fprintf(stderr, "createLCPTable returned %i\n", result);
#endif
		free(r3tw.sa);
		return result; 
	}	

	/* allocate memory structures for r3 tree construction */
	result = r3th_allocate(r3t, &r3tw);
	if(result)
	{
#ifdef DEBUG2
		fprintf(stderr, "r3th_allocate returned %i\n", result);
#endif
		return result;
	}

#ifdef DEBUG_PRINTPHASE
	printf("traversing bottom-up ...\n");
	fflush(stdout);
#endif

	result = r3th_traverseBottomUp(r3t, &r3tw);
	if(result)
	{
#ifdef DEBUG2
		fprintf(stderr, "r3th_traverseBottomUp returned %i\n", result);
#endif
		return result;
	}

#ifdef DEBUG_PRINTPHASE
	printf("computing parent array ...\n");
	fflush(stdout);
#endif

	result = r3th_computeParent(r3t, &r3tw);
	if(result)
	{
#ifdef DEBUG2
		fprintf(stderr, "r3th_computeParent returned %i\n", result);
#endif
		return result;
	}

#ifdef DEBUG_PRINTPHASE
	printf("compacting union forest ...\n");
	fflush(stdout);
#endif

	result = r3th_compactUnionForest(r3t, &r3tw);
	if(result)
	{
#ifdef DEBUG2
		fprintf(stderr, "r3th_compactUnionForest returned %i\n", result);
#endif
		return result;
	}

#ifdef DEBUG_PRINTPHASE
	printf("traversing top-down ...\n");
	fflush(stdout);
#endif

	result = r3th_traverseTopDown(r3t, &r3tw);
	if(result)
	{
#ifdef DEBUG2
		fprintf(stderr, "r3th_traverseTopDown returned %i\n", result);
#endif
		return result;
	}

#ifdef DEBUG_PRINTPHASE
	printf("computing map array ...\n");
	fflush(stdout);
#endif

	result = r3th_computeMap(r3t, &r3tw);
	if(result)
	{
#ifdef DEBUG2
		fprintf(stderr, "r3th_computeMap returned %i\n", result);
#endif
		return result;
	}
	
#ifdef DEBUG2
//	output_union_trees_html(r3t, "output/ut.html"); 
//	output_intarray(r3t->map, r3t->textLen, "output/map.txt");
//	output_intarray(r3t->bp, r3t->size, "output/bp.txt");
//	output_union_trees(r3t, &r3tw, "output/ut.txt");
//	output_FLAGUINTarray(r3t->uf, r3tw.uf_size, "output/uf.txt");
//	output_FLAGUINTarray(r3t->bptables, r3tw.bptables_size, "output/bptables.txt");

//	output_FLAGUINTarray(r3t->parent1, r3t->size, "output/parent.txt");
//	output_intarray(r3tw.fd, r3t->size, "output/fd.txt");
#endif

	return 0;
}

/* free memory allocated by r3t_create */
int r3t_destroy(t_r3t* r3t)
{
	if(!r3t)
		return ERROR_INTEGRITY;
	if(r3t->lcp) {
		free(r3t->lcp);
		r3t->lcp = NULL;
	}
	if(r3t->up1) {
		free(r3t->up1);
		r3t->up1 = NULL;
	}
	if(r3t->up2) {
		free(r3t->up2);
		r3t->up2 = NULL;
	}
	if(r3t->bp) {
		free(r3t->bp);
		r3t->bp = NULL;
	}
	if(r3t->bptables) {
		free(r3t->bptables);
		r3t->bptables = NULL;
	}
	if(r3t->uf) {
		free(r3t->uf);
		r3t->uf = NULL;
	}
	if(r3t->map) {
		free(r3t->map);
		r3t->map = NULL;
	}
	r3t->text = NULL; // do not free text, because it is external to our framework
	r3t->textLen = 0;
	r3t->size = 0;
	return 0;
}

/* find all pairs (p2, l) such that (p1, p2, l) is maximal repeat in 
 * r3t->text with l>=k . r3t has to be initialised by r3t_create.
 * number of found pairs is returned in count
 * pairs are returned in form (p2[0], l[0]), (p2[1], l[1]), ..., (p2[count-1],l[count-1])
 * arrays p2 and l should be allocated for count_limit items
 * count never exceeds count_limit
 * */
int r3t_findPairs(t_r3t* r3t, int p1, int k, int* count, int count_limit, int* p2, int* l)
{
	int v,i, uf_i, temp, lc, lcp;
	int visited[SIGMA_SIZE];
	FLAGUINT* bptab;
	FLAGUINT* uf;
	BYTE* text;
	UINT a, b;
	
	if( // check sanity of parameters
		!r3t ||
		!count ||
		!p2 ||
		!l ||	
		p1 < 0 ||
		p1 >= r3t->textLen ||
		k < 0 ||
		k >= r3t->textLen ||
		!r3t->bp || 
		!r3t->lcp || 
		!r3t->lc1 || 
		!r3t->map || 
		!r3t->up1 || 
		!r3t->up2 || 
		!r3t->text ||
		r3t->textLen <= 0 ||
		r3t->textLen > FLAGUINT_MAX ||
		r3t->size <= 0 ||
		r3t->size > r3t->textLen ||
		r3t->bptables_size < 0 ||
		r3t->bptables_size > r3t->textLen*3 ||
		r3t->uf_size < 0 ||
		r3t->uf_size > (r3t->textLen-1)*3 ||
		(r3t->bp[r3t->size -1] != UNDEF && (!r3t->uf || !r3t->bptables) )  
	)
	   return ERROR_INTEGRITY; 
		
	bptab = r3t->bptables;
	uf = r3t->uf;
	text = r3t->text;
	a = p1 == 0 ? 0x100 : text[p1-1];
	
	for(i=0; i<SIGMA_SIZE; i++)
		visited[i] = UNDEF;
		
	*count = 0;
	v = r3t->map[p1];

	lcp = r3t->lcp[v].value;
	
	while( (v != UNDEF) && (lcp >=  k) ) {
		temp = r3t->bp[v];
		if(temp != UNDEF) {
			i = temp-1; 
			do {
				i++;
				uf_i = bptab[i].value;
				temp = uf[uf_i].value;
				lc = text[uf[temp].value -1];
				if(lc != a) {
					r3th_combineUnionSubtree(uf, uf_i, visited[lc], lcp, count, count_limit, p2, l);
					if(*count >= count_limit)
						return 0;
					visited[lc] = uf_i;
				}
			} while(bptab[i].flag2 != 1);
		}
		b = r3t->up1[v].flag1 ? LC_CENT : r3t->lc1[v];
		if(b != a) {
			v = r3t->up1[v].value;
		} else {
			v = r3t->up2[v];
		}
		lcp = r3t->lcp[v].value;
	}
	return 0;
}

/* create r3 tree for empty string
 * */
int r3th_createEmpty(t_r3t* r3t)
{
	if(!r3t)
		return ERROR_INTEGRITY;
		
	r3t->bp = (int*) malloc(sizeof(int));
	r3t->bptables = NULL;
	r3t->bptables_size = 0;
	r3t->lc1 = (BYTE*) malloc(sizeof(BYTE));
	r3t->lcp = (FLAGUINT*) malloc(sizeof(FLAGUINT));
	r3t->map = (int*) malloc(sizeof(int));
	r3t->up1 = (FLAGUINT*) malloc(sizeof(FLAGUINT));
	r3t->up2 = (int*) malloc(sizeof(int));
	r3t->peakmem = 0; // don't care about peakmem
	r3t->size = 1;
	r3t->totalmem = 0; // don't care about totalmem
	r3t->uf = NULL;
	r3t->uf_size = 0;
		
	if( !r3t->bp ||
		!r3t->lc1 ||
		!r3t->lcp ||
		!r3t->map ||
		!r3t->up1 ||
		!r3t->up2)
		return ERROR_MEM;
	
	r3t->bp[0] = UNDEF;
	r3t->lc1[0] = 0;
	r3t->lcp[0].value = 0;
	r3t->lcp[0].flag1 = 1;
	r3t->lcp[0].flag2 = 1;
	r3t->map[0] = 0;
	r3t->up1[0].value = UNDEF;
	r3t->up1[0].flag1 = 1;
	r3t->up1[0].flag2 = 0;
	r3t->up2[0] = UNDEF;

	return 0;
}

/* compute r3t->up1 array from fd array stored in r3tw->lcp
 * post-condition:
 * r3t->up1[i] is index of parent of node i
 * */
int r3th_computeParent(t_r3t* r3t, t_r3t_work* r3tw)
{
	int i;
	int* fd = r3tw->fd;
	FLAGUINT* parent = r3t->up1;
	int* stack = r3tw->sa;
	int top = 0;
	stack[0] = r3t->size-1;
	parent[r3t->size-1].value = UNDEF;
	parent[r3t->size-1].flag1 = 0;
	parent[r3t->size-1].flag2 = 0;  // reserved (not in use)
	for(i=r3t->size-2; i>=0; i--)
	{
		parent[i].value = stack[top];
		parent[i].flag1 = 0;
		parent[i].flag2 = 0; // reserved (not in use)
		top++;
		stack[top] = i;
		while(top >=0 && fd[stack[top]] == i)
			top--;
	}
	r3t->up2 = r3tw->sa;
	r3t->map = r3tw->fd;
	r3tw->sa = NULL;
	r3tw->fd = NULL;
	return 0;
}

/* compact union forest
 * 
 * after call to r3th_allocate r3t->uf is allocated to 3*(textLen-1) bytes
 * after r3th_traverseBottomUp is called, we know size of each union tree
 * and there may be gaps between union trees stored at different parts of r3t->uf array, 
 * pointed to by pointers from r3tw->ut_ptr table.
 * this function compacts these gaps in r3t->uf array. it also has to correct values in 
 * navigator items of union trees and entries in bptables.
 * */
int r3th_compactUnionForest(t_r3t* r3t, t_r3t_work* r3tw)
{
	int temp2, temp, a, last, total_shift;
	FLAGUINT* i;
	FLAGUINT* bptab_end;
	FLAGUINT* uf;
	BYTE* lc;
	BYTE* text;
	int shift[SIGMA_SIZE];
	
	if(!r3t || !r3tw)
		return ERROR_INTEGRITY;
	if(!r3t->bptables || !r3t->uf || !r3t->map || !r3t->text)
		return ERROR_INTEGRITY;

	bptab_end = r3t->bptables + r3t->bptables_size;
	uf = r3t->uf;
	lc = (BYTE*) r3t->map; // reuse
	text = r3t->text;

	for(i=r3t->bptables; i<bptab_end; i++) {
		temp = (*i).value;
		temp2 = uf[temp].value;
		uf[temp].flag1 = 0;
		uf[temp-1].flag1 = 0;
		temp = uf[temp2].value;
		*lc = text[temp -1];
		lc++;
	}

	last = 0; // last nonempty union tree
	while(last < SIGMA_SIZE && !r3tw->ut_cnt[last])
		last++;	
		
	if(last == SIGMA_SIZE) { // all union trees are empty
		if(r3t->uf) {
			free(r3t->uf);
			r3t->uf = NULL;
		}
		return 0;
	}
	
	shift[last] = 0;
	a = last+1;
	while(a<SIGMA_SIZE) {
		while(a < SIGMA_SIZE && !r3tw->ut_cnt[a])
			a++;	
		if(a >= SIGMA_SIZE)
			break;
		shift[a] = shift[last] + r3tw->ut_ptr[a]-(r3tw->ut_ptr[last]+r3tw->ut_cnt[last]);
		memmove(r3t->uf+r3tw->ut_ptr[a]-shift[a], r3t->uf+r3tw->ut_ptr[a], r3tw->ut_cnt[a]*sizeof(FLAGUINT));
		last = a;
		a++;		
	}
	total_shift = r3t->uf_size-(r3tw->ut_ptr[last]+r3tw->ut_cnt[last]) + shift[last];
	r3t->uf = realloc(r3t->uf, (r3t->uf_size-total_shift)*sizeof(FLAGUINT));
	if(!r3t->uf)
		return ERROR_MEM;
	uf = r3t->uf;
	r3t->uf_size -= total_shift;
	r3t->totalmem -= total_shift*sizeof(FLAGUINT);
	
	lc = (BYTE*) r3t->map;
	for(i=r3t->bptables; i<bptab_end; i++) {
		(*i).value -= shift[*lc];
		temp = (*i).value;
		if(!uf[temp].flag1) {
			r3t->uf[temp].value -= shift[*lc];
			uf[temp].flag1 = 1;
		}
		temp2 = temp-1;
		if(uf[temp2].flag2 && !uf[temp2].flag1) { //u-navigator node, we have to check if it doesn't contain unreferenced bucket,
			r3t->uf[temp2].value -= shift[*lc]; // which has to be shifted too
			uf[temp2].flag1 = 1;
		}
		lc++;
	}

	return 0;
}

/* compute r3t->map array
 * post-condition:
 * r3t->map[p] is index of node v such that b(v, a) contains p for some a 
 * */
int r3th_computeMap(t_r3t* r3t, t_r3t_work* r3tw)
{
	int v, j, k, u;
	FLAGUINT* bptab = r3t->bptables;
	FLAGUINT* uf = r3t->uf;
	int* bp = r3t->bp;
	int* map = r3t->map;
	FLAGUINT* lcp = r3t->lcp;
	int r3tsize = r3t->size;
	
	for(v=0; v<r3tsize; v++) {
		if(lcp[v].flag1)
			map[0] = v;
		if(bp[v] == UNDEF)
			break;
		j = bp[v]-1;
		do {
			j++;
			if(bptab[j].flag1) {
				u = bptab[j].value; 
				if(uf[u-1].flag2) // u-navigator item
					u--;
				for(k=uf[u].value; k<u; k++)
					map[uf[k].value] = v;
			}
		} while(bptab[j].flag2 == 0);	
	}
	return 0;
}

/* combine value lcp with each value from union subtree node u
 * skip visited nodes. uf is union forest array. 
 * stores combined pairs in arrays p2 and l. count never exceeds count_limit
 * */
void r3th_combineUnionSubtree(FLAGUINT* uf, int u, int visited, int lcp, int* count, int count_limit, int* p2, int* l)
{
	int i, fd;
	if(u == visited)
		return;
	fd = uf[u].value;
	for(i=u-1; i>=fd; i--) {
		if(!(uf[i].flag2)) { // add pair (uf[i].value, p2)
			l[*count] = lcp;
			p2[*count] = uf[i].value;
			(*count)++;
			if((*count) >= count_limit)
				return;
		} else {
			if( i == visited )
				i = uf[i].value -1; // skip visited subtree
		}
	}
}

/* analyze lcp table to determine memory requirements 
 * 
 * fills nodeCnt and maxStackBottomUp in r3t_work structure
 * requires 4n additional memory 
 * */
int r3th_analyzeLcpTable(t_r3t* r3t, t_r3t_work* r3t_work) 
{
	int i;
	int cnt, top;
	int* stack_lcp;
	int maxStack = 0;
	int nodeCnt = 0;
	int n;
	int* lcp;

	if(!r3t || !r3t_work)
		return ERROR_INTEGRITY;

	lcp = r3t_work->lcp;
	n = r3t->textLen;
	
	if(!lcp || (n < 1) || (n > FLAGUINT_MAX))
		return ERROR_INTEGRITY;

	stack_lcp = (int*) malloc( sizeof(int) * n );
	if(!stack_lcp)
		return ERROR_MEM;
	stack_lcp[0] = 0;

	cnt = 1;
	top = 0;

	for(i=1; i<n; i++) // main loop
	{
		while(lcp[i] < stack_lcp[top]) // interval is ending here, process it and create r3 node
		{
			cnt--; top--; // pop
			nodeCnt++;
		}
		if(lcp[i] > stack_lcp[top]) // create new interval
		{
			stack_lcp[cnt] = lcp[i]; // push
			cnt++; top++;
			if(cnt > maxStack)
				maxStack = cnt;
		}
	}

	while(cnt)
	{
		cnt--; top--; // pop
		nodeCnt++;
	}

	r3t_work->maxStackBottomUp = maxStack;
	r3t_work->nodeCnt = nodeCnt;
	free(stack_lcp);
	return 0;
}


/* processes lcp interval and creates new r3 tree node
 * */
int r3th_processIntervalBottomUp(	t_r3t*		r3t, 
									t_r3t_work*	r3tw,
									int			lcp1, 
									int			lb,
									int			rb,
									int			fd1 )
{
	int i,j,k,r,lc;
	UINT temp, tempc;
	UINT bucket_s[SIGMA_SIZE]; // bucket sizes
	int bucket_p[SIGMA_SIZE]; // bucket pointers
	FLAGUINT firstTree[SIGMA_SIZE];
	FLAGUINT spec_bucket;
	FLAGUINT* uf;
	FLAGUINT bp_table[SIGMA_SIZE]; // flag1 == own flag
//	int result;
	int* fd;
	FLAGUINT* sa;
	BYTE* text;
	int textLen;	
	FLAGUINT* bptab;
	int* ut_cnt;
	int* ut_ptr;

	if(!r3t || !r3tw)
		return ERROR_INTEGRITY;
	
	fd = r3tw->lcp; // r3tw->lcp array will be rewritten by fd array
	sa = (FLAGUINT*) (r3tw->sa);
	text = r3t->text;
	textLen = r3t->textLen;
	bptab = r3t->bptables;
	uf = r3t->uf;
	ut_cnt = r3tw->ut_cnt; // ut_cnt[a] number of items in UT for lc a
	ut_ptr = r3tw->ut_ptr; // start index of UT for lc a in UF

	if(!text || !sa || !fd )
		return ERROR_INTEGRITY;
		
	r = r3t->size; // to understand later comments, let v be r3 tree node represented by this row
	(r3t->size)++;
	fd[r] = fd1;

	/* compute firstBucket array */	
	for(i=0; i<SIGMA_SIZE; i++) {
		firstTree[i].value = UNDEF;
		firstTree[i].flag1 = 0;
		firstTree[i].flag2 = 0;
		bp_table[i].value = UNDEF;
		bp_table[i].flag1 = 0;
		bp_table[i].flag2 = 0;
	}
	// spec bucket tells us whether b+(v, CENT) is empty
	spec_bucket.value = 0;
	spec_bucket.flag1 = 0; // 1 if b(v,CENT) not empty 
	spec_bucket.flag2 = 0; // 1 if b+(v, CENT) not empty
	for(i=r-1; i>=fd[r]; i=fd[i]-1)
	{
		spec_bucket.flag2 |= r3t->lcp[i].flag2;
			
		if(r3t->bp[i] != UNDEF) {
			j = r3t->bp[i]-1;
			do {
				j++;
				k = bptab[j].value;
				temp = uf[k].value;
				lc = text[uf[temp].value-1];
				if(firstTree[lc].value != UNDEF)
					firstTree[lc].flag1 = 1; // indicate union
				firstTree[lc].value = k;
			} while(bptab[j].flag2 != 1);
		}
	}
	
	/* firstBucket[a] = UNDEF if there is no child c such that b+(c,a) != emptyset
	 * firstBucket[a] = k if there is child c such that b+(c,a) != emptyset and k is
	 * union tree index of first such child
	 * firstBucket[a].flag1 = 1 if there is more than one such child
	 * in that case we have to add new u-navigator item at the end with 
	 * value = UF[k].value (index of first value item of child c)
	 * */


	/* determine bucket sizes */
	/* bucket_s[a] = size of b(v,a) */
	memset((void*)bucket_s, 0, SIGMA_SIZE*sizeof(UINT));
	for(i=lb; i<=rb; i++)
	{
		while( i <= rb && sa[i].flag1 )
			i= sa[i].value; // skip children positions
		
		if(i > rb) break;
			
		temp = (int) (sa[i].value);
		if(temp)
		{
			tempc = text[temp-1];
			bucket_s[tempc]++;
		}
		else
		{
			spec_bucket.flag1 = 1; 
		}
	}
	spec_bucket.value = lcp1;
	if(spec_bucket.flag1)
		spec_bucket.flag2 = 1;
	r3t->lcp[r] = spec_bucket; // lcp value + zero flags

	/* allocate new buckets in union trees */
	for(i=0; i<SIGMA_SIZE; i++)
		bucket_p[i] = ut_ptr[i] + ut_cnt[i];

	/* construct bp-table for this lcp-interval */	
	for(i=0;i<SIGMA_SIZE;i++)
	{
		if(firstTree[i].value == UNDEF) // b+(v,i) - b(v,i) is empty 
		{
			if(bucket_s[i]) // b+(v,i) == b(v,i)
			{
				temp = bucket_p[i] + bucket_s[i]; // position of new b-navigator item
				uf[temp].value = bucket_p[i]; 
				uf[temp].flag1 = 0; // reserved use
				uf[temp].flag2 = 1; // save navigator item
				bp_table[i].value = temp; // store pointer to navigator item into bp_table
				bp_table[i].flag1 = 1;
				ut_cnt[i] += bucket_s[i]+1; // increment size of ut
			}
			else // b+(v,i) is empty
			{
				bp_table[i].value = UNDEF;
			}
		}
		else // b+(v,i) - b(v,i) is not empty
		{
			if(bucket_s[i]) // b(v,i) is not empty
			{
				temp = bucket_p[i] + bucket_s[i]; // position of new b-navigator item
				uf[temp].value = bucket_p[i];
				uf[temp].flag1 = 0; // reserved use
				uf[temp].flag2 = 1;
				bp_table[i].value = temp+1; // bp_table points to new u-navigator item
				bp_table[i].flag1 = 1;
				ut_cnt[i] += bucket_s[i]+2;
				tempc = firstTree[i].value; // first b-navigator item to be unioned
				uf[temp+1] = uf[tempc];
			}
			else // b(v,i) is empty
			{
				tempc = firstTree[i].value; // first b-navigator item to be unioned
				if(firstTree[i].flag1) // we have to union
				{
					temp = bucket_p[i]; // position of new u-navigator item
					uf[temp] = uf[tempc];
					bp_table[i].value = temp;
					ut_cnt[i]++;
				}
				else
				{
					bp_table[i].value = tempc;
				}
			}
		}
	}

	
	// store bp_table to r3t->bptables
	tempc = r3t->bptables_size;
	for(i=0; i<SIGMA_SIZE; i++) {
		if(bp_table[i].value != UNDEF) {
			bptab[tempc] = bp_table[i];
			tempc++;
		}
	}
	if(!(tempc - r3t->bptables_size))
		r3t->bp[r] = UNDEF;
	else 
		r3t->bp[r] = r3t->bptables_size;
		
	r3t->bptables_size = tempc;
	bptab[tempc-1].flag2 = 1; // set endflag
	

	/* fill buckets with positions */
	for(i=lb; i<=rb; i++)
	{
		while( i <= rb && sa[i].flag1 )
			i = sa[i].value; // skip children positions

		if(i > rb) break;
			
		temp = sa[i].value;
		if(temp)
		{
			tempc = text[temp-1];
			((int*)uf)[bucket_p[tempc]] = temp; // sets also flag1 and flag2 to 0
			bucket_p[tempc]++;
		}
	}
	
	sa[lb].value = rb+1;
	sa[lb].flag1 = 1; // mark interval as processed
	
	return 0;
}

/* allocates memory structures needed for r3 tree
 * requires r3t->text and r3t->textLen to be defined
 * */
int r3th_allocate(t_r3t* r3t, t_r3t_work* r3tw)
{
	int i;
	int* ut_cnt;
//	double alpha;
	int* ut_ptr;
	
	if(!r3t ||
	   !r3t->text ||
	   r3t->textLen < 0 ||
	   r3t->textLen > FLAGUINT_MAX ||
	   !r3tw )
		return ERROR_INTEGRITY;

	ut_cnt = r3tw->ut_cnt;
	ut_ptr = r3tw->ut_ptr;
	r3t->totalmem = r3t->textLen*sizeof(int)*2; // r3tw->lcp, r3tw->sa
	r3t->bptables_size = 0;
	
	// determine maxStack and nodeCnt
	r3th_analyzeLcpTable(r3t, r3tw);

	// null all values (to conveniently use destroyR3 for freeing in case of abort process)
	r3t->size=0;
	r3t->lcp = NULL;
	r3t->up1 = NULL;
	r3t->up2 = NULL;
	r3t->bp = NULL;
	r3t->bptables = NULL;
	r3t->uf = NULL;
	r3t->map = NULL;
	r3t->lc1 = NULL;
	
	// size 4N
	r3t->lcp = (FLAGUINT *) malloc( sizeof(FLAGUINT)*(r3tw->nodeCnt) );
	if(!(r3t->lcp))
	{
		r3t_destroy(r3t);
		return ERROR_MEM;
	}
	r3t->totalmem += sizeof(FLAGUINT)*(r3tw->nodeCnt);

	// size 4N
	r3t->up1 = (FLAGUINT *) malloc( sizeof(FLAGUINT)*(r3tw->nodeCnt) );
	if(!(r3t->up1))
	{
		r3t_destroy(r3t);
		return ERROR_MEM;
	}
	r3t->totalmem += sizeof(FLAGUINT)*(r3tw->nodeCnt);

	// size 4N
	r3t->bp = (int*) malloc( sizeof(int)*(r3tw->nodeCnt) );
	if(!(r3t->bp))
	{
		r3t_destroy(r3t);
		return ERROR_MEM;
	}
	r3t->totalmem += sizeof(FLAGUINT)*(r3tw->nodeCnt);

	// count symbols in text
	memset(ut_cnt, 0, SIGMA_SIZE*sizeof(int));
	(r3t->textLen)--; // don't count last symbol
	for(i=0; i<r3t->textLen; i++)
		ut_cnt[r3t->text[i]]++;
	(r3t->textLen)++;

	// compute locations of union trees in union forest
	ut_ptr[0]=0;
	for(i=1; i<SIGMA_SIZE; i++) 
		ut_ptr[i] = ut_ptr[i-1] + ut_cnt[i-1]*3; 
	
	memset(ut_cnt, 0, SIGMA_SIZE*sizeof(int));
	
	// size 12N
	r3t->uf_size = (r3t->textLen-1)*3;
	r3t->uf = (FLAGUINT*) malloc((r3t->uf_size)*sizeof(FLAGUINT));
	if(!(r3t->uf))
	{
		r3t_destroy(r3t);
		return ERROR_MEM;
	}
	r3t->totalmem += (r3t->uf_size)*sizeof(FLAGUINT);
	
	// size 12N
	r3t->bptables = (FLAGUINT*) malloc((r3t->textLen)*sizeof(FLAGUINT)*3);
	if(!(r3t->bptables))
	{
		r3t_destroy(r3t);
		return ERROR_MEM;
	}
	r3t->totalmem += (r3t->textLen)*sizeof(FLAGUINT)*3;	
	


	return 0;	
}

/* find entry in bp-table of node v, such that this entry is
 * superset of entry from bptab, with same left context
 * (entry in bp-table represents union tree node i.e. b+ set)
 * */
int r3th_findBpEntry(t_r3t* r3t, FLAGUINT* bptab, int v, int* ret)
{
	int j, unode2, fd, lc, unode1;
	
//	if(!r3t || !bptab || !ret)
//		return ERROR_INTEGRITY;
		
	*ret = UNDEF;
	
	if(r3t->bp[v] == UNDEF) 
		return 0;
		
	j = r3t->bp[v]-1;
	do {
		j++;
		unode2 = r3t->bptables[j].value; // index of an item in union forest
		fd = r3t->uf[unode2].value;
		lc = r3t->text[r3t->uf[fd].value-1];
		unode1 = bptab[lc].value;
		if(unode1 == UNDEF) {
			*ret = unode2;
			return 0;
		}
		else {
			if(unode2 > unode1 && r3t->uf[unode2].value <= r3t->uf[unode1].value) {
				*ret = unode2;
				return 0;
			}
		}		
	} while(r3t->bptables[j].flag2 != 1);
	return 0;	
}

/* find entry in bp-table of node v, such that this entry is
 * superset of entry from bptab, with same left context and
 * left context is different from lc1
 * (entry in bp-table represents union tree node i.e. b+ set) 
 * */
int r3th_findBpEntryDifferentFromLc(t_r3t* r3t, FLAGUINT* bptab, int v, int lc1, int* ret)
{
	int j, unode2, fd, lc, unode1;
	
//	if(!r3t || !bptab || !ret)
//		return ERROR_INTEGRITY;
		
	*ret = UNDEF;
	
	if(r3t->bp[v] == UNDEF) 
		return 0;
		
	j = r3t->bp[v]-1;
	do {
		j++;
		unode2 = r3t->bptables[j].value; // index of an item in union forest
		fd = r3t->uf[unode2].value;
		lc = r3t->text[r3t->uf[fd].value-1];
		if(lc != lc1) {
			unode1 = bptab[lc].value;
			if(unode1 == UNDEF) {
				*ret = unode2;
				return 0;
			}
			else {
				if(unode2 > unode1 && r3t->uf[unode2].value <= r3t->uf[unode1].value) {
					*ret = unode2;
					return 0;
				}
			}		
		}			
	} while(r3t->bptables[j].flag2 != 1);
	return 0;	
}

/* find two different entries in bp-table of node v, that are supersets
 * of entries from bptab
 * (entry in bp-table represents union tree node i.e. b+ set)
 * */
int r3th_findBpEntries(t_r3t* r3t, FLAGUINT* bptab, int v, int* ret1, int* ret2)
{
	int j, unode2, fd, lc, unode1;
	
//	if(!r3t || !bptab || !ret1 || !ret2)
//		return ERROR_INTEGRITY;
		
	*ret1 = UNDEF;
	*ret2 = UNDEF;
	
	if(r3t->bp[v] == UNDEF) 
		return 0;
		
	j = r3t->bp[v]-1;
	do {
		j++;
		unode2 = r3t->bptables[j].value; // index of an item in union forest
		fd = r3t->uf[unode2].value;
		lc = r3t->text[r3t->uf[fd].value-1];
		unode1 = bptab[lc].value;
		if(unode1 == UNDEF) {
			*ret1 = unode2;
			break;
		}
		else {
			if(unode2 > unode1 && r3t->uf[unode2].value <= r3t->uf[unode1].value) {
				*ret1 = unode2;
				break;
			}
		}		
	} while(r3t->bptables[j].flag2 != 1);

	while(r3t->bptables[j].flag2 != 1) {
		j++;
		unode2 = r3t->bptables[j].value; // index of an item in union forest
		fd = r3t->uf[unode2].value;
		lc = r3t->text[r3t->uf[fd].value-1];
		unode1 = bptab[lc].value;
		if(unode1 == UNDEF) {
			*ret2 = unode2;
			break;
		}
		else {
			if(unode2 > unode1 && r3t->uf[unode2].value <= r3t->uf[unode1].value) {
				*ret2 = unode2;
				break;
			}
		}		
	}
	return 0;	
}

/* top down traversal of r3 tree
 * computes r3t-up2, r3t->lc1 arrays
 * */
int r3th_traverseTopDown(t_r3t* r3t, t_r3t_work* r3tw)
{
	int v, u, result, unode1, unode2, lc, temp;
	FLAGUINT* up1;
	FLAGUINT* lcp;
	FLAGUINT* uf;
	int* up2;
	BYTE* lc1;
	FLAGUINT bptab[SIGMA_SIZE];
	BYTE* text;
	FLAGUINT* bptables;
	
	if(!r3t || !r3tw )
		return  ERROR_INTEGRITY;
		
	// size N
	r3t->lc1 = (BYTE*) malloc( r3tw->nodeCnt );
	if(!(r3t->lc1))
		return ERROR_MEM;
	r3t->totalmem += (r3tw->nodeCnt);		
		
	if(	!r3t->bptables || 
		!r3t->bp || 
		!r3t->uf || 
		!r3t->text || 
		!r3t->up1 ||
		!r3t->up2 ||
		!r3t->lc1 ||
		r3t->textLen < 0 ||
		r3t->textLen > FLAGUINT_MAX)
		return ERROR_INTEGRITY;

	up1 = r3t->up1;
	up2 = r3t->up2;
	lcp = r3t->lcp;
	bptables = r3t->bptables;
	text = r3t->text;
	lc1 = r3t->lc1;
	uf = r3t->uf;
	up1[r3t->size-1].value = UNDEF;
	up1[r3t->size-1].flag1 = 0;
	up1[r3t->size-1].flag2 = 0;  // reserved (not in use)
	up2[r3t->size-1] = UNDEF;
	
	for(v=r3t->size-2; v>=0; v--)
	{
		result = r3th_loadBpTable(r3t, bptab, v);
		if(result)
			return result;
		u = up1[v].value;
		if(	!lcp[v].flag2 && lcp[u].flag2 ) {
			 // 0 is extra position in u
			up1[v].flag1 = 1;
			
			result = r3th_findBpEntry(r3t, bptab, u, &unode1);
			if(result)
				return result;
				
			if(unode1 == UNDEF) {
				if(up1[u].flag1)
					up2[v] = up2[u];
				else
					up2[v] = up1[u].value;
			} else {
				up2[v] = u;
			}
		} else {
			result = r3th_findBpEntries(r3t, bptab, u, &unode1, &unode2);
			if(result)
				return result;
#ifdef DEBUG2			
			if(unode1 == UNDEF)
				return ERROR_INTEGRITY;				
#endif
			temp = uf[unode1].value;
			lc = text[uf[temp].value-1];
			lc1[v] = lc; 
			if(unode2 == UNDEF) {
				if(up1[u].flag1 || lc1[u] != lc)
					up2[v] = up1[u].value; 
				else 
					up2[v] = up2[u]; 
			} else {
				up2[v] = u;
			}
		}
	}		
	return 0;
}

/* bottom up traversal of lcp-interval tree
 * builds r3t->lcp, r3t->bp, r3t->bptables, r3t->uf
 * needs 8*nodeCnt extra space 
 * */
int r3th_traverseBottomUp(t_r3t* r3t, t_r3t_work* r3tw)
{
	int i, lb, result, lcp_i;
	int cnt, top;
	int* stack_lcp;
	int* stack_lb;
	int* stack_fd;
	int* lcp;
	BYTE* text;
	int textLen;
	int reset_fd = 0;
	
	
	if(!r3t || !r3tw)
		return ERROR_INTEGRITY;
	
	stack_lcp = (int*) r3t->up1; // this array will be reused
	stack_lb = (int*) malloc((r3tw->nodeCnt+1)*sizeof(int));
	stack_fd = (int*) malloc((r3tw->nodeCnt+1)*sizeof(int));
	if(!stack_lb || !stack_fd)
		return ERROR_MEM;
	r3t->totalmem += (r3tw->nodeCnt+1)*sizeof(int)*2;
	
	lcp = r3tw->lcp;
	text = r3t->text;
	textLen = r3t->textLen;

	if(!text || !(r3tw->sa) || !lcp || !stack_lcp)
		return ERROR_INTEGRITY;
	
	r3t->size = 0;
	reset_fd = 1;
	
	stack_lcp[0] = 0;
	stack_lb[0] = 0;
	stack_fd[0] = 0;

	cnt = 1;
	top = 0;

	for(i=1; i<textLen; i++) // main loop
	{
		lcp_i = lcp[i]; // lcp array may be rewritten by r3th_processIntervalBottomUp
		lb = i-1;       // up to index i, so we have to save lcp[i] value
		while(lcp_i < stack_lcp[top]) 
		{
			cnt--; top--; // ( stack_lcp[cnt], stack_lb[cnt], stack_fd[cnt] ) = pop()
			result = r3th_processIntervalBottomUp(
				r3t,
				r3tw, 
				stack_lcp[cnt],
				stack_lb[cnt],
				i-1, // rb
				stack_fd[cnt] );
			if(result)
				return result;
	
			lb = stack_lb[cnt];
			if(lcp_i == stack_lcp[top]) 
				reset_fd = 1;
			else
				reset_fd = 0; // last_fd = stack_fd[cnt];
		}

		if(lcp_i > stack_lcp[top])
		{
			stack_lcp[cnt] = lcp_i; // push ( lcp[i], lb, fd )
			stack_lb[cnt] = lb;
			if(reset_fd)
				stack_fd[cnt] = r3t->size;
			else
				reset_fd = 1; // stack_fd[cnt] = last_fd;
			cnt++; top++;
		}
	}

	while(top)
	{
		cnt--; top--; // pop

		result = r3th_processIntervalBottomUp(
			r3t,
			r3tw, 
			stack_lcp[cnt],
			stack_lb[cnt],
			textLen-1, // rb
			stack_fd[cnt] );
			
		if(result)
			return result;
	
	}

	/* process root lcp-interval (0,0,textLen-1) */
	result = r3th_processIntervalBottomUp(
		r3t,
		r3tw, 
		0, // lcpval
		0, // lb
		textLen-1, // rb
		0 ); //fd

	if(result)
		return result;	

	r3t->peakmem = r3t->totalmem; // now we are using maximal amount of memory

	// now it's time for little compactation
	free(stack_lb);
	free(stack_fd);
	r3t->totalmem -= (r3tw->nodeCnt+1)*sizeof(int)*2;
	r3tw->fd = r3tw->lcp;
	r3tw->lcp = NULL;
	r3tw->sa = realloc(r3tw->sa, (r3t->size)*sizeof(int));
	if(!(r3tw->sa))
		return ERROR_MEM;
	r3t->totalmem -= (r3t->textLen-r3tw->nodeCnt)*sizeof(int);
	if(r3t->bptables_size) {
		r3t->bptables = realloc(r3t->bptables, (r3t->bptables_size)*sizeof(FLAGUINT));
		if(!(r3t->bptables))
			return ERROR_MEM;
		r3t->totalmem -= (r3t->textLen*3-r3t->bptables_size)*sizeof(FLAGUINT);
	} else {
		if(r3t->bptables) {
			free(r3t->bptables);
			r3t->bptables = NULL;
		}
		r3t->totalmem -= r3t->textLen*3*sizeof(FLAGUINT);
	}
	
	return 0;
}

/* loads bp-table stored in node v into dest_table location
 * */
int r3th_loadBpTable(t_r3t* r3t, FLAGUINT* dest_table, int v)
{
	int i, j, k, lc;
	
	if(!dest_table || !r3t)
		return ERROR_INTEGRITY;
	if(	!r3t->bp || 
		!r3t->text || 
		r3t->textLen < 0 ||
		r3t->textLen > FLAGUINT_MAX)
		return ERROR_INTEGRITY;

	for(i=0; i<SIGMA_SIZE; i++) {
		dest_table[i].flag1 = 0;
		dest_table[i].flag2 = 0;
		dest_table[i].value = UNDEF;
	}
	
	if(r3t->bp[v]== UNDEF)
		return 0; // empty bp-table

	if(!r3t->bptables || !r3t->uf)
		return ERROR_INTEGRITY;
	
	i = r3t->bp[v] -1;
	do {
		i++;
#ifdef DEBUG2		
		if(i < 0 || i >= (r3t->textLen)*3)
			return ERROR_INTEGRITY;
#endif
		j = r3t->bptables[i].value; // index of an item in union forest
#ifdef DEBUG2		
		if(j < 0 || j >= (r3t->textLen-1)*3 || !r3t->uf[j].flag2)
			return ERROR_INTEGRITY;
#endif
		k = r3t->uf[j].value;
#ifdef DEBUG2		
		if(k < 0 || k >= j || r3t->uf[k].flag2)
			return ERROR_INTEGRITY;
#endif
		lc = r3t->text[r3t->uf[k].value-1];
		dest_table[lc].value = j;
		dest_table[lc].flag1 = r3t->bptables[i].flag1;
	} while(r3t->bptables[i].flag2 != 1);

	return 0;
}

/* this function returns bucket b(v,a) when we supply entry b of bp-table for
 * r3 node v that represents pointer to union tree with left context a
 * contents of bucket is returned in array array of size array_size
 * */
int r3th_getBucket1(t_r3t* r3t, FLAGUINT b, int** array, int* array_size)
{
	FLAGUINT* uf; 
	t_aarray a;
	int i;
	int u = b.value;

	if(!r3t ||
	   !array ||
	   !array_size)
	   return ERROR_INTEGRITY;
	if(!r3t->uf ||
		u < 0 ||
		u > FLAGUINT_MAX)
		return ERROR_INTEGRITY;
	uf=r3t->uf;

	if(u == UNDEF || !b.flag1)
	{
		*array = NULL;
		*array_size = 0;
		return 0;
	}
	if(!uf[u].flag2) 
		return ERROR_INTEGRITY;
	if(uf[u-1].flag2) // we are in u-navigator node
		u--;
			
	aar_init(&a, AAR_SMALL_ALLOC, AAR_SMALL_ALLOC);
	for(i=uf[u].value; i<u; i++) {
		if(uf[i].flag2)
			return ERROR_INTEGRITY;
		aar_append(&a, uf[i].value);
	}

	aar_compact(&a);
	(*array) = a.array;
	(*array_size) = a.size;
	return 0;
}

 
/* this function returns set b+(v,a) - b(v,a) when we supply entry b of bp-table for
 * r3 node v that represents pointer to union tree with left context a
 * contents of the set is returned in array array of size array_size
 * */
int r3th_getBucket2(t_r3t* r3t, FLAGUINT b, int** array, int* array_size)
{
	FLAGUINT* uf; 
	t_aarray a;
	int u= b.value;
	int i, temp;

	if(!r3t ||
	   !array ||
	   !array_size)
	   return ERROR_INTEGRITY;
	if(!r3t->uf ||
		u < 0 ||
		u > FLAGUINT_MAX)
		return ERROR_INTEGRITY;
	uf=r3t->uf;

	if(u == UNDEF)
	{
		*array = NULL;
		*array_size = 0;
		return 0;
	}

	if(!uf[u].flag2) 
		return ERROR_INTEGRITY;
	
	if(b.flag1) {
		if(!uf[u-1].flag2) {
			*array = NULL; // u is b-navigator item, and therefore b+(u, LC(u))==b(u, LC(u))
			*array_size = 0;
			return 0;
		}
		aar_init(&a, AAR_SMALL_ALLOC, AAR_SMALL_ALLOC);
		temp = uf[u-1].value-1; // last but one child of u
		for(i=uf[u].value; i<temp; i++) {
			if(!uf[i].flag2)
				aar_append(&a, uf[i].value);
		}
		aar_compact(&a);
		(*array) = a.array;
		(*array_size) = a.size;
		return 0;
	}else{
		aar_init(&a, AAR_SMALL_ALLOC, AAR_SMALL_ALLOC);
		for(i=uf[u].value; i<u; i++) {
			if(!uf[i].flag2)
				aar_append(&a, uf[i].value);
		}
		aar_compact(&a);
		(*array) = a.array;
		(*array_size) = a.size;
		return 0;
	}
}
