/*  r3.h
 * 
 *  R3lib
 * 
 *  R3 tree definition and routines
 * 
 *  Copyright (C) 2006-2007  Michal Linhard <michal@linhard.sk>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  as published by the Free Software Foundation; either version 2.1
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */ 
#ifndef R3_H_
#define R3_H_
#include "conf.h" 

/************************************************************************************/
/* LIBRARY INTERFACE - R3 TREE DATA STRUCTURES                                                          */  
/************************************************************************************/

/* R3 tree */
typedef struct s_r3t
{
	FLAGUINT*	lcp;     // value = lcp value flag1 = b(v,cent)=emptyset? flag2 = b+(v,cent)=emptyset?
	FLAGUINT*	up1;
	int*		up2;
	int*		bp;      // idx to bptables
	FLAGUINT*	bptables;// bp-table for given node value = idx to uf, flag1 = ownflag, flag2 = endflag
	FLAGUINT*	uf;      // union forest value = flag2 ? firstdesc : position ;
	int*		map;     // map[p] is idx of v such that p in b(v,LC(p))
	BYTE*		text;    // text for which r3 tree is constructed
	BYTE*		lc1;
	int			textLen; // length of text with endmarker $
	int 		size;    // number of lcp-intervals i.e. number of lcp, parent, bp items
	int 		bptables_size; // size of bptables array
	int			uf_size; // size of union forest
	UINT		totalmem; // total memory usage
	UINT		peakmem; // peak memory usage during construction
} t_r3t;

/************************************************************************************/
/* LIBRARY INTERFACE - R3 TREE ROUTINES                                                                 */  
/************************************************************************************/

/* create r3 tree for given text */
int r3t_create(t_r3t* r3t, BYTE* text, int textLen);

/* free memory allocated by r3t */
int r3t_destroy(t_r3t* r3t);

/* find all pairs (p2, l) such that (p1, p2, l) is maximal repeat in 
 * r3t->text with l>=k */
int r3t_findPairs(t_r3t* r3t, int p1, int k, int* count, int count_limit, int* p2, int* l);

/************************************************************************************/
/* INTERNAL DATA STRUCTURES                                                         */  
/************************************************************************************/

/* R3 tree working structure, used during construction */
typedef struct s_r3t_work {
	int nodeCnt; // number of r3 nodes
	int maxStackBottomUp; // max size of stack during bottom up traversal
	int ut_cnt[SIGMA_SIZE]; // counts of symbols in string -> sizes of union trees
	int ut_ptr[SIGMA_SIZE]; // points to start of union tree in union forest
	int* sa;
	int* lcp;
	int* fd;
} t_r3t_work;

typedef struct s_repeat {
	int p1;
	int p2;
	int l;
} t_repeat;

/************************************************************************************/
/* INTERNAL ROUTINES                                                                */  
/************************************************************************************/
int r3th_processIntervalBottomUp(	t_r3t*		r3t, 
									t_r3t_work*	r3tw,
									int			lcpval, 
									int			lb,
									int			rb,
									int			fd );
int r3th_computeParent(t_r3t* r3t, t_r3t_work* r3tw);
int r3th_loadBpTable(t_r3t* r3t, FLAGUINT* dest_table, int v);
void r3th_combineUnionSubtree(FLAGUINT* uf, int u, int visited, int lcp, int* count, int count_limit, int* p2, int* l);
int r3th_allocate(t_r3t* r3t, t_r3t_work* r3tw);
int r3th_traverseBottomUp(t_r3t* r3t, t_r3t_work* r3tw);
int r3th_traverseTopDown(t_r3t* r3t, t_r3t_work* r3tw);
int r3th_computeMap(t_r3t* r3t, t_r3t_work* r3tw);
int r3th_computeParent(t_r3t* r3t, t_r3t_work* r3tw);
int r3th_compactUnionForest(t_r3t* r3t, t_r3t_work* r3tw);
int r3th_createEmpty(t_r3t* r3t);
int r3th_getBucket1(t_r3t* r3t, FLAGUINT b, int** array, int* array_size);
int r3th_getBucket2(t_r3t* r3t, FLAGUINT b, int** array, int* array_size);
int r3th_findBpEntries(t_r3t* r3t, FLAGUINT* bptab, int v, int* ret1, int* ret2);
int r3th_findBpEntry(t_r3t* r3t, FLAGUINT* bptab, int v, int* ret);
int r3th_findBpEntryDifferentFromLc(t_r3t* r3t, FLAGUINT* bptab, int v, int lc1, int* ret);

#endif /*R3_H_*/
