/*  debug.c
 * 
 *  R3lib
 * 
 *  Debug routines
 * 
 *  Copyright (C) 2006-2007  Michal Linhard <michal@linhard.sk>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  as published by the Free Software Foundation; either version 2.1
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 */ 
#include "../conf.h"
#include "../bitstr.h"
#include "debug.h"
#include "test.h"
#include "../suffixArray.h"
#include "../r3.h"

void output_work_ut_info(t_r3t_work* r3tw, char* filename)
{
	int a;
	FILE *f = fopen(filename, "w");
	
	if(!r3tw)
		return;
	if(!f) 
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	
	for(a=0; a<SIGMA_SIZE; a++)
	{
		if(r3tw->ut_cnt[a])
		{
			fprintf(f, "ut_cnt[%i]", a);
			if(a>=32) // printable character
				fprintf(f, " (%c)", a);
			fprintf(f, "==%i, ut_ptr[%i]==%i\n", r3tw->ut_cnt[a], a, r3tw->ut_ptr[a]);
		}
	}
	
	fclose(f);	
}

void output_union_trees(t_r3t* r3t, t_r3t_work* r3tw, char* filename)
{
	int a,i;
	FILE *f = fopen(filename, "w");
	if(!r3t || !r3tw)
		return;
	if(!f) 
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	
	fprintf(f, "   i       a[i]  f1  f2\n");
	 
	for(a=0; a<SIGMA_SIZE; a++)
	{
		if(r3tw->ut_cnt[a])
		{
			fprintf(f, "----ut[%i]", a);
			if(a>=32) // printable character
				fprintf(f, " (%c)", a);
			fprintf(f, "----\n");

			for(i=r3tw->ut_ptr[a]; i<r3tw->ut_cnt[a]+r3tw->ut_ptr[a]; i++)
			{
				fprintf(f, "%4i %10i %3i %3i\n", i, r3t->uf[i].value, r3t->uf[i].flag1, r3t->uf[i].flag2 );
			}
		}
	}
	
	fclose(f);	
}

void output_union_trees_html(t_r3t* r3t, char* filename)
{
	int a,i,u, result;
	FILE *f;
	FLAGUINT bptab[SIGMA_SIZE];
	
	if(!r3t || !filename)
		return;
	
	f = fopen(filename, "w");
	if(!f) 
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	
	result = r3th_loadBpTable(r3t, bptab, r3t->size-1);
	if(result)
		return;
	
	// print html file header
	fprintf(f, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n"
				"<html>\n"
				"<head>\n"
				"<title>Union Trees</title>\n"
				"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\">\n"
				"</head>\n"
				"<body>\n"
				"<p>UNION TREES</p>\n");
	 
	for(a=0; a<SIGMA_SIZE; a++)
	{
		u=bptab[a].value;
		if(u != UNDEF)
		{
			if(u < 0 || 
			u >= (r3t->textLen-1)*3 || 
			!(r3t->uf[u].flag2) ||
			r3t->uf[u].value >= u) {
				fprintf(stderr, "integrity error\n");
				return;
			}
			fprintf(f, 
				"<table width=\"50%%\" border=\"1\" cellpadding=\"2\" cellspacing=\"0\" bordercolor=\"#666666\">\n"
  				"<tr bgcolor=\"#99CCFF\">\n" 
    			"<td width=\"10%%\">&nbsp;</td>\n"
				"<td width=\"90%%\">ut[%i]",a);
			if(a>=32) // printable character
				fprintf(f, " (%c)", a);
			fprintf(f, "</td>\n</tr>\n");

			for(i=r3t->uf[u].value; i<=u; i++)
			{
				fprintf(f, "<tr");
				if(r3t->uf[i].flag2)
					fprintf(f, " bgcolor=\"#999999\"");
				fprintf(f, 
					">\n<td>%i</td>\n"
					"<td>%i</td>\n"
					"</tr>", i, (r3t->uf[i].value) );
			}
			
			fprintf(f, "</table>\n");
		}
	}
	
	fprintf(f, "<p>&nbsp;</p></body></html>\n");
	
	fclose(f);	
}


int output_r3t_html_file_file(char* input_filename, char* output_filename)
{
	BYTE* text;
	int textLen;
	t_r3t r3t;
	int result;
	
	if(!input_filename || !output_filename)
		return ERROR_INTEGRITY;

	result = load_data(&text, &textLen, input_filename);
	if(result) {
		return result; }

    result = r3t_create(&r3t, text, textLen); 
    if(result) { 	
		free(text);
    	return result; 
    }
    output_r3t_html_file(&r3t,output_filename);
	free(text);
	r3t_destroy(&r3t);
    return 0;
}


/* output_r3t_html_file
 * output r3 tree to html file
 * */
int output_r3t_html_file(t_r3t* r3t, char* output_filename)
{
	int i, j, a, result;
	FLAGUINT bptab[SIGMA_SIZE];
	int* bucket;
	int bucket_cnt;

	FILE* fout = fopen(output_filename, "w");
	if(!fout)
	{
		fprintf(stderr, "\nerror opening output file '%s'.\n", output_filename);
		return ERROR_FILEOPEN; 
	}
	
	if(!r3t) {
		fprintf(stderr, "integrity error\n");
		return ERROR_INTEGRITY;
	}

	/* print html head */
	fprintf(fout,  
		"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n"
		"<html>\n"
		"<head>\n"
		"<title>R3 Tree</title>\n"
		"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=iso-8859-1\">\n"
		"</head>\n");

	/* print r3 node table header */
	fprintf(fout,  
		"<body>\n"
		"<p>R3 TREE NODES (lcp-intervals)</p>\n"
		"<table width=\"100%%\" border=\"1\" cellpadding=\"2\" cellspacing=\"0\" bordercolor=\"#666666\">\n"
  		"<tr bgcolor=\"#33CCFF\">\n"
		    "<td width=\"10%%\">id</td>\n"
		    "<td width=\"10%%\">lcp</td>\n"
		    "<td width=\"10%%\">parent1</td>\n" 
		    "<td width=\"10%%\">parent2</td>\n" 
		    "<td width=\"10%%\">lc1</td>\n" 
		    "<td width=\"10%%\">bp-table</td>\n" 
		"</tr>\n");
		
	/* print nodes in table rows */
	for(i=0; i<r3t->size; i++)
	{
		
		fprintf(fout,
			"<a name=\"r3_%i\"></a>\n"  
			"<tr>\n"
				"<td><div align=\"right\">%i</div></td>\n"
				"<td><div align=\"right\">%i</div></td>\n", i, i, r3t->lcp[i].value);
				
		if(r3t->up1[i].value != UNDEF)
			fprintf(fout, "<td><a href=\"#r3_%i\">%i</a></td>\n", r3t->up1[i].value, r3t->up1[i].value);
		else			
			fprintf(fout, "<td>UNDEF</td>\n");

		if(r3t->up2[i] != UNDEF)
			fprintf(fout, "<td><a href=\"#r3_%i\">%i</a></td>\n", r3t->up2[i], r3t->up2[i]);
		else			
			fprintf(fout, "<td>UNDEF</td>\n");

		if(r3t->up1[i].flag1)
			fprintf(fout, "<td>cent</td>\n");
		else {			
			if(r3t->lc1[i] < 32)
				fprintf(fout, "<td>%i</td>\n", r3t->lc1[i]);
			else
				fprintf(fout, "<td>%i (%c)</td>\n", r3t->lc1[i],  r3t->lc1[i]);
		}		

		if(r3t->bp[i] != UNDEF)
			fprintf(fout, "<td><a href=\"#bp_%i\">%i</a></td>\n", i, i);
		else			
			fprintf(fout, "<td>UNDEF</td>\n");

		fprintf(fout,"</tr>\n");
	}
		
	/* print r3 node table footer */
	fprintf(fout,  
		"</table>\n"
		"<p>BPTABLES</p>\n");

	/* print bp-tables */
	for(i=0; i<r3t->size; i++)
	{
		result = r3th_loadBpTable(r3t, bptab, i);
		if(result) {
			fprintf(stderr, "loadBpTable error\n");
			return result;
		}

		fprintf(fout,
			"<a name=\"bp_%i\"></a>" 
			"<table width=\"80%%\" border=\"1\" cellpadding=\"2\" cellspacing=\"0\" bordercolor=\"#666666\">"
				"<tr bgcolor=\"#FFCC99\">"
					"<td width=\"8%%\">a</td>"
					"<td width=\"46%%\">b(%i,a)</td>"
					"<td width=\"46%%\">b+(%i,a)-b(%i,a)</td>"
				"</tr>", i, i, i, i);
				
		if(r3t->lcp[i].flag1)
			fprintf(fout, "<tr>\n<td>cent</td>\n<td>0</td>\n<td>&nbsp;</td>\n</tr>\n");

		if(r3t->lcp[i].flag2 && !r3t->lcp[i].flag1)
			fprintf(fout, "<tr>\n<td>cent</td>\n<td>&nbsp;</td>\n<td>0</td>\n</tr>\n");
					
		for(a=0; a<SIGMA_SIZE; a++)
			if(bptab[a].value != UNDEF)
			{
				if(a < 32)
					fprintf(fout, "<tr>\n<td>%i</td>\n<td>", a);
				else
					fprintf(fout, "<tr>\n<td>'%c'</td>\n<td>", a);
				
				r3th_getBucket1(r3t, bptab[a], &bucket, &bucket_cnt);
				
				for(j=0; j<bucket_cnt; j++)
					fprintf(fout, "%i ", bucket[j]);
				
				if(bucket)
					free(bucket);

				if(bucket_cnt==0)
					fprintf(fout, "&nbsp;");	

				fprintf(fout, "</td>\n<td>");

				r3th_getBucket2(r3t, bptab[a], &bucket, &bucket_cnt);
				
				for(j=0; j<bucket_cnt; j++)
					fprintf(fout, "%i ", bucket[j]);
				
				if(bucket)
					free(bucket);

				if(bucket_cnt==0)
					fprintf(fout, "&nbsp;");	
	
				fprintf(fout, "</td>\n</tr>\n");
			}

		fprintf(fout, "</table>\n"); 
	}

	/* print html file footer */ 
	fprintf(fout,  
		"<p>&nbsp;</p>"
		"</body>"
		"</html>");
				
	fclose(fout); 
	return 0;	
}
	
/* output_sa_and_lcp_for_file
 * 
 * takes input_file, creates sa and lcp on it and outputs in user readable format
 * into output_file
 * useful for debugging with small text files
 * 
 * */
void output_sa_lcp_from_file(char* input_file, char* output_file)
{
	BYTE* text;
	int textLen;
	int* suffixArray;
	int* lcp;
	if((!input_file) || (!output_file))
	{
		fprintf(stderr, "null filename\n");
		return;
	}

	if( load_data(&text, &textLen, input_file) )	
	{
		fprintf(stderr, "error loading file '%s'.\n", input_file);
		return; 
	}

    if( createSuffixArray(text, textLen, &suffixArray) ) 
    { 	
    	free(text);
		fprintf(stderr, "createSuffixArray error.\n");
    	return; 
    }
		
	if( createLCPTable(text, textLen, suffixArray, &lcp) )
	{
    	free(text);
    	free(suffixArray);
		fprintf(stderr, "createLCPTable error.\n");
		return; 
	}	

	output_sa_lcp(text, textLen, suffixArray, lcp, output_file);
	free(text);
	free(suffixArray);
	free(lcp);
	return;	
}

/* output_sa_lcp_from_file_trunc
 * 
 * takes input_file, creates sa and lcp on it and outputs in user readable format
 * into output_file, truncates suffixes after n characters
 * this can handle larger files (doesn't print whole suffixes)
 * 
 * */
void output_sa_lcp_from_file_trunc(char* input_file, char* output_file, int n)
{
	BYTE* text;
	int textLen;
	int* suffixArray;
	int* lcp;
	if((!input_file) || (!output_file))
	{
		fprintf(stderr, "null filename\n");
		return;
	}

	if( load_data(&text, &textLen, input_file) )	
	{
		fprintf(stderr, "error loading file '%s'.\n", input_file);
		return; 
	}

    if( createSuffixArray(text, textLen, &suffixArray) ) 
    { 	
    	free(text);
		fprintf(stderr, "createSuffixArray error.\n");
    	return; 
    }
		
	if( createLCPTable(text, textLen, suffixArray, &lcp) )
	{
    	free(text);
    	free(suffixArray);
		fprintf(stderr, "createLCPTable error.\n");
		return; 
	}	

	output_sa_lcp_trunc(text, textLen, suffixArray, lcp, n, output_file);
	free(text);
	free(suffixArray);
	free(lcp);
	return;	
}

/* works probably only with optimalisation turned off durnig compilation
 * */
void test_writability(BYTE* mem, UINT count)
{
	UINT i;
	BYTE origval;
	BYTE testval = (BYTE) 'A'; 
	for(i=0;i<count;i++)
	{
		origval = mem[i];
		mem[i] = testval;
		mem[i] = origval;
	}
}

/* output suffix array values to file
 * print max 10 characters for each suffix
 * */
void output_sa_to_file_10(BYTE* text, UINT textLen, UINT* suffixArray, char* filename)
{
	int i,j;
	FILE *f = fopen(filename, "w");
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	for(i=0; i<textLen; i++)
	{
		fprintf(f,"\n%4i: %4i\t",i,suffixArray[i]);
		// print suffix
		for(j=suffixArray[i]; (j < suffixArray[i]+10) && (j < textLen); j++)
		{
			if(text[j] >= 0x20) // printable?
			  fprintf(f,"%c",text[j]);
			else
			  fprintf(f,"\\%i",text[j]);
		}
	}
	fclose(f);
}

void output_sa_to_file(BYTE* text, UINT textLen, UINT* suffixArray, char* filename)
{
	int i,j;
	FILE *f = fopen(filename, "w");
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	for(i=0; i<textLen; i++)
	{
		fprintf(f,"\n%4i: %4i\t",i,suffixArray[i]);
		// print suffix
		for(j=suffixArray[i]; j < textLen; j++)
		{
			if(text[j] >= 0x20) // printable?
			  fprintf(f,"%c",text[j]);
			else
			  fprintf(f,"\\%i",text[j]);
		}
	}
	fclose(f);
}

void output_sa_to_file2(BYTE* text, UINT textLen, UINT arrayLen, UINT* suffixArray, char* filename)
{
	int i,j;
	FILE *f = fopen(filename, "w");
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	for(i=0; i<arrayLen; i++)
	{
		fprintf(f,"\n%4i: %4i\t",i,suffixArray[i]);
		// print suffix
		for(j=suffixArray[i]; j < textLen; j++)
		{
			if(text[j] >= 0x20) // printable?
			  fprintf(f,"%c",text[j]);
			else
			  fprintf(f,"\\%i",text[j]);
		}
	}
	fclose(f);
}

void output_sa_to_file3(int* text, int textLen, int* suffixArray, char* filename)
{
	int i,j;
	FILE *f = fopen(filename, "w");
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	for(i=0; i<textLen; i++)
	{
		fprintf(f,"\n%4i: %4i\t",i,suffixArray[i]);
		// print suffix
		for(j=suffixArray[i]; j < textLen; j++)
		{
			  fprintf(f,"%i ", text[j]);
		}
	}
	fclose(f);
}

void output_intarray(int* array, int arrayLen, char* filename)
{
	int i;
	FILE *f = fopen(filename, "w");
	
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	for(i=0; i<arrayLen; i++)
	{
		fprintf(f,"%4i: %4i\n",i,array[i]);
	}
	fclose(f);
}

void output_FLAGUINTarray(FLAGUINT* array, int arrayLen, char* filename)
{
	int i;
	FILE *f = fopen(filename, "w");
	
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	fprintf(f, "   i       a[i]  f1  f2\n"); 
	for(i=0; i<arrayLen; i++)
	{
		fprintf(f,"%4i %10i %3i %3i\n", i, array[i].value, array[i].flag1, array[i].flag2);
	}
	fclose(f);
}



void output_dist(int* dist, int distSize, int* distCnt, int maxDist, char* filename)
{
	int i;
	FILE *f = fopen(filename, "w");
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	fprintf(f,"Dist:\n");
	for(i=0; i<distSize; i++)
	{
		fprintf(f,"%4i: %4i\n",i,dist[i]);
	}
	fprintf(f,"DistCnt:\n");
	for(i=0; i<=maxDist; i++)
	{
		fprintf(f,"%4i: %4i\n",i,distCnt[i]);
	}
	fclose(f);
}

void output_lists(int* list, int* boolBuckList, int buckListSize, char* filename)
{
	int i;
	FILE *f = fopen(filename, "w");
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	fprintf(f,"list and buckets:\n");
	for(i=0; i<buckListSize; i++)
	{
		fprintf(f,"%4i: %4i\t%4i\n",i,list[i], boolBuckList[i]);
	}
	fclose(f);
}

void output_array_buck(int* list, int listSize, int* boolBuckets, char* filename)
{
	int i;
	FILE *f = fopen(filename, "w");
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	for(i=0; i<listSize; i++)
	{
		fprintf(f,"%4i: %4i\t%4i\n",i,list[i], boolBuckets[i]);
	}
	fclose(f);
}

void output_sort_suffix_state(BYTE* text, UINT textLen, UINT arrayLen, UINT* suffixArray, int* skipVal, 
t_bitstr* suffixType, t_bitstr* BuckA, char* filename)
{
	int i,j;
	FILE *f = fopen(filename, "w");
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	for(i=0; i<arrayLen; i++)
	{
		fprintf(f,"\n%4i: %4i\t%4i\t",i,suffixArray[i], skipVal[i]);
		//print dist
		if(suffixArray[i] != textLen-1)
		{
			j=suffixArray[i]+1;
			while( ( bs_getVal(suffixType, j) != bs_getVal(suffixType, suffixArray[i]) ) && (j < textLen))
				j++;
			fprintf(f,"%4i\t",(j-suffixArray[i]));
		}
		else
		{	 
			fprintf(f,"    \t");
		}
		// print BuckA
		fprintf(f,"%2i\t", bs_getVal(BuckA, i) );
		// print suffix
		for(j=suffixArray[i]; j < textLen; j++)
		{
			if(text[j] >= 0x20) // printable?
			  fprintf(f,"%c",text[j]);
			else
			  fprintf(f,"\\%i",text[j]);
		}
	}
	fclose(f);
}


void output_data(BYTE* text, int textLen, int* suffixArray, char* filename)
{
	int i,j;
	FILE *f = fopen(filename, "w");
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	for(i=0; i<textLen; i++)
	{
		fprintf(f,"\n%4i: %4i\t",i,suffixArray[i]);
		// print suffix
		for(j=suffixArray[i]; (j < suffixArray[i]+10) && (j < textLen); j++)
		{
			if(text[j] >= 0x20) // printable?
			  fprintf(f,"%c",text[j]);
			else
			  fprintf(f,"\\%i",text[j]);
		}
	}
	fclose(f);
}

void output_data2(BYTE* text, int n, int textLen, int* suffixArray, int* lcp, char* filename)
{
	int i;
	FILE *f = fopen(filename, "w");
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	for(i=0; i<n; i++)
	{
		fprintf(f,"\n%4i",suffixArray[i]);
		if(suffixArray[i])
			fprintf(f," %c", text[suffixArray[i]-1]);
	}
	fclose(f);
}

/* output_sa_lcp_trunc
 * outputs suffix array and lcp table
 * to an user readable text file
 * useful for debugging with small text files
 * truncate suffixes after n chars
 * */
void output_sa_lcp_trunc(BYTE* text, int textLen, int* suffixArray, int* lcp, int n, char* filename)
{
	int i,j;
	FILE *f = fopen(filename, "w");
	if(!text || !suffixArray || !lcp || (textLen == 0) || (textLen > FLAGUINT_MAX))
	{
		fprintf(stderr, "bad arguments.\n");
		return;
	}
	
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	for(i=0; i<textLen; i++)
	{
		fprintf(f,"\n%4i: %4i\t",i,suffixArray[i]);
		// print suffix
		for(j=suffixArray[i]; j < suffixArray[i]+n; j++)
		{
			if(j < textLen)
			{
				if(text[j] >= 0x20) // printable?
				  fprintf(f,"%c",text[j]);
				else
				  fprintf(f,"\\%i",text[j]);
			}
			else
				  fprintf(f," ");
		}
		fprintf(f,"\t\tLCP: %4i ",lcp[i]);
	}
	fclose(f);
}

/* output_sa_lcp
 * outputs suffix array and lcp table
 * to an user readable text file
 * useful for debugging with small text files
 * */
void output_sa_lcp(BYTE* text, int textLen, int* suffixArray, int* lcp, char* filename)
{
	int i,j;
	FILE *f = fopen(filename, "w");
	if(!text || !suffixArray || !lcp || (textLen == 0) || (textLen > FLAGUINT_MAX))
	{
		fprintf(stderr, "bad arguments.\n");
		return;
	}
	
	if(!f)
	{
		fprintf(stderr, "error opening output file '%s'\n", filename);
		return;
	}
	for(i=0; i<textLen; i++)
	{
		fprintf(f,"\n%4i: %4i\tLCP%4i\t",i,suffixArray[i], lcp[i]);
		// print suffix
		for(j=suffixArray[i]; j < textLen; j++)
		{
			if(text[j] >= 0x20) // printable?
			  fprintf(f,"%c",text[j]);
			else
			  fprintf(f,"\\%i",text[j]);
		}
	}
	fclose(f);
}

