/*  test.c
 * 
 *  R3lib
 * 
 *  Main framework for calling test routines
 *
 *  Copyright (C) 2006-2007  Michal Linhard <michal@linhard.sk>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  as published by the Free Software Foundation; either version 2.1
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA. 
 * 
 */
#include <stdio.h>
#include <string.h> // strlen
#include "../conf.h"
#include "../suffixArray.h"
#include "debug.h"
#include "test.h"
#include "test_sa.h"
#include "test_r3.h"
#include "analysis.h"

#define REP_LOAD 100

#define TESTDATA_SMALL_CNT    19
#define TESTDATA_TEXT_CNT      8
#define TESTDATA_BIN_CNT       6

char* testdata_small[TESTDATA_SMALL_CNT] = 
{ 
	"testdata/small/file01.txt", 
	"testdata/small/file02.txt",
	"testdata/small/file03.txt",
	"testdata/small/file04.txt",
	"testdata/small/file05.txt",
	"testdata/small/file06.txt",
	"testdata/small/file07.txt",
	"testdata/small/file08.txt",
	"testdata/small/file09.txt",
	"testdata/small/file10.txt",
	"testdata/small/file11.txt",
	"testdata/small/file12.txt",
	"testdata/small/file13.txt",
	"testdata/small/file14.txt",
	"testdata/small/file15.txt",
	"testdata/small/file16.txt",
	"testdata/small/file17.txt",
	"testdata/small/file18.txt",
	"testdata/small/file19.txt"
};

char* testdata_text[TESTDATA_TEXT_CNT] = 
{ 
	"testdata/text/19033.txt",
	"testdata/text/1ws0610.txt",
	"testdata/text/1ws1710.txt",
	"testdata/text/1ws3410.txt",
	"testdata/text/1ws1810.txt",
	"testdata/text/1ws2510.txt",
	"testdata/text/2ws2610.txt",
	"testdata/text/rbcru10.txt"
};

char* testdata_bin[TESTDATA_BIN_CNT] = 
{ 
	"testdata/bin/zlib123.zip",
	"testdata/bin/184722357.jpg",
	"testdata/bin/183899999.jpg",
	"testdata/bin/180907088.jpg",
	"testdata/bin/Azureus_2.5.0.4a_Win32.setup.exe",
	"testdata/bin/gimp-2.2.9.tar.gz"
};	

/* this function demonstrates usage of R3lib functions
 * */
int example()
{
   int result, i;
   t_r3t r3t; // r3 tree structure
   int* p2; // storage for p2 component of (p2, l) pair
   int* l; // storage for l component of (p2, l) pair
   int pair_cnt; // number of found pairs
   int k; // minimal repeat length
   int p1; // first component of maximal repeat
   int limit; // maximal number of returned repeats

   /* input text */
   /* PATTERN occurs at positions 4, 16, 28 and 40 */
   BYTE* text = "abcdPATTERNabceaPATTERNbcfabPATTERNcgabcPATTERNhabc";
	
   /* size of input */
   int textLen = strlen(text);
	
   /* create r3 tree for given input */
   result = r3t_create(&r3t, text, textLen); 
   if(result) 	
      return result;

   /* allocate space for query results */
   p2 = (int*) malloc(textLen*sizeof(int));
   l = (int*) malloc(textLen*sizeof(int));
   if(!p2 || !l)
      return -1;
    
   /* define parameters for findPairs query */
   p1 = 4; /* first component of maximal repeat */
   k = 7;  /* minimal repeat length */
   limit = textLen; /* maximal number of returned repeats */
    
   /* find all pairs (p2, l) such that (p1, p2, l) is 
    * maximal repeat in text and l >= k */
   result = r3t_findPairs(&r3t, p1, k, &pair_cnt, limit, p2, l);		
   if(result)
      return result;
		
   /* now we should have 
    *  
    * pair_cnt == 3
    * (p2[0], l[0]) == (16, 7)
    * (p2[1], l[1]) == (28, 7)
    * (p2[2], l[2]) == (40, 7)
    *  
    */
   printf("*** Example of findPairs query ***\n");	
   printf("Text: '%s'\n", text);
   printf("Position: %i\n", p1);
   printf("Min. length: %i\n", k);
	
   for(i=0; i<pair_cnt; i++) {
      /* (p1, p2[i], l[i]) now represents maximal repeat */
      printf("found repeat: (%i, %i, %i)\n", p1, p2[i], l[i]);
   }
	
   /* free resources held by r3 tree structure */
   r3t_destroy(&r3t); 
   return 0;
}

/* call this function to perform test routines for r3lib library
* 
* */
int test() 
{
	int result = 0;

    /* use this line if you want output to a file */
//	FILE* log = fopen("output/log.txt", "w");
	/* use this line if you want output to stdout */
	FILE* log = stdout;
	
    //output_r3t_html_file_file("testdata/small/file14.txt", "output/r3ta.html");
	
    /* testing suffix arrays */
//	result = test_sa_files(log, testdata_small, TESTDATA_SMALL_CNT );
//	if(result)
//		fprintf(log, "test_sa_files() returned %i\n", result);

    /* testing lcp tables */
//	result = test_lcp_files(log, testdata_small, TESTDATA_SMALL_CNT );
//	if(result)
//		fprintf(log, "test_lcp_files() returned %i\n", result);

    /* print suffix array with lcp table for given file, truncate suffixes (for debugging) */
//	output_sa_lcp_from_file_trunc("testdata/small/file04.txt", "output/sa1.txt", 20);

    /* print suffix array with lcp table for given file (for debugging) */
//	output_sa_lcp_from_file("testdata/small/file04.txt", "output/sa2.txt");

	/* print r3 tree for given file into html file (for debugging) */
//	output_r3_html_file("testdata/small/file13.txt", "output/r3.html");

//	result = testr3t_file(log, "testdata/small/file14.txt");
//	if(result)
//		fprintf(log, "testr3t_file() returned %i\n", result);
	
//	output_r3t_html_file_file("testdata/small/file14.txt", "output/r3t.html");

//    /* test r3 trees */
	result = testr3t_files(log, testdata_small, TESTDATA_SMALL_CNT);
	if(result)
		fprintf(log, "testr3t_files() returned %i\n", result);

//	result = testr3t_files(log, testdata_text, TESTDATA_TEXT_CNT);
//	if(result)
//		fprintf(log, "testr3t_files() returned %i\n", result);

//	result = testr3t_files(log, testdata_bin, TESTDATA_BIN_CNT);
//	if(result)
//		fprintf(log, "testr3t_files() returned %i\n", result);
		
	/* run example function */
//	example();
	
//	fclose(log);
	return result;
}

/* main routine for running statistic computations
 * */
int statistics()
{
	int result;

//	result =  statistics_from_files(stdout, testdata_small, TESTDATA_SMALL_CNT, "output/stat_small.csv");
//	if(result) {
//		fprintf(stderr, "statistics_from_files returned %i\n", result);
//		return result;
//	}

//	result =  statistics_from_files(stdout, testdata_text, TESTDATA_TEXT_CNT, "output/stat_text.csv");
//	if(result) {
//		fprintf(stderr, "statistics_from_files returned %i\n", result);
//		return result;
//	}
//
//	result =  statistics_from_files(stdout, testdata_bin, TESTDATA_BIN_CNT, "output/stat_bin.csv");
//	if(result) {
//		fprintf(stderr, "statistics_from_files returned %i\n", result);
//		return result;
//	}

//	result = statistics_complete_from_file("testdata/bin/zlib123.zip", "output/stat_complete.txt", 8);
//	if(result) {
//		fprintf(stderr, "statistics_complete_from_file returned %i\n", result);
//		return result;
//	}
	
	return 0;
}

/* load data from file to memory, last byte is unocupied
 * 
 * */
int load_data(BYTE** text, UINT* textLen, char* filename)
{
	FILE* f;
	int result;
	if(!filename) 
		return ERROR_BADARG;
	f = fopen(filename, "rb");
	if(!f)  
		return ERROR_FILEOPEN;
	if(fseek(f, 0, SEEK_END))
	{
		fclose(f);
		return ERROR_FILEREAD;
	}
	result = (int) ftell(f);
	if(result == -1)
	{
		fclose(f);
		return ERROR_FILEREAD;
	}
	*textLen = (UINT) result;
	if(fseek(f, 0, SEEK_SET))
	{
		fclose(f);
		return ERROR_FILEREAD;
	}
    *text = (BYTE*) malloc((*textLen)+1);
    if(!(*text))
    {
    	fclose(f);
    	return ERROR_MEM;
    }
    result = fread(*text, 1, *textLen, f);
    if(result != *textLen) 
    {
    	fclose(f);
    	return ERROR_INTEGRITY;
    }
    (*text)[*textLen]=0; // for easier display as string
    fclose(f);
    return 0;
}

/* load repeats from .repeats file
 * */
int load_repeats(t_repeat** repeats, int* cnt, char* filename)
{
	FILE* f;
	int i, allocated, result;
	t_repeat* rep;
	
	if(!filename) 
		return ERROR_BADARG;
		
	f = fopen(filename, "r");
	if(!f)  
		return ERROR_FILEOPEN;
		
	allocated = REP_LOAD;
	rep = (t_repeat*) malloc(REP_LOAD*sizeof(t_repeat));
	if(!rep) 
		return ERROR_MEM;
		
	i=0;
	while(!feof(f)) {
		result = fscanf(f, "%i %i %i", &(rep[i].p1), &(rep[i].p2), &(rep[i].l) );
		if(result != 3) break;
		i++;
		if(i == allocated) {
			allocated += REP_LOAD;
			rep = (t_repeat*) realloc(rep, allocated*sizeof(t_repeat));
			if(!rep)
				return ERROR_MEM;
		}
	}
	rep = (t_repeat*) realloc(rep, i*sizeof(t_repeat));
	if(!rep)
		return ERROR_MEM;
	*cnt = i;
	*repeats = rep;
    fclose(f);
    return 0;
}

int test_bitstr(FILE* log)
{
	int n = 4;
	BYTE *bs_mem = (BYTE*) malloc(1);
	t_bitstr bs;
	int i;
	
	bs_init(&bs, bs_mem, n);
	
	bs_setAll(&bs, 0);
	bs_setVal(&bs, 0, 1);
	bs_setVal(&bs, 0, 1);
	bs_setVal(&bs, 3, 1);
	bs_setVal(&bs, 0, 1);
	bs_setVal(&bs, 3, 1);
	bs_setVal(&bs, 0, 1);
	bs_setVal(&bs, 3, 1);
	
	fprintf(log, "values of bitstr: \n");
	for(i=0;i<n;i++)
		fprintf(log, "%4i: %4i\n", i, bs_getVal(&bs, i));
		
	return 0;
}

void test_realloc(FILE* log)
{
	int big = 200;
	int small = 100;
	fprintf(log, "allocating %i bytes...\n", big);
	BYTE *mem = (BYTE*) malloc(big);
	fprintf(log, "writing %i bytes...\n", big);
	test_writability(mem, big);

	fprintf(log, "reallocating %i bytes to %i bytes...\n", big, small);
	mem = realloc(mem, 100);
	
	fprintf(log, "writing %i bytes...\n", small);
	test_writability(mem, 100);
	free(mem);

	fprintf(log, "done.\n");
}

