/*  test_sa.c
 * 
 *  R3lib
 * 
 *  Testing routines for suffix array 
 *
 *  Copyright (C) 2006-2007  Michal Linhard <michal@linhard.sk>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  as published by the Free Software Foundation; either version 2.1
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *  
 */
#include "../conf.h" 
#include "../suffixArray.h"
#include "test_sa.h"
#include "test.h"
 
/* test_sa_files 
 * test createSuffixArray on files specified by filenames
 * output results to log file
 * */
int test_sa_files(FILE* log, char** filenames, int cnt)
{
	int i, result_or, result;
	if(!log || !filenames || (cnt < 0))
		return ERROR_INTEGRITY;
		
	result_or = 0;
	for(i=0; i<cnt; i++)
	{
		if(!(filenames[i]))
			return ERROR_INTEGRITY;
		result = test_sa_file(log, filenames[i]);
		fflush(log);
		if(result)
			result_or = ERROR_TEST;
	}
	
	return result_or;
}

/* test_lcp_files 
 * test createLCPTable on files specified by filenames
 * output results to log file
 * */
int test_lcp_files(FILE* log, char** filenames, int cnt)
{
	int i, result_or, result;
	if(!log || !filenames || (cnt < 0))
		return ERROR_INTEGRITY;
		
	result_or = 0;
	for(i=0; i<cnt; i++)
	{
		if(!(filenames[i]))
			return ERROR_INTEGRITY;
		result = test_lcp_file(log, filenames[i]);
		fflush(log);
		if(result)
			result_or = ERROR_TEST;
	}
	
	return result_or;
}
 
/* test_sa_file 
 * test createSuffixArray on a file
 * output results to log file
 * */
int test_sa_file(FILE* log, char* filename)
{
	BYTE* text;
	int textLen;
	int* suffixArray;
	int result;
#ifdef DEBUG1
	char* newfilename;
#endif	
	if(!log)
		return ERROR_INTEGRITY;

	fprintf(log, "testing input: '%s' ... ", filename);
	result = load_data(&text, &textLen, filename);
	if(result) {
		fprintf(log, "\nerror loading file.\n");
		return result; }
	else
		fprintf(log, "size: %u\n", textLen);

    result = createSuffixArray(text, textLen, &suffixArray); 
    if(result) { 	
		fprintf(log, "createSuffixArray returned error: %i\n", result);
    	return result; }
	else
		result = testSuffixArray(text, textLen, suffixArray, log);

	if(result) {
		fprintf(log, "testSuffixArray returned error: %i\n", result);

#ifdef DEBUG1
		newfilename = (char *) malloc(strlen(filename)+4);
		strcpy(newfilename, filename);
		strcat(newfilename, ".sa");
		fprintf(log, "outputting array to: '%s'\n", newfilename);
		output_sa_to_file(text, textLen, suffixArray, newfilename);
#endif
		return result; }	
	else 
		fprintf(log, "O.K.\n");	
	
	free(text);
	free(suffixArray);
	return 0;
}
 
 
/* test_lcp_file 
 * test createLCPTable on a file
 * output results to log file
 * */
int test_lcp_file(FILE* log, char* filename)
{
	BYTE* text;
	int textLen;
	int* suffixArray;
	int* lcp;
	int result;
	if(!log)
		return ERROR_INTEGRITY;

	fprintf(log, "testing input: '%s' ... ", filename);
	result = load_data(&text, &textLen, filename);
	if(result) {
		fprintf(log, "\nerror loading file.\n");
		return result; }
	else
		fprintf(log, "size: %u\n", textLen);

    result = createSuffixArray(text, textLen, &suffixArray); 
    if(result) { 	
		fprintf(log, "createSuffixArray returned error: %i\n", result);
    	return result; }
	else
		result = testSuffixArray(text, textLen, suffixArray, log);

	if(result) {
		fprintf(log, "testSuffixArray returned error: %i\n", result);
		return result; }	
	else 
		result = createLCPTable(text, textLen, suffixArray, &lcp);
		
	if(result) {
		fprintf(log, "createLCPTable returned error: %i\n", result);
		return result; }	
	else 
		result = testLCPTable(text, textLen, suffixArray, lcp, log);

	if(result) {
		fprintf(log, "testLCPTable returned error: %i\n", result);
		return result; }	
	else 
		fprintf(log, "O.K.\n");	
	
	free(text);
	free(lcp);
	free(suffixArray);
	return 0;
}
 
/* testLCPTable
 * 
 * test given lcp table
 * 
 * */
int testLCPTable(BYTE* text, UINT textLen, UINT* suffixArray, UINT* lcp, FILE *log)
{
	int result;
	int i,j,l;
	UINT p1, p2;
	
	// argument sanity check
	if( (!log) || // pointers should not be NULL
	    (!text) ||
	    (!suffixArray) ||
	    (!lcp) ||
	    (textLen == 0) || // texLen >= 1
	    (textLen > FLAGUINT_MAX) ) // do not try to process too large input
		return ERROR_BADARG;

	result = 0;
	for(i=0;i<textLen-1;i++)
	{
		p1 = suffixArray[i];
		p2 = suffixArray[i+1];
		if(lcp[i+1] >= textLen) {
			fprintf(log, "testLCPTable: lcp[%4i] = %4i (value greater than textLen[%4i])\n",i+1, lcp[i+1], textLen);
			result = ERROR_INTEGRITY;
		}
		l = (int) lcp[i+1];
		for(j=0; j < l; j++)
		{
			if(text[p1+j] != text[p2+j])
			{
				fprintf(log, "testLCPTable: text[%4i+%4i]!=text[%4i+%4i] (sa[%4i]=%4i, sa[%4i]=%4i) (prefix length < lcp)\n", p1, j, p2, j, i, p1, i+1, p2);
				result = ERROR_INTEGRITY;
			}
		}
		if((p1+l < textLen)&&(p2+l < textLen)&&(text[p1+l] == text[p2+l]))
		{
			fprintf(log, "testLCPTable: text[%4i+%4i]==text[%4i+%4i] (sa[%4i]=%4i, sa[%4i]=%4i) (prefix length > lcp)\n", p1, l, p2, l, i, p1, i+1, p2);
			result = ERROR_INTEGRITY;
		}
	}

	return result;
}



/* testConvertLCP
 * function that tests correctness of 
 * analyzeLCP and convertLCP functions
 *
 * returns -1 on errror and 0 on success 
 * */

//int testConvertLCP(FILE* log)
//{
//	int i, temp;
//	int lcpOrig[TESTN];
//	int* lcp = (int*)malloc(TESTN*4);
//	int minLength = 5;
//	int maxStack, maxChild, lcpOver1, lcpOver2, maxRoot, procCnt;
//	int code;
//	void* newLcp;
//	void* lcpEx;
//	int nextEx;
//	int result;
//
//	printf("testing lcp compression (n=%i, minLength=%i)\n",TESTN, minLength);
//	fprintf(log, "testing lcp compression (n=%i, minLength=%i)\n",TESTN, minLength);
//
//	
////---------- LCP_CODE10 ---------------------------------------------------------------------------------
//	printf("testing LCP_CODE10 scheme\n");
//	fprintf(log,"testing LCP_CODE10 scheme\n");
//	// everything has to be under or equal minLength+254
//	for(i=1;i<TESTN;i++)
//		lcpOrig[i] = minLength;
//	lcpOrig[0] = 0;
//
//	lcpOrig[10] = minLength+254; // let's have some boundary cases
//	lcpOrig[20] = 0;
//
//	for(i=0;i<TESTN;i++) //copy original lcp to working lcp array
//		lcp[i] = lcpOrig[i];
//	
//	analyzeLCP(lcp, TESTN, minLength, &maxStack, &maxChild, &lcpOver1, &lcpOver2, &maxRoot, &procCnt);
//	printf("lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	fprintf(log, "lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	result = convertLCP(lcp, TESTN, minLength, lcpOver1, lcpOver2, &code, &newLcp, &lcpEx);
//	lcp = NULL; // we shouldn't use lcp after call to convertLCP
//	if(result)
//	{
//		printf("ERROR: convertLCP returned error! result = %i\n", result);
//		fprintf(log,"ERROR: convertLCP returned error! result = %i\n", result);
//		return result;
//	}
//	if(code != LCP_CODE10)
//	{
//		printf("ERROR: convertLCP returned bad coding! code = %i\n", code);
//		fprintf(log,"ERROR: convertLCP returned bad coding! code = %i\n", code);
//		return -1;
//	}
//	
//	nextEx = 0;
//	for(i=1;i<TESTN;i++) //compare original lcp to working lcp array
//		if(get_lcp10(newLcp, lcpEx, &nextEx, minLength, i) != lcpOrig[i])
//		{
//			printf("ERROR: convertLCP modified data!\n");
//			fprintf(log,"ERROR: convertLCP modified data!\n");
//			return -1;
//		}
//	
////---------- LCP_CODE12 ---------------------------------------------------------------------------------
//	printf("testing LCP_CODE12 scheme\n");
//	fprintf(log,"testing LCP_CODE12 scheme\n");
//	// realloc arrays
//	if(newLcp)
//		free(newLcp);		
//	if(lcpEx)
//		free(lcpEx);	
//		
//	lcp = (int*)malloc(TESTN*sizeof(int));	
//
//	for(i=1;i<TESTN;i++)
//		lcpOrig[i] = minLength;
//	lcpOrig[0] = 0;
//
//	lcpOrig[10] = minLength+254; // let's have some boundary cases
//	lcpOrig[20] = 0;
//	lcpOrig[40] = minLength+255;
//	lcpOrig[42] = minLength+256;
//	lcpOrig[43] = minLength+0xfffe;
//
//	for(i=0;i<TESTN;i++) //copy original lcp to working lcp array
//	{
//		lcp[i] = lcpOrig[i];
////		printf("%i, ", lcp[i]);
//	}
////	printf("\n");
//
//	analyzeLCP(lcp, TESTN, minLength, &maxStack, &maxChild, &lcpOver1, &lcpOver2, &maxRoot, &procCnt);
//	printf("lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	fprintf(log, "lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	result = convertLCP(lcp, TESTN, minLength, lcpOver1, lcpOver2, &code, &newLcp, &lcpEx);
//	lcp = NULL; // we shouldn't use lcp after call to convertLCP
//	if(result)
//	{
//		printf("ERROR: convertLCP returned error! result = %i\n", result);
//		fprintf(log,"ERROR: convertLCP returned error! result = %i\n", result);
//		return result;
//	}
//	if(code != LCP_CODE12)
//	{
//		printf("ERROR: convertLCP returned bad coding! code = %i\n", code);
//		fprintf(log,"ERROR: convertLCP returned bad coding! code = %i\n", code);
//		return -1;
//	}
//
//	nextEx = 0;
//	for(i=0;i<TESTN;i++) //compare original lcp to working lcp array
//	{
//		temp = get_lcp12(newLcp, lcpEx, &nextEx, minLength, i);
//		if(temp != lcpOrig[i])
//		{
//			printf("ERROR: convertLCP modified data! (i=%i,orig=%i,get=%i)\n", i, lcpOrig[i], temp);
//			fprintf(log,"ERROR: convertLCP modified data!\n");
//			return -1;
//		}
//	}
//
////---------- LCP_CODE14 ---------------------------------------------------------------------------------
//	printf("testing LCP_CODE14 scheme\n");
//	fprintf(log,"testing LCP_CODE14 scheme\n");
//	// realloc arrays
//	if(newLcp)
//		free(newLcp);		
//	if(lcpEx)
//		free(lcpEx);	
//		
//	lcp = (int*)malloc(TESTN*sizeof(int));	
//
//	for(i=1;i<TESTN;i++)
//		lcpOrig[i] = minLength;
//	lcpOrig[0] = 0;
//
//	lcpOrig[10] = minLength+254; // let's have some boundary cases
//	lcpOrig[20] = 0;
//	lcpOrig[40] = minLength+255;
//	lcpOrig[42] = minLength+256;
//	lcpOrig[43] = minLength+0xfffe;
//
//	lcpOrig[50] = minLength+0xffff;
//	lcpOrig[51] = minLength+0xffff+1;
//
//	for(i=0;i<TESTN;i++) //copy original lcp to working lcp array
//		lcp[i] = lcpOrig[i];
//
//	analyzeLCP(lcp, TESTN, minLength, &maxStack, &maxChild, &lcpOver1, &lcpOver2, &maxRoot, &procCnt);
//	printf("lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	fprintf(log, "lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	result = convertLCP(lcp, TESTN, minLength, lcpOver1, lcpOver2, &code, &newLcp, &lcpEx);
//	lcp = NULL; // we shouldn't use lcp after call to convertLCP
//	if(result)
//	{
//		printf("ERROR: convertLCP returned error! result = %i\n", result);
//		fprintf(log,"ERROR: convertLCP returned error! result = %i\n", result);
//		return result;
//	}
//	if(code != LCP_CODE14)
//	{
//		printf("ERROR: convertLCP returned bad coding! code = %i\n", code);
//		fprintf(log,"ERROR: convertLCP returned bad coding! code = %i\n", code);
//		return -1;
//	}
//
//	nextEx = 0;
//	for(i=0;i<TESTN;i++) //compare original lcp to working lcp array
//		if(get_lcp14(newLcp, lcpEx, &nextEx, minLength, i) != lcpOrig[i])
//		{
//			printf("ERROR: convertLCP modified data!\n");
//			fprintf(log,"ERROR: convertLCP modified data!\n");
//			return -1;
//		}
////---------- LCP_CODE20 ---------------------------------------------------------------------------------
//	printf("testing LCP_CODE20 scheme\n");
//	fprintf(log,"testing LCP_CODE20 scheme\n");
//	// realloc arrays
//	if(newLcp)
//		free(newLcp);		
//	if(lcpEx)
//		free(lcpEx);	
//		
//	lcp = (int*)malloc(TESTN*sizeof(int));	
//
//	for(i=1;i<TESTN;i++)
//		lcpOrig[i] = minLength+255;
//	
//	lcpOrig[0] = 0;
//	lcpOrig[10] = minLength+254; // let's have some boundary cases
//	lcpOrig[20] = 0;
//	lcpOrig[40] = minLength+255;
//	lcpOrig[42] = minLength+256;
//	lcpOrig[43] = minLength+0xfffe;
//
//	for(i=0;i<TESTN;i++) //copy original lcp to working lcp array
//		lcp[i] = lcpOrig[i];
//
//	analyzeLCP(lcp, TESTN, minLength, &maxStack, &maxChild, &lcpOver1, &lcpOver2, &maxRoot, &procCnt);
//	printf("lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	fprintf(log, "lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	result = convertLCP(lcp, TESTN, minLength, lcpOver1, lcpOver2, &code, &newLcp, &lcpEx);
//	lcp = NULL; // we shouldn't use lcp after call to convertLCP
//	if(result)
//	{
//		printf("ERROR: convertLCP returned error! result = %i\n", result);
//		fprintf(log,"ERROR: convertLCP returned error! result = %i\n", result);
//		return result;
//	}
//	if(code != LCP_CODE20)
//	{
//		printf("ERROR: convertLCP returned bad coding! code = %i\n", code);
//		fprintf(log,"ERROR: convertLCP returned bad coding! code = %i\n", code);
//		return -1;
//	}
//
//	nextEx = 0;
//	for(i=1;i<TESTN;i++) //compare original lcp to working lcp array
//		if(get_lcp20(newLcp, lcpEx, &nextEx, minLength, i) != lcpOrig[i])
//		{
//			printf("ERROR: convertLCP modified data!\n");
//			fprintf(log,"ERROR: convertLCP modified data!\n");
//			return -1;
//		}
//		
////---------- LCP_CODE24 ---------------------------------------------------------------------------------
//	printf("testing LCP_CODE24 scheme\n");
//	fprintf(log,"testing LCP_CODE24 scheme\n");
//	// realloc arrays
//	if(newLcp)
//		free(newLcp);		
//	if(lcpEx)
//		free(lcpEx);	
//		
//	lcp = (int*)malloc(TESTN*sizeof(int));	
//
//	for(i=1;i<TESTN;i++)
//		lcpOrig[i] = minLength+255;
//	lcpOrig[0] = 0;
//
//	lcpOrig[10] = minLength+254; // let's have some boundary cases
//	lcpOrig[20] = 0;
//	lcpOrig[40] = minLength+255;
//	lcpOrig[42] = minLength+256;
//	lcpOrig[43] = minLength+0xfffe;
//
//	lcpOrig[50] = minLength+0xffff;
//	lcpOrig[51] = minLength+0xffff+1;
//
//	for(i=0;i<TESTN;i++) //copy original lcp to working lcp array
//		lcp[i] = lcpOrig[i];
//
//	analyzeLCP(lcp, TESTN, minLength, &maxStack, &maxChild, &lcpOver1, &lcpOver2, &maxRoot, &procCnt);
//	printf("lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	fprintf(log, "lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	result = convertLCP(lcp, TESTN, minLength, lcpOver1, lcpOver2, &code, &newLcp, &lcpEx);
//	lcp = NULL; // we shouldn't use lcp after call to convertLCP
//	if(result)
//	{
//		printf("ERROR: convertLCP returned error! result = %i\n", result);
//		fprintf(log,"ERROR: convertLCP returned error! result = %i\n", result);
//		return result;
//	}
//	if(code != LCP_CODE24)
//	{
//		printf("ERROR: convertLCP returned bad coding! code = %i\n", code);
//		fprintf(log,"ERROR: convertLCP returned bad coding! code = %i\n", code);
//		return -1;
//	}
//
//	nextEx = 0;
//	for(i=1;i<TESTN;i++) //compare original lcp to working lcp array
//		if(get_lcp24(newLcp, lcpEx, &nextEx, minLength, i) != lcpOrig[i])
//		{
//			printf("ERROR: convertLCP modified data!\n");
//			fprintf(log,"ERROR: convertLCP modified data!\n");
//			return -1;
//		}
//		
////---------- LCP_CODE40 ---------------------------------------------------------------------------------
//	printf("testing LCP_CODE40 scheme\n");
//	fprintf(log,"testing LCP_CODE40 scheme\n");
//	// realloc arrays
//	if(newLcp)
//		free(newLcp);		
//	if(lcpEx)
//		free(lcpEx);	
//		
//	lcp = (int*)malloc(TESTN*sizeof(int));	
//
//	for(i=1;i<TESTN;i++)
//		lcpOrig[i] = minLength+0xffff;
//	lcpOrig[0] = 0;
//	
//	lcpOrig[10] = minLength+254; // let's have some boundary cases
//	lcpOrig[20] = 0;
//	lcpOrig[40] = minLength+255;
//	lcpOrig[42] = minLength+256;
//	lcpOrig[43] = minLength+0xfffe;
//
//	lcpOrig[50] = minLength+0xffff+1; 
//	lcpOrig[51] = 0x7FFFffff; // what's upper boundary for lcp?
//
//	for(i=0;i<TESTN;i++) //copy original lcp to working lcp array
//		lcp[i] = lcpOrig[i];
//
//	analyzeLCP(lcp, TESTN, minLength, &maxStack, &maxChild, &lcpOver1, &lcpOver2, &maxRoot, &procCnt);
//	printf("lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	fprintf(log, "lcpOver1 = %i, lcpOver2 = %i\n", lcpOver1, lcpOver2);
//	result = convertLCP(lcp, TESTN, minLength, lcpOver1, lcpOver2, &code, &newLcp, &lcpEx);
//	lcp = NULL; // we shouldn't use lcp after call to convertLCP
//	if(result)
//	{
//		printf("ERROR: convertLCP returned error! result = %i\n", result);
//		fprintf(log,"ERROR: convertLCP returned error! result = %i\n", result);
//		return result;
//	}
//	if(code != LCP_CODE40)
//	{
//		printf("ERROR: convertLCP returned bad coding! code = %i\n", code);
//		fprintf(log,"ERROR: convertLCP returned bad coding! code = %i\n", code);
//		return -1;
//	}
//
//	nextEx = 0;
//	for(i=1;i<TESTN;i++) //compare original lcp to working lcp array
//		if(get_lcp40(newLcp, lcpEx, &nextEx, minLength, i) != lcpOrig[i])
//		{
//			printf("ERROR: convertLCP modified data!\n");
//			fprintf(log,"ERROR: convertLCP modified data!\n");
//			return -1;
//		}		
//	return 0;
//}
