import javax.swing.*;
import java.util.ArrayList;
import java.util.Arrays;

/*trieda pomocou ktorej detegujeme parne kruznice v grafe*/
public class CycleDetector {
    ArrayList<ArrayList<Integer>> graph;
    ArrayList<Boolean> inTree;
    ArrayList<Integer> parent;
    ArrayList<ArrayList<Integer>> baseCycles;
    ArrayList<ArrayList<Integer>> cycles;
    ArrayList<Integer> dissolvableCycle = new ArrayList<>();
    int N;
    boolean computingOnlyDetection = false;
    boolean foundInBaseCycles = false;

    CycleDetector(ArrayList<ArrayList<Integer>> graph) {
        this.graph = new ArrayList<>();
        this.parent = new ArrayList<>();
        this.inTree = new ArrayList<>();
        N = graph.size();
        for (int i = 0; i < N; i++) {
            this.graph.add(new ArrayList<>());
            this.graph.get(i).addAll(graph.get(i));
            this.parent.add(-1);
            this.inTree.add(false);
        }
        cycles = new ArrayList<>();
        baseCycles = new ArrayList<>();
    }

    /*metoda vytvori zoznam parnych kruznic v grafe:
    * 1)vypocita zakladne kruznice
    * 2)nasledne urobi z kazdej n-tice zakladnych kruznic maticu, ktora je xorom vsetkych zakladnych kruznic  z danej
    * n-tice (xor znamena, ze do vysledku zoberiem hranu len vtedy, ak je sucastou prave jednej z dvoch kruznic, na
    * ktorych je aplikovana operacia)
    * 3)skontroluje, ci xor matica tvori jednu suvislu kruznicu a ci je tato kruznica parna*/
    void computeCycles() {
        if (baseCycles.isEmpty()) {
            computeBaseCycles();
        }
        int numBaseCycles = baseCycles.size();

        //bitstring na prejdenie cez vsetky tice zakladnych kruznic
        int bitstring[] = new int[numBaseCycles];

        for (int i = 2; i <= numBaseCycles; i++) {
            Arrays.fill(bitstring, 0, i, 1);
            Arrays.fill(bitstring, i, numBaseCycles, 0);

            // iteruj cez vsetky moznosti, ako mozes vybrat i prvkov zo zakladnych kruznic
            do {

                // vytvor xor maticu na zaklade aktualneho bitstringu
                ArrayList<ArrayList<Integer>> cycleMatrix = getAllZeroMatrix();
                int nEdges = 0;
                for (int j = 0; j < numBaseCycles; j++) {
                    if (bitstring[j] == 1) {
                        cycleMatrix = xorMatrices(cycleMatrix, makeAdjMatrix(baseCycles.get(j)));
                        nEdges = nEdges + baseCycles.get(j).size();
                    }
                }

                // ak beries len dvojicu kruznic, pozri, ci maju nejaku spolocnu hranu - ak ano, pridaj
                if (i == 2) {
                    if (nEdges > getNumEdges(cycleMatrix)) {
                        ArrayList<Integer> newCycle = fromMatrixToCycle(cycleMatrix);
                        if (newCycle.size() % 2 == 0) {
                            cycles.add(newCycle);
                        }
                    }
                } else {
                    //ak skombinujes viac kruznic, over validnost matice prehladavanim do hlbky
                    if (validateCycleMatrix(cycleMatrix)) {
                        ArrayList<Integer> newCycle = fromMatrixToCycle(cycleMatrix);
                        if (newCycle.size() % 2 == 0) {
                            cycles.add(newCycle);
                        }
                    }
                }
                //prejde na predoslu permutaciu bistringu (permutacie idu lexikograficky opacne - 11000...)
            } while (prev_permutation(bitstring));
        }
        for (ArrayList<Integer> cyc : cycles) {
            //System.out.println(cyc);
        }
    }

    /*metoda postupne tvori cykly, tak isto ako metoda computeCycsle,
    * zaroven vsak hned po vytvoreni noveho cyklu overuje, ci je cyklus rozpustitelny.
    * Ak je vrati True a do premennej dissolvableCycle cyklus ulozi.*/
    boolean computeCyclesAndTryToDissolve() {
        computingOnlyDetection = true;
        if (baseCycles.isEmpty()) {
            computeBaseCycles();
        }
        if(foundInBaseCycles) return true;
        int numBaseCycles = baseCycles.size();

        //bitstring na prejdenie cez vsetky tice zakladnych kruznic
        int bitstring[] = new int[numBaseCycles];

        for (int i = 2; i <= numBaseCycles; i++) {
            Arrays.fill(bitstring, 0, i, 1);
            Arrays.fill(bitstring, i, numBaseCycles, 0);

            // iteruj cez vsetky moznosti, ako mozes vybrat i prvkov zo zakladnych kruznic
            do {

                // vytvor xor maticu na zaklade aktualneho bitstringu
                ArrayList<ArrayList<Integer>> cycleMatrix = getAllZeroMatrix();
                int nEdges = 0;
                for (int j = 0; j < numBaseCycles; j++) {
                    if (bitstring[j] == 1) {
                        cycleMatrix = xorMatrices(cycleMatrix, makeAdjMatrix(baseCycles.get(j)));
                        nEdges = nEdges + baseCycles.get(j).size();
                    }
                }

                // ak beries len dvojicu kruznic, pozri, ci maju nejaku spolocnu hranu - ak ano, pridaj
                if (i == 2) {
                    if (nEdges > getNumEdges(cycleMatrix)) {
                        ArrayList<Integer> newCycle = fromMatrixToCycle(cycleMatrix);
                        if (isCycleDissolvable(newCycle)) {
                            return true;
                        }
                    }
                } else {
                    //ak skombinujes viac kruznic, over validnost matice prehladavanim do hlbky
                    if (validateCycleMatrix(cycleMatrix)) {
                        ArrayList<Integer> newCycle = fromMatrixToCycle(cycleMatrix);
                        if (isCycleDissolvable(newCycle)) {
                            return true;
                        }
                    }
                }
                //prejde na predoslu permutaciu bistringu (permutacie idu lexikograficky opacne - 11000...)
            } while (prev_permutation(bitstring));
        }
        for (ArrayList<Integer> cyc : cycles) {
            //System.out.println(cyc);
        }
        return false;
    }

    /*vytvori xor dvoch matic incidencie*/
    ArrayList<ArrayList<Integer>> xorMatrices(ArrayList<ArrayList<Integer>> m1, ArrayList<ArrayList<Integer>> m2) {
        ArrayList<ArrayList<Integer>> xorMatrix = getAllZeroMatrix();
        for (int i = 0; i < N; i++) {
            for (int j = 0; j < N; j++) {
                if (m1.get(i).get(j) != m2.get(i).get(j)) {
                    xorMatrix.get(i).set(j, 1);
                }
            }
        }
        return xorMatrix;
    }

    /*vytvori zoznam zakladnych kruznic:
    * 1) pomocou prehladavania do hlbky tvori spanning tree
    * 2) ak narazi na vrchol, ktory uz je v strome, ulozi si cestu zo seba a z daneho vrcholu ako zakladnu kruznicu */
    void computeBaseCycles() {
        ArrayList<Integer> queue = new ArrayList<>();
        //kedze matica je symetricka, potrebuje si kontrolovat, ktore hrany uz pouzil ako dokoncovacie v cykle
        ArrayList<ArrayList<Boolean>> usedForCycle = new ArrayList<>();
        for (int i = 0; i < N; i++) {
            ArrayList<Boolean> allFalse = new ArrayList<>();
            for (int j = 0; j < N; j++) {
                allFalse.add(false);
            }
            usedForCycle.add(allFalse);
        }
        //zaciatok prehladavania do hlbky
        queue.add(0);
        inTree.set(0, true);

        while (!queue.isEmpty()) {
            int node = queue.get(0);
            queue.remove(0);
            for (int i = 0; i < N; i++) {
                if (graph.get(node).get(i) == 1) {
                    if (inTree.get(i)) {
                        if (parent.get(node) != i && !usedForCycle.get(i).get(node)) {
                            addBaseCycle(i, node);
                            if(computingOnlyDetection && isCycleDissolvable(baseCycles.get(baseCycles.size()-1))) {
                                foundInBaseCycles = true;
                                return;
                            }
                            usedForCycle.get(i).set(node, true);
                            usedForCycle.get(node).set(i, true);
                        }
                    } else {
                        inTree.set(i, true);
                        parent.set(i, node);
                        queue.add(i);
                    }
                }
            }
        }

    }

    ArrayList<ArrayList<Integer>> getAllZeroMatrix() {
        ArrayList<ArrayList<Integer>> adjMatrix = new ArrayList<>();
        for (int i = 0; i < N; i++) {
            ArrayList<Integer> allZeros = new ArrayList<>();
            for (int j = 0; j < N; j++) {
                allZeros.add(0);
            }
            adjMatrix.add(allZeros);
        }
        return adjMatrix;
    }

    /*vytvori maticu susednosti z cyklu ulozenehom ako zoznam vrcholov*/
    ArrayList<ArrayList<Integer>> makeAdjMatrix(ArrayList<Integer> cycle) {
        ArrayList<ArrayList<Integer>> adjMatrix = getAllZeroMatrix();
        int cSize = cycle.size();
        for (int i = 0; i < cSize - 1; i++) {
            adjMatrix.get(cycle.get(i)).set(cycle.get(i + 1), 1);
            adjMatrix.get(cycle.get(i + 1)).set(cycle.get(i), 1);
        }
        adjMatrix.get(cycle.get(0)).set(cycle.get(cycle.size() - 1), 1);
        adjMatrix.get(cycle.get(cycle.size() - 1)).set(cycle.get(0), 1);
        return adjMatrix;
    }

    /*vrati cestu do korena spanning tree*/
    ArrayList<Integer> path(int node) {
        ArrayList<Integer> path = new ArrayList<>();
        path.add(node);
        while (parent.get(node) != -1) {
            node = parent.get(node);
            path.add(node);
        }
        return path;
    }

    /*z dvoch vrcholov v strome najde cesty do korena a vytvori zakladny cyklus*/
    void addBaseCycle(int i, int j) {
        ArrayList<Integer> iPath = path(i);
        ArrayList<Integer> jPath = path(j);
        int last = -1;
        while (iPath.get(iPath.size() - 1) == jPath.get(jPath.size() - 1)) {
            last = iPath.remove(iPath.size() - 1);
            jPath.remove(jPath.size() - 1);
        }
        int iSize = iPath.size();
        ArrayList<Integer> cycle = new ArrayList<>();
        for (int k = 0; k < iSize; k++) {
            cycle.add(iPath.get(k));
        }
        cycle.add(last);
        while (!jPath.isEmpty()) {
            cycle.add(jPath.remove(jPath.size() - 1));
        }
        baseCycles.add(cycle);
        //ak je parny, pridaj do vyslednych cyklov a ak iba hladas ci existuje rozpustna kruznica, tak rovno
        //skontroluj rospustitelnost
        if(cycle.size() % 2 == 0) {
            cycles.add(cycle);
        }
    }

    /*kontroluje, ci matica susednosti indikuje jednu suvislu kruznicu*/
    boolean validateCycleMatrix(ArrayList<ArrayList<Integer>> cycle) {
        int pathLength = 0;
        for (int i = 0; i < N; ++i) {
            for (int j = 0; j < N; ++j) {
                //najdi prvu jednosku v matici
                if (cycle.get(i).get(j) == 1) {
                    ++pathLength;
                    ArrayList<Integer> aVisited = new ArrayList<>();
                    aVisited.add(i);
                    //zacni rekurziu v susednom vrchole
                    pathLength = validateCycleMatrix_recursion(cycle, pathLength, j, i, aVisited);
                    //skontroluj, ci cesta, ktoru si nasiel je taka ista dlha, ako je pocet hran v matici
                    //ak je, mame suvislu kruznicu, ak nie je, matica obsahuje viac nesuvislych kruznic
                    return pathLength + 1 == getNumEdges(cycle);
                }
            }
        }
        System.out.println("Given Cycle Matrix does not contain any edges!");
        return false;
    }

    /*rekurzia pri validacii kruznice*/
    int validateCycleMatrix_recursion(ArrayList<ArrayList<Integer>> cycle, int pathLength,
                                       int i, int previousNode, ArrayList<Integer> aVisited) {
        if (pathLength > 500)
            System.out.println("Graph::validateCycleMatrix_recursion(): Maximum recursion level reached.");

        //v matici najdi dalsiu hranu, ktora nejde do predchodcu a pokracuj v prehladavani
        for (int j = 0; j < N; ++j) {
            if (cycle.get(i).get(j) == 1 && j != previousNode) {
                if (aVisited.contains(j)) return pathLength; //nasiel som nejaku kruznicu
                ++pathLength;
                aVisited.add(j);
                pathLength = validateCycleMatrix_recursion(cycle, pathLength, j, i, aVisited);
                return pathLength;
            }
        }
        System.out.println("Graph::validateCycleMatrix_recursion(): Found a dead end!");
        return 0;
    }

    /*vrati pocet hran v kruznici*/
    int getNumEdges(ArrayList<ArrayList<Integer>> cycle) {
        int numEdges = 0;
        for (int i = 0; i < N; ++i) {
            for (int j = 0; j < N; ++j) {
                if (cycle.get(i).get(j) == 1) numEdges++;
            }
        }
        return numEdges / 2;
    }

    /*prekonvertuje matitu cyklu na zoznam vrcholov*/
    ArrayList<Integer> fromMatrixToCycle(ArrayList<ArrayList<Integer>> matrix) {
        int nEdges = getNumEdges(matrix);
        boolean found = false;
        int previous = -1;
        int current = -1;
        int i = 0;
        //najde prvu jednotku v matici
        while (!found && i < N) {
            int j = 0;
            while (j < N) {
                if (matrix.get(i).get(j) == 1) {
                    found = true;
                    previous = i;
                    current = j;
                    break;
                }
                ++j;
            }
            i++;
        }
        ArrayList<Integer> cycle = new ArrayList<>();
        cycle.add(previous);
        cycle.add(current);

        //v cykle prechadza maticou a pridava vrcholy do kruznice, kym sa nevrati na zaciatok
        while (current != cycle.get(0)) {
            for (int k = 0; k < N; k++) {
                if (matrix.get(current).get(k) == 1 && k != previous) {
                    cycle.add(k);
                    previous = current;
                    current = k;
                    break;
                }
            }
        }
        cycle.remove(cycle.size() - 1);
        return cycle;
    }

    /*najde lexikograficky predchadzajucu permutaciu daneho pola*/
    boolean prev_permutation(int[] arr) {
        int len = arr.length;
        int i = len - 1;

        // 1. find largest i where arr[i - 1] > arr[i]
        while (i > 0) {
            if (arr[i - 1] > arr[i]) break;
            i--;
        }

        if (i <= 0) return false;

        // 2. find largest j where arr[i - 1] > arr[j] and j >= i
        int j = len - 1;
        while (j >= i) {
            if (arr[i - 1] > arr[j]) break;
            j--;
        }

        // 3. swap elements between arr[i-1] and arr[j]
        swap(i - 1, j, arr);

        // 4. reverse elements from i to end of array
        len--;
        while (i < len) {
            swap(i, len, arr);
            len--;
            i++;
        }
        return true;
    }

    void swap(int x, int y, int[] arr) {
        int temp = arr[x];
        arr[x] = arr[y];
        arr[y] = temp;
    }

    boolean isCycleDissolvable(ArrayList<Integer> cycle) {
        if (cycle.size() % 2 == 0) {
            CycleDissolver dissolver = new CycleDissolver(graph, cycle, true);
            ArrayList<ArrayList<Integer>> resultGraph = dissolver.getDissolvedGraph();
            GraphShrinker shrinker = new GraphShrinker(resultGraph);
            ColorabilityTester tester1 = new ColorabilityTester(shrinker.graph);
            boolean G1Colorable = tester1.isColorable();
            dissolver = new CycleDissolver(graph, cycle, false);
            resultGraph = dissolver.getDissolvedGraph();
            shrinker = new GraphShrinker(resultGraph);
            ColorabilityTester tester2 = new ColorabilityTester(shrinker.graph);
            boolean G2Colorable = tester2.isColorable();
            if (!G1Colorable && !G2Colorable) {
                dissolvableCycle.addAll(cycle);
                return true;
            }
        }
        return false;
    }
}


